// Object.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _OBJECT_H_
#define _OBJECT_H_

#include <cassert>
#include <TypeInfo.h>
#include <string>
#include <tchar.h>

#ifdef _UNICODE
#define tstring	std::wstring
#else
#define tstring	std::string
#endif

namespace Manah {
	namespace Windows {
		class CDumpContext;
	}
}


// CObject class definition and implementation
/////////////////////////////////////////////////////////////////////////////

namespace Manah {

class CObject {
	// methods
protected:
#ifdef _DEBUG
	virtual void AssertValid() const {
		assert(this != 0);
	}
#else
	virtual void AssertValid() const {}
#endif /* _DEBUG */
};


// CDumpContext class definition
/////////////////////////////////////////////////////////////////////////////

namespace Windows {

class CDumpContext {
	// constructor
public:
	CDumpContext(const TCHAR* lpszFileName = 0) throw(std::exception);

	// data member
private:
	TCHAR		m_szFileName[MAX_PATH];	// error log

	// methods
public:
	void			Flush() throw();
	CDumpContext&	operator <<(const CObject* pObject) throw();
	CDumpContext&	operator <<(const CObject& obj) throw();
	CDumpContext&	operator <<(const char* lpszString) throw();
	CDumpContext&	operator <<(const wchar_t* lpszString) throw();
	CDumpContext&	operator <<(const void* p) throw();
	CDumpContext&	operator <<(unsigned long ul) throw();
	CDumpContext&	operator <<(int n) throw();
	CDumpContext&	operator <<(double d) throw();
	CDumpContext&	operator <<(float f) throw();
	CDumpContext&	operator <<(long l) throw();
	CDumpContext&	operator <<(unsigned int n) throw();
	CDumpContext&	operator <<(char ch) throw();
	CDumpContext&	operator <<(unsigned char ch) throw();
	CDumpContext&	operator <<(wchar_t ch) throw();
	CDumpContext&	operator <<(__int64 n) throw();
	CDumpContext&	operator <<(unsigned __int64 n) throw();
	CDumpContext&	operator <<(const std::string& str) throw();
	CDumpContext&	operator <<(const std::wstring& str) throw();
	void			HexDump(const TCHAR* lpszLine, BYTE* pb, int nBytes, int nWidth = 0x10) throw();
};

namespace {
	CDumpContext	dout;
}

#if !defined(_DEBUG)
inline CDumpContext::CDumpContext(const TCHAR* lpszFileName /* = 0 */) {}
inline void CDumpContext::Flush() {}
inline void CDumpContext::HexDump(const TCHAR* lpszLine, BYTE* pb, int nBytes, int nWidth /* = 16 */) {}
inline CDumpContext& CDumpContext::operator <<(const CObject* pObject) {return *this;}
inline CDumpContext& CDumpContext::operator <<(const CObject& obj) {return *this;}
inline CDumpContext& CDumpContext::operator <<(const char* lpszString) {return *this;}
inline CDumpContext& CDumpContext::operator <<(const wchar_t* lpszString) {return *this;}
inline CDumpContext& CDumpContext::operator <<(const void* p) {return *this;}
inline CDumpContext& CDumpContext::operator <<(unsigned long n) {return *this;}
inline CDumpContext& CDumpContext::operator <<(int n) {return *this;}
inline CDumpContext& CDumpContext::operator <<(double d) {return *this;}
inline CDumpContext& CDumpContext::operator <<(float f) {return *this;}
inline CDumpContext& CDumpContext::operator <<(long l) {return *this;}
inline CDumpContext& CDumpContext::operator <<(unsigned int n) {return *this;}
inline CDumpContext& CDumpContext::operator <<(char ch) {return *this;}
inline CDumpContext& CDumpContext::operator <<(unsigned char ch) {return *this;}
inline CDumpContext& CDumpContext::operator <<(wchar_t ch) {return *this;}
inline CDumpContext& CDumpContext::operator <<(__int64 n) {return *this;}
inline CDumpContext& CDumpContext::operator <<(unsigned __int64 n) {return *this;}
inline CDumpContext& CDumpContext::operator <<(const std::string& str) {return *this;}
inline CDumpContext& CDumpContext::operator <<(const std::wstring& str) {return *this;}
#endif

} /* namespace Windows */
} /* namespace Manah */


// modern types
/////////////////////////////////////////////////////////////////////////////

typedef unsigned char		uchar;
typedef unsigned short		ushort;
typedef unsigned int		uint;
typedef unsigned long		ulong;
typedef unsigned __int64	ulonglong;


// Macros
/////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG
#define ASSERT(exp)	assert(exp)
#define VERIFY(exp)	assert(exp)
#else
#define ASSERT(exp)
#define VERIFY(exp)	(exp)
#endif /* _DEBUG */

#define toBoolean(exp)				((exp) != 0)

#ifndef LVS_EX_LABELTIP	// VC6
	#define LVS_EX_LABELTIP	0x4000
#endif
#ifndef ListView_SetCheckState	// VC6
	#define ListView_SetCheckState(hwndLV, i, fCheck)	\
	ListView_SetItemState(hwndLV, i,					\
	INDEXTOSTATEIMAGEMASK((fCheck)? 2 : 1), LVIS_STATEIMAGEMASK)
#endif


// CDumpContext class implementation
/////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG
using Manah::CObject;
using Manah::Windows::CDumpContext;

inline CDumpContext::CDumpContext(const TCHAR* lpszFileName /* = 0 */) throw(std::exception) {
	if(lpszFileName == 0)
		*m_szFileName = 0;
	else
		throw std::exception("File log is not supported!");
		// _tcscpy(m_szFileName, lpszFileName);
}

inline void CDumpContext::Flush() {
	/* not implemented */
}

inline void CDumpContext::HexDump(const TCHAR* lpszLine, BYTE* pb, int nBytes, int nWidth /* = 0x10 */) {
	ASSERT(this != 0);

	TCHAR*	pszOutput;	// all line
	TCHAR	szByte[4];

	pszOutput = new TCHAR[static_cast<size_t>((_tcslen(lpszLine) + 3 * nWidth + 2) * static_cast<float>(nBytes / nWidth))];
	_tcscpy(pszOutput, lpszLine);
	for(int i = 0; i < nBytes; i++){
		wsprintf(szByte, _T(" %d"), pb);
		_tcscat(pszOutput, szByte);
		if(i % nWidth == 0){
			_tcscat(pszOutput, _T("\n"));
			_tcscat(pszOutput, lpszLine);
		}
	}
	::OutputDebugString(_T("\n>----Dump is started"));
	::OutputDebugString(pszOutput);
	::OutputDebugString(_T("\n>----Dump is done"));
	delete[] pszOutput;
}

inline CDumpContext& CDumpContext::operator <<(const CObject* pObject) throw() {
	ASSERT(this != 0);
	return *this << reinterpret_cast<long>(pObject);
}

inline CDumpContext& CDumpContext::operator <<(const CObject& obj) throw() {
	ASSERT(this != 0);
	return *this << &obj;
}

inline CDumpContext& CDumpContext::operator <<(const char* lpszString) throw() {
	ASSERT(this != 0);

	if(lpszString == 0)
		::OutputDebugStringA("(NULL)");
	else
		::OutputDebugStringA(lpszString);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(const wchar_t* lpszString) throw() {
	ASSERT(this != 0);

	if(lpszString == 0)
		::OutputDebugStringW(L"(NULL)");
	else
		::OutputDebugStringW(lpszString);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(const void* p) throw() {
	ASSERT(this != 0);

	TCHAR	szOutput[33];

	if(p == NULL)
		::OutputDebugString(_T("(NULL)"));
	else{
		wsprintf(szOutput, _T("$%lX"), reinterpret_cast<long>(p));
		::OutputDebugString(szOutput);
	}

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(char ch) throw() {
	ASSERT(this != 0);

	char	sz[] = {ch, 0};
	::OutputDebugStringA(sz);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(unsigned char ch) throw() {
	ASSERT(this != 0);

	char	sz[] = {ch, 0};
	::OutputDebugStringA(sz);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(wchar_t ch) throw() {
	ASSERT(this != 0);

	wchar_t	wsz[] = {ch, 0};
	::OutputDebugStringW(wsz);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(int n) throw() {
	ASSERT(this != 0);

	TCHAR	szOutput[33];

	wsprintf(szOutput, _T("%d"), n);
	::OutputDebugString(szOutput);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(float f) throw() {
	ASSERT(this != 0);

	TCHAR	szOutput[33];

	wsprintf(szOutput, _T("%f"), f);
	::OutputDebugString(szOutput);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(double d) throw() {
	ASSERT(this != 0);

	TCHAR	szOutput[33];

	wsprintf(szOutput, _T("%lf"), d);
	::OutputDebugString(szOutput);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(long l) throw() {
	ASSERT(this != 0);

	TCHAR	szOutput[33];

	wsprintf(szOutput, _T("%ld"), l);
	::OutputDebugString(szOutput);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(unsigned long l) throw() {
	ASSERT(this != 0);

	TCHAR	szOutput[33];

	wsprintf(szOutput, _T("%lu"), l);
	::OutputDebugString(szOutput);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(unsigned int n) throw() {
	ASSERT(this != 0);

	TCHAR	szOutput[33];

	wsprintf(szOutput, _T("0x%X"), n);
	::OutputDebugString(szOutput);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(__int64 n) throw() {
	ASSERT(this != 0);

	TCHAR	szOutput[65];

	wsprintf(szOutput, _T("%l64d"), n);
	::OutputDebugString(szOutput);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(unsigned __int64 n) throw() {
	ASSERT(this != 0);

	TCHAR	szOutput[65];

	wsprintf(szOutput, _T("%l64u"), n);
	::OutputDebugString(szOutput);

	return *this;
}

inline CDumpContext& CDumpContext::operator <<(const std::string& str) throw() {
	ASSERT(this != 0);
	::OutputDebugStringA(str.c_str());
	return *this;
}

inline CDumpContext& CDumpContext::operator <<(const std::wstring& str) throw() {
	ASSERT(this != 0);
	::OutputDebugStringW(str.c_str());
	return *this;
}

#endif /* _DEBUG */

#endif /* _MANAH_OBJECT_H_ */

/* [EOF] */