// win_utils.h
// (c) 2003 exeal

#ifndef _WIN_UTILS_H_
#define _WIN_UTILS_H_

#include "Object.h"
#include "DC.h"


namespace Manah {
namespace Windows {

// this header contains following classes:
class CPoint;
class CSize;
class CRect;
class CClipboard;


// CPoint class definition
/////////////////////////////////////////////////////////////////////////////

class CPoint : public tagPOINT {
	// constructors
public:
	CPoint();
	CPoint(int xInit, int yInit);
	CPoint(const POINT& pt);
	CPoint(const SIZE& size);
	CPoint(DWORD dwPoint);

	// operators
public:
	bool	operator ==(const POINT& pt) const;
	bool	operator !=(const POINT& size) const;
	CPoint&	operator +=(const POINT& rhs);
	CPoint&	operator +=(const SIZE& rhs);
	CPoint&	operator -=(const POINT& rhs);
	CPoint&	operator -=(const SIZE& rhs);
	CPoint	operator +(const POINT& pt) const;
	CPoint	operator +(const SIZE& size) const;
	CRect	operator +(const CRect& rect) const;
	CPoint	operator -() const;
	CPoint	operator -(const POINT& pt) const;
	CPoint	operator -(const SIZE& size) const;
	CRect	operator -(const RECT& rect) const;

	// methods
public:
	void	Offset(int xOffset, int yOffset);
	void	Offset(const POINT& pt);
	void	Offset(const SIZE& size);
};


// CSize class definition
/////////////////////////////////////////////////////////////////////////////

class CSize : public tagSIZE {
	// constructors
public:
	CSize();
	CSize(int xInit, int yInit);
	CSize(const SIZE& size);
	CSize(const POINT& pt);
	CSize(DWORD dwSize);

	// operators
public:
	bool	operator ==(const SIZE& size) const;
	bool	operator !=(const SIZE& size) const;
	CSize&	operator +=(const SIZE& rhs);
	CSize&	operator -=(const SIZE& rhs);
	CSize	operator +(const SIZE& size) const;
	CPoint	operator +(const POINT& pt) const;
	CRect	operator +(const RECT& rect) const;
	CSize	operator -() const;
	CSize	operator -(const SIZE& size) const;
	CPoint	operator -(const POINT& pt) const;
	CRect	operator -(const RECT& rect) const;
};


// CRect class definition
/////////////////////////////////////////////////////////////////////////////

class CRect : public tagRECT {
	// constructors
public:
	CRect();
	CRect(int leftInit, int topInit, int rightInit, int bottomInit);
	CRect(const RECT& rect);
	CRect(const RECT* lpRect);
	CRect(const POINT& pt, const SIZE& size);
	CRect(const POINT& ptLeftTop, const POINT& ptRightBottom);

	// operators
public:
			operator const RECT*() const;
			operator RECT*() const;
	CRect&	operator =(const RECT& rhs);
	bool	operator ==(const RECT& rhs) const;
	bool	operator !=(const RECT& rhs) const;
	CRect&	operator +=(const POINT& rhs);
	CRect&	operator +=(const SIZE& rhs);
	CRect&	operator +=(const RECT& rhs);
	CRect&	operator -=(const POINT& rhs);
	CRect&	operator -=(const SIZE& rhs);
	CRect&	operator -=(const RECT& rhs);
	CRect&	operator &=(const RECT& rhs);
	CRect&	operator |=(const RECT& rhs);
	CRect	operator +(const POINT& pt) const;
	CRect	operator +(const SIZE& size) const;
	CRect	operator +(const RECT& rect) const;
	CRect	operator -(const POINT& pt) const;
	CRect	operator -(const SIZE& size) const;
	CRect	operator -(const RECT& rect) const;
	CRect	operator &(const RECT& rect) const;
	CRect	operator |(const RECT& rect) const;

	// methods
public:
	int		Width() const;
	int		Height() const;
	CSize	Size() const;
	CPoint	TopLeft() const;
	CPoint	BottomRight() const;
	CPoint	CenterPoint() const;
	bool	IsRectEmpty() const;
	bool	IsRectNull() const;
	bool	PtInRect(const POINT& pt) const;
	void	SetRect(int left, int top, int right, int bottom);
	void	SetRectEmpty();
	void	CopyRect(const RECT& rect);
	bool	EqualRect(const RECT& rect) const;
	void	InflateRect(int x, int y);
	void	InflateRect(const SIZE& size);
	void	InflateRect(const RECT& rect);
	void	InflateRect(int left, int top, int right, int bottom);
	void	DeflateRect(int x, int y);
	void	DeflateRect(const SIZE& size);
	void	DeflateRect(const RECT& rect);
	void	DeflateRect(int left, int top, int right, int bottom);
	void	NormalizeRect();
	void	OffsetRect(int x, int y);
	void	OffsetRect(const POINT& pt);
	void	OffsetRect(const SIZE& size);
	bool	SubtractRect(const RECT& rect1, const RECT& rect2);
	bool	IntersectRect(const RECT& rect1, const RECT& rect2);
	bool	UnionRect(const RECT& rect1, const RECT& rect2);
};


// CClipboard class definition
/////////////////////////////////////////////////////////////////////////////

class CClipboard {
	// methods
public:
	/* open and read the clipboard text */
	static DWORD	GetClipboardTextSize(HWND hWnd);	// return SIZE (byte length) of clipboard text
	static bool		ReadClipboardText(					// read multi-bytes text from the clipboard
		HWND hWnd, char* pszText, size_t cch);			//  as format CF_TEXT
	static bool		ReadClipboardText(					// read wide-characters from the clipboard
		HWND hWnd, wchar_t* pszText, size_t cch);		//  as format CF_UNICODETEXT

	/* open and set the clipboard with CF_TEXT format text */
	static bool	SetClipboardText(		// set multi-bytes text to the clipboard
		HWND hWnd, const char* pszText, size_t cch = -1);
	static bool	SetClipboardText(		// set wide-characters to the clipboard
		HWND hWnd, const wchar_t* pszText, size_t cch = -1);

	/* basic clipboard operations (getter will return permanent object) */
	static void*	GetClipboardData(HWND hWnd,
						UINT nFormat, DWORD* pdwSize) throw(invalid_argument);
	static bool		SetClipboardData(HWND hWnd,
						UINT nFormat, const void* pData, DWORD dwSize) throw(invalid_argument);
	static void*	GetFirstAvailableClipboardData(HWND hWnd,
						UINT* pnFormat, DWORD* pdwSize) throw(invalid_argument);
};


// CPoint class implementation
/////////////////////////////////////////////////////////////////////////////

inline CPoint::CPoint() {
}

inline CPoint::CPoint(int xInit, int yInit) {
	x = xInit;
	y = yInit;
}

inline CPoint::CPoint(const POINT& pt) {
	x = pt.x;
	y = pt.y;
}

inline CPoint::CPoint(const SIZE& size) {
	x = size.cx;
	y = size.cy;
}

inline CPoint::CPoint(DWORD dwPoint) {
	x = LOWORD(dwPoint);
	y = HIWORD(dwPoint);
}

inline bool CPoint::operator !=(const POINT& pt) const {
	return (x != pt.x || y != pt.y);
}

inline CPoint CPoint::operator +(const POINT& pt) const {
	return CPoint(*this) += pt;
}

inline CPoint CPoint::operator +(const SIZE& size) const {
	return CPoint(*this) + size;
}

inline CRect CPoint::operator +(const CRect& rect) const {
	return CRect(rect.left + x, rect.top + y, rect.right + x, rect.bottom + y);
}

inline CPoint& CPoint::operator +=(const POINT& rhs) {
	x += rhs.x;
	y += rhs.y;
	return *this;
}

inline CPoint& CPoint::operator +=(const SIZE& rhs) {
	x += rhs.cx;
	y += rhs.cy;
	return *this;
}

inline CPoint CPoint::operator -() const {
	return CPoint(-x, -y);
}

inline CPoint CPoint::operator -(const POINT& pt) const {
	return CPoint(*this) -= pt;
}

inline CPoint CPoint::operator -(const SIZE& size) const {
	return CPoint(*this) -= size;
}

inline CRect CPoint::operator -(const RECT& rect) const {
	return CRect(rect.left - x, rect.top - y, rect.right - x, rect.bottom - y);
}

inline CPoint& CPoint::operator -=(const POINT& rhs) {
	x -= rhs.x;
	y -= rhs.y;
	return *this;
}

inline CPoint& CPoint::operator -=(const SIZE& rhs) {
	x -= rhs.cx;
	y -= rhs.cy;
	return *this;
}

inline bool CPoint::operator ==(const POINT& pt) const {
	return (x == pt.x && y == pt.y);
}

inline void CPoint::Offset(int xOffset, int yOffset) {
	x += xOffset;
	y += yOffset;
}

inline void CPoint::Offset(const POINT& pt) {
	*this += pt;
}

inline void CPoint::Offset(const SIZE& size) {
	*this += size;
}


// CSize class implementation
/////////////////////////////////////////////////////////////////////////////

inline CSize::CSize() {
}

inline CSize::CSize(int cxInit, int cyInit) {
	cx = cxInit;
	cy = cyInit;
}

inline CSize::CSize(const SIZE& size) {
	cx = size.cx;
	cy = size.cy;
}

inline CSize::CSize(const POINT& pt) {
	cx = pt.x;
	cy = pt.y;
}

inline CSize::CSize(DWORD dwSize) {
	cx = LOWORD(dwSize);
	cy = HIWORD(dwSize);
}

inline bool CSize::operator !=(const SIZE& size) const {
	return (cx != size.cx || cy != size.cy);
}

inline CSize CSize::operator +(const SIZE& size) const {
	return CSize(*this) += size;
}

inline CPoint CSize::operator +(const POINT& pt) const {
	return CSize(pt.x + cx, pt.y + cy);
}

inline CRect CSize::operator +(const RECT& rect) const {
	return CRect(rect.left + cx, rect.top + cy, rect.right + cx, rect.bottom + cy);
}

inline CSize& CSize::operator +=(const SIZE& rhs) {
	cx += rhs.cx;
	cy += rhs.cy;
	return *this;
}

inline CSize CSize::operator -() const {
	return CSize(-cx, -cy);
}

inline CSize CSize::operator -(const SIZE& size) const {
	return CSize(*this) -= size;
}

inline CPoint CSize::operator -(const POINT& pt) const {
	return CSize(pt.x - cx, pt.y - cy);
}

inline CRect CSize::operator -(const RECT& rect) const {
	return CRect(rect.left - cx, rect.top - cy, rect.right - cx, rect.bottom - cy);
}

inline CSize& CSize::operator -=(const SIZE& rhs) {
	cx -= rhs.cx;
	cy -= rhs.cy;
	return *this;
}

inline bool CSize::operator ==(const SIZE& size) const {
	return (cx == size.cx && cy == size.cy);
}


// CRect class implementation
/////////////////////////////////////////////////////////////////////////////

inline CRect::CRect() {
}

inline CRect::CRect(int leftInit, int topInit, int rightInit, int bottomInit) {
	left = leftInit;
	top = topInit;
	right = rightInit;
	bottom = bottomInit;
}

inline CRect::CRect(const RECT& rect) {
	left = rect.left;
	top = rect.top;
	right = rect.right;
	bottom = rect.bottom;
}

inline CRect::CRect(const RECT* lpRect) {
	left = lpRect->left;
	top = lpRect->top;
	right = lpRect->right;
	bottom = lpRect->bottom;
}

inline CRect::CRect(const POINT& pt, const SIZE& size) {
	left = pt.x;
	top = pt.y;
	right = pt.x + size.cx;
	bottom = pt.y + size.cy;
}

inline CRect::CRect(const POINT& ptLeftTop, const POINT& ptRightBottom) {
	left = ptLeftTop.x;
	top = ptLeftTop.y;
	right = ptRightBottom.x;
	bottom = ptRightBottom.y;
}

inline bool CRect::operator !=(const RECT& rhs) const {
	return !EqualRect(rhs);
}

inline CRect CRect::operator &(const RECT& rect) const {
	return CRect(*this) &= rect;
}

inline CRect& CRect::operator &=(const RECT& rhs) {
	IntersectRect(*this, rhs);
	return *this;
}

inline CRect CRect::operator +(const POINT& pt) const {
	return CRect(*this) += pt;
}

inline CRect CRect::operator +(const SIZE& size) const {
	return CRect(*this) += size;
}

inline CRect CRect::operator +(const RECT& rect) const {
	return CRect(*this) += rect;
}

inline CRect& CRect::operator +=(const POINT& rhs) {
	OffsetRect(rhs);
	return *this;
}

inline CRect& CRect::operator +=(const SIZE& rhs) {
	InflateRect(rhs);
	return *this;
}

inline CRect& CRect::operator +=(const RECT& rhs) {
	left += rhs.left;
	top += rhs.top;
	right += rhs.right;
	bottom += rhs.bottom;
	return *this;
}

inline CRect CRect::operator -(const POINT& pt) const {
	return CRect(*this) -= pt;
}

inline CRect CRect::operator -(const SIZE& size) const {
	return CRect(*this) -= size;
}

inline CRect CRect::operator -(const RECT& rect) const {
	return CRect(*this) -= rect;
}

inline CRect& CRect::operator -=(const POINT& rhs) {
	OffsetRect(-rhs.x, -rhs.y);
	return *this;
}

inline CRect& CRect::operator -=(const SIZE& rhs) {
	DeflateRect(rhs);
	return *this;
}

inline CRect& CRect::operator -=(const RECT& rhs) {
	left -= rhs.left;
	top -= rhs.top;
	right -= rhs.right;
	bottom -= rhs.bottom;
	return *this;
}

inline CRect& CRect::operator =(const RECT& rhs) {
	SetRect(rhs.left, rhs.top, rhs.right, rhs.bottom);
}

inline bool CRect::operator ==(const RECT& rhs) const {
	return EqualRect(rhs);
}

inline CRect::operator const RECT *() const {
	return this;
}

inline CRect::operator RECT *() const {
	return const_cast<CRect*>(this);
}

inline CRect CRect::operator |(const RECT& rect) const {
	return CRect(*this) |= rect;
}

inline CRect& CRect::operator |=(const RECT& rhs) {
	UnionRect(*this, rhs);
	return *this;
}

inline CPoint CRect::BottomRight() const {
	return CPoint(right, bottom);
}

inline CPoint CRect::CenterPoint() const {
	return CPoint((right - left) / 2, (bottom - top) / 2);
}

inline void CRect::CopyRect(const RECT& rect) {
	*this = rect;
}

inline void CRect::DeflateRect(int x, int y) {
	InflateRect(-x, -y);
}

inline void CRect::DeflateRect(const SIZE& size) {
	InflateRect(-size.cx, -size.cy);
}

inline void CRect::DeflateRect(const RECT& rect) {
	SetRect(left + rect.left, top + rect.top, right - rect.right, bottom - rect.bottom);
}

inline void CRect::DeflateRect(int left_, int top_, int right_, int bottom_) {
	SetRect(left + left_, top + top_, right - right_, bottom - bottom_);
}

inline bool CRect::EqualRect(const RECT& rect) const {
	return toBoolean(::EqualRect(*this, &rect));
}

inline int CRect::Height() const {
	return bottom - top;
}

inline void CRect::InflateRect(int x, int y) {
	::InflateRect(this, x, y);
}

inline void CRect::InflateRect(const SIZE& size) {
	::InflateRect(this, size.cx, size.cy);
}

inline void CRect::InflateRect(const RECT& rect) {
	SetRect(left - rect.left, top - rect.top, right + rect.right, bottom + rect.bottom);
}

inline void CRect::InflateRect(int left_, int top_, int right_, int bottom_) {
	SetRect(left - left_, top - top_, right + right_, bottom + bottom_);
}

inline bool CRect::IntersectRect(const RECT& rect1, const RECT& rect2) {
	return toBoolean(::IntersectRect(*this, &rect1, &rect2));
}

inline bool CRect::IsRectEmpty() const {
	return toBoolean(::IsRectEmpty(this));
}

inline bool CRect::IsRectNull() const {
	return toBoolean(left == 0 && top == 0 && right == 0 && bottom == 0);
}

inline void CRect::NormalizeRect() {
	if(top > bottom)
		swap(top, bottom);
	if(left > right)
		swap(left, right);
}

inline void CRect::OffsetRect(int x, int y) {
	::OffsetRect(*this, x, y);
}

inline void CRect::OffsetRect(const POINT& pt) {
	::OffsetRect(*this, pt.x, pt.y);
}

inline void CRect::OffsetRect(const SIZE& size) {
	::OffsetRect(*this, size.cx, size.cy);
}

inline bool CRect::PtInRect(const POINT& pt) const {
	return toBoolean(::PtInRect(*this, pt));
}

inline void CRect::SetRect(int left, int top, int right, int bottom) {
	::SetRect(*this, left, top, right, bottom);
}

inline void CRect::SetRectEmpty() {
	::SetRectEmpty(*this);
}

inline CSize CRect::Size() const {
	return CSize(right - left, bottom - top);
}

inline bool CRect::SubtractRect(const RECT& rect1, const RECT& rect2) {
	return toBoolean(::SubtractRect(this, &rect1, &rect2));
}

inline CPoint CRect::TopLeft() const {
	return CPoint(left, top);
}

inline bool CRect::UnionRect(const RECT& rect1, const RECT& rect2) {
	return toBoolean(::UnionRect(this, &rect1, &rect2));
}

inline int CRect::Width() const {
	return right - left;
}


// CClipboard class implementation
/////////////////////////////////////////////////////////////////////////////

inline void* CClipboard::GetClipboardData(HWND hWnd, UINT nFormat, DWORD* pdwSize) throw(invalid_argument) { 
	void*	pData = 0;
	DWORD	dwSize;
	HGLOBAL	hGlobal = 0;

	if(pdwSize == 0)
		throw invalid_argument("Third argument can not be null!");
	if(!::IsClipboardFormatAvailable(nFormat) || !::OpenClipboard(hWnd))
		return 0;
	hGlobal = ::GetClipboardData(nFormat);
	if(hGlobal == 0)
		return ::CloseClipboard(), 0;
	dwSize = ::GlobalSize(hGlobal);
	pData = malloc(dwSize);
	if(pData == 0)
		return ::CloseClipboard, ::GlobalUnlock(hGlobal), 0;
	memcpy(pData, GlobalLock(hGlobal), dwSize);
	::GlobalUnlock(hGlobal);
	::CloseClipboard();

	*pdwSize = dwSize;

	return pData;
}

inline DWORD CClipboard::GetClipboardTextSize(HWND hWnd) {
	LONG	dwSize;
	HGLOBAL	hGlobal;

	if(!::IsClipboardFormatAvailable(CF_TEXT))
		return -1;

	::OpenClipboard(hWnd);
	hGlobal = ::GetClipboardData(CF_TEXT);
	if(hGlobal == 0){
		::CloseClipboard();
		return -1;
	}
	dwSize = ::GlobalSize(hGlobal);
	::GlobalUnlock(hGlobal);
	::CloseClipboard();

	return dwSize;
}

inline void* CClipboard::GetFirstAvailableClipboardData(
		HWND hWnd, UINT* pnFormat, DWORD* pdwSize) throw(invalid_argument) {
	UINT	nFormat;
	void*	pData = 0;

	if(pnFormat == 0 || pdwSize == 0)
		throw invalid_argument("Second and third argument can not be null!");
	if(!::OpenClipboard(hWnd))
		return 0;
	nFormat = ::EnumClipboardFormats(0);
	if(nFormat == 0)
		return 0;
	pData = CClipboard::GetClipboardData(hWnd, nFormat, pdwSize);
	*pnFormat = (pData != 0) ? nFormat : 0;
	return pData;
}

inline bool CClipboard::ReadClipboardText(HWND hWnd, char* pszText, size_t cch) {
	HGLOBAL	hGlobal;
	char*	pszString;

	if(!::IsClipboardFormatAvailable(CF_TEXT))
		return false;

	::OpenClipboard(hWnd);
	hGlobal = ::GetClipboardData(CF_TEXT);
	if(hGlobal == 0)
		return ::CloseClipboard(), false;
	pszString = static_cast<char*>(::GlobalLock(hGlobal));
	strncpy(pszText, pszString, cch);
	::GlobalUnlock(hGlobal);
	::CloseClipboard();

	return true;
}

inline bool CClipboard::ReadClipboardText(HWND hWnd, wchar_t* pszText, size_t cch) {
	HGLOBAL		hGlobal;
	char*		psz = 0;
	wchar_t*	pwsz = 0;

	if(::IsClipboardFormatAvailable(CF_UNICODETEXT)) {
		::OpenClipboard(hWnd);
		hGlobal = ::GetClipboardData(CF_UNICODETEXT);
		if(hGlobal == 0)
			return ::CloseClipboard(), false;
		pwsz = static_cast<wchar_t*>(::GlobalLock(hGlobal));
		wcsncpy(pszText, pwsz, cch);
	} else if(::IsClipboardFormatAvailable(CF_TEXT)) {
		::OpenClipboard(hWnd);
		hGlobal = ::GetClipboardData(CF_TEXT);
		if(hGlobal == 0)
			return ::CloseClipboard(), false;
		psz = static_cast<char*>(::GlobalLock(hGlobal));
		::MultiByteToWideChar(CP_ACP, 0, psz, cch * 2, pszText, cch);
	} else
		return false;
	::GlobalUnlock(hGlobal);
	::CloseClipboard();

	return true;
}

inline bool CClipboard::SetClipboardData(
		HWND hWnd, UINT nFormat, const void* pData, DWORD dwSize) throw(invalid_argument) {
	HGLOBAL	hGlobal = 0;
	void*	pTempData = 0;
	bool	bReturn;

	if(pData == 0)
		throw invalid_argument("Third argument can not be null!");
	if(!::OpenClipboard(hWnd))
		return 0;
	::EmptyClipboard();
	hGlobal = ::GlobalAlloc(GHND, dwSize);
	if(hGlobal == 0)
		return ::CloseClipboard(), false;
	pTempData = ::GlobalLock(hGlobal);
	memcpy(pTempData, pData, dwSize);
	::GlobalUnlock(hGlobal);
	bReturn = (::SetClipboardData(nFormat, hGlobal) != 0);
	::CloseClipboard();

	return bReturn;
}

inline bool CClipboard::SetClipboardText(HWND hWnd, const char* pszText, size_t cch /* = -1 */) {
	HGLOBAL	hGlobal;
	char*	pszString;

	if(cch == -1)
		cch = strlen(pszText);
	hGlobal = ::GlobalAlloc(GHND, cch + 1);
	if(hGlobal == 0)
		return false;
	pszString = static_cast<char*>(::GlobalLock(hGlobal));
	strncpy(pszString, pszText, cch);
	::GlobalUnlock(hGlobal);
	if(!::OpenClipboard(hWnd)){
		::GlobalFree(hGlobal);
		return false;
	}
	::EmptyClipboard();
	::SetClipboardData(CF_TEXT, hGlobal);
	::CloseClipboard();

	return true;
}

inline bool CClipboard::SetClipboardText(HWND hWnd, const wchar_t* pszText, size_t cch /* = -1 */) {
	HGLOBAL		hGlobal;
	wchar_t*	pwszString;

	if(cch == -1)
		cch = wcslen(pszText);
	hGlobal = ::GlobalAlloc(GHND, (cch + 1) * 2);
	if(hGlobal == 0)
		return false;
	pwszString = static_cast<wchar_t*>(::GlobalLock(hGlobal));
	wcsncpy(pwszString, pszText, cch);
	::GlobalUnlock(hGlobal);
	if(!::OpenClipboard(hWnd)){
		::GlobalFree(hGlobal);
		return false;
	}
	::EmptyClipboard();
	::SetClipboardData(CF_UNICODETEXT, hGlobal);
	::CloseClipboard();

	return true;
}

} /* namespace Windows */
} /* namespace Manah */

#endif /* _WIN_UTILS_H_ */

/* [EOF] */