// TabCtrl.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _TAB_CTRL_H_
#define _TAB_CTRL_H_
#include "Window.h"
#include "ImageList.h"
#include "ToolTipCtrl.h"


// CTabCtrl class definition
/////////////////////////////////////////////////////////////////////////////

namespace Manah {
namespace Windows {
namespace Controls {

class CTabCtrl : public CWindow {
	// \bh
public:
	/* 쐬 */
	bool	Attach(HWND hWnd, bool bSubClass = false);
	bool	Create(bool bSubClass, DWORD dwStyle, const RECT* lpRect, HWND hwndParent, UINT id);
	bool	CreateEx(bool bSubClass, DWORD dwExStyle, DWORD dwStyle, const RECT* lpRect, HWND hwndParent, UINT id);

	/*  */
	CImageList*	GetImageList() const;
	CImageList*	SetImageList(const CImageList* pImageList);
	int			GetItemCount() const;
	bool		GetItem(int iItem, TCITEM* pTabCtrlItem) const;
	bool		SetItem(int iItem, const TCITEM* pTabCtrlItem);
	bool		SetItemExtra(int nBytes);
	bool		GetItemRect(int iItem, LPRECT lpRect) const;
	int			GetCurSel() const;
	int			SetCurFocus(int iItem);
	int			SetCurSel(int iItem);
	SIZE		SetItemSize(SIZE size);
	void		SetPadding(SIZE size);
	int			GetRowCount() const;
	HWND		GetToolTips() const;
	void		SetToolTips(HWND hwndTip);
	int			GetCurFocus() const;
	int			SetMinTabWidth(int cx);
	DWORD		GetExtendedStyle() const;
	DWORD		SetExtendedStyle(DWORD dwNewStyle, DWORD dwExMask = 0);
	bool		GetItemState(int iItem, DWORD dwMask, DWORD& dwState) const;
	bool		SetItemState(int iItem, DWORD dwMask, DWORD dwState);

	/*  */
	bool	InsertItem(int iItem, const TCITEM* pItem);
	bool	InsertItem(int iItem, const TCHAR* lpszItem);
	bool	InsertItem(int iItem, const TCHAR* lpszItem, int iImage);
	bool	InsertItem(UINT nMask, int iItem, const TCHAR* lpszItem, int iImage, LPARAM lParam);
	bool	DeleteItem(int iItem);
	bool	DeleteAllItems();
	void	AdjustRect(bool bLarger, LPRECT lpRect);
	void	RemoveImage(int iImage);
	int		HitTest(TCHITTESTINFO* pHitTestInfo) const;
	void	DeselectAll(bool bExcludeFocus);
	bool	HighlightItem(int iItem, bool bHighlight = true);

protected:
	virtual void	DrawItem(LPDRAWITEMSTRUCT lpDrawItemStruct);
	static LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);

};

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */


// CTabCtrl class implementation
/////////////////////////////////////////////////////////////////////////////

using Manah::Windows::Controls::CTabCtrl;
using Manah::Windows::CImageList;

inline void CTabCtrl::AdjustRect(bool bLarger, LPRECT lpRect) {
	AssertValidAsWindow();
	TabCtrl_AdjustRect(m_hWnd, bLarger, lpRect);
}

inline bool CTabCtrl::Attach(HWND hWnd, bool bSubClass /* = false */) {
	AssertValid();

	if(IsWindow() || !::IsWindow(hWnd) || m_OldProc != 0)
		return false;
	m_hWnd = hWnd;

	if(bSubClass) {
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(CTabCtrl::WndProc)));
		assert(m_OldProc != CTabCtrl::WndProc);
		m_lOldUserData = GetWindowLong(GWL_USERDATA);
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline bool CTabCtrl::Create(bool bSubClass, DWORD dwStyle, const RECT* lpRect, HWND hwndParent, UINT id) {
	AssertValid();

	m_hWnd = ::CreateWindow(WC_TABCONTROL, _T(""),
		dwStyle, lpRect->left, lpRect->top, lpRect->right - lpRect->left, lpRect->bottom - lpRect->top,
		hwndParent, reinterpret_cast<HMENU>(id), reinterpret_cast<HINSTANCE>(::GetModuleHandle(0)), 0L);
	if(m_hWnd == 0)
		return false;
	if(bSubClass){
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(CTabCtrl::WndProc)));
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline bool CTabCtrl::CreateEx(bool bSubClass, DWORD dwExStyle, DWORD dwStyle, const RECT* lpRect, HWND hwndParent, UINT id) {
	AssertValid();

	m_hWnd = ::CreateWindowEx(dwExStyle, WC_TABCONTROL, _T(""),
		dwStyle, lpRect->left, lpRect->top, lpRect->right - lpRect->left, lpRect->bottom - lpRect->top,
		hwndParent, reinterpret_cast<HMENU>(id), reinterpret_cast<HINSTANCE>(::GetModuleHandle(0)), 0L);
	if(m_hWnd == 0)
		return false;
	if(bSubClass){
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(CTabCtrl::WndProc)));
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline bool CTabCtrl::DeleteAllItems() {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_DeleteAllItems(m_hWnd));
}

inline void CTabCtrl::DeselectAll(bool bExcludeFocus) {
	AssertValidAsWindow();
	TabCtrl_DeselectAll(m_hWnd, bExcludeFocus);
}

inline bool CTabCtrl::DeleteItem(int iItem) {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_DeleteItem(m_hWnd, iItem));
}

inline void CTabCtrl::DrawItem(LPDRAWITEMSTRUCT lpDrawItemStruct) {
}

inline int CTabCtrl::GetCurFocus() const {
	AssertValidAsWindow();
	return TabCtrl_GetCurFocus(m_hWnd);
}

inline int CTabCtrl::GetCurSel() const {
	AssertValidAsWindow();
	return TabCtrl_GetCurSel(m_hWnd);
}

inline DWORD CTabCtrl::GetExtendedStyle() const {
	AssertValidAsWindow();
	return TabCtrl_GetExtendedStyle(m_hWnd);
}

inline CImageList* CTabCtrl::GetImageList() const {
	AssertValidAsWindow();
	return CImageList::FromHandle(TabCtrl_GetImageList(m_hWnd));
}

inline bool CTabCtrl::GetItem(int iItem, TCITEM* pTabCtrlItem) const {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_GetItem(m_hWnd, iItem, pTabCtrlItem));
}

inline int CTabCtrl::GetItemCount() const {
	AssertValidAsWindow();
	return TabCtrl_GetItemCount(m_hWnd);
}

inline bool CTabCtrl::GetItemRect(int iItem, LPRECT lpRect) const {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_GetItemRect(m_hWnd, iItem, lpRect));
}

inline bool CTabCtrl::GetItemState(int iItem, DWORD dwMask, DWORD& dwState) const {
	AssertValidAsWindow();

	TCITEM	item;
	item.mask = TCIF_STATE;
	item.dwStateMask = dwMask;
	if(!GetItem(iItem, &item))
		return false;
	dwState = item.dwState;
	return true;
}

inline int CTabCtrl::GetRowCount() const {
	AssertValidAsWindow();
	return TabCtrl_GetRowCount(m_hWnd);
}

inline HWND CTabCtrl::GetToolTips() const {
	AssertValidAsWindow();
	return TabCtrl_GetToolTips(m_hWnd);
}

inline bool CTabCtrl::HighlightItem(int iItem, bool bHighlight /* = true */) {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_HighlightItem(m_hWnd, iItem, bHighlight));
}

inline int CTabCtrl::HitTest(TCHITTESTINFO* pHitTestInfo) const {
	AssertValidAsWindow();
	return TabCtrl_HitTest(m_hWnd, pHitTestInfo);
}

inline bool CTabCtrl::InsertItem(int iItem, const TCITEM* pItem) {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_InsertItem(m_hWnd, iItem, pItem));
}

inline bool CTabCtrl::InsertItem(int iItem, const TCHAR* lpszItem) {
	AssertValidAsWindow();
	TCITEM	item;

	item.mask = TCIF_TEXT;
	item.pszText = const_cast<TCHAR*>(lpszItem);
	item.cchTextMax = _tcslen(lpszItem);
	return toBoolean(TabCtrl_InsertItem(m_hWnd, iItem, &item));
}

inline bool CTabCtrl::InsertItem(int iItem, const TCHAR* lpszItem, int iImage) {
	AssertValidAsWindow();
	TCITEM	item;

	item.mask = TCIF_IMAGE | TCIF_TEXT;
	item.pszText = const_cast<TCHAR*>(lpszItem);
	item.cchTextMax = _tcslen(lpszItem);
	item.iImage = iImage;
	return toBoolean(TabCtrl_InsertItem(m_hWnd, iItem, &item));
}

inline bool CTabCtrl::InsertItem(UINT nMask, int iItem, const TCHAR* lpszItem, int iImage, LPARAM lParam) {
	AssertValidAsWindow();
	TCITEM	item;

	item.mask = nMask;
	item.pszText = const_cast<TCHAR*>(lpszItem);
	item.cchTextMax = _tcslen(lpszItem);
	item.iImage = iImage;
	item.lParam = lParam;
	return toBoolean(TabCtrl_InsertItem(m_hWnd, iItem, &item));
}

inline void CTabCtrl::RemoveImage(int iImage) {
	AssertValidAsWindow();
	TabCtrl_RemoveImage(m_hWnd, iImage);
}

inline int CTabCtrl::SetCurSel(int iItem) {
	AssertValidAsWindow();
	return TabCtrl_SetCurSel(m_hWnd, iItem);
}

inline int CTabCtrl::SetCurFocus(int iItem) {
	AssertValidAsWindow();
	return TabCtrl_SetCurFocus(m_hWnd, iItem);
}

inline DWORD CTabCtrl::SetExtendedStyle(DWORD dwNewStyle, DWORD dwExMask /* = 0 */) {
	AssertValidAsWindow();
	return SendMessage(TCM_SETEXTENDEDSTYLE, dwExMask, dwNewStyle);
}

inline CImageList* CTabCtrl::SetImageList(const CImageList* pImageList) {
	AssertValidAsWindow();
	return CImageList::FromHandle(TabCtrl_SetImageList(m_hWnd, pImageList->GetSafeHandle()));
}

inline bool CTabCtrl::SetItem(int iItem, const TCITEM* pTabCtrlItem) {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_SetItem(m_hWnd, iItem, pTabCtrlItem));
}

inline bool CTabCtrl::SetItemExtra(int nBytes) {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_SetItemExtra(m_hWnd, nBytes));
}

inline SIZE CTabCtrl::SetItemSize(SIZE size) {
	AssertValidAsWindow();
	DWORD	dw = TabCtrl_SetItemSize(m_hWnd, size.cx, size.cy);
	SIZE	s = {LOWORD(dw), HIWORD(dw)};
	return s;
}

inline bool CTabCtrl::SetItemState(int iItem, DWORD dwMask, DWORD dwState) {
	AssertValidAsWindow();
	TCITEM	item;
	item.mask = TCIF_STATE;
	item.dwState = dwState;
	item.dwStateMask = dwMask;
	return SetItem(iItem, &item);
}

inline int CTabCtrl::SetMinTabWidth(int cx) {
	AssertValidAsWindow();
	return TabCtrl_SetMinTabWidth(m_hWnd, cx);
}

inline void CTabCtrl::SetPadding(SIZE size) {
	AssertValidAsWindow();
	TabCtrl_SetPadding(m_hWnd, size.cx, size.cy);
}

inline void CTabCtrl::SetToolTips(HWND hwndTip) {
	AssertValidAsWindow();
	TabCtrl_SetToolTips(m_hWnd, hwndTip);
}

inline LRESULT CALLBACK CTabCtrl::WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	CTabCtrl*	pTabCtrl = 0;

	if(message == WM_MATTACH) {
		pTabCtrl = reinterpret_cast<CTabCtrl*>(lParam);
		pTabCtrl->SetWindowLong(GWL_USERDATA, lParam);
		assert(hWnd == pTabCtrl->m_hWnd);
		return pTabCtrl->DispatchEvent(WM_MATTACH, 0, 0L);
	} else {
		pTabCtrl = reinterpret_cast<CTabCtrl*>(::GetWindowLong(hWnd, GWL_USERDATA));
		return (pTabCtrl != 0) ? pTabCtrl->DispatchEvent(message, wParam, lParam) : FALSE;
	}
}

#endif /* _TAB_CTRL */

/* [EOF] */