// ListCtrl.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _LIST_CTRL_H_
#define _LIST_CTRL_H_

#include "Window.h"
#include "Edit.h"
#include "ImageList.h"


// CListCtrl class definition
/////////////////////////////////////////////////////////////////////////////

namespace Manah {
namespace Windows {
namespace Controls {

class CListCtrl : public CWindow {
	// \bh
public:
	/* 쐬 */
	bool	Attach(HWND hWnd, bool bSubClass = false);
	bool	Create(bool bSubClass, DWORD dwStyle, const RECT& rect, HWND hwndParent, UINT id);
	bool	CreateEx(bool bSubClass, DWORD dwExStyle,
				DWORD dwStyle, const RECT& rect, HWND hwndParent, UINT id);

	/*  */
	SIZE		ApproximateViewRect(const SIZE& size, int iCount = -1) const;
	COLORREF	GetBkColor() const;
	bool		GetBkImage(LVBKIMAGE& image) const;
	UINT		GetCallbackMask() const;
	bool		GetCheck(int iItem) const;
	bool		GetColumn(int iCol, LVCOLUMN& column) const;
	bool		GetColumnOrderArray(INT* piArray, int iCount = -1) const;
	int			GetColumnWidth(int iCol) const;
	int			GetCountPerPage() const;
	CEdit*		GetEditControl() const;
	DWORD		GetExtendedStyle() const;
	HWND		GetHeaderControl() const;
	HCURSOR		GetHotCursor() const;
	int			GetHotItem() const;
	DWORD		GetHoverTime() const;
	CImageList*	GetImageList(int nImageListType) const;
	bool		GetItem(LVITEM& item) const;
	int			GetItemCount() const;
	DWORD		GetItemData(int iItem) const;
	bool		GetItemPosition(int iItem, LPPOINT lpPoint) const;
	bool		GetItemRect(int iItem, RECT& rect, UINT nCode) const;
	UINT		GetItemState(int iItem, UINT nMask) const;
	int			GetItemText(int iItem, int iSubItem, TCHAR* lpszText, int nLength) const;
	int			GetNextItem(int iItem, int nFlag) const;
	bool		GetOrigin(LPPOINT lpPoint) const;
	UINT		GetSelectedCount() const;
	int			GetSelectionMark() const;
	int			GetStringWidth(const TCHAR* lpszText) const;
	bool		GetSubItemRect(int iItem, int iSubItem, int nArea, RECT& rect) const;
	COLORREF	GetTextBkColor() const;
	COLORREF	GetTextColor() const;
	int			GetTopIndex() const;
	bool		GetViewRect(RECT& rect) const;
	void		GetWorkAreas(int nWorkAreas, LPRECT lpRect) const;
	bool		SetBkColor(COLORREF clr);
	bool		SetBkImage(const LVBKIMAGE& image);
	bool		SetBkImage(HBITMAP hBMP, bool fTile = true, int xOffsetPercent = 0, int yOffsetPercent = 0);
	bool		SetBkImage(const TCHAR* lpszURL, bool fTile = true,
					int xOffsetPercent = 0, int yOffsetPercent = 0);
	bool		SetCallbackMask(UINT nMask);
	bool		SetCheck(int iItem, bool fCheck = true);
	bool		SetColumn(int iCol, const LVCOLUMN& column);
	bool		SetColumnOrderArray(int iCount, INT* piArray);
	bool		SetColumnWidth(int iCol, int cx);
	DWORD		SetExtendedStyle(DWORD dwNewStyle);
	DWORD		SetExtendedStyleEx(DWORD dwExMask, DWORD dwExStyle);
	HCURSOR		SetHotCursor(HCURSOR hc);
	int			SetHotItem(int iIndex);
	DWORD		SetHoverTime(DWORD dwHoverTime = static_cast<DWORD>(-1));
	SIZE		SetIconSpacing(int cx, int cy);
	SIZE		SetIconSpacing(const SIZE& size);
	CImageList*	SetImageList(CImageList* pImageList, int nImageListType);
	bool		SetItem(const LVITEM& item);
	bool		SetItem(int iItem, int iSubItem, UINT nMask, const TCHAR* lpszItem,
					int iImage, UINT nState, UINT nStateMask, LPARAM lParam);
	void		SetItemCount(int nItems);
	void		SetItemCountEx(int iCount, DWORD dwFlags = LVSICF_NOINVALIDATEALL);
	bool		SetItemData(int iItem, DWORD dwData);
	bool		SetItemPosition(int iItem, const POINT& pt);
	bool		SetItemState(int iItem, const LVITEM& item);
	bool		SetItemState(int iItem, UINT nState, UINT nMask);
	bool		SetItemText(int iItem, int iSubItem, const TCHAR* lpszText);
	int			SetSelectionMark(int iIndex);
	bool		SetTextBkColor(COLORREF clr);
	bool		SetTextColor(COLORREF clr);
	void		SetWorkAreas(int nWorkAreas, const RECT* lpRect);
	int			SubItemHitTest(LPLVHITTESTINFO pInfo);

	/*  */
	bool		Arrange(UINT nCode);
	CImageList*	CreateDragImage(int iItem, LPPOINT lpPoint);	// returned pointer is merpanent
	bool		DeleteAllItems();
	bool		DeleteColumn(int iCol);
	bool		DeleteItem(int iItem);
	CEdit*		EditLabel(int iItem);
	bool		EnsureVisible(int iItem, bool bPartialOK);
	int			FindItem(LVFINDINFO* pFindInfo, int iStart = -1) const;
	int			HitTest(LVHITTESTINFO& hitTestInfo) const;
	int			HitTest(const POINT& pt, UINT* pFlags = 0) const;
	int			InsertColumn(int iCol, const LVCOLUMN& column);
	int			InsertColumn(int iCol, const TCHAR* lpszColumnHeading,
					int nFormat = LVCFMT_LEFT, int nWidth = -1, int iSubItem = -1);
	int			InsertItem(const LVITEM& item);
	int			InsertItem(int iItem, const TCHAR* lpszText);
	int			InsertItem(int iItem, const TCHAR* lpszText, int iImage);
	int			InsertItem(UINT nMask, int iItem, const TCHAR* lpszText,
					UINT nState, UINT nStateMask, int iImage, LPARAM lParam);
	bool		RedrawItems(int iFirst, int iLast);
	bool		Scroll(const SIZE& size);
	bool		SortItems(PFNLVCOMPARE pfnCompare, DWORD dwData);
	bool		Update(int iItem);
protected:
	OVERRIDE_STANDARD_DISPATCH_EVENT(CListCtrl);
};


// CListCtrl class implementation
/////////////////////////////////////////////////////////////////////////////

#define UNCONST_THIS	const_cast<CListCtrl*>(this)

inline SIZE CListCtrl::ApproximateViewRect(const SIZE& size, int iCount /* = -1 */) const {
	AssertValidAsWindow();

	SIZE	size_;
	DWORD	dwSize = static_cast<DWORD>(UNCONST_THIS->SendMessage(
		LVM_APPROXIMATEVIEWRECT, iCount, MAKELONG(size.cx, size.cy)));
	size_.cx = LOWORD(dwSize);
	size_.cy = HIWORD(dwSize);

	return size_;
}

inline bool CListCtrl::Arrange(UINT nCode) {
	return toBoolean(SendMessage(LVM_ARRANGE, nCode));
}

inline bool CListCtrl::Attach(HWND hWnd, bool bSubClass /* = false */) {
	AssertValid();

	if(IsWindow() || !::IsWindow(hWnd) || m_OldProc != 0)
		return false;
	m_hWnd = hWnd;

	if(bSubClass) {
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(STANDARD_WINDOW_CLASS_WNDPROC(CListCtrl))));
		assert(m_OldProc != STANDARD_WINDOW_CLASS_WNDPROC(CListCtrl));
		m_lOldUserData = GetWindowLong(GWL_USERDATA);
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline bool CListCtrl::Create(bool bSubClass, DWORD dwStyle, const RECT& rect, HWND hwndParent, UINT id) {
	AssertValid();

	m_hWnd = ::CreateWindow(WC_LISTVIEW, _T(""),
		dwStyle, rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top,
		hwndParent, reinterpret_cast<HMENU>(id), reinterpret_cast<HINSTANCE>(::GetModuleHandle(0)), 0L);
	if(m_hWnd == 0)
		return false;
	if(bSubClass){
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(STANDARD_WINDOW_CLASS_WNDPROC(CListCtrl))));
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline Manah::Windows::CImageList* CListCtrl::CreateDragImage(int iItem, LPPOINT lpPoint) {
	return CImageList::FromHandlePermanent(reinterpret_cast<HIMAGELIST>(
		SendMessage(LVM_CREATEDRAGIMAGE, iItem, reinterpret_cast<LPARAM>(lpPoint))));
}

inline bool CListCtrl::CreateEx(bool bSubClass, DWORD dwExStyle,
		DWORD dwStyle, const RECT& rect, HWND hwndParent, UINT id) {
	AssertValid();

	m_hWnd = ::CreateWindowEx(dwExStyle, WC_LISTVIEW, _T(""),
		dwStyle, rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top,
		hwndParent, reinterpret_cast<HMENU>(id), reinterpret_cast<HINSTANCE>(::GetModuleHandle(0)), 0L);
	if(m_hWnd == 0)
		return false;
	if(bSubClass) {
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(STANDARD_WINDOW_CLASS_WNDPROC(CListCtrl))));
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline bool CListCtrl::DeleteAllItems() {
	return toBoolean(SendMessage(LVM_DELETEALLITEMS));
}

inline bool CListCtrl::DeleteColumn(int iCol) {
	return toBoolean(SendMessage(LVM_DELETECOLUMN, iCol));
}

inline bool CListCtrl::DeleteItem(int iItem) {
	return toBoolean(SendMessage(LVM_DELETEITEM, iItem));
}

inline LRESULT CListCtrl::DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam) {
	if(PreTranslateMessage(message, wParam, lParam))
		return false;
	return CWindow::DispatchEvent(message, wParam, lParam);
}

inline CEdit* CListCtrl::EditLabel(int iItem) {
	static CEdit	oEdit;
	oEdit.Attach(reinterpret_cast<HWND>(SendMessage(LVM_EDITLABEL, iItem)));
	return &oEdit;
}

inline bool CListCtrl::EnsureVisible(int iItem, bool bPartialOK) {
	return (SendMessage(LVM_ENSUREVISIBLE, iItem, bPartialOK) != 0);
}

inline int CListCtrl::FindItem(LVFINDINFO* pFindInfo, int iStart /* = -1 */) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_FINDITEM, iStart, reinterpret_cast<LPARAM>(pFindInfo)));
}

inline COLORREF CListCtrl::GetBkColor() const {
	AssertValidAsWindow();
	return static_cast<COLORREF>(::SendMessage(m_hWnd, LVM_GETBKCOLOR, 0, 0));
}

inline bool CListCtrl::GetBkImage(LVBKIMAGE& image) const {
	return toBoolean(UNCONST_THIS->SendMessage(LVM_GETBKIMAGE, 0, reinterpret_cast<LPARAM>(&image)));
}

inline UINT CListCtrl::GetCallbackMask() const {
	return static_cast<UINT>(UNCONST_THIS->SendMessage(LVM_GETCALLBACKMASK));
}

inline bool CListCtrl::GetCheck(int iItem) const {
	return toBoolean(((static_cast<UINT>(UNCONST_THIS->SendMessage(LVM_GETITEMSTATE, iItem, LVIS_STATEIMAGEMASK)) >> 12) - 1));
}

inline bool CListCtrl::GetColumn(int iCol, LVCOLUMN& column) const {
	return toBoolean(UNCONST_THIS->SendMessage(LVM_GETCOLUMN, iCol, reinterpret_cast<LPARAM>(&column)));
}

inline bool CListCtrl::GetColumnOrderArray(LPINT piArray, int iCount /* = -1 */) const {
	AssertValidAsWindow();
	if(iCount == -1)
		iCount = sizeof(*piArray) / sizeof(int);
	return (::SendMessage(m_hWnd, LVM_GETCOLUMNORDERARRAY, iCount, reinterpret_cast<LPARAM>(piArray)) != 0);
}

inline int CListCtrl::GetColumnWidth(int iCol) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETCOLUMNWIDTH, iCol, 0L));
}

inline int CListCtrl::GetCountPerPage() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETCOUNTPERPAGE, 0, 0L));
}

inline CEdit* CListCtrl::GetEditControl() const {
	AssertValidAsWindow();
	static CEdit	oEdit;
	HWND			hEdit;
	
	hEdit = reinterpret_cast<HWND>(::SendMessage(m_hWnd, LVM_GETEDITCONTROL, 0, 0L));
	if(hEdit == 0)
		return 0;
	oEdit.Attach(hEdit);
	return &oEdit;
}

inline DWORD CListCtrl::GetExtendedStyle() const {
	AssertValidAsWindow();
	return static_cast<DWORD>(::SendMessage(m_hWnd, LVM_GETEXTENDEDLISTVIEWSTYLE, 0, 0L));
}

inline HWND CListCtrl::GetHeaderControl() const {
	AssertValidAsWindow();
	return reinterpret_cast<HWND>(::SendMessage(m_hWnd, LVM_GETHEADER, 0, 0L));
}

inline HCURSOR CListCtrl::GetHotCursor() const {
	AssertValidAsWindow();
	return reinterpret_cast<HCURSOR>(::SendMessage(m_hWnd, LVM_GETHOTCURSOR, 0, 0L));
}

inline int CListCtrl::GetHotItem() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETHOTITEM, 0, 0L));
}

inline DWORD CListCtrl::GetHoverTime() const {
	AssertValidAsWindow();
	return static_cast<DWORD>(::SendMessage(m_hWnd, LVM_GETHOVERTIME, 0, 0L));
}

inline Manah::Windows::CImageList* CListCtrl::GetImageList(int nImageListType) const {
	AssertValidAsWindow();
	return CImageList::FromHandle(
		reinterpret_cast<HIMAGELIST>(::SendMessage(m_hWnd, LVM_GETIMAGELIST, nImageListType, 0L)));
}

inline bool CListCtrl::GetItem(LVITEM& item) const {
	return toBoolean(UNCONST_THIS->SendMessage(LVM_GETITEM, 0, reinterpret_cast<LPARAM>(&item)));
}

inline int CListCtrl::GetItemCount() const {
	return static_cast<int>(UNCONST_THIS->SendMessage(LVM_GETITEMCOUNT));
}

inline DWORD CListCtrl::GetItemData(int iItem) const {
	LVITEM	item;
	item.mask = LVIF_PARAM;
	item.iItem = iItem;
	item.iSubItem = 0;
	GetItem(item);
	return item.lParam;
}

inline bool CListCtrl::GetItemPosition(int iItem, LPPOINT lpPoint) const {
	AssertValidAsWindow();
	if(lpPoint == 0)
		return false;
	return (::SendMessage(m_hWnd, LVM_GETITEMPOSITION, iItem, reinterpret_cast<LPARAM>(lpPoint)) != 0);
}

inline bool CListCtrl::GetItemRect(int iItem, RECT& rect, UINT nCode) const {
	rect.left = nCode;
	return toBoolean(UNCONST_THIS->SendMessage(LVM_GETITEMRECT, iItem, reinterpret_cast<LPARAM>(&rect)));
}

inline UINT CListCtrl::GetItemState(int iItem, UINT nMask) const {
	return static_cast<UINT>(UNCONST_THIS->SendMessage(LVM_GETITEMSTATE, iItem, nMask));
}

inline int CListCtrl::GetItemText(int iItem, int iSubItem, TCHAR* lpszText, int nLength) const {
	AssertValidAsWindow();

	LVITEM	item;
	item.iSubItem = iSubItem;
	item.pszText = lpszText;
	item.cchTextMax = nLength;
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETITEMTEXT, iItem, reinterpret_cast<LPARAM>(&item)));
}

inline int CListCtrl::GetNextItem(int iItem, int nFlag) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETNEXTITEM, iItem, nFlag));
}

inline bool CListCtrl::GetOrigin(LPPOINT lpPoint) const {
	AssertValidAsWindow();
	if(lpPoint == 0)
		return false;
	return (::SendMessage(m_hWnd, LVM_GETORIGIN, 0, reinterpret_cast<LPARAM>(lpPoint)) != 0);
}

inline UINT CListCtrl::GetSelectedCount() const {
	AssertValidAsWindow();
	return static_cast<UINT>(::SendMessage(m_hWnd, LVM_GETSELECTEDCOUNT, 0, 0L));
}

inline int CListCtrl::GetSelectionMark() const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETSELECTIONMARK, 0, 0L));
}

inline int CListCtrl::GetStringWidth(const TCHAR* lpszText) const {
	AssertValidAsWindow();
	return static_cast<int>(::SendMessage(m_hWnd, LVM_GETSTRINGWIDTH, 0, reinterpret_cast<LPARAM>(lpszText)));
}

inline bool CListCtrl::GetSubItemRect(int iItem, int iSubItem, int nArea, RECT& rect) const {
	rect.left = nArea;
	rect.top = iSubItem;
	return toBoolean(UNCONST_THIS->SendMessage(LVM_GETSUBITEMRECT, iItem, reinterpret_cast<LPARAM>(&rect)));
}

inline COLORREF CListCtrl::GetTextBkColor() const {
	return static_cast<COLORREF>(UNCONST_THIS->SendMessage(LVM_GETTEXTBKCOLOR));
}

inline COLORREF CListCtrl::GetTextColor() const {
	return static_cast<COLORREF>(UNCONST_THIS->SendMessage(LVM_GETTEXTCOLOR));
}

inline int CListCtrl::GetTopIndex() const {
	return static_cast<int>(UNCONST_THIS->SendMessage(LVM_GETTOPINDEX));
}

inline bool CListCtrl::GetViewRect(RECT& rect) const {
	return toBoolean(UNCONST_THIS->SendMessage(LVM_GETVIEWRECT, 0, reinterpret_cast<LPARAM>(&rect)));
}

inline void CListCtrl::GetWorkAreas(int nWorkAreas, RECT* lpRect) const {
	if(lpRect == 0)
		return;
	UNCONST_THIS->SendMessage(LVM_GETWORKAREAS, nWorkAreas, reinterpret_cast<LPARAM>(lpRect));
}

inline int CListCtrl::HitTest(LVHITTESTINFO& hitTestInfo) const {
	return static_cast<int>(UNCONST_THIS->SendMessage(LVM_HITTEST, 0, reinterpret_cast<LPARAM>(&hitTestInfo)));
}

inline int CListCtrl::HitTest(const POINT& pt, UINT* pFlags /* = 0 */) const {
	LVHITTESTINFO	lvti;
	lvti.pt = pt;
	lvti.flags = (pFlags != 0) ? *pFlags : 0;
	return HitTest(lvti);
}

inline int CListCtrl::InsertColumn(int iCol, const LVCOLUMN& column) {
	return static_cast<int>(SendMessage(LVM_INSERTCOLUMN, iCol, reinterpret_cast<LPARAM>(&column)));
}

inline int CListCtrl::InsertColumn(int iCol, const TCHAR* lpszColumnHeading, int nFormat /* = LVCFMT_LEFT */, int nWidth /* = -1 */, int iSubItem /* = -1 */) {
	LVCOLUMN	lvcol = {
		LVCF_FMT | LVCF_TEXT
			| ((iSubItem != -1) ? LVCF_SUBITEM : 0)
			| ((nWidth != -1) ? LVCF_WIDTH : 0),
		nFormat, nWidth, const_cast<TCHAR*>(lpszColumnHeading), _tcslen(lpszColumnHeading), iSubItem
	};
	return InsertColumn(iCol, lvcol);
}

inline int CListCtrl::InsertItem(const LVITEM& item) {
	return static_cast<int>(SendMessage(LVM_INSERTITEM, 0, reinterpret_cast<LPARAM>(&item)));
}

inline int CListCtrl::InsertItem(int iItem, const TCHAR* lpszText) {
	LVITEM	item;

	ZeroMemory(&item, sizeof(LVITEM));
	item.mask = LVIF_TEXT;
	item.iItem = iItem;
	item.pszText = const_cast<TCHAR*>(lpszText);
	item.cchTextMax = _tcslen(lpszText);

	return InsertItem(item);
}

inline int CListCtrl::InsertItem(int iItem, const TCHAR* lpszText, int iImage) {
	LVITEM	item;

	ZeroMemory(&item, sizeof(LVITEM));
	item.mask = LVIF_TEXT | LVIF_IMAGE;
	item.iItem = iItem;
	item.pszText = const_cast<TCHAR*>(lpszText);
	item.cchTextMax = _tcslen(lpszText);
	item.iImage = iImage;

	return InsertItem(item);
}

inline int CListCtrl::InsertItem(UINT nMask, int iItem, const TCHAR* lpszText, UINT nState, UINT nStateMask, int iImage, LPARAM lParam) {
	LVITEM	item = {
		nMask, iItem, 0, nState, nStateMask, const_cast<TCHAR*>(lpszText), _tcslen(lpszText), iImage, lParam
	};

	return InsertItem(item);
}

inline bool CListCtrl::RedrawItems(int iFirst, int iLast) {
	return toBoolean(SendMessage(LVM_REDRAWITEMS, iFirst, iLast));
}

inline bool CListCtrl::Scroll(const SIZE& size) {
	return toBoolean(SendMessage(LVM_SCROLL, size.cx, size.cy));
}

inline bool CListCtrl::SetBkColor(COLORREF clr) {
	return toBoolean(SendMessage(LVM_SETBKCOLOR, 0, clr));
}

inline bool CListCtrl::SetBkImage(const LVBKIMAGE& image) {
	return toBoolean(SendMessage(LVM_SETBKIMAGE, 0, reinterpret_cast<LPARAM>(&image)));
}

inline bool CListCtrl::SetBkImage(HBITMAP hBMP, bool fTile /* = true */, int xOffsetPercent /* = 0 */, int yOffsetPercent /* = 0 */) {
	LVBKIMAGE	bkImage = {
		LVBKIF_SOURCE_HBITMAP | ((xOffsetPercent == 0 && yOffsetPercent == 0) ?
			LVBKIF_STYLE_TILE : LVBKIF_STYLE_NORMAL),
		hBMP, 0, 0, xOffsetPercent, yOffsetPercent
	};
	return SetBkImage(bkImage);
}

inline bool CListCtrl::SetBkImage(const TCHAR* lpszURL, bool fTile /* = true */, int xOffsetPercent /* = 0 */, int yOffsetPercent /* = 0 */) {
	LVBKIMAGE	bkImage = {
		LVBKIF_SOURCE_URL | ((xOffsetPercent == 0 && yOffsetPercent == 0) ?
			LVBKIF_STYLE_TILE : LVBKIF_STYLE_NORMAL),
		0, const_cast<TCHAR*>(lpszURL), _tcslen(lpszURL), xOffsetPercent, yOffsetPercent
	};
	return SetBkImage(bkImage);
}

inline bool CListCtrl::SetCallbackMask(UINT nMask) {
	return toBoolean(SendMessage(LVM_SETCALLBACKMASK, nMask));
}

inline bool CListCtrl::SetCheck(int iItem, bool fCheck /* = true */) {
	LVITEM	item = {0, iItem, 0, INDEXTOSTATEIMAGEMASK(fCheck - 1), LVIS_STATEIMAGEMASK, 0, 0, 0, 0};
	return toBoolean(SendMessage(LVM_SETITEMSTATE, iItem, reinterpret_cast<LPARAM>(&item)));
}

inline bool CListCtrl::SetColumn(int iCol, const LVCOLUMN& column) {
	return toBoolean(SendMessage(LVM_SETCOLUMN, iCol, reinterpret_cast<LPARAM>(&column)));
}

inline bool CListCtrl::SetColumnOrderArray(int iCount, INT* piArray) {
	return toBoolean(SendMessage(LVM_SETCOLUMNORDERARRAY, iCount, reinterpret_cast<LPARAM>(piArray)));
}

inline bool CListCtrl::SetColumnWidth(int iCol, int cx) {
	return toBoolean(SendMessage(LVM_SETCOLUMNWIDTH, iCol, cx));
}

inline DWORD CListCtrl::SetExtendedStyle(DWORD dwNewStyle) {
	return static_cast<DWORD>(SendMessage(LVM_SETEXTENDEDLISTVIEWSTYLE, 0, dwNewStyle));
}

inline DWORD CListCtrl::SetExtendedStyleEx(DWORD dwExMask, DWORD dwExStyle) {
	return static_cast<DWORD>(SendMessage(LVM_SETEXTENDEDLISTVIEWSTYLE, dwExMask, dwExStyle));
}

inline HCURSOR CListCtrl::SetHotCursor(HCURSOR hc) {
	return reinterpret_cast<HCURSOR>(SendMessage(LVM_SETHOTCURSOR, 0, reinterpret_cast<LPARAM>(hc)));
}

inline int CListCtrl::SetHotItem(int iIndex) {
	return static_cast<int>(SendMessage(LVM_SETHOTITEM, iIndex));
}

inline DWORD CListCtrl::SetHoverTime(DWORD dwHoverTime /* = static_cast<DWORD>(-1) */) {
	return static_cast<DWORD>(SendMessage(LVM_SETHOVERTIME, 0, dwHoverTime));
}

inline SIZE CListCtrl::SetIconSpacing(int cx, int cy) {
	SIZE	size;
	DWORD	dwSize;

	dwSize = SendMessage(LVM_SETICONSPACING, 0, MAKELONG(cx, cy));
	size.cx = LOWORD(dwSize);
	size.cy = HIWORD(dwSize);
	return size;
}

inline SIZE CListCtrl::SetIconSpacing(const SIZE& size) {
	SIZE		size_;
	const DWORD	dwSize = SendMessage(LVM_SETICONSPACING, 0, MAKELONG(size.cx, size.cy));

	size_.cx = LOWORD(dwSize);
	size_.cy = HIWORD(dwSize);
	return size_;
}

inline Manah::Windows::CImageList* CListCtrl::SetImageList(Manah::Windows::CImageList* pImageList, int nImageListType) {
	return CImageList::FromHandle(reinterpret_cast<HIMAGELIST>(
		SendMessage(LVM_SETIMAGELIST, nImageListType, reinterpret_cast<LPARAM>(pImageList->GetSafeHandle()))));
}

inline bool CListCtrl::SetItem(const LVITEM& item) {
	return toBoolean(SendMessage(LVM_SETITEM, 0, reinterpret_cast<LPARAM>(&item)));
}

inline void CListCtrl::SetItemCount(int nItems) {
	SendMessage(LVM_SETITEMCOUNT, nItems);
}

inline void CListCtrl::SetItemCountEx(int iCount, DWORD dwFlags /* = LVSICF_NOINVALIDATEALL */) {
	SendMessage(LVM_SETITEMCOUNT, iCount, dwFlags);
}

inline bool CListCtrl::SetItemData(int iItem, DWORD dwData) {
	LVITEM	item;
	item.mask = LVIF_PARAM;
	item.iItem = iItem;
	item.iSubItem = 0;
	item.lParam = dwData;
	return SetItem(item);
}

inline bool CListCtrl::SetItem(int iItem, int iSubItem, UINT nMask, const TCHAR* lpszItem, int iImage, UINT nState, UINT nStateMask, LPARAM lParam) {
	LVITEM	item = {nMask, iItem, iSubItem, nState, nStateMask, const_cast<TCHAR*>(lpszItem), _tcslen(lpszItem), iImage};
	return toBoolean(SendMessage(LVM_SETITEM, 0, reinterpret_cast<LPARAM>(&item)));
}

inline bool CListCtrl::SetItemPosition(int iItem, const POINT& pt) {
	return toBoolean(SendMessage(LVM_SETITEMPOSITION32, iItem, MAKELPARAM(pt.x, pt.y)));
}

inline bool CListCtrl::SetItemState(int iItem, const LVITEM& item) {
	return toBoolean(SendMessage(LVM_SETITEMSTATE, iItem, reinterpret_cast<LPARAM>(&item)));
}

inline bool CListCtrl::SetItemText(int iItem, int iSubItem, const TCHAR* lpszText) {
	LVITEM	item;
	item.iSubItem = iSubItem;
	item.pszText = const_cast<TCHAR*>(lpszText);
	return toBoolean(SendMessage(LVM_SETITEMTEXT, iItem, reinterpret_cast<LPARAM>(&item)));
}

inline int CListCtrl::SetSelectionMark(int iIndex) {
	return static_cast<int>(SendMessage(LVM_SETSELECTIONMARK, 0, iIndex));
}

inline bool CListCtrl::SetItemState(int iItem, UINT nState, UINT nMask) {
	LVITEM	item;
	item.state = nState;
	item.stateMask = nMask;
	return toBoolean(SendMessage(LVM_SETITEMSTATE, iItem, reinterpret_cast<LPARAM>(&item)));
}

inline bool CListCtrl::SetTextBkColor(COLORREF clr) {
	return toBoolean(SendMessage(LVM_SETTEXTBKCOLOR, 0, static_cast<LPARAM>(clr)));
}

inline bool CListCtrl::SetTextColor(COLORREF clr) {
	return toBoolean(SendMessage(LVM_SETTEXTCOLOR, 0, static_cast<LPARAM>(clr)));
}

inline void CListCtrl::SetWorkAreas(int nWorkAreas, const RECT* lpRect) {
	SendMessage(LVM_SETWORKAREAS, nWorkAreas, reinterpret_cast<LPARAM>(lpRect));
}

inline bool CListCtrl::SortItems(PFNLVCOMPARE pfnCompare, DWORD dwData) {
	return toBoolean(SendMessage(LVM_SORTITEMS, dwData, reinterpret_cast<LPARAM>(pfnCompare)));
}

inline int CListCtrl::SubItemHitTest(LPLVHITTESTINFO pInfo) {
	return static_cast<int>(SendMessage(LVM_SUBITEMHITTEST, 0, reinterpret_cast<LPARAM>(pInfo)));
}

inline bool CListCtrl::Update(int iItem) {
	return toBoolean(SendMessage(LVM_UPDATE, iItem));
}

#undef UNCONST_THIS

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _LIST_CTRL_H_ */

/* [EOF] */