// GdiObject.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _GDI_OBJECT_H_
#define _GDI_OBJECT_H_

#include "win_utils.h"

namespace Manah {
namespace Windows {
namespace GDI {


// CGdiObject class definition
/////////////////////////////////////////////////////////////////////////////

class CGdiObject : public CObject {
	// constructors
public:
	CGdiObject();
	virtual ~CGdiObject();
private:
	CGdiObject(const CGdiObject& rhs);

	// operator
private:
	operator =(const CGdiObject& rhs);

	// methods
public:
	virtual bool	CreateStockObject(int index) = 0;
	bool	DeleteObject();
	bool	UnrealizeObject();
protected:
	virtual bool	_Attach(HGDIOBJ hGdiObj);
	virtual HGDIOBJ	_Detach();

	// data members
protected:
	HGDIOBJ	m_hGdiObject;
	bool	m_bAttached;
	bool	m_bIsStockObject;
};


// CBitmap class definition
/////////////////////////////////////////////////////////////////////////////

class CBitmap : public CGdiObject {
	// operator
public:
	operator HBITMAP() const;

	// methods
public:
	bool	CreateStockObject(int index);
	bool	Attach(HBITMAP hBitmap);
	HBITMAP	Detach();

	HBITMAP	GetSafeHandle() const;
	bool	LoadBitmap(const TCHAR* lpszResourceName);
	bool	LoadBitmap(uint nResourceId);
	bool	LoadOEMBitmap(uint nBitmapId);
	bool	LoadMappedBitmap(uint nBitmapId, uint nFlags = 0, LPCOLORMAP lpColorMap = 0, int nMapSize = 0);
	bool	CreateBitmap(int nWidth, int nHeight, uint nPlanes, uint nBitCount, const void* lpBits);
	bool	CreateBitmapIndirect(const BITMAP& bitmap);
	template<bool bDCIsDisposable>
	bool	CreateCompatibleBitmap(const CDC<bDCIsDisposable>& dc, int nWidth, int nHeight);
	template<bool bDCIsDisposable>
	bool	CreateDiscardableBitmap(const CDC<bDCIsDisposable>& dc, int nWidth, int nHeight);
	bool	GetBitmap(BITMAP& bitmap) const;
	DWORD	SetBitmapBits(DWORD dwCount, const void* lpBits);
	DWORD	GetBitmapBits(DWORD dwCount, void* lpBits) const;
	CSize	SetBitmapDimension(int nWidth, int nHeight);
	CSize	GetBitmapDimension() const;
};


// CBrush class definition
/////////////////////////////////////////////////////////////////////////////

class CBrush : public CGdiObject {
	// constructors
public:
	CBrush();
	CBrush(COLORREF clr);
	CBrush(int index, COLORREF clr);
	CBrush(const CBitmap& bitmap);

	// operator
public:
	operator HBRUSH() const;

	// methods
public:
	bool	CreateStockObject(int index);
	bool	Attach(HBRUSH hBrush);
	HBRUSH	Detach();

	HBRUSH	GetSafeHandle() const;
	bool	CreateSolidBrush(COLORREF clr);
	bool	CreateHatchBrush(int index, COLORREF clr);
	bool	CreateBrushIndirect(const LOGBRUSH& logbrush);
	bool	CreatePatternBrush(const CBitmap& bitmap);
	bool	CreateDIBPatternBrush(HGLOBAL hGlobal, uint nUsage);
	bool	CreateDIBPatternBrush(const void* lpPackedDIB, uint nUsage);
	bool	CreateSysColorBrush(int index);
	bool	GetLogBrush(LOGBRUSH& logbrush) const;
};


// CFont class definition
/////////////////////////////////////////////////////////////////////////////

class CFont : public CGdiObject {
	// operator
public:
	operator HFONT() const;

	// methods
public:
	bool	CreateStockObject(int index);
	bool	Attach(HFONT hFont);
	HFONT	Detach();

	HFONT	GetSafeHandle() const;
	bool	CreateFont(int nWidth, int nHeight, int nEscapement, int nOrientation, int nWeight,
				bool bItalic, bool bUnderline, bool bStrikeOut, BYTE nCharset, BYTE nOutPrecision,
				BYTE nClipPrecision, BYTE nQuality, BYTE nPitchAndFamily, const TCHAR* lpszFaceName);
	bool	CreateFontIndirect(const LOGFONT& logfont);
//	bool	CreatePointFont(int nPointSize, const TCHAR* lpszFaceName, CDC* pDC = 0);
//	bool	CreatePointFontIndirect(const LOGFONT& logfont, CDC* pDC = 0);
	bool	GetLogFont(LOGFONT& logfont) const;
};


// CPalette class definition
/////////////////////////////////////////////////////////////////////////////

class CPalette : public CGdiObject {
	// operator
public:
	operator HPALETTE() const;

	// methods
public:
	bool		CreateStockObject(int index);
	bool		Attach(HPALETTE hPalette);
	HPALETTE	Detach();

	HPALETTE	GetSafeHandle() const;
	bool		CreatePalette(const LOGPALETTE& logpalette);
	template<bool bDCIsDisposable>
	bool		CreateHalftonePalette(CDC<bDCIsDisposable>& dc);
	void		AnimatePalette(uint iStart, uint cEntries, const PALETTEENTRY& palettecolors);
	uint		GetNearestPaletteIndex(COLORREF clr) const;
	bool		ResizePalette(uint cEntries);
	int			GetEntryCount() const;
	uint		GetPaletteEntries(uint iStart, uint cEntries, PALETTEENTRY& palettecolors) const;
	uint		SetPaletteEntries(uint iStart, uint cEntries, const PALETTEENTRY& palettecolors);
};


// CPen class definition
/////////////////////////////////////////////////////////////////////////////

class CPen : public CGdiObject {
	// constructors
public:
	CPen();
	CPen(int nPenStyle, int nWidth, COLORREF clr);
	CPen(int nPenStyle, int nWidth, const LOGBRUSH& logbrush, int cStyles = 0, const DWORD* pdwStyles = 0);

	// operator
public:
	operator HPEN() const;

	// methods
public:
	bool	CreateStockObject(int index);
	bool	Attach(HPEN hPen);
	HPEN	Detach();

	HPEN	GetSafeHandle() const;
	bool	CreatePen(int nPenStyle, int nWidth, COLORREF clr);
	bool	CreatePen(int nPenStyle, int nWidth,
				const LOGBRUSH& logbrush, int cStyles = 0, const DWORD* pdwStyles = 0);
	bool	CreatePenIndirect(const LOGPEN& logpen);
	bool	GetLogPen(LOGPEN& logpen) const;
	bool	GetExtLogPen(EXTLOGPEN& extlogpen) const;
};


// CRgn class definition
/////////////////////////////////////////////////////////////////////////////

class CRgn : public CGdiObject {
	// operator
public:
	operator HRGN() const;

	// methods
public:
	bool	CreateStockObject(int index);
	bool	Attach(HRGN hRgn);
	HRGN	Detach();

	HRGN	GetSafeHandle() const;
	bool	CreateRectRgn(int nLeft, int nTop, int nRight, int nBottom);
	bool	CreateRectRgnIndirect(const RECT& rect);
	bool	CreateEllipticRgn(int nLeft, int nTop, int nRight, int nBottom);
	bool	CreateEllipticRgnIndirect(const RECT& rect);
	bool	CreatePolygonRgn(const POINT* lpPoints, int cPoints, int nPolyFillMode);
	bool	CreatePolyPolygonRgn(const POINT* lpPoints, const int* pcPoly, int cPoints, int nPolyFillMode);
	bool	CreateRoundRectRgn(int x1, int y1, int x2, int y2, int x3, int y3);
	int		CombineRgn(const CRgn& oRgn1, const CRgn& oRgn2, int nCombineMode);
	int		CopyRgn(const CRgn& oRgn);
	template<bool bDCIsDisposable>
	bool	CreateFromPath(const CDC<bDCIsDisposable>& dc);
	bool	CreateFromData(const XFORM* lpXForm, int cDatas, const RGNDATA* pRgnData);
	bool	EqualRgn(const CRgn& oRgn) const;
	int		GetRegionData(LPRGNDATA lpRgnData, int cDatas) const;
	int		GetRgnBox(RECT* lpRect) const;
	int		OffsetRgn(int x, int y);
	int		OffsetRgn(const POINT& pt);
	bool	PtInRegion(int x, int y) const;
	bool	PtInRegion(const POINT& pt) const;
	bool	RectInRegion(const RECT& rect) const;
	bool	SetRectRgn(int nLeft, int nTop, int nRight, int nBottom);
	bool	SetRectRgn(const RECT& rect);
};


// CGdiObject class implementation
/////////////////////////////////////////////////////////////////////////////

inline CGdiObject::CGdiObject() : m_hGdiObject(0), m_bAttached(false), m_bIsStockObject(false) {
}

inline CGdiObject::~CGdiObject() {
	DeleteObject();
}

inline bool CGdiObject::_Attach(HGDIOBJ hGdiObj) {
	if(m_hGdiObject != 0) {
		m_hGdiObject = hGdiObj;
		m_bAttached = true;
		return m_hGdiObject != 0;
	}
	return false;
}

inline bool CGdiObject::DeleteObject() {
	if(m_hGdiObject != 0 && !m_bIsStockObject) {
		m_bAttached = false;
		return toBoolean(::DeleteObject(m_hGdiObject));
	}
	return false;
}

inline HGDIOBJ CGdiObject::_Detach() {
	if(!m_bAttached)
		return false;
	m_bAttached = false;
	HGDIOBJ	hGdiObj = m_hGdiObject;
	m_hGdiObject = 0;
	return hGdiObj;
}

inline bool CGdiObject::UnrealizeObject() {
	return toBoolean(::UnrealizeObject(m_hGdiObject));
}


// CBitmap class implementation
/////////////////////////////////////////////////////////////////////////////

inline CBitmap::operator HBITMAP() const {
	return static_cast<HBITMAP>(m_hGdiObject);
}

inline bool CBitmap::Attach(HBITMAP hBitmap) {
	return CGdiObject::_Attach(hBitmap);
}

inline bool CBitmap::CreateBitmap(int nWidth, int nHeight, uint nPlanes, uint nBitCount, const void* lpBits) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateBitmap(nWidth, nHeight, nPlanes, nBitCount, lpBits);
	return m_hGdiObject != 0;
}

inline bool CBitmap::CreateBitmapIndirect(const BITMAP& bitmap) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateBitmapIndirect(&(const_cast<BITMAP&>(bitmap)));
	return m_hGdiObject != 0;
}

template<bool bDCIsDisposable>
inline bool CBitmap::CreateCompatibleBitmap(const CDC<bDCIsDisposable>& dc, int nWidth, int nHeight) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateCompatibleBitmap(dc, nWidth, nHeight);
	return m_hGdiObject != 0;
}

template<bool bDCIsDisposable>
inline bool CBitmap::CreateDiscardableBitmap(const CDC<bDCIsDisposable>& dc, int nWidth, int nHeight) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateDiscardableBitmap(dc, nWidth, nHeight);
	return m_hGdiObject != 0;
}

inline bool CBitmap::CreateStockObject(int index) {
	return false;
}

inline HBITMAP CBitmap::Detach() {
	return static_cast<HBITMAP>(CGdiObject::_Detach());
}

inline bool CBitmap::GetBitmap(BITMAP& bitmap) const {
	return ::GetObject(m_hGdiObject, sizeof(HBITMAP), &bitmap) != 0;
}

inline DWORD CBitmap::GetBitmapBits(DWORD dwCount, void* lpBits) const {
	return ::GetBitmapBits(static_cast<HBITMAP>(m_hGdiObject), dwCount, lpBits);
}

inline CSize CBitmap::GetBitmapDimension() const {
	SIZE	size;
	::GetBitmapDimensionEx(static_cast<HBITMAP>(m_hGdiObject), &size);
	return size;
}

inline HBITMAP CBitmap::GetSafeHandle() const {
	return static_cast<HBITMAP>((this != 0) ? m_hGdiObject : 0);
}

inline bool CBitmap::LoadBitmap(const TCHAR* lpszResourceName) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::LoadBitmap(::GetModuleHandle(0), lpszResourceName);
	return m_hGdiObject != 0;
}

inline bool CBitmap::LoadBitmap(uint nResourceId) {
	return LoadBitmap(MAKEINTRESOURCE(nResourceId));
}

inline bool CBitmap::LoadMappedBitmap(uint nBitmapId, uint nFlags /* = 0 */, LPCOLORMAP lpColorMap /* = 0 */, int nMapSize /* = 0 */) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateMappedBitmap(::GetModuleHandle(0), nBitmapId, nFlags, lpColorMap, nMapSize);
	return m_hGdiObject != 0;
}

inline bool CBitmap::LoadOEMBitmap(uint nBitmapId) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::LoadBitmap(0, MAKEINTRESOURCE(nBitmapId));
	return m_hGdiObject != 0;
}

inline DWORD CBitmap::SetBitmapBits(DWORD dwCount, const void* lpBits) {
	return ::SetBitmapBits(static_cast<HBITMAP>(m_hGdiObject), dwCount, lpBits);
}

inline CSize CBitmap::SetBitmapDimension(int nWidth, int nHeight) {
	SIZE	size;
	::SetBitmapDimensionEx(static_cast<HBITMAP>(m_hGdiObject), nWidth, nHeight, &size);
	return size;
}


// CBrush class implementation
/////////////////////////////////////////////////////////////////////////////

inline CBrush::CBrush() {
}

inline CBrush::CBrush(COLORREF clr) {
	CreateSolidBrush(clr);
}

inline CBrush::CBrush(int index, COLORREF clr) {
	CreateHatchBrush(index, clr);
}

inline CBrush::CBrush(const CBitmap& bitmap) {
	CreatePatternBrush(bitmap);
}

inline CBrush::operator HBRUSH() const {
	return static_cast<HBRUSH>(m_hGdiObject);
}

inline bool CBrush::Attach(HBRUSH hBrush) {
	return CGdiObject::_Attach(hBrush);
}

inline bool CBrush::CreateBrushIndirect(const LOGBRUSH& logbrush) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateBrushIndirect(&logbrush);
	return (m_hGdiObject != 0);
}

inline bool CBrush::CreateDIBPatternBrush(HGLOBAL hGlobal, uint nUsage) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateDIBPatternBrush(hGlobal, nUsage);
	return (m_hGdiObject != 0);
}

inline bool CBrush::CreateDIBPatternBrush(const void* lpPackedDIB, uint nUsage) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateDIBPatternBrushPt(lpPackedDIB, nUsage);
	return (m_hGdiObject != 0);
}

inline bool CBrush::CreateHatchBrush(int index, COLORREF clr) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateHatchBrush(index, clr);
	return (m_hGdiObject != 0);
}

inline bool CBrush::CreatePatternBrush(const CBitmap& bitmap) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreatePatternBrush(bitmap);
	return (m_hGdiObject != 0);
}

inline bool CBrush::CreateSolidBrush(COLORREF clr) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateSolidBrush(clr);
	return (m_hGdiObject != 0);
}

inline bool CBrush::CreateStockObject(int index) {
	if(m_hGdiObject != 0 || index > HOLLOW_BRUSH)
		return false;
	m_hGdiObject = ::GetStockObject(index);
	if(m_hGdiObject != 0) {
		m_bIsStockObject = true;
		return true;
	}
	return false;
}

inline bool CBrush::CreateSysColorBrush(int index) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::GetSysColorBrush(index);
	return (m_hGdiObject != 0);
}

inline HBRUSH CBrush::Detach() {
	return static_cast<HBRUSH>(CGdiObject::_Detach());
}

inline HBRUSH CBrush::GetSafeHandle() const {
	return static_cast<HBRUSH>((this != 0) ? m_hGdiObject : 0);
}

inline bool CBrush::GetLogBrush(LOGBRUSH& logbrush) const {
	return ::GetObject(m_hGdiObject, sizeof(HBRUSH), &logbrush) != 0;
}


// CFont class implementation
/////////////////////////////////////////////////////////////////////////////

inline CFont::operator HFONT() const {
	return static_cast<HFONT>(m_hGdiObject);
}

inline bool CFont::Attach(HFONT hFont) {
	return CGdiObject::_Attach(hFont);
}

inline bool CFont::CreateFont(int nWidth, int nHeight, int nEscapement, int nOrientation, int nWeight,
		bool bItalic, bool bUnderline, bool bStrikeOut, BYTE nCharset, BYTE nOutPrecision, BYTE nClipPrecision,
		BYTE nQuality, BYTE nPitchAndFamily, const TCHAR* lpszFaceName) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateFont(nWidth, nHeight, nEscapement, nOrientation, nWeight,
				bItalic, bUnderline, bStrikeOut, nCharset, nOutPrecision, nClipPrecision,
				nQuality, nPitchAndFamily, lpszFaceName);
	return (m_hGdiObject != 0);
}

inline bool CFont::CreateFontIndirect(const LOGFONT& logfont) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateFontIndirect(&logfont);
	return (m_hGdiObject != 0);
}

inline bool CFont::CreateStockObject(int index) {
	if(m_hGdiObject != 0 || index < ANSI_FIXED_FONT || index > DEFAULT_GUI_FONT)
		return false;
	m_hGdiObject = ::GetStockObject(index);
	if(m_hGdiObject != 0) {
		m_bIsStockObject = true;
		return true;
	}
	return false;
}

inline HFONT CFont::Detach() {
	return static_cast<HFONT>(CGdiObject::_Detach());
}

inline bool CFont::GetLogFont(LOGFONT& logfont) const {
	return ::GetObject(m_hGdiObject, sizeof(LOGFONT), &logfont) != 0;
}

inline HFONT CFont::GetSafeHandle() const {
	return static_cast<HFONT>((this != 0) ? m_hGdiObject : 0);
}


// CPalette class implementation
/////////////////////////////////////////////////////////////////////////////

inline CPalette::operator HPALETTE() const {
	return static_cast<HPALETTE>(m_hGdiObject);
}

inline void CPalette::AnimatePalette(uint iStart, uint cEntries, const PALETTEENTRY& palettecolors) {
	::AnimatePalette(static_cast<HPALETTE>(m_hGdiObject), iStart, cEntries, &palettecolors);
}

inline bool CPalette::Attach(HPALETTE hPalette) {
	return CGdiObject::_Attach(hPalette);
}

template<bool bDCIsDisposable> inline bool CPalette::CreateHalftonePalette(CDC<bDCIsDisposable>& dc) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateHalftonePalette(dc);
	return (m_hGdiObject != 0);
}

inline bool CPalette::CreatePalette(const LOGPALETTE& logpalette) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreatePalette(&logpalette);
	return (m_hGdiObject != 0);
}

inline bool CPalette::CreateStockObject(int index) {
	if(m_hGdiObject != 0 || index != DEFAULT_PALETTE)
		return false;
	m_hGdiObject = ::GetStockObject(index);
	return (m_hGdiObject != 0);
}

inline HPALETTE CPalette::Detach() {
	return static_cast<HPALETTE>(CGdiObject::_Detach());
}

inline int CPalette::GetEntryCount() const {
	return ::GetPaletteEntries(static_cast<HPALETTE>(m_hGdiObject), 0, 0, 0);
}

inline uint CPalette::GetNearestPaletteIndex(COLORREF clr) const {
	return ::GetNearestPaletteIndex(static_cast<HPALETTE>(m_hGdiObject), clr);
}

inline uint CPalette::GetPaletteEntries(uint iStart, uint cEntries, PALETTEENTRY& paletteentry) const {
	return ::GetPaletteEntries(static_cast<HPALETTE>(m_hGdiObject), iStart, cEntries, &paletteentry);
}

inline HPALETTE CPalette::GetSafeHandle() const {
	return static_cast<HPALETTE>((this != 0) ? m_hGdiObject : 0);
}

inline bool CPalette::ResizePalette(uint cEntries) {
	return toBoolean(::ResizePalette(static_cast<HPALETTE>(m_hGdiObject), cEntries));
}

inline uint CPalette::SetPaletteEntries(uint iStart, uint cEntries, const PALETTEENTRY& paletteentry) {
	return ::SetPaletteEntries(static_cast<HPALETTE>(m_hGdiObject), iStart, cEntries, &paletteentry);
}


// CPen class implementation
/////////////////////////////////////////////////////////////////////////////

inline CPen::CPen() {
}

inline CPen::CPen(int nPenStyle, int nWidth, COLORREF clr) {
	CreatePen(nPenStyle, nWidth, clr);
}

inline CPen::CPen(int nPenStyle, int nWidth,
		const LOGBRUSH& logbrush, int cStyles /* = 0 */, const DWORD* pdwStyles /* = 0 */) {
	CreatePen(nPenStyle, nWidth, logbrush, cStyles, pdwStyles);
}

inline CPen::operator HPEN() const {
	return static_cast<HPEN>(m_hGdiObject);
}

inline bool CPen::Attach(HPEN hPen) {
	return CGdiObject::_Attach(hPen);
}

inline bool CPen::CreatePen(int nPenStyle, int nWidth, COLORREF clr) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreatePen(nPenStyle, nWidth, clr);
	return (m_hGdiObject != 0);
}

inline bool CPen::CreatePen(int nPenStyle, int nWidth,
		const LOGBRUSH& logbrush, int cStyles /* = 0 */, const DWORD* pdwStyles /* = 0*/) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::ExtCreatePen(nPenStyle, nWidth, &logbrush, cStyles, pdwStyles);
	return (m_hGdiObject != 0);
}

inline bool CPen::CreatePenIndirect(const LOGPEN& logpen) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreatePenIndirect(&logpen);
	return (m_hGdiObject != 0);
}

inline bool CPen::CreateStockObject(int index) {
	if(m_hGdiObject != 0 || index < BLACK_PEN || index > WHITE_PEN)
		return false;
	m_hGdiObject = ::GetStockObject(index);
	return (m_hGdiObject != 0);
}

inline HPEN CPen::Detach() {
	return static_cast<HPEN>(CGdiObject::_Detach());
}

inline bool CPen::GetExtLogPen(EXTLOGPEN& extlogpen) const {
	return toBoolean(::GetObject(m_hGdiObject, sizeof(EXTLOGPEN), &extlogpen));
}

inline bool CPen::GetLogPen(LOGPEN& logpen) const {
	return toBoolean(::GetObject(m_hGdiObject, sizeof(LOGPEN), &logpen));
}

inline HPEN CPen::GetSafeHandle() const {
	return static_cast<HPEN>((this != 0) ? m_hGdiObject : 0);
}


// CRgn class implementation
/////////////////////////////////////////////////////////////////////////////

inline CRgn::operator HRGN() const {
	return static_cast<HRGN>(m_hGdiObject);
}

inline bool CRgn::Attach(HRGN hRgn) {
	return CGdiObject::_Attach(hRgn);
}

inline int CRgn::CombineRgn(const CRgn& oRgn1, const CRgn& oRgn2, int nCombineMode) {
	if(m_hGdiObject != 0)
		return false;
	return ::CombineRgn(static_cast<HRGN>(m_hGdiObject), oRgn1, oRgn2, nCombineMode);
}

inline bool CRgn::CreateEllipticRgn(int nLeft, int nTop, int nRight, int nBottom) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateEllipticRgn(nLeft, nTop, nRight, nBottom);
	return (m_hGdiObject != 0);
}

inline bool CRgn::CreateEllipticRgnIndirect(const RECT& rect) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateEllipticRgnIndirect(&rect);
	return (m_hGdiObject != 0);
}

inline bool CRgn::CreateFromData(const XFORM* lpXForm, int cDatas, const RGNDATA* pRgnData) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::ExtCreateRegion(lpXForm, cDatas, pRgnData);
	return (m_hGdiObject != 0);
}

template<bool bDCIsDisposable> inline bool CRgn::CreateFromPath(const CDC<bDCIsDisposable>& dc) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::PathToRegion(dc);
	return (m_hGdiObject != 0);
}

inline bool CRgn::CreatePolygonRgn(const POINT* lpPoints, int cPoints, int nPolyFillMode) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreatePolygonRgn(lpPoints, cPoints, nPolyFillMode);
	return (m_hGdiObject != 0);
}

inline bool CRgn::CreatePolyPolygonRgn(const POINT* lpPoints, const int* pcPoly, int cPoints, int nPolyFillMode) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreatePolyPolygonRgn(lpPoints, pcPoly, cPoints, nPolyFillMode);
	return (m_hGdiObject != 0);
}

inline bool CRgn::CreateRectRgn(int nLeft, int nTop, int nRight, int nBottom) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateRectRgn(nLeft, nTop, nRight, nBottom);
	return (m_hGdiObject != 0);
}

inline bool CRgn::CreateRectRgnIndirect(const RECT& rect) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateRectRgnIndirect(&rect);
	return (m_hGdiObject != 0);
}

inline bool CRgn::CreateRoundRectRgn(int x1, int y1, int x2, int y2, int x3, int y3) {
	if(m_hGdiObject != 0)
		return false;
	m_hGdiObject = ::CreateRoundRectRgn(x1, y1, x2, y2, x3, y3);
	return (m_hGdiObject != 0);
}

inline bool CRgn::CreateStockObject(int index) {
	return false;
}

inline HRGN CRgn::Detach() {
	return static_cast<HRGN>(CGdiObject::_Detach());
}

inline bool CRgn::EqualRgn(const CRgn& oRgn) const {
	return toBoolean(::EqualRgn(static_cast<HRGN>(m_hGdiObject), oRgn));
}

inline int CRgn::GetRegionData(LPRGNDATA lpRgnData, int cDatas) const {
	return toBoolean(::GetRegionData(static_cast<HRGN>(m_hGdiObject), cDatas, lpRgnData));
}

inline int CRgn::GetRgnBox(RECT* lpRect) const {
	return ::GetRgnBox(static_cast<HRGN>(m_hGdiObject), lpRect);
}

inline HRGN CRgn::GetSafeHandle() const {
	return (this != 0) ? static_cast<HRGN>(m_hGdiObject) : 0;
}

inline int CRgn::OffsetRgn(int x, int y) {
	return ::OffsetRgn(static_cast<HRGN>(m_hGdiObject), x, y);
}

inline int CRgn::OffsetRgn(const POINT& pt) {
	return ::OffsetRgn(static_cast<HRGN>(m_hGdiObject), pt.x, pt.y);
}

inline bool CRgn::PtInRegion(int x, int y) const {
	return toBoolean(::PtInRegion(static_cast<HRGN>(m_hGdiObject), x, y));
}

inline bool CRgn::PtInRegion(const POINT& pt) const {
	return toBoolean(::PtInRegion(static_cast<HRGN>(m_hGdiObject), pt.x, pt.y));
}

inline bool CRgn::RectInRegion(const RECT& rect) const {
	return toBoolean(::RectInRegion(static_cast<HRGN>(m_hGdiObject), &rect));
}

inline bool CRgn::SetRectRgn(int nLeft, int nTop, int nRight, int nBottom) {
	return toBoolean(::SetRectRgn(static_cast<HRGN>(m_hGdiObject), nLeft, nTop, nRight, nBottom));
}

inline bool CRgn::SetRectRgn(const RECT& rect) {
	return toBoolean(::SetRectRgn(static_cast<HRGN>(m_hGdiObject), rect.left, rect.top, rect.right, rect.bottom));
}

} /* namespace GDI */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _GDI_OBJECT_H_ */

/* [EOF] */