// File.h

#ifndef _FILE_H_
#define _FILE_H_
#include "Object.h"


// CFile class definition
/////////////////////////////////////////////////////////////////////////////

namespace Manah {
namespace Windows {
namespace IO {

class CFile;
class CFileException;

class CFile : CObject {
	// 
public:
	enum OpenFlag {
		modeRead		= 0x0000,	// ǂݎp
		modeWrite		= 0x0001,	// ݐp
		modeReadWrite	= 0x0002,	// ǂݏ\
		shareCompat		= 0x0000,	// gp
		shareExclusive	= 0x0010,	// vZX̓ǂݏ
		shareDenyWrite	= 0x0020,	// vZX݂̏
		shareDenyRead	= 0x0030,	// vZX̓ǂݎ
		shareDenyNone	= 0x0040,	// vZXƋL
		modeNoInherit	= 0x0080,	// qvZXŃt@CpȂ
		modeCreate		= 0x1000,	// (t@CɂĂ) t@C쐬
		modeNoTruncate	= 0x2000,	// modeCreate Ƃ̑gݍ킹Ŋ̃t@CjȂ
//		typeText		= 0x4000,
//		typeBinary		= 0x8000
	};

	// f[^o
private:
	HANDLE	m_hFile;		// file handle
protected:
	tstring	m_strFileName;	// full path
	bool	m_bAutoClose;	// whether close file when deleted object

	// RXgN^
public:
	CFile();
	CFile(HANDLE hFile);
	CFile(const tstring& strFileName, UINT nFlags) throw(CFileException);
	virtual ~CFile();

	// \bh
public:
	/* \z */
	virtual void	Abort();
	virtual CFile*	Duplicate() const throw(CFileException);
	virtual bool	Open(const tstring& strFileName, UINT nFlags, bool bThrow = true) throw(CFileException);
	virtual void	Close() throw(CFileException);

	/* Xg[ */
	virtual DWORD	Read(void* lpBuffer, DWORD dwCount) throw(CFileException);
	virtual void	Write(const void* lpBuffer, DWORD dwCount) throw(CFileException);
	virtual void	Flush() throw(CFileException);

	/* bN */
	virtual void	LockRange(DWORD dwPos, DWORD dwCount) throw(CFileException);
	virtual void	UnlockRange(DWORD dwPos, DWORD dwCount) throw(CFileException);

	/*  */
	virtual DWORD	GetPosition() const throw(CFileException);
	virtual DWORD	GetFileSize(DWORD* lpFileSizeHigh) const throw(CFileException);
	virtual bool	GetFileTime(LPFILETIME lpCreationTime, LPFILETIME lpLastAccessTime, LPFILETIME lpLastWriteTime) const throw(CFileException);
	virtual DWORD	GetCompressedFileSize(DWORD* lpFileSizeHigh) const throw(CFileException);
	virtual bool	IsOpened() const;
};


// CFileException class definition
/////////////////////////////////////////////////////////////////////////////

class CFileException : std::runtime_error {
	// RXgN^
public:
	explicit CFileException(const std::string& what_arg) : runtime_error(what_arg) {}
};


// Global elements
/////////////////////////////////////////////////////////////////////////////

inline std::string GenerateErrorMessage() {
	std::string	strMessage;
	void*		lpszMessage = 0;
	DWORD		n = ::GetLastError();

	::FormatMessageA(FORMAT_MESSAGE_ALLOCATE_BUFFER
		| FORMAT_MESSAGE_FROM_SYSTEM
		| FORMAT_MESSAGE_IGNORE_INSERTS,
		0, n, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		reinterpret_cast<char*>(&lpszMessage), 0, 0);
	strMessage = static_cast<char*>(lpszMessage);
	::LocalFree(lpszMessage);

	return strMessage;
}


// CFile class implementation
/////////////////////////////////////////////////////////////////////////////

inline CFile::CFile() : m_hFile(0), m_bAutoClose(false) {
}

inline CFile::CFile(HANDLE hFile) : m_hFile(hFile), m_bAutoClose(false) {
}

inline CFile::CFile(const tstring& strFileName, UINT nFlags) : m_hFile(0) {
	try {
		Open(strFileName, nFlags);
	} catch(CFileException& /* e */) {
		throw;
	}
}

inline CFile::~CFile() {
	if(m_hFile != 0 && m_bAutoClose)
		Close();
}

inline void CFile::Abort() {
	AssertValid();

	if(m_hFile != 0) {
		::CloseHandle(m_hFile);
		m_hFile = 0;
	}
	m_strFileName.erase();
}

inline void CFile::Close() {
	AssertValid();

	if(m_hFile != 0) {
		if(!::CloseHandle(m_hFile))
			throw CFileException(GenerateErrorMessage());

		m_hFile = 0;
	}
	m_bAutoClose = false;
	m_strFileName.erase();
}

inline CFile* CFile::Duplicate() const {
	AssertValid();

	if(m_hFile == 0)
		return 0;

	CFile*	pFile = new CFile();
	HANDLE	hFile;
	if (!::DuplicateHandle(::GetCurrentProcess(), m_hFile,
			::GetCurrentProcess(), &hFile, 0, false, DUPLICATE_SAME_ACCESS)){
		delete pFile;
		throw CFileException(GenerateErrorMessage());
	}
	pFile->m_hFile = hFile;
	assert(pFile->m_hFile != 0);
	pFile->m_bAutoClose = m_bAutoClose;

	return pFile;
}

inline void CFile::LockRange(DWORD dwPos, DWORD dwCount) {
	AssertValid();
	if(!::LockFile(m_hFile, dwPos, 0, dwCount, 0))
		throw CFileException(GenerateErrorMessage());
}

inline void CFile::Flush() {
	AssertValid();

	if(!::FlushFileBuffers(m_hFile))
		throw CFileException(GenerateErrorMessage());
}

inline DWORD CFile::GetFileSize(DWORD* lpFileSizeHigh) const {
	AssertValid();

	DWORD	dwSize = ::GetFileSize(m_hFile, lpFileSizeHigh);
	if(dwSize == static_cast<DWORD>(-1) && ::GetLastError() != NO_ERROR)
		throw CFileException(GenerateErrorMessage());

	return dwSize;
}

inline bool CFile::GetFileTime(LPFILETIME lpCreationTime,
		LPFILETIME lpLastAccessTime, LPFILETIME lpLastWriteTime) const throw(CFileException) {
	AssertValid();

	bool	b = toBoolean(::GetFileTime(m_hFile, lpCreationTime, lpLastAccessTime, lpLastWriteTime));
	if(!b && ::GetLastError() != NO_ERROR)
		throw CFileException(GenerateErrorMessage());

	return b;
}

inline DWORD CFile::GetCompressedFileSize(DWORD* lpFileSizeHigh) const {
	AssertValid();

	DWORD	dwSize = ::GetCompressedFileSize(m_strFileName.c_str(), lpFileSizeHigh);
	if(dwSize == static_cast<DWORD>(-1) && ::GetLastError() != NO_ERROR)
		throw CFileException(GenerateErrorMessage());

	return dwSize;
}

inline DWORD CFile::GetPosition() const {
	AssertValid();

	DWORD dwPos = ::SetFilePointer(m_hFile, 0, 0, FILE_CURRENT);
	if(dwPos == static_cast<DWORD>(-1))
		throw CFileException(GenerateErrorMessage());

	return dwPos;
}

inline bool CFile::IsOpened() const {
	AssertValid();
	return (m_hFile != 0);
}

inline bool CFile::Open(const tstring& strFileName, UINT nFlags, bool bThrow /* = true */) {
	AssertValid();

	if(m_hFile != 0 /*|| (nFlags & typeText) || (nFlags & typeBinary)*/) {
		if(bThrow)
			throw CFileException("File is already opened.");
		else
			return false;
	}

	HANDLE	hFile;
	DWORD	dwAccess, dwShareMode, dwCreateFlag;

	switch(nFlags & 0x03) {
	case modeRead:		dwAccess = GENERIC_READ;					break;
	case modeWrite:		dwAccess = GENERIC_WRITE;					break;
	case modeReadWrite:	dwAccess = GENERIC_READ | GENERIC_WRITE;	break;
	default:			assert(false);
	}

	switch(nFlags & 0x70) {
	case shareCompat:
	case shareExclusive:	dwShareMode = 0;									break;
	case shareDenyWrite:	dwShareMode = FILE_SHARE_READ;						break;
	case shareDenyRead:		dwShareMode = FILE_SHARE_WRITE;						break;
	case shareDenyNone:		dwShareMode = FILE_SHARE_READ | FILE_SHARE_WRITE;	break;
	}

	if(nFlags & modeCreate) {
		if (nFlags & modeNoTruncate)
			dwCreateFlag = OPEN_ALWAYS;
		else
			dwCreateFlag = CREATE_ALWAYS;
	} else
		dwCreateFlag = OPEN_EXISTING;

	hFile = ::CreateFile(strFileName.c_str(), dwAccess,
		dwShareMode, 0, dwCreateFlag, FILE_ATTRIBUTE_NORMAL, 0);
	if(hFile == INVALID_HANDLE_VALUE) {
		if(bThrow)
			throw CFileException(GenerateErrorMessage());
		else
			return false;
	}

	m_hFile = hFile;
	m_bAutoClose = true;

	return true;
}

inline DWORD CFile::Read(void* lpBuffer, DWORD dwCount) {
	AssertValid();
	assert(lpBuffer != 0);

	if(dwCount == 0)
		return 0;

	DWORD	dwRead;
	if(!::ReadFile(m_hFile, lpBuffer, dwCount, &dwRead, 0))
		throw CFileException(GenerateErrorMessage());

	return dwRead;
}

inline void CFile::UnlockRange(DWORD dwPos, DWORD dwCount) {
	AssertValid();
	if(!::UnlockFile(m_hFile, dwPos, 0, dwCount, 0))
		throw CFileException(GenerateErrorMessage());
}

inline void CFile::Write(const void* lpBuffer, DWORD dwCount) {
	AssertValid();
	assert(lpBuffer != 0);

	if(dwCount == 0)
		return;

	DWORD dwWritten;
	if(!::WriteFile(m_hFile, lpBuffer, dwCount, &dwWritten, 0))
		throw CFileException(GenerateErrorMessage());
}

} /* namespace IO */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _FILE_H_ */

/* [EOF] */