// Edit.h
/////////////////////////////////////////////////////////////////////////////

#ifndef __EDIT_H__
#define __EDIT_H__

#include "Window.h"


// CEdit class definition
/////////////////////////////////////////////////////////////////////////////

namespace Manah {
namespace Windows {
namespace Controls {

class CEdit : public CWindow {
	// RXgN^
public:
	CEdit();
	CEdit(const CEdit& rhs);
	virtual ~CEdit();

public:
	bool	Attach(HWND hWnd, bool bSubClass = false);
	bool	Create(bool bSubClass, DWORD dwStyle, const RECT& rect, HWND hwndParent, UINT id);
	bool	CreateEx(bool nSubClass, DWORD dwExStyle,
				DWORD dwStyle, const RECT& rect, HWND hwndParent, UINT id);

	bool				CanUndo() const;
	int					CharFromPos(const POINT& pt) const;
	void				EmptyUndoBuffer();
	bool				FmtLines(bool bAddEOL);
	int					GetFirstVisibleLine() const;
	HLOCAL				GetHandle() const;
	UINT				GetLimitText() const;
	int					GetLine(int nIndex, TCHAR* lpszBuffer) const;
	int					GetLine(int nIndex, TCHAR* lpszBuffer, int nMaxLength) const;
	int					GetLineCount() const;
	DWORD				GetMargins() const;
	bool				GetModify() const;
	void				GetRect(RECT& rect) const;
	DWORD				GetSel() const;
	void				GetSel(int& nStartChar, int& nEndChar) const;
	int					GetThumb() const;
	TCHAR				GetPasswordChar() const;
	EDITWORDBREAKPROC	GetWordBreakProc() const;
	void				LimitText(int nChars = 0);
	int					LineFromChar(int nIndex = -1) const;
	int					LineIndex(int nIndex = -1) const;
	int					LineLength(int nLine = -1) const;
	void				LineScroll(int nLines, int nChars = 0);
	POINT				PosFromChar(UINT nChar) const;
	void				ReplaceSel(const TCHAR* lpszNewText, bool nCanUndo = false);
	void				SetHandle(HLOCAL hBuffer);
	void				SetLimitText(UINT nMax);
	void				SetMargins(UINT nLeft, UINT nRight);
	void				SetModify(bool bModified = true);
	void				SetPasswordChar(TCHAR ch);
	void				SetReadOnly(bool bReadOnly = true);
	void				SetRect(const RECT& rect);
	void				SetRectNP(const RECT& rect);
	void				SetSel(DWORD dwSelection, bool bNoScroll = false);
	void				SetSel(int nStartChar, int nEndChar, bool bNoScroll = false);
	void				SetTabStops();
	bool				SetTabStops(const int& cxEachStop);
	bool				SetTabStops(int nTabStop, INT* rgTabStops);
	void				SetWordBreakProc(EDITWORDBREAKPROC ewbp);
#ifdef EM_GETIMESTATUS
	DWORD				GetImeStatus(DWORD dwType) const;
	DWORD				SetImeStatus(DWORD dwType, DWORD dwData);
#endif
#ifdef EM_GETCUEBANNER
	bool				GetCueBanner(WCHAR* lpszText, int cchText) const;
	bool				HideBalloonTip();
	bool				SetCueBanner(const WCHAR* lpszText);
	bool				ShowBalloonTip(const EDITBALLOONTIP& ebt);
#endif

	// ǉ\bh
	void			AdjustOtherWindow(HWND hWnd);
	bool			CanPaste() const;
	void			EnableContextMenu(bool bEnable = true);
	void			EnableIMEContext(bool bEnable = true);
	void			EnableSubmit(bool bEnable = true);
	void			LimitInputChar(const TCHAR* lpszAcceptChars);
	UINT			GetSelCount() const;
	const TCHAR*	GetSelection() const;
	bool			IsReadOnly() const;
	bool			ScrollCaret();
	void			SetFont(HFONT hFont, bool bReDraw = true);

	// I[o[Ch
protected:
	OVERRIDE_STANDARD_DISPATCH_EVENT(CEdit);

	// f[^o
private:
	bool	m_bEnableContextMenu;
	HIMC	m_hIMEContext;
	TCHAR*	m_pszAcceptChars;
	bool	m_bReturnToCommand;
};


// CEdit class implementation
/////////////////////////////////////////////////////////////////////////////

#define UNCONST_THIS	const_cast<CEdit*>(this)

inline CEdit::CEdit() : m_bEnableContextMenu(true), m_hIMEContext(0), m_pszAcceptChars(0), m_bReturnToCommand(false) {
}

inline CEdit::CEdit(const CEdit& rhs) : Manah::Windows::Controls::CWindow(rhs) {
	m_bEnableContextMenu = rhs.m_bEnableContextMenu;
	m_hIMEContext = 0;
	m_pszAcceptChars = new TCHAR[_tcslen(rhs.m_pszAcceptChars) + 1];
	_tcscpy(m_pszAcceptChars, rhs.m_pszAcceptChars);
	m_bReturnToCommand = rhs.m_bReturnToCommand;
}

inline CEdit::~CEdit() {
	if(m_hIMEContext != 0)
		::ImmAssociateContext(m_hWnd, m_hIMEContext);
	if(m_pszAcceptChars != 0)
		delete[] m_pszAcceptChars;
}

inline bool CEdit::Attach(HWND hWnd, bool bSubClass /* = false */) {
	AssertValid();

	if(IsWindow() || !::IsWindow(hWnd) || m_OldProc != 0)
		return false;
	m_hWnd = hWnd;

	if(bSubClass){
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(STANDARD_WINDOW_CLASS_WNDPROC(CEdit))));
		assert(m_OldProc != STANDARD_WINDOW_CLASS_WNDPROC(CEdit));
		m_lOldUserData = GetWindowLong(GWL_USERDATA);
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline bool CEdit::Create(bool bSubClass, DWORD dwStyle, const RECT& rect, HWND hwndParent, UINT id) {
	AssertValid();

	m_hWnd = ::CreateWindow(
		_T("Edit"), _T("EditWindow"), dwStyle,
		rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top,
		hwndParent, reinterpret_cast<HMENU>(id),
		reinterpret_cast<HINSTANCE>(::GetWindowLong(hwndParent, GWL_HINSTANCE)), 0L);
	if(m_hWnd == 0)
		return false;
	if(bSubClass){
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(STANDARD_WINDOW_CLASS_WNDPROC(CEdit))));
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline bool CEdit::CreateEx(bool bSubClass, DWORD dwExStyle, DWORD dwStyle, const RECT& rect, HWND hwndParent, UINT id) {
	AssertValid();

	m_hWnd = ::CreateWindowEx(dwExStyle,
		_T("Edit"), _T("EditWindow"), dwStyle,
		rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top,
		hwndParent, reinterpret_cast<HMENU>(id),
		reinterpret_cast<HINSTANCE>(::GetWindowLong(hwndParent, GWL_HINSTANCE)), 0L);
	if(m_hWnd == 0)
		return false;
	if(bSubClass){
		m_OldProc = reinterpret_cast<WNDPROC>(SetWindowLong(
			GWL_WNDPROC, reinterpret_cast<LPARAM>(STANDARD_WINDOW_CLASS_WNDPROC(CEdit))));
		SendMessage(WM_MATTACH, 0, reinterpret_cast<LPARAM>(this));
	}

	return true;
}

inline bool CEdit::CanUndo() const {
	return toBoolean(UNCONST_THIS->SendMessage(EM_CANUNDO, 0, 0));
}

inline bool CEdit::CanPaste() const {
	return ((IsClipboardFormatAvailable(CF_TEXT)
		|| IsClipboardFormatAvailable(CF_OEMTEXT)) && (!IsReadOnly()));
}

inline LRESULT CEdit::DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam) {
	if(PreTranslateMessage(message, wParam, lParam))
		return false;

	switch(message){
	case WM_CHAR:
		if(m_pszAcceptChars != 0){
			if(_tcschr(m_pszAcceptChars, static_cast<TCHAR>(wParam)) == 0){
				::MessageBeep(MB_OK);
				return false;
			}
		}
		break;
	case WM_CONTEXTMENU:
		if(!m_bEnableContextMenu)
			return false;
		break;
	case WM_KEYDOWN:
		if(static_cast<int>(wParam) == VK_RETURN && m_bReturnToCommand){
			::SendMessage(::GetParent(m_hWnd),
				WM_COMMAND, MAKEWPARAM(GetDlgCtrlID(), IDOK), reinterpret_cast<LPARAM>(m_hWnd));
			return true;
		}
		break;
	}

	return CWindow::DispatchEvent(message, wParam, lParam);
}

inline void CEdit::EnableContextMenu(bool bEnable /* = true */) {
	m_bEnableContextMenu = bEnable;
}

inline void CEdit::EnableIMEContext(bool bEnable /* = true */) {
	AssertValidAsWindow();

	if(bEnable){
		if(m_hIMEContext != 0)
			::ImmAssociateContext(m_hWnd, m_hIMEContext);
		else
			return;
	}else{
		if(m_hIMEContext != 0)
			return;
		else
			m_hIMEContext = ::ImmAssociateContext(m_hWnd, 0);
	}
}

inline void CEdit::EnableSubmit(bool bEnable /* = true */) {
	m_bReturnToCommand = bEnable;
}

inline int CEdit::GetLineCount() const {
	return static_cast<int>(UNCONST_THIS->SendMessage(EM_GETLINECOUNT));
}

inline bool CEdit::GetModify() const {
	return toBoolean(UNCONST_THIS->SendMessage(EM_GETMODIFY));
}

inline void CEdit::SetModify(bool bModified /* = true */) {
	SendMessage(EM_SETMODIFY, bModified);
}

inline int CEdit::GetThumb() const {
	return UNCONST_THIS->SendMessage(EM_GETTHUMB);
}

inline void CEdit::GetRect(RECT& rect) const {
	UNCONST_THIS->SendMessage(EM_GETRECT, 0, reinterpret_cast<LPARAM>(&rect));
}

inline DWORD CEdit::GetSel() const {
	return static_cast<DWORD>(UNCONST_THIS->SendMessage(EM_GETSEL));
}

inline void CEdit::GetSel(int& nStartChar, int& nEndChar) const {
	UNCONST_THIS->SendMessage(EM_GETSEL, nStartChar, nEndChar);
}

inline HLOCAL CEdit::GetHandle() const {
	return reinterpret_cast<HGLOBAL>(UNCONST_THIS->SendMessage(EM_GETHANDLE));
}

inline void CEdit::SetHandle(HLOCAL hBuffer) {
	SendMessage(EM_SETHANDLE, reinterpret_cast<WPARAM>(hBuffer));
}

inline void CEdit::SetMargins(UINT nLeft, UINT nRight) {
	SendMessage(EM_SETMARGINS, EC_LEFTMARGIN | EC_RIGHTMARGIN, MAKELONG(nLeft, nRight));
}

inline DWORD CEdit::GetMargins() const {
	return static_cast<DWORD>(UNCONST_THIS->SendMessage(EM_GETMARGINS));
}

inline void CEdit::SetLimitText(UINT nMax) {
	SendMessage(EM_SETLIMITTEXT, nMax);
}

inline UINT CEdit::GetLimitText() const {
	return static_cast<UINT>(UNCONST_THIS->SendMessage(EM_GETLIMITTEXT));
}

inline POINT CEdit::PosFromChar(UINT nChar) const {
	DWORD dw;
	POINT point;

	dw =  UNCONST_THIS->SendMessage(EM_POSFROMCHAR, nChar);
	point.x = LOWORD(dw);
	point.y = HIWORD(dw);

	return point;
}

inline int CEdit::CharFromPos(const POINT& pt) const {
	return static_cast<int>(UNCONST_THIS->SendMessage(EM_CHARFROMPOS, 0, MAKELPARAM(pt.x, pt.y)));
}

inline int CEdit::GetLine(int nIndex, TCHAR* lpszBuffer) const {
	return static_cast<int>(UNCONST_THIS->SendMessage(EM_GETLINE, nIndex, reinterpret_cast<LPARAM>(lpszBuffer)));
}

inline int CEdit::GetLine(int nIndex, TCHAR* lpszBuffer, int nMaxLength) const {
	return static_cast<int>(UNCONST_THIS->SendMessage(EM_GETLINE, nIndex, reinterpret_cast<LPARAM>(lpszBuffer)));
}

inline TCHAR CEdit::GetPasswordChar() const {
	return static_cast<TCHAR>(UNCONST_THIS->SendMessage(EM_GETPASSWORDCHAR));
}

inline int CEdit::GetFirstVisibleLine() const {
	return static_cast<int>(UNCONST_THIS->SendMessage(EM_GETFIRSTVISIBLELINE));
}

inline void CEdit::EmptyUndoBuffer() {
	SendMessage(EM_EMPTYUNDOBUFFER);
}

inline bool CEdit::FmtLines(bool bAddEOL) {
	return toBoolean(SendMessage(EM_FMTLINES, bAddEOL));
}

inline void CEdit::LimitText(int nChars /* = 0 */) {
	SendMessage(EM_LIMITTEXT, nChars);
}

inline void CEdit::LimitInputChar(const TCHAR* lpszAcceptChars) {
	if(m_pszAcceptChars != 0)
		delete[] m_pszAcceptChars;
	if(lpszAcceptChars == 0)
		m_pszAcceptChars = 0;
	else {
		m_pszAcceptChars = new TCHAR[_tcslen(lpszAcceptChars) + 1];
		_tcscpy(m_pszAcceptChars, lpszAcceptChars);
	}
}

inline int CEdit::LineFromChar(int nIndex /* = -1 */) const {
	return static_cast<int>(UNCONST_THIS->SendMessage(EM_LINEFROMCHAR, nIndex));
}

inline int CEdit::LineIndex(int nIndex /* = -1 */) const {
	return static_cast<int>(UNCONST_THIS->SendMessage(EM_LINEINDEX, nIndex));
}

inline int CEdit::LineLength(int nLine /* = -1 */) const {
	return static_cast<int>(
		UNCONST_THIS->SendMessage(EM_LINELENGTH, UNCONST_THIS->SendMessage(EM_LINEINDEX, nLine)));
}

inline void CEdit::LineScroll(int nLines, int nChars /* = 0 */) {
	SendMessage(EM_LINESCROLL, nChars, nLines);
}

inline void CEdit::ReplaceSel(const TCHAR* lpszNewText, bool bCanUndo /* = false */) {
	SendMessage(EM_REPLACESEL, bCanUndo, reinterpret_cast<LPARAM>(lpszNewText));
}

inline void CEdit::SetPasswordChar(TCHAR ch) {
	SendMessage(EM_SETPASSWORDCHAR, ch);
}

inline void CEdit::SetRect(const RECT& rect) {
	SendMessage(EM_SETRECT, 0, reinterpret_cast<LPARAM>(&rect));
}

inline void CEdit::SetRectNP(const RECT& rect) {
	SendMessage(EM_SETRECTNP, 0, reinterpret_cast<LPARAM>(&rect));
}

inline void CEdit::SetSel(DWORD dwSelection, bool bNoScroll /* = false */) {
	SendMessage(EM_SETSEL, LOWORD(dwSelection), HIWORD(dwSelection));
}

inline void CEdit::SetSel(int nStartChar, int nEndChar, bool bNoScroll /* = false */) {
	SendMessage(EM_SETSEL, nStartChar, nEndChar);
}

inline void CEdit::SetTabStops() {
	SendMessage(EM_SETTABSTOPS, 32);
}

inline bool CEdit::SetTabStops(const int& cxEachStop) {
	return toBoolean(SendMessage(EM_SETTABSTOPS, cxEachStop));
}

inline bool CEdit::SetTabStops(int nTabStops, INT* rgTabStops) {
	return toBoolean(SendMessage(EM_SETTABSTOPS, nTabStops, reinterpret_cast<LPARAM>(rgTabStops)));
}

inline void CEdit::SetReadOnly(bool bReadOnly /* = true */) {
	SendMessage(EM_SETREADONLY, bReadOnly);
}

inline void CEdit::AdjustOtherWindow(HWND hWnd) {
	DWORD	dwSelection;
	POINT	point;
	RECT	rect;

	dwSelection = GetSel();
	point = PosFromChar(LOWORD(dwSelection));
	ClientToScreen(point);
	::GetWindowRect(hWnd, &rect);

	if(::PtInRect(&rect, point)) {
		if(point.y > rect.bottom - rect.top)
			::OffsetRect(&rect, 0, point.y - rect.bottom - 20);
		else if(point.y + rect.bottom - rect.top < ::GetSystemMetrics(SM_CYSCREEN))
			::OffsetRect(&rect, 0, 40 + point.y - rect.top);
		::MoveWindow(hWnd, rect.left, rect.top, rect.right, rect.bottom, true);
	}
}

inline bool CEdit::IsReadOnly() const {
	return toBoolean(GetWindowLong(GWL_STYLE) & ES_READONLY);
}

inline UINT CEdit::GetSelCount() const {
	DWORD dwSel;

	dwSel = UNCONST_THIS->SendMessage(EM_GETSEL, 0, 0L);
	return HIWORD(dwSel) - LOWORD(dwSel);
}

// ĂԂ
inline LPCTSTR CEdit::GetSelection() const {
	int				nTextLength;
	UINT			nSelLength;
	DWORD			dwSel;
	static TCHAR	szWholeText[64 * 1024];

	dwSel = UNCONST_THIS->SendMessage(EM_GETSEL);
	nSelLength = HIWORD(dwSel) - LOWORD(dwSel);
	nTextLength = GetWindowTextLength();
	GetWindowText(szWholeText, nTextLength + 1);
	memmove(szWholeText, szWholeText + LOWORD(dwSel), sizeof(_TCHAR) * nSelLength);
	*(szWholeText + nSelLength) = '\0';

	return szWholeText;
}

inline bool CEdit::ScrollCaret() {
	return toBoolean(SendMessage(EM_SCROLLCARET));
}

inline void CEdit::SetFont(HFONT hFont, bool bRedraw /* = true */) {
	SendMessage(WM_SETFONT, reinterpret_cast<WPARAM>(hFont), bRedraw);
}

inline EDITWORDBREAKPROC CEdit::GetWordBreakProc() const {
	return reinterpret_cast<EDITWORDBREAKPROC>(UNCONST_THIS->SendMessage(EM_GETWORDBREAKPROC));
}

inline void CEdit::SetWordBreakProc(EDITWORDBREAKPROC ewbp) {
	SendMessage(EM_SETWORDBREAKPROC, 0, reinterpret_cast<LPARAM>(ewbp));
}

#ifdef EM_GETIMESTATUS
inline DWORD CEdit::GetImeStatus(DWORD type) const {
	return UNCONST_THIS->SendMessage(EM_GETIMESTATUS, type);
}

inline DWORD CEdit::SetImeStatus(DWORD type, DWORD data) {
	return SendMessage(EM_SETIMESTATUS, type, data);
}
#endif /* EM_GETIMESTATUS */

#ifdef EM_GETCUEBANNER
inline bool CEdit::GetCueBanner(WCHAR* lpszText, int cchText) const {
	AssertValidAsWindow();
	return toBoolean(Edit_GetCueBannerText(m_hWnd, lpszText, cchText));
}

inline bool CEdit::SetCueBanner(const WCHAR* lpszText) {
	AssertValidAsWindow();
	return toBoolean(Edit_SetCueBannerText(m_hWnd, lpszText));
}

inline bool CEdit::ShowBalloonTip(const EDITBALLOONTIP& ebt) {
	AssertValidAsWindow();
	return toBoolean(Edit_ShowBalloonTip(m_hWnd, &ebt));
}

inline bool CEdit::HideBalloonTip() {
	AssertValidAsWindow();
	return toBoolean(Edit_HideBalloonTip(m_hWnd));
}
#endif /* EM_GETCUEBANNER */

#undef UNCONST_THIS

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#endif /* __EDIT_H__ */

/* EOF */