// DocumentView.h
// (c) 2003-2004 exeal

#ifndef _DOCUMENT_VIEW_H_
#define _DOCUMENT_VIEW_H_
#include "Window.h"
#include <list>


namespace Manah {
namespace Windows {

// CDocument class definition
/////////////////////////////////////////////////////////////////////////////

class CView;
class CController;

typedef std::list<CView*>::const_iterator	VIEW_POSITION;

class CDocument : public CObject {
	// RXgN^
public:
	CDocument(CController* pController = 0);
	virtual ~CDocument();	// destructor destories all containing views
private:
	CDocument(const CDocument& rhs);

	// Zq
private:
	operator =(const CDocument& rhs);

	// \bh
public:
	CController*	GetController() const;
	// r[
	void					AddView(CView& view);
	virtual VIEW_POSITION	GetFirstViewPosition() const;
	virtual CView*			GetNextView(VIEW_POSITION& vp) const;
	unsigned long			GetViewCount() const;
	void					RemoveView(CView& view);
	void					SetupAllViews(LPARAM lParam = 0L);
	void					UpdateAllViews(CObject* pSender, LPARAM lParam = 0L, void* pHint = 0);
	// hLg
	tstring			GetExtensionName() const;
	const tstring&	GetPathName() const;
	const tstring&	GetTitle() const;
	virtual bool	IsModified() const;
	virtual void	SetModified(bool bModified = true);
	void			SetPathName(const tstring& strPathName);
	void			SetTitle(const tstring& strTitle);

protected:
	virtual void	OnChangedViewList();
	virtual void	OnCloseDocument();

	// f[^o
protected:
	CController*		m_pController;
	tstring				m_strTitle;
	tstring				m_strPathName;
	bool				m_bModified;
	std::list<CView*>	m_views;
};


// CView class definition
/////////////////////////////////////////////////////////////////////////////

class CView : public Manah::Windows::Controls::CWindow {
	friend class CDocument;

	// RXgN^
public:
	CView();
	CView(const CView& rhs);	// creates new view belongs to no documents
	virtual ~CView();

	// \bh
public:
	virtual bool	Create(HWND hwndParent, const RECT& rect,	// always fail (return false)
						DWORD dwStyle, DWORD dwExStyle) throw(std::runtime_error);
	CDocument*		GetDocument() const;
protected:
	virtual void	OnDocumentSetup(LPARAM lHint = 0);
	virtual void	OnUpdate(CObject* pSender, LPARAM lHint, void* pHint);

	// f[^o
protected:
	CDocument*	m_pDocument;
};


// CController class definition
/////////////////////////////////////////////////////////////////////////////

class CController : public Manah::Windows::Controls::CWindow {
	// RXgN^
public:
	CController();
	~CController();
private:
	CController(const CController& rhs);	// never been invoked
	operator =(const CController& rhs);		// never been invoked

	// \bh
public:
	virtual bool	Create(HWND hwndParent, const RECT& rect,
						DWORD dwStyle, DWORD dwExStyle, DWORD dwViewStyle, DWORD dwViewExStyle);
	CDocument*		GetDocument() const;
	int				GetPadding() const;
	void			SetPadding(int nPadding);
protected:
	virtual void		_AdjustView();
	virtual CDocument*	_NewDocumentInstance() = 0;
	virtual CView*		_NewViewInstance() = 0;

	// bZ[Wnh
protected:
	virtual void OnSize(UINT nType, int cx, int cy);

protected:
	friend class WndProcProvider<CController, false>;

	// f[^o
protected:
	CDocument*	m_pDocument;
	CView*		m_pView;
private:
	int			m_nPadding;
};


// CDocument class implementation
/////////////////////////////////////////////////////////////////////////////

inline CDocument::CDocument(CController* pController /* = 0 */) : m_pController(pController), m_bModified(false) {
}

inline CDocument::~CDocument() {
	for(std::list<CView*>::iterator it = m_views.begin(); it != m_views.end(); ++it) {
		if((*it)->IsWindow())
			(*it)->DestroyWindow();
		delete *it;
	}
	m_views.erase(m_views.begin(), m_views.end());
}

inline void CDocument::AddView(CView& view) {
	AssertValid();
	view.m_pDocument = this;
	m_views.push_back(const_cast<CView*>(&view));
}

inline CController* CDocument::GetController() const {
	AssertValid();
	return m_pController;
}

inline tstring CDocument::GetExtensionName() const {
	AssertValid();

	tstring::size_type	i = m_strTitle.rfind(_T("."));
	if(i == tstring::npos)
		return _T("");
	return m_strTitle.substr(i + 1);
}

inline VIEW_POSITION CDocument::GetFirstViewPosition() const {
	AssertValid();
	return m_views.begin();
}

inline CView* CDocument::GetNextView(VIEW_POSITION& vp) const {
	AssertValid();
	if(vp == m_views.end())
		return 0;

	CView*	pView = *vp;
	++vp;
	return pView;
}

inline const tstring& CDocument::GetPathName() const {
	AssertValid();
	return m_strPathName;
}

inline const tstring& CDocument::GetTitle() const {
	AssertValid();
	return m_strTitle;
}

inline unsigned long CDocument::GetViewCount() const {
	AssertValid();
	return m_views.size();
}

inline bool CDocument::IsModified() const {
	AssertValid();
	return m_bModified;
}

inline void CDocument::RemoveView(CView& view) {
	AssertValid();
	view.m_pDocument = 0;
	m_views.remove(&view);
}

inline void CDocument::SetModified(bool bModified /* = true */) {
	AssertValid();
	m_bModified = bModified;
}

inline void CDocument::SetPathName(const tstring& strPathName) {
	AssertValid();
	m_strPathName = strPathName;
}

inline void CDocument::SetTitle(const tstring& strTitle) {
	AssertValid();
	m_strTitle = const_cast<tstring&>(strTitle);
}

inline void CDocument::SetupAllViews(LPARAM lParam /* = 0L */) {
	AssertValid();

	std::list<CView*>::iterator	it = m_views.begin();
	while(it != m_views.end()) {
		(*it)->OnDocumentSetup();
		++it;
	}
}

inline void CDocument::UpdateAllViews(CObject* pSender, LPARAM lParam /* = 0L */, void* pHint /* = 0 */) {
	AssertValid();
	
	std::list<CView*>::iterator	it = m_views.begin();
	while(it != m_views.end()) {
		(*it)->OnUpdate(pSender, lParam, pHint);
		++it;
	}
}

inline void CDocument::OnChangedViewList() {
	if(m_views.empty())
		OnCloseDocument();
}

inline void CDocument::OnCloseDocument() {
}


// CView class implementation
/////////////////////////////////////////////////////////////////////////////

inline CView::CView() : m_pDocument(0) {
}

inline CView::CView(const CView& rhs) : CWindow(rhs), m_pDocument(0) {
}

inline CView::~CView() {
}

inline bool CView::Create(HWND hwndParent, const RECT& rect,
		DWORD dwStyle, DWORD dwExStyle) throw(std::runtime_error) {
	return false;
}

inline CDocument* CView::GetDocument() const {
	AssertValid();
	return m_pDocument;
}

inline void CView::OnDocumentSetup(LPARAM lHint /* = 0 */) {
}

inline void CView::OnUpdate(CObject* pSender, LPARAM lHint, void* pHint) {
	InvalidateRect(0);
}


// CController class implementation
/////////////////////////////////////////////////////////////////////////////

inline CController::CController() : m_pDocument(0), m_pView(0), m_nPadding(0) {
}

inline CController::~CController() {
	delete m_pDocument;
}

inline void CController::_AdjustView() {
	AssertValidAsWindow();
	if(m_pView == 0 || !m_pView->IsWindow())
		return;

	RECT	rect;

	GetClientRect(rect);
	::InflateRect(&rect, -m_nPadding, -m_nPadding);
	m_pView->MoveWindow(rect);
}

inline bool CController::Create(HWND hwndParent, const RECT& rect,
		DWORD dwStyle, DWORD dwExStyle, DWORD dwViewStyle, DWORD dwViewExStyle) {
	AssertValid();
	assert(::IsWindow(hwndParent));
	assert(m_pDocument == 0);

	if(IsWindow())
		return false;

	HINSTANCE	hInstance = reinterpret_cast<HINSTANCE>(::GetWindowLong(hwndParent, GWL_HINSTANCE));
	WNDCLASSEX	wc;
	wc.cbSize = sizeof(WNDCLASSEX);
	if(!::GetClassInfoEx(hInstance, _T("ControllerPane"), &wc)) {
		wc.style			= CS_BYTEALIGNCLIENT | CS_BYTEALIGNWINDOW | CS_DBLCLKS;
		wc.lpfnWndProc		= reinterpret_cast<WNDPROC>(WINDOW_CLASS_WNDPROC(CController));
		wc.cbClsExtra		= 0;
		wc.cbWndExtra		= sizeof(long);
		wc.hInstance		= ::GetModuleHandle(0);
		wc.hIcon			= 0;
		wc.hIconSm			= 0;
		wc.hCursor			= ::LoadCursor(0, IDC_ARROW);
		wc.hbrBackground	= ::GetSysColorBrush(COLOR_3DFACE);
		wc.lpszClassName	= _T("ControllerPane");
		wc.lpszMenuName		= 0;
		if(!::RegisterClassEx(&wc))
			return false;
	}

	m_hWnd = ::CreateWindowEx(dwExStyle, _T("ControllerPane"), _T(""), dwStyle,
			rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top,
			hwndParent, 0, hInstance, reinterpret_cast<void*>(this));
	if(m_hWnd == 0)
		return false;

	m_pDocument = _NewDocumentInstance();
	m_pView = _NewViewInstance();
	m_pDocument->AddView(*m_pView);
	if(m_pView->Create(m_hWnd, rect, dwViewStyle, dwViewExStyle))
		return true;
	else {
		delete m_pDocument;
		return false;
	}
}

inline CDocument* CController::GetDocument() const {
	AssertValid();
	return m_pDocument;
}

inline int CController::GetPadding() const {
	AssertValid();
	return m_nPadding;
}

inline void CController::SetPadding(int nPadding) {
	AssertValid();
	assert(nPadding >= 0);

	m_nPadding = nPadding;
	if(IsWindowVisible())
		_AdjustView();
}

inline void CController::OnSize(UINT nType, int cx, int cy) {
	_AdjustView();
}

} // namespace Windows
} // namespace Manah

#endif /* _DOCUMENT_VIEW_H_ */

/* [EOF] */