// DC.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _DC_H_
#define _DC_H_

#include "Object.h"

namespace Manah {
namespace Windows {
	namespace Controls {
		class CWindow;
	}
namespace GDI {

// CDC class definition
/////////////////////////////////////////////////////////////////////////////

template<bool bDisposable = true>
class CDC : public CObject {
	// RXgN^
public:
	CDC();
	CDC(const CDC<bDisposable>& rhs);
	virtual ~CDC();

	// Zq
public:
	operator HDC() const;
private:
	template<bool bDisposable>
	operator =(const CDC<bDisposable> rhs);

	// \bh
public:
	/* 쐬 */
	bool		Attach(HDC hDC);
	bool		CreateCompatibleDC(HDC hDC);
	bool		DeleteDC();
	HDC			Detach();
	HBITMAP		GetCurrentBitmap() const;
	HBRUSH		GetCurrentBrush() const;
	HFONT		GetCurrentFont() const;
	HPALETTE	GetCurrentPalette() const;
	HPEN		GetCurrentPen() const;
	HWND		GetWindow() const;

	/* foCXReLXg */
	int				GetDeviceCaps(int iIndex) const;
	HDC				GetSafeHdc() const;
	virtual bool	RestoreDC(int nSavedDC);
	virtual int		SaveDC();

	/* `c[ */
	int		EnumObjects(int nObjectType, GOBJENUMPROC lpObjectFunc, LPARAM lpData);
	POINT	GetBrushOrg() const;
	POINT	SetBrushOrg(int x, int y);
	POINT	SetBrushOrg(const POINT& pt);

	/* I */
	HBITMAP			SelectObject(HBITMAP hBitmap);
	HBRUSH			SelectObject(HBRUSH hBrush);
	virtual HFONT	SelectObject(HFONT hFont);
	HPEN			SelectObject(HPEN hPen);
	HGDIOBJ			SelectStockObject(int nObject);

	/* Fƃpbg */
	COLORREF	GetNearestColor(COLORREF clr) const;
	UINT		RealizePalette();
	HPALETTE	SelectPalette(HPALETTE hPalette, bool bForceBackground);
	void		UpdateColors();

	/*  */
	COLORREF	GetBkColor() const;
	int			GetBkMode() const;
	bool		GetColorAdjustment(COLORADJUSTMENT& colorAdjust) const;
	int			GetPolyFillMode() const;
	int			GetROP2() const;
	int			GetStretchBltMode() const;
	COLORREF	GetTextColor() const;
	COLORREF	SetBkColor(COLORREF clr);
	int			SetBkMode(int nBkMode);
	bool		SetColorAdjustment(const COLORADJUSTMENT& colorAdjust);
	int			SetPolyFillMode(int nPolyFillMode);
	int			SetROP2(int nDrawMode);
	int			SetStretchBltMode(int nStretchMode);
	COLORREF	SetTextColor(COLORREF clr);

	/* ̈ */
	bool	FillRgn(HRGN hRgn, HBRUSH hBrush);
	bool	FrameRgn(HRGN hRgn, HBRUSH hBrush, int nWidth, int nHeight);
	bool	InvertRgn(HRGN hRgn);
	bool	PaintRgn(HRGN hRgn);

	/* o */
	bool	AngleArc(int x, int y, int nRadius, float fStartAngle, float fSweepAngle);
	bool	Arc(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4);
	bool	Arc(const RECT& rect, const POINT& ptStart, const POINT& ptEnd);
	bool	ArcTo(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4);
	bool	ArcTo(const RECT& rect, const POINT& ptStart, const POINT& ptEnd);
	int		GetArcDirection() const;
	POINT	GetCurrentPosition() const;
	bool	LineTo(int x, int y);
	bool	LineTo(const POINT& pt);
	POINT	MoveTo(int x, int y);
	POINT	MoveTo(const POINT& pt);
	bool	PolyBezier(const POINT* lpPoints, int nCount);
	bool	PolyBezierTo(const POINT* lpPoints, int nCount);
	bool	PolyDraw(const POINT* lpPoints, const BYTE* lpTypes, int nCount);
	bool	Polyline(LPPOINT lpPoints, int nCount);
	bool	PolylineTo(const POINT* lpPoints, int nCount);
	bool	PolyPolyline(const POINT* lpPoints, const DWORD* lpPolyPoints, int nCount);
	int		SetArcDirection(int nArcDirection);

	/* P` */
//	void	Draw3dRect(const RECT* lpRect, COLORREF clrTopLeft, COLORREF clrRightBottom);
//	void	Draw3dRect(int x, int y, int cx, int cy, COLORREF clrTopLeft, COLORREF clrRightBottom);
//	void	DrawDragIcon(const RECT* lpRect, SIZE size,
//				const RECT* lpRectLast, SIZE sizeLast, HBRUSH hBrush, HBRUSH hLastBrush);
	bool	DrawEdge(const RECT& rect, UINT nEdge, UINT grfFlags);
	bool	DrawFrameControl(const RECT& rect, UINT nType, UINT nState);
	bool	DrawIcon(int x, int y, HICON hIcon);
	bool	DrawIcon(const POINT& pt, HICON hIcon);
	void	FillRect(const RECT& rect, HBRUSH hBrush);
	void	FillSolidRect(const RECT& rect, COLORREF clr);
	void	FillSolidRect(int x, int y, int cx, int cy, COLORREF clr);
	void	FrameRect(const RECT& rect, HBRUSH hBrush);
	void	InvertRect(const RECT& rect);

	/* ȉ~Ƒp` */
	bool	Chord(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4);
	bool	Chord(const RECT& rect, const POINT& ptStart, const POINT& ptEnd);
	void	DrawFocusRect(const RECT& rect);
	bool	Ellipse(int x1, int y1, int x2, int y2);
	bool	Ellipse(const RECT& rect);
	bool	Pie(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4);
	bool	Pie(const RECT& rect, const POINT& ptStart, const POINT& ptEnd);
	bool	Polygon(const POINT* lpPoints, int nCount);
	bool	PolyPolygon(const POINT* lpPoints, const int* lpPolyCounts, int nCount);
	bool	Rectangle(int x1, int y1, int x2, int y2);
	bool	Rectangle(const RECT& rect);
	bool	RoundRect(int x1, int y1, int x2, int y2, int x3, int y3);
	bool	RoundRect(const RECT& rect, const POINT& pt);

	/* rbg}bv */
	bool		BitBlt(int x, int y, int nWidth, int nHeight, HDC hSrcDC, int xSrc, int ySrc, DWORD dwRop);
	bool		ExtFloodFill(int x, int y, COLORREF clr, UINT nFillType);
	bool		FloodFill(int x, int y, COLORREF clr);
	COLORREF	GetPixel(int x, int y) const;
	COLORREF	GetPixel(const POINT& pt) const;
	bool		MaskBlt(int x, int y, int nWidth, int nHeight,
					HDC hDC, int xSrc, int ySrc, HBITMAP hBitmap, int xMask, int yMask, DWORD dwRop);
	bool		PatBlt(int x, int y, int nWidth, int nHeight, DWORD dwRop);
	bool		PlgBlt(const POINT* lpPoint, HDC hDC,
					int xSrc, int ySrc, int nWidth, int nHeight, HBITMAP hBitmap, int xMask, int yMask);
	COLORREF	SetPixel(int x, int y, COLORREF clr);
	COLORREF	SetPixel(const POINT& pt, COLORREF clr);
	bool		SetPixelV(int x, int y, COLORREF clr);
	bool		SetPixelV(const POINT& pt, COLORREF clr);
	bool		StretchBlt(int x, int y, int nWidth, int nHeight,
					HDC hSrcDC, int xSrc, int ySrc, int nSrcWidth, int nSrcHeight, DWORD dwRop);

	/* eLXg */
	virtual int		DrawText(const TCHAR* lpszText, int cchText, const RECT& rect, UINT nFormat);
	virtual bool	ExtTextOut(int x, int y, UINT nOptions, const RECT* lpRect,
						const TCHAR* lpszText, UINT cchText, const int* lpDxWidths);
	SIZE			GetCharacterPlacement(const TCHAR* lpszText, int nCount,
						int nMaxExtent, GCP_RESULTS& results, DWORD dwFlags) const;
	SIZE			GetTabbedTextExtent(const TCHAR* lpszText,
						int cchText, int nTabPositions, int* lpnTabStopPositions) const;
	UINT			GetTextAlign() const;
	int				GetTextCharacterExtra() const;
	SIZE			GetTextExtent(const TCHAR* lpszText, int cchText) const;
	bool			GetTextExtentExPoint(const TCHAR* pgiln, int cgi,
						int nMaxExtent, LPINT lpnFit, LPINT alpDx, LPSIZE lpSize) const;
	bool			GetTextExtentExPointI(LPWORD pgiln, int cgi,
						int nMaxExtent, LPINT lpnFit, LPINT alpDx, LPSIZE lpSize) const;
	bool			GetTextExtentPointI(LPWORD pgiln, int cgi, LPSIZE lpSize) const;
	int				GetTextFace(int cchFaceName, TCHAR* lpszFaceName) const;
	bool			GetTextMetrics(TEXTMETRIC& metrics) const;
	virtual bool	GrayString(HBRUSH hBrush, GRAYSTRINGPROC lpfnOutput,
						LPARAM lpData, int cchText, int x, int y, int nWidth, int nHeight);
	virtual SIZE	TabbedTextOut(int x, int y, const TCHAR* lpszText,
						int cchText, int nTabPositions, int* lpnTabStopPositions, int nTabOrigin);
	virtual bool	TextOut(int x, int y, const TCHAR* lpszText, int cchText);
	UINT			SetTextAlign(UINT nFlags);
	int				SetTextCharacterExtra(int nCharExtra);
	int				SetTextJustification(int nBreakExtra, int nBreakCount);

	/* tHg */
	bool	GetCharABCWidths(UINT nFirstChar, UINT nLastChar, ABC& abc) const;
	bool	GetCharABCWidths(UINT nFirstChar, UINT nLastChar, ABCFLOAT& abcf) const;
	bool	GetCharWidth(UINT nFirstChar, UINT nLastChar, int* lpBuffer) const;
	bool	GetCharWidth(UINT nFirstChar, UINT nLastChar, float* lpBuffer) const;
	DWORD	GetFontData(DWORD dwTable, DWORD dwOffset, LPVOID lpData, DWORD cbData) const;
	DWORD	GetFontLanguageInfo() const;
#if(_WIN32_WINNT >= 0x0500)
	DWORD	GetFontUnicodeRanges(GLYPHSET& glyphset) const;
#endif
	DWORD	GetGlyphOutline(UINT nChar, UINT nFormat,
				LPGLYPHMETRICS lpgm, DWORD cbBuffer, LPVOID lpData, const MAT2* lpmat2) const;
	DWORD	GetKerningPairs(DWORD dwPairs, LPKERNINGPAIR lpkrnpair) const;
	DWORD	SetMapperFlags(DWORD dwFlag);
	UINT	GetOutlineTextMetrics(UINT cbData, LPOUTLINETEXTMETRIC lpotm) const;

	/* XN[ */
	bool	ScrollDC(int dx, int dy,
				const RECT& scrollRect, const RECT& clipRect, HRGN hUpdateRgn, RECT* lpUpdateRect);

	/* \ */
	void	AssertValidAsDC() const;

	// f[^o
protected:
	HDC		m_hDC;
private:
	bool	m_bAttached;
};


// CPaintDC class definition
/////////////////////////////////////////////////////////////////////////////

class CPaintDC : public CDC<false> {
	// RXgN^
public:
	CPaintDC(HWND hWnd) : m_hWnd(hWnd), m_bCreatedByCWindow(false) {
		assert(::IsWindow(m_hWnd));
		m_hDC = ::BeginPaint(hWnd, &m_paint);
		assert(m_hDC != 0);
	}
	CPaintDC(const CPaintDC& rhs) : CDC<false>(rhs), m_hWnd(rhs.m_hWnd), m_bCreatedByCWindow(rhs.m_bCreatedByCWindow) {
		const_cast<CPaintDC&>(rhs).m_hWnd = 0;
	}
private:
	CPaintDC(HWND hWnd, PAINTSTRUCT& paint) : m_hWnd(hWnd), m_paint(paint), m_bCreatedByCWindow(true) {
		m_hDC = m_paint.hdc;
		assert(::IsWindow(m_hWnd));
		assert(m_hDC != 0);
	}

public:
	~CPaintDC() {
		if(!m_bCreatedByCWindow)
			::EndPaint(m_hWnd, &m_paint);
	}

	// Zq
private:
	operator HDC() const;

	// \bh
public:
	const PAINTSTRUCT&	GetPaintStruct() const {
		return m_paint;
	}

	// f[^o
private:
	HWND		m_hWnd;
	PAINTSTRUCT	m_paint;
	bool		m_bCreatedByCWindow;

	friend class Controls::CWindow;
};


// CWindowDC class definition
/////////////////////////////////////////////////////////////////////////////

class CWindowDC : public CDC<false> {
	// RXgN^
private:
	CWindowDC(HWND hWnd, HDC hDC) : m_hWnd(hWnd) {
		m_hDC = hDC;
		assert(::IsWindow(m_hWnd));
		assert(m_hDC != 0);
	}
public:
	CWindowDC(const CWindowDC& rhs) : CDC<false>(rhs), m_hWnd(rhs.m_hWnd) {
		const_cast<CWindowDC&>(rhs).m_hWnd = 0;
	}
public:
	~CWindowDC() {
		if(m_hWnd != 0)
			::ReleaseDC(m_hWnd, m_hDC);
	}

	// Zq
private:
	operator HDC() const;

	// f[^o
private:
	HWND	m_hWnd;

	friend class Controls::CWindow;
};


// CDC class implementation
/////////////////////////////////////////////////////////////////////////////

template<bool bDisposable> inline CDC<bDisposable>::CDC() : m_hDC(0), m_bAttached(false) {
}

template<bool bDisposable> inline CDC<bDisposable>::CDC(const CDC<bDisposable>& rhs) : m_hDC(rhs.m_hDC) {
	const_cast<CDC<bDisposable>&>(rhs).m_hDC = 0;
}

template<> inline CDC<true>::~CDC() {
	if(m_hDC != 0 && !m_bAttached)
		::DeleteDC(Detach());
}

template<> inline CDC<false>::~CDC() {
}

template<bool bDisposable> inline CDC<bDisposable>::operator HDC() const {
	return m_hDC;
}

template<bool bDisposable>
inline bool CDC<bDisposable>::AngleArc(int x, int y, int nRadius, float fStartAngle, float fSweepAngle) {
	AssertValidAsDC();
	return toBoolean(::AngleArc(m_hDC, x, y, nRadius, fStartAngle, fSweepAngle));
}

template<bool bDisposable>
inline bool CDC<bDisposable>::Arc(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4) {
	AssertValidAsDC();
	return toBoolean(::Arc(m_hDC, x1, y1, x2, y2, x3, y3, x4, y4));
}

template<bool bDisposable>
inline bool CDC<bDisposable>::Arc(const RECT& rect, const POINT& ptStart, const POINT& ptEnd) {
	AssertValidAsDC();
	return toBoolean(::Arc(m_hDC, rect.left, rect.top,
		rect.right, rect.bottom, ptStart.x, ptStart.y, ptEnd.x, ptEnd.y));
}

template<bool bDisposable>
inline bool CDC<bDisposable>::ArcTo(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4) {
	AssertValidAsDC();
	return toBoolean(::ArcTo(m_hDC, x1, y1, x2, y2, x3, y3, x4, y4));
}

template<bool bDisposable> inline bool CDC<bDisposable>::ArcTo(const RECT& rect, const POINT& ptStart, const POINT& ptEnd) {
	AssertValidAsDC();
	return toBoolean(::ArcTo(m_hDC, rect.left, rect.top,
		rect.right, rect.bottom, ptStart.x, ptStart.y, ptEnd.x, ptEnd.y));
}

template<bool bDisposable> inline void CDC<bDisposable>::AssertValidAsDC() const {
	AssertValid();
	assert(m_hDC != 0);
}

template<> inline bool CDC<true>::Attach(HDC hDC) {
	AssertValid();
	assert(m_hDC == 0);

	if(hDC == 0)
		return false;
	m_hDC = hDC;
	m_bAttached = true;
	return true;
}

template<bool bDisposable>
inline bool CDC<bDisposable>::BitBlt(int x, int y, int nWidth, int nHeight, HDC hSrcDC, int xSrc, int ySrc, DWORD dwRop) {
	AssertValidAsDC();
	return toBoolean(::BitBlt(m_hDC, x, y, nWidth, nHeight, hSrcDC, xSrc, ySrc, dwRop));
}

template<bool bDisposable>
inline bool CDC<bDisposable>::Chord(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4) {
	AssertValidAsDC();
	return toBoolean(::Chord(m_hDC, x1, y1, x2, y2, x3, y3, x4, y4));
}

template<bool bDisposable> inline bool CDC<bDisposable>::Chord(const RECT& rect, const POINT& ptStart, const POINT& ptEnd) {
	AssertValidAsDC();
	return toBoolean(::Chord(m_hDC, rect.left, rect.top,
		rect.right, rect.bottom, ptStart.x, ptStart.y, ptEnd.x, ptEnd.y));
}

template<> inline bool CDC<true>::CreateCompatibleDC(HDC hDC) {
	AssertValid();
	assert(m_hDC == 0);

	m_hDC = ::CreateCompatibleDC(hDC);
	return m_hDC != 0;
}

template<> inline bool CDC<true>::DeleteDC() {
	AssertValid();

	if(m_hDC == 0)
		return false;
	return toBoolean(::DeleteDC(Detach()));
}

template<> inline HDC CDC<true>::Detach() {
	AssertValid();

	HDC	hDC = m_hDC;

	m_hDC = 0;
	m_bAttached = false;
	return hDC;
}

template<bool bDisposable> inline bool CDC<bDisposable>::DrawEdge(const RECT& rect, UINT nEdge, UINT grfFlags) {
	AssertValidAsDC();
	return toBoolean(::DrawEdge(m_hDC, const_cast<RECT*>(&rect), nEdge, grfFlags));
}

template<bool bDisposable> inline void CDC<bDisposable>::DrawFocusRect(const RECT& rect) {
	AssertValidAsDC();
	::DrawFocusRect(m_hDC, &rect);
}

template<bool bDisposable> inline bool CDC<bDisposable>::DrawFrameControl(const RECT& rect, UINT nType, UINT nState) {
	AssertValidAsDC();
	return toBoolean(::DrawFrameControl(m_hDC, const_cast<RECT*>(&rect), nType, nState));
}

template<bool bDisposable> inline bool CDC<bDisposable>::DrawIcon(int x, int y, HICON hIcon) {
	AssertValidAsDC();
	return toBoolean(::DrawIcon(m_hDC, x, y, hIcon));
}

template<bool bDisposable> inline bool CDC<bDisposable>::DrawIcon(const POINT& pt, HICON hIcon) {
	AssertValidAsDC();
	return toBoolean(::DrawIcon(m_hDC, pt.x, pt.y, hIcon));
}

template<bool bDisposable> inline int CDC<bDisposable>::DrawText(const TCHAR* lpszText, int cchText, const RECT& rect, UINT nFormat) {
	AssertValidAsDC();
	return ::DrawText(m_hDC, lpszText, cchText, const_cast<RECT*>(&rect), nFormat);
}

template<bool bDisposable> inline bool CDC<bDisposable>::Ellipse(int x1, int y1, int x2, int y2) {
	AssertValidAsDC();
	return toBoolean(::Ellipse(m_hDC, x1, y1, x2, y2));
}

template<bool bDisposable> inline bool CDC<bDisposable>::Ellipse(const RECT& rect) {
	AssertValidAsDC();
	return toBoolean(::Ellipse(m_hDC, rect.left, rect.top, rect.right, rect.bottom));
}

template<bool bDisposable>
inline int CDC<bDisposable>::EnumObjects(int nObjectType, GOBJENUMPROC lpObjectFunc, LPARAM lpData) {
	AssertValidAsDC();
	return ::EnumObjects(m_hDC, nObjectType, lpObjectFunc, lpData);
}

template<bool bDisposable> inline bool CDC<bDisposable>::ExtFloodFill(int x, int y, COLORREF clr, UINT nFillType) {
	AssertValidAsDC();
	return toBoolean(::ExtFloodFill(m_hDC, x, y, clr, nFillType));
}

template<bool bDisposable>
inline bool CDC<bDisposable>::ExtTextOut(int x, int y, UINT nOptions,
		const RECT* lpRect, const TCHAR* lpszText, UINT cchText, const int* lpDxWidths) {
	AssertValidAsDC();
	return toBoolean(::ExtTextOut(m_hDC, x, y, nOptions, lpRect, lpszText, cchText, lpDxWidths));
}

template<bool bDisposable> inline bool CDC<bDisposable>::FillRgn(HRGN hRgn, HBRUSH hBrush) {
	AssertValidAsDC();
	return toBoolean(::FillRgn(m_hDC, hRgn, hBrush));
}

template<bool bDisposable> inline void CDC<bDisposable>::FillSolidRect(const RECT& rect, COLORREF clr) {
	AssertValidAsDC();

	const COLORREF	clrOrg = GetBkColor();

	SetBkColor(clr);
	ExtTextOut(0, 0, ETO_OPAQUE, &rect, _T(""), 0, 0);
	SetBkColor(clrOrg);
}

template<bool bDisposable> inline void CDC<bDisposable>::FillSolidRect(int x, int y, int cx, int cy, COLORREF clr) {
	AssertValidAsDC();
	
	COLORREF	clrOrg = GetBkColor();
	RECT		rect;

	::SetRect(&rect, x, y, x + cx, y + cy);
	SetBkColor(clr);
	ExtTextOut(0, 0, ETO_OPAQUE, &rect, _T(""), 0, 0);
	SetBkColor(clrOrg);
}

template<bool bDisposable> inline bool CDC<bDisposable>::FloodFill(int x, int y, COLORREF clr) {
	AssertValidAsDC();
	return toBoolean(::FloodFill(m_hDC, x, y, clr));
}

template<bool bDisposable> inline bool CDC<bDisposable>::FrameRgn(HRGN hRgn, HBRUSH hBrush, int nWidth, int nHeight) {
	AssertValidAsDC();
	return toBoolean(::FrameRgn(m_hDC, hRgn, hBrush, nWidth, nHeight));
}

template<bool bDisposable> inline int CDC<bDisposable>::GetArcDirection() const {
	AssertValidAsDC();
	return ::GetArcDirection(m_hDC);
}

template<bool bDisposable> inline void CDC<bDisposable>::FillRect(const RECT& rect, HBRUSH hBrush) {
	AssertValidAsDC();
	::FillRect(m_hDC, &rect, hBrush);
}

template<bool bDisposable> inline void CDC<bDisposable>::FrameRect(const RECT& rect, HBRUSH hBrush) {
	AssertValidAsDC();
	::FrameRect(m_hDC, &rect, hBrush);
}

template<bool bDisposable> inline COLORREF CDC<bDisposable>::GetBkColor() const {
	AssertValidAsDC();
	return ::GetBkColor(m_hDC);
}

template<bool bDisposable> inline int CDC<bDisposable>::GetBkMode() const {
	AssertValidAsDC();
	return ::GetBkMode(m_hDC);
}

template<bool bDisposable> inline POINT CDC<bDisposable>::GetBrushOrg() const {
	AssertValidAsDC();

	POINT	pt;
	::GetBrushOrgEx(m_hDC, &pt);
	return pt;
}

template<bool bDisposable>
inline SIZE CDC<bDisposable>::GetCharacterPlacement(const TCHAR* lpszText,
		int nCount, int nMaxExtent, GCP_RESULTS& results, DWORD dwFlags) const {
	AssertValidAsDC();
	DWORD	dw = ::GetCharacterPlacement(m_hDC, lpszText, nCount, nMaxExtent, &results, dwFlags);
	SIZE	size = {LOWORD(dw), HIWORD(dw)};
	return size;
}

template<bool bDisposable> inline bool CDC<bDisposable>::GetCharABCWidths(UINT nFirstChar, UINT nLastChar, ABC& abc) const {
	AssertValidAsDC();
	return toBoolean(::GetCharABCWidths(m_hDC, nFirstChar, nLastChar, &abc));
}

template<bool bDisposable> inline bool CDC<bDisposable>::GetCharABCWidths(UINT nFirstChar, UINT nLastChar, ABCFLOAT& abcf) const {
	AssertValidAsDC();
	return toBoolean(::GetCharABCWidthsFloat(m_hDC, nFirstChar, nLastChar, &abcf));
}

template<bool bDisposable> inline bool CDC<bDisposable>::GetCharWidth(UINT nFirstChar, UINT nLastChar, int* lpBuffer) const {
	AssertValidAsDC();
	return toBoolean(::GetCharWidth32(m_hDC, nFirstChar, nLastChar, lpBuffer));
}

template<bool bDisposable> inline bool CDC<bDisposable>::GetCharWidth(UINT nFirstChar, UINT nLastChar, float* lpBuffer) const {
	AssertValidAsDC();
	return toBoolean(::GetCharWidthFloat(m_hDC, nFirstChar, nLastChar, lpBuffer));
}

template<bool bDisposable> inline bool CDC<bDisposable>::GetColorAdjustment(COLORADJUSTMENT& colorAdjust) const {
	AssertValidAsDC();
	return toBoolean(::GetColorAdjustment(m_hDC, &colorAdjust));
}

template<bool bDisposable> inline HBITMAP CDC<bDisposable>::GetCurrentBitmap() const {
	AssertValidAsDC();
	return reinterpret_cast<HBITMAP>(::GetCurrentObject(m_hDC, OBJ_BITMAP));
}

template<bool bDisposable> inline HBRUSH CDC<bDisposable>::GetCurrentBrush() const {
	AssertValidAsDC();
	return reinterpret_cast<HBRUSH>(::GetCurrentObject(m_hDC, OBJ_BRUSH));
}

template<bool bDisposable> inline HFONT CDC<bDisposable>::GetCurrentFont() const {
	AssertValidAsDC();
	return reinterpret_cast<HFONT>(::GetCurrentObject(m_hDC, OBJ_FONT));
}

template<bool bDisposable> inline HPALETTE CDC<bDisposable>::GetCurrentPalette() const {
	AssertValidAsDC();
	return reinterpret_cast<HPALETTE>(::GetCurrentObject(m_hDC, OBJ_PAL));
}

template<bool bDisposable> inline HPEN CDC<bDisposable>::GetCurrentPen() const {
	AssertValidAsDC();
	return reinterpret_cast<HPEN>(::GetCurrentObject(m_hDC, OBJ_PEN));
}

template<bool bDisposable> inline POINT CDC<bDisposable>::GetCurrentPosition() const {
	AssertValidAsDC();

	POINT	pt;
	::GetCurrentPositionEx(m_hDC, &pt);
	return pt;
}

template<bool bDisposable> inline int CDC<bDisposable>::GetDeviceCaps(int iIndex) const {
	AssertValidAsDC();
	return ::GetDeviceCaps(m_hDC, iIndex);
}

template<bool bDisposable> inline DWORD CDC<bDisposable>::GetFontData(DWORD dwTable, DWORD dwOffset, LPVOID lpData, DWORD cbData) const {
	AssertValidAsDC();
	return ::GetFontData(m_hDC, dwTable, dwOffset, lpData, cbData);
}

template<bool bDisposable> inline DWORD CDC<bDisposable>::GetFontLanguageInfo() const {
	AssertValidAsDC();
	return ::GetFontLanguageInfo(m_hDC);
}

#if(_WIN32_WINNT >= 0x0500)
template<bool bDisposable> inline DWORD CDC<bDisposable>::GetFontUnicodeRanges(GLYPHSET& glyphset) const {
	AssertValidAsDC();
	return ::GetFontUnicodeRanges(m_hDC, &glyphset);
}
#endif

template<bool bDisposable>
inline DWORD CDC<bDisposable>::GetGlyphOutline(UINT nChar, UINT nFormat,
		LPGLYPHMETRICS lpgm, DWORD cbBuffer, LPVOID lpData, const MAT2* lpmat2) const {
	AssertValidAsDC();
	return ::GetGlyphOutline(m_hDC, nChar, nFormat, lpgm, cbBuffer, lpData, lpmat2);
}

template<bool bDisposable> inline DWORD CDC<bDisposable>::GetKerningPairs(DWORD dwPairs, LPKERNINGPAIR lpkrnpair) const {
	AssertValidAsDC();
	return ::GetKerningPairs(m_hDC, dwPairs, lpkrnpair);
}

template<bool bDisposable> inline COLORREF CDC<bDisposable>::GetNearestColor(COLORREF clr) const {
	AssertValidAsDC();
	return ::GetNearestColor(m_hDC, clr);
}

template<bool bDisposable> inline UINT CDC<bDisposable>::GetOutlineTextMetrics(UINT cbData, LPOUTLINETEXTMETRIC lpotm) const {
	AssertValidAsDC();
	return ::GetOutlineTextMetrics(m_hDC, cbData, lpotm);
}

template<bool bDisposable> inline COLORREF CDC<bDisposable>::GetPixel(int x, int y) const {
	AssertValidAsDC();
	return ::GetPixel(m_hDC, x, y);
}

template<bool bDisposable> inline COLORREF CDC<bDisposable>::GetPixel(const POINT& pt) const {
	AssertValidAsDC();
	return ::GetPixel(m_hDC, pt.x, pt.y);
}

template<bool bDisposable> inline int CDC<bDisposable>::GetPolyFillMode() const {
	AssertValidAsDC();
	return ::GetPolyFillMode(m_hDC);
}

template<bool bDisposable> inline int CDC<bDisposable>::GetROP2() const {
	AssertValidAsDC();
	return ::GetROP2(m_hDC);
}

template<bool bDisposable> inline HDC CDC<bDisposable>::GetSafeHdc() const {
	return (this == 0) ? 0 : m_hDC;
}

template<bool bDisposable> inline int CDC<bDisposable>::GetStretchBltMode() const {
	AssertValidAsDC();
	return ::GetStretchBltMode(m_hDC);
}

template<bool bDisposable>
inline SIZE CDC<bDisposable>::GetTabbedTextExtent(const TCHAR* lpszText,
		int cchText, int nTabPositions, int* lpnTabStopPositions) const {
	AssertValidAsDC();

	SIZE	size;
	DWORD	dw = ::GetTabbedTextExtent(m_hDC, lpszText, cchText, nTabPositions, lpnTabStopPositions);

	size.cx = LOWORD(dw);
	size.cy = HIWORD(dw);
	return size;
}

template<bool bDisposable> inline UINT CDC<bDisposable>::GetTextAlign() const {
	AssertValidAsDC();
	return ::GetTextAlign(m_hDC);
}

template<bool bDisposable> inline int CDC<bDisposable>::GetTextCharacterExtra() const {
	AssertValidAsDC();
	return ::GetTextCharacterExtra(m_hDC);
}

template<bool bDisposable> inline COLORREF CDC<bDisposable>::GetTextColor() const {
	AssertValidAsDC();
	return ::GetTextColor(m_hDC);
}

template<bool bDisposable> inline SIZE CDC<bDisposable>::GetTextExtent(const TCHAR* lpszText, int cchText) const {
	AssertValidAsDC();

	SIZE	size;

	::GetTextExtentPoint32(m_hDC, lpszText, cchText, &size);
	return size;
}

template<bool bDisposable>
inline bool CDC<bDisposable>::GetTextExtentExPoint(const TCHAR* pgiln,
		int cgi, int nMaxExtent, LPINT lpnFit, LPINT alpDx, LPSIZE lpSize) const {
	AssertValidAsDC();
	return toBoolean(::GetTextExtentExPoint(m_hDC, pgiln, cgi, nMaxExtent, lpnFit, alpDx, lpSize));
}

#if(_WIN32_WINNT >= 0x0500)
template<bool bDisposable>
inline bool CDC<bDisposable>::GetTextExtentExPointI(LPWORD pgiln,
		int cgi, int nMaxExtent, LPINT lpnFit, LPINT alpDx, LPSIZE lpSize) const {
	AssertValidAsDC();
	return toBoolean(::GetTextExtentExPointI(m_hDC, pgiln, cgi, nMaxExtent, lpnFit, alpDx, lpSize));
}
#endif

#if(_WIN32_WINNT >= 0x0500)
template<bool bDisposable> inline bool CDC<bDisposable>::GetTextExtentPointI(LPWORD pgiln, int cgi, LPSIZE lpSize) const {
	AssertValidAsDC();
	return toBoolean(::GetTextExtentPointI(m_hDC, pgiln, cgi, lpSize));
}
#endif

template<bool bDisposable> inline int CDC<bDisposable>::GetTextFace(int cchFaceName, TCHAR* lpszFaceName) const {
	AssertValidAsDC();
	return ::GetTextFace(m_hDC, cchFaceName, lpszFaceName);
}

template<bool bDisposable> inline bool CDC<bDisposable>::GetTextMetrics(TEXTMETRIC& metrics) const {
	AssertValidAsDC();
	return toBoolean(::GetTextMetrics(m_hDC, &metrics));
}

template<bool bDisposable> inline HWND CDC<bDisposable>::GetWindow() const {
	AssertValidAsDC();
	return ::WindowFromDC(m_hDC);
}

template<bool bDisposable>
inline bool CDC<bDisposable>::GrayString(HBRUSH hBrush,
		GRAYSTRINGPROC lpfnOutput, LPARAM lpData, int cchText, int x, int y, int nWidth, int nHeight) {
	AssertValidAsDC();
	return toBoolean(::GrayString(m_hDC, hBrush, lpfnOutput, lpData, cchText, x, y, nWidth, nHeight));
}

template<bool bDisposable> inline void CDC<bDisposable>::InvertRect(const RECT& rect) {
	AssertValidAsDC();
	::InvertRect(m_hDC, &rect);
}

template<bool bDisposable> inline bool CDC<bDisposable>::InvertRgn(HRGN hRgn) {
	AssertValidAsDC();
	return toBoolean(::InvertRgn(m_hDC, hRgn));
}

template<bool bDisposable> inline bool CDC<bDisposable>::LineTo(int x, int y) {
	AssertValidAsDC();
	return toBoolean(::LineTo(m_hDC, x, y));
}

template<bool bDisposable> inline bool CDC<bDisposable>::LineTo(const POINT& pt) {
	AssertValidAsDC();
	return toBoolean(::LineTo(m_hDC, pt.x, pt.y));
}

template<bool bDisposable>
inline bool CDC<bDisposable>::MaskBlt(int x, int y, int nWidth, int nHeight,
		HDC hDC, int xSrc, int ySrc, HBITMAP hBitmap, int xMask, int yMask, DWORD dwRop) {
	AssertValidAsDC();
	return toBoolean(::MaskBlt(m_hDC, x, y, nWidth, nHeight, hDC, xSrc, ySrc, hBitmap, xMask, yMask, dwRop));
}

template<bool bDisposable> inline POINT CDC<bDisposable>::MoveTo(int x, int y) {
	AssertValidAsDC();

	POINT	pt;
	::MoveToEx(m_hDC, x, y, &pt);
	return pt;
}

template<bool bDisposable> inline POINT CDC<bDisposable>::MoveTo(const POINT& pt) {
	AssertValidAsDC();

	POINT	pt_;
	::MoveToEx(m_hDC, pt.x, pt.y, &pt_);
	return pt_;
}

template<bool bDisposable> inline bool CDC<bDisposable>::PaintRgn(HRGN hRgn) {
	AssertValidAsDC();
	return toBoolean(::PaintRgn(m_hDC, hRgn));
}

template<bool bDisposable> inline bool CDC<bDisposable>::PatBlt(int x, int y, int nWidth, int nHeight, DWORD dwRop) {
	AssertValidAsDC();
	return toBoolean(::PatBlt(m_hDC, x, y, nWidth, nHeight, dwRop));
}

template<bool bDisposable> inline bool CDC<bDisposable>::Pie(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4) {
	AssertValidAsDC();
	return toBoolean(::Pie(m_hDC, x1, y1, x2, y2, x3, y3, x4, y4));
}

template<bool bDisposable> inline bool CDC<bDisposable>::Pie(const RECT& rect, const POINT& ptStart, const POINT& ptEnd) {
	AssertValidAsDC();
	return toBoolean(::Pie(m_hDC, rect.left, rect.top,
		rect.right, rect.bottom, ptStart.x, ptStart.y, ptEnd.x, ptEnd.y));
}

template<bool bDisposable> inline bool CDC<bDisposable>::PlgBlt(const POINT* lpPoint, HDC hDC,
		int xSrc, int ySrc, int nWidth, int nHeight, HBITMAP hBitmap, int xMask, int yMask) {
	AssertValidAsDC();
	return toBoolean(::PlgBlt(m_hDC, lpPoint, hDC, xSrc, ySrc, nWidth, nHeight, hBitmap, xMask, yMask));
}

template<bool bDisposable> inline bool CDC<bDisposable>::PolyBezier(const POINT* lpPoints, int nCount) {
	AssertValidAsDC();
	return toBoolean(::PolyBezier(m_hDC, lpPoints, nCount));
}

template<bool bDisposable> inline bool CDC<bDisposable>::PolyBezierTo(const POINT* lpPoints, int nCount) {
	AssertValidAsDC();
	return toBoolean(::PolyBezierTo(m_hDC, lpPoints, nCount));
}

template<bool bDisposable> inline bool CDC<bDisposable>::PolyDraw(const POINT* lpPoints, const BYTE* lpTypes, int nCount) {
	AssertValidAsDC();
	return toBoolean(::PolyDraw(m_hDC, lpPoints, lpTypes, nCount));
}

template<bool bDisposable> inline bool CDC<bDisposable>::Polygon(const POINT* lpPoints, int nCount) {
	AssertValidAsDC();
	return toBoolean(::Polygon(m_hDC, lpPoints, nCount));
}

template<bool bDisposable> inline bool CDC<bDisposable>::Polyline(LPPOINT lpPoints, int nCount) {
	AssertValidAsDC();
	return toBoolean(::Polyline(m_hDC, lpPoints, nCount));
}

template<bool bDisposable> inline bool CDC<bDisposable>::PolylineTo(const POINT* lpPoints, int nCount) {
	AssertValidAsDC();
	return toBoolean(::PolylineTo(m_hDC, lpPoints, nCount));
}

template<bool bDisposable> inline bool CDC<bDisposable>::PolyPolygon(const POINT* lpPoints, const int* lpPolyCounts, int nCount) {
	AssertValidAsDC();
	return toBoolean(::PolyPolygon(m_hDC, lpPoints, lpPolyCounts, nCount));
}

template<bool bDisposable> inline bool CDC<bDisposable>::PolyPolyline(const POINT* lpPoints, const DWORD *lpPolyPoints, int nCount) {
	AssertValidAsDC();
	return toBoolean(::PolyPolyline(m_hDC, lpPoints, lpPolyPoints, nCount));
}

template<bool bDisposable> inline UINT CDC<bDisposable>::RealizePalette() {
	AssertValidAsDC();
	return ::RealizePalette(m_hDC);
}

template<bool bDisposable> inline bool CDC<bDisposable>::Rectangle(int x1, int y1, int x2, int y2) {
	AssertValidAsDC();
	return toBoolean(::Rectangle(m_hDC, x1, y1, x2, y2));
}

template<bool bDisposable> inline bool CDC<bDisposable>::Rectangle(const RECT& rect) {
	AssertValidAsDC();
	return toBoolean(::Rectangle(m_hDC, rect.left, rect.top, rect.right, rect.bottom));
}

template<bool bDisposable> inline bool CDC<bDisposable>::RestoreDC(int nSavedDC) {
	AssertValidAsDC();
	return toBoolean(::RestoreDC(m_hDC, nSavedDC));
}

template<bool bDisposable> inline bool CDC<bDisposable>::RoundRect(int x1, int y1, int x2, int y2, int x3, int y3) {
	AssertValidAsDC();
	return toBoolean(::RoundRect(m_hDC, x1, y1, x2, y2, x3, y3));
}

template<bool bDisposable> inline bool CDC<bDisposable>::RoundRect(const RECT& rect, const POINT& pt) {
	AssertValidAsDC();
	return toBoolean(::RoundRect(m_hDC, rect.left, rect.top, rect.right, rect.bottom, pt.x, pt.y));
}

template<bool bDisposable> inline int CDC<bDisposable>::SaveDC() {
	AssertValidAsDC();
	return ::SaveDC(m_hDC);
}

template<bool bDisposable>
inline bool CDC<bDisposable>::ScrollDC(int dx, int dy,
		const RECT& scrollRect, const RECT& clipRect, HRGN hUpdateRgn, RECT* lpUpdateRect) {
	AssertValidAsDC();
	return toBoolean(::ScrollDC(m_hDC, dx, dy, &scrollRect, &clipRect, hUpdateRgn, lpUpdateRect));
}

template<bool bDisposable> inline HBITMAP CDC<bDisposable>::SelectObject(HBITMAP hBitmap) {
	AssertValidAsDC();
	return reinterpret_cast<HBITMAP>(::SelectObject(m_hDC, hBitmap));
}

template<bool bDisposable> inline HBRUSH CDC<bDisposable>::SelectObject(HBRUSH hBrush) {
	AssertValidAsDC();
	return reinterpret_cast<HBRUSH>(::SelectObject(m_hDC, hBrush));
}

template<bool bDisposable> inline HFONT CDC<bDisposable>::SelectObject(HFONT hFont) {
	AssertValidAsDC();
	return reinterpret_cast<HFONT>(::SelectObject(m_hDC, hFont));
}

template<bool bDisposable> inline HPEN CDC<bDisposable>::SelectObject(HPEN hPen) {
	AssertValidAsDC();
	return reinterpret_cast<HPEN>(::SelectObject(m_hDC, hPen));
}

template<bool bDisposable> inline HPALETTE CDC<bDisposable>::SelectPalette(HPALETTE hPalette, bool bForceBackground) {
	AssertValidAsDC();
	return ::SelectPalette(m_hDC, hPalette, bForceBackground);
}

template<bool bDisposable> inline HGDIOBJ CDC<bDisposable>::SelectStockObject(int nObject) {
	AssertValidAsDC();
	return ::SelectObject(m_hDC, ::GetStockObject(nObject));
}

template<bool bDisposable> inline int CDC<bDisposable>::SetArcDirection(int nArcDirection) {
	AssertValidAsDC();
	return ::SetArcDirection(m_hDC, nArcDirection);
}

template<bool bDisposable> inline COLORREF CDC<bDisposable>::SetBkColor(COLORREF clr) {
	AssertValidAsDC();
	return ::SetBkColor(m_hDC, clr);
}

template<bool bDisposable> inline int CDC<bDisposable>::SetBkMode(int nBkMode) {
	AssertValidAsDC();
	return ::SetBkMode(m_hDC, nBkMode);
}

template<bool bDisposable> inline POINT CDC<bDisposable>::SetBrushOrg(int x, int y) {
	AssertValidAsDC();

	POINT	pt;
	::SetBrushOrgEx(m_hDC, x, y, &pt);
	return pt;
}

template<bool bDisposable> inline POINT CDC<bDisposable>::SetBrushOrg(const POINT& pt) {
	AssertValidAsDC();

	POINT	pt_;
	::SetBrushOrgEx(m_hDC, pt.x, pt.y, &pt_);
	return pt_;
}

template<bool bDisposable> inline bool CDC<bDisposable>::SetColorAdjustment(const COLORADJUSTMENT& colorAdjust) {
	AssertValidAsDC();
	return toBoolean(::SetColorAdjustment(m_hDC, &colorAdjust));
}

template<bool bDisposable> inline DWORD CDC<bDisposable>::SetMapperFlags(DWORD dwFlag) {
	AssertValidAsDC();
	return ::SetMapperFlags(m_hDC, dwFlag);
}

template<bool bDisposable> inline COLORREF CDC<bDisposable>::SetPixel(int x, int y, COLORREF clr) {
	AssertValidAsDC();
	return ::SetPixel(m_hDC, x, y, clr);
}

template<bool bDisposable> inline COLORREF CDC<bDisposable>::SetPixel(const POINT& pt, COLORREF clr) {
	AssertValidAsDC();
	return ::SetPixel(m_hDC, pt.x, pt.y, clr);
}

template<bool bDisposable> inline bool CDC<bDisposable>::SetPixelV(int x, int y, COLORREF clr) {
	AssertValidAsDC();
	return toBoolean(::SetPixelV(m_hDC, x, y, clr));
}

template<bool bDisposable> inline bool CDC<bDisposable>::SetPixelV(const POINT& pt, COLORREF clr) {
	AssertValidAsDC();
	return toBoolean(::SetPixelV(m_hDC, pt.x, pt.y, clr));
}

template<bool bDisposable> inline int CDC<bDisposable>::SetPolyFillMode(int nPolyFillMode) {
	AssertValidAsDC();
	return ::SetPolyFillMode(m_hDC, nPolyFillMode);
}

template<bool bDisposable> inline int CDC<bDisposable>::SetROP2(int nDrawMode) {
	AssertValidAsDC();
	return ::SetROP2(m_hDC, nDrawMode);
}

template<bool bDisposable> inline int CDC<bDisposable>::SetStretchBltMode(int nStretchMode) {
	AssertValidAsDC();
	return ::SetStretchBltMode(m_hDC, nStretchMode);
}

template<bool bDisposable> inline UINT CDC<bDisposable>::SetTextAlign(UINT nFlags) {
	AssertValidAsDC();
	return ::SetTextAlign(m_hDC, nFlags);
}

template<bool bDisposable> inline int CDC<bDisposable>::SetTextCharacterExtra(int nCharExtra) {
	AssertValidAsDC();
	return ::SetTextCharacterExtra(m_hDC, nCharExtra);
}

template<bool bDisposable> inline COLORREF CDC<bDisposable>::SetTextColor(COLORREF clr) {
	AssertValidAsDC();
	return ::SetTextColor(m_hDC, clr);
}

template<bool bDisposable> inline int CDC<bDisposable>::SetTextJustification(int nBreakExtra, int nBreakCount) {
	AssertValidAsDC();
	return ::SetTextJustification(m_hDC, nBreakExtra, nBreakExtra);
}

template<bool bDisposable> inline bool CDC<bDisposable>::StretchBlt(int x, int y, int nWidth, int nHeight,
		HDC hSrcDC, int xSrc, int ySrc, int nSrcWidth, int nSrcHeight, DWORD dwRop) {
	AssertValidAsDC();
	return toBoolean(::StretchBlt(m_hDC,
		x, y, nWidth, nHeight, hSrcDC, xSrc, ySrc, nSrcWidth, nSrcHeight, dwRop));
}

template<bool bDisposable>
inline SIZE CDC<bDisposable>::TabbedTextOut(int x, int y,
		const TCHAR* lpszText, int cchText, int nTabPositions, int* lpnTabStopPositions, int nTabOrigin) {
	AssertValidAsDC();
	SIZE	size;
	long	n = ::TabbedTextOut(m_hDC, x, y,
		lpszText, cchText, nTabPositions, lpnTabStopPositions, nTabOrigin);

	size.cx = LOWORD(n);
	size.cy = HIWORD(n);
	return size;
}

template<bool bDisposable> inline bool CDC<bDisposable>::TextOut(int x, int y, const TCHAR* lpszText, int cchText) {
	AssertValidAsDC();
	return toBoolean(::TextOut(m_hDC, x, y, lpszText, cchText));
}

template<bool bDisposable> inline void CDC<bDisposable>::UpdateColors() {
	AssertValidAsDC();
	::UpdateColors(m_hDC);
}


// CDC class implementation
/////////////////////////////////////////////////////////////////////////////


} /* namespace GDI */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _DC_H_ */

/* [EOF] */