//
//  PgSQLResultSetProtocol.h
//  PostgresSQL
//
//  Created by Pascal on Thu Dec 26 2002.
//  Copyright (c) 2002 P3 Consulting. All rights reserved.
//

/*!
@header PgSQLResultSetProtocol.h

*/

#import "libpq-fe.h"
#import "pgtypes.h"

@protocol PgSQLResultSetProtocol

/*! 
   @method rowsAffected
   @result How many rows were affected by latest query executed (SELECT).
*/
-(int)rowsAffected;
/*! 
   @method rowsAffectedByCommand
   @result how many rows were affected by latestquery executed (UPDATE, DELETE, INSERT).
*/
-(int)rowsAffectedByCommand;
/*! 
   @method nextRow
   @abstract position the internal index to be on the next row.
   @result YES if successful, NO if the current row is no more valid (after the last row).
*/
-(BOOL)nextRow;
/*! 
   @method previousRow
   @abstract position the internal index result to be on the previous row.
   @result YES if successful, NO if the current row is no more valid (before the first row).
*/
-(BOOL)previousRow;
/*! 
   @method afterLastRow
   @abstract position the internal index to be after the last row.
*/
-(void)afterLastRow;
/*! 
   @method beforeFirstRow
   @abstract position the internal index to be before the first row.
*/
-(void)beforeFirstRow;
/*!
	@method	firstRow
	@abstract position the internal index to be on the first row.
*/
-(void)firstRow;
/*!
	@method	lastRow
	@abstract position the internal index to be on the last row.
*/
-(void)lastRow;
/*!
	@method	goRow
	@abstract To position the internal index to be on the specified row.<br>
	Note calling <code>nextRow</code> and <code>previousRow</code> will move the cursor up and down resp.<br>
	If you want to read row N: <code>[conn goRow:N-1]</code> and then <code>[conn nextRow]</code>
	@param rowIndex where to position the internal index.
*/
-(void)goRow:(int)rowIndex;
/*!
	@method	goRowRelative
	@abstract position the internal index to be on the specified row relatively to the current one.<br>
	<code>nextRow</code> and <code>previousRow</code> and equivalent to <code>goRowRelative:1</code> and <code>goRowRelative:-1</code> resp.
	@param delta to add to internal row index. (If delta is negative we move backward).
*/
-(void)goRowRelative:(int)deltaIndex;

/*! 
   @method isColumnNULL
   @result YES if column at <code>fieldIndex</code> is NULL
*/
-(BOOL)isColumnNULL:(int)fieldIndex;
/*! 
   @method resultColumnCount
   @result the number of columns of latest executed command.
*/
-(int)resultColumnCount;
/*! 
   @method resultColumnName
   @param <code>col</code> column index of name to fetch.
   @result the name of the column at index col.
*/
-(const char *)resultColumnName:(int)col;
/*! 
   @method resultColumnTypeName
   @param <code>col</code> column index of type to fetch.
   @result the name of the type of the column at index col.
*/
-(const char *)resultColumnTypeName:(int)col;
/*! 
   @method resultColumnType
   @param <code>col</code> column index of name to fetch.
   @result the type oid of the column at index col.
*/
-(Oid)resultColumnType:(int)col;

/*!
	@method	curRowField
	@param <code>fieldIndex</code> index of the field to fetch
	@abstract returns field at fieldIndex of current row as an <code>id</code> (NSString, NSNumber, etc.).
	@result	object or nil if any problem occurs.
*/
-(id)curRowField:(int)fieldIndex;

/*!
	@method	resultRowAsArray
	@abstract returns current row as an array of NSObject
	@result	NSArray of field values of current row or nil if an error occurs.
*/
-(NSArray *)resultRowAsArray;

/*! 
   @method resultAsDictionary
   @abstract returns a dictionnary [autorelease] build from result of last query
	@discussion <br>
	resultAsDictionnary	returns a dictionary structured as follows<br>
	<pre><code>&lt;dict&gt;
			&lt;key&gt;columns/key&gt;
			&lt;array&gt;
				&lt;dict&gt;
					&lt;key&gt;name&lt;/key&gt;
					&lt;string&gt;columnTitle&lt;/string&gt;
					&lt;key&gt;typeName&lt;/key&gt;
					&lt;string&gt;columnTypeName&lt;/string&gt;
					&lt;key&gt;typeOid&lt;/key&gt;
					&lt;integer&gt;columnTypeOid&lt;/integer&gt;
				&lt;/dict&gt;
				...
					
			&lt;/array&gt;
			&lt;key&gt;fields/key&gt;
			&lt;array&gt;
				&lt;string&gt;columnFieldValue&lt;/string&gt;
				...

			&lt;/array&gt;
			&lt;/dict&gt;
		</code></pre>
			Since PostgreSQL allows creation of new types, we don't provide yet
			an automatic conversion from the PostgreSQL type to the plist one.

			NB<br>
				resultAsDictionary preserves the <code>curRowIndex</code> used by <code>nextRow</code> and <code>previousRow</code> methods<br>
				This means it can be used inside a<br>
				<code>while ([psql nextRow])	{<br>
					...<br>
				}</code><br>
				loop as a debugging tool (for example if an error condition occurs dump the result as a dict in a file)
				without changing the behavior of the loop.
*/
-(NSDictionary *)resultAsDictionary;
/*! 
   @method resultTableName
   @abstract should return the name of the table of the column at index <code>col</code> in latest result.<br>
   Not yet supported byPostgreSQL.
   @result "NOT_SUPPORTED_BY_POSTGRESQL"
*/
-(const char *)resultTableName:(int)col;

@end