/*
*	writeset.h (wb)
*	
*	Contains data structures and prototypes relating to write sets and
*	write set processing
*
*  Created by: Win Bausch and Bettina Kemme
*
*/

#ifndef WRITESET_H
#define WRITESET_H

#include "postgres.h"
#include "postgres_ext.h"
#include "c.h"

#include "access/attnum.h"
#include "access/htup.h"
#include "catalog/pg_index.h"
#include "executor/tuptable.h"
#include "lib/dllist.h"
#include "libpq/bufferedSock.h"
#include "nodes/nodes.h"
#include "utils/rel.h"

#include "storage/lock.h"


#define TUPLE_LIMIT 20
#define WS_BUF_SIZE 8192

//#define WS_DEBUG_LOCAL
//#define WS_DEBUG_REMOTE
//#define WS_SER_DEBUG
//#define WS_PRINT
//#define WS_PRINT_LOW_LEVEL

#ifdef WS_PRINT_LOW_LEVEL
#ifndef WS_PRINT
#define WS_PRINT
#endif
#endif

typedef struct SizedList{
	Dllist 	*list;
	int		size;
} SizedList;

/*
* IdxInfo contains informatioon about prim.key index
* and is used by the remote bknd to find target tuples
*
*	indexName - name of the index
*	indexOid - oid of the index
*	numKeys - number of keys of the index
*	attrNums - attribute numbers of the pkey attributes
*				in the indexed relation
*	equalProcs - procedure oids for the '=' operation
*				for the different index attribute types
*				Used by remote bknd to de index scan
*
*/

typedef struct IdxInfo{
	char			indexName[NAMEDATALEN];
	Oid				indexOid;
	int 			numKeys;
	AttrNumber		attrNums[INDEX_MAX_KEYS];
	RegProcedure	equalProcs[INDEX_MAX_KEYS];
} IdxInfo;

/*
*	RelInfo contains info about a target relationm
*
*	name - the target relation name
*	relOid - the target relation oid
*	indexinfo - primary key index info
*	queryList - list of query info structs containing
*				queries that modify this relation
*/
typedef struct RelInfo{
	char		name[NAMEDATALEN+1];
	Oid			relOid;
	IdxInfo		indexinfo;
	SizedList	querylist;
} RelInfo;

/*
*	TupleCollection contains info relating to tuple-level
*	replication
*
*	rel - points back to the associated relation info object
*	numAttrs - total number of attributes in the relation
*	changeArray - contains 1 char for each attribute, indicates wheter
*					the attr has been changed by the query or not
*					'r' means changed, ' ' means unchanged
*	maxTuples - the maximum number of tuples this struct can contain
*	numTuples - the number of tuples this struct contains at the moment
*
*	pkey_values - array of strings containing the primary key values
*					of each tuple in the collection
  *	newtup_data - array of strings containing the modified attribute's
*					new values for each tuple in the collection
*/
typedef struct TupleCollection{
	RelInfo 	*rel;
	
	int			numAttrs;
	char		*changeArray;

	int 		maxTuples;
	int			numTuples;

	char		**pkey_values;
	char		**newtup_data;
} TupleCollection;

/*
* CmdInfo contains general info  about a query
*
*	cmd - the query type
*	sect - the query's execution section
*	stmt - the query string 
*/
typedef struct CmdInfo{
	CmdType cmd;
	int		sect;//(oi)
	char 	*stmt;
}CmdInfo;

/*
* QueryInfo contains  info about a query
*
*	info - misc info about query
*	tcoll - all tuples modified by this query
*/
typedef struct QueryInfo{
	CmdInfo			info;
	TupleCollection	tcoll;
} QueryInfo;


/*
* Write set structure
*
*	targetrels - list of RelationInfo structs of all relations
*					modified in the current write set
*	currQuery - the currently processed query info struct
*				(see WriteSetAddQuery)
*	critSect - local backend: is used to distribute the execution 
*				section numbers when the write set is marshaled
*			   remote backend: holds the highest excecution section
*				number in the current write set.				
*/
typedef  struct WriteSet{
	SizedList 	targetrels;
	CmdInfo		*currQuery;
	int			critSect; 
} WriteSet;

typedef  struct TxnId{
	uint32		hostid;
	uint32		xid;
} TxnId;

typedef WriteSet *WriteSetPtr;

					  
extern int WriteSetWaitOnLock(char **relNames, int granted[],
							  int numNames);


extern void WriteSetInit(WriteSetPtr ws);
extern void WriteSetFree(WriteSetPtr ws);
extern void WriteSetAddQuery(WriteSetPtr ws, char *query);
extern void WriteSetCollectTuple(ItemPointer origTuple, TupleTableSlot *slot, CmdInfo *currQuery);
extern HeapTuple WriteSetFormTuple(Relation rel, HeapTuple oldTuple, 
												char *newtup_data, TupleDesc tdesc);
extern void WriteSetGetDatum(Datum *values, char *data, int numTargets, 
											AttrNumber *TargetNums, TupleDesc tdesc);
extern void WriteSetGetChangedAtts(List *targetlist, char *changeArray, int numAttrs);
extern void WriteSetSerialize(WriteSetPtr ws, ebufptr extbuf);
extern void WriteSetUnserialize(WriteSetPtr ws, bufsockptr bsock);
extern void WriteSetProcess(bool doTxnDelim, WriteSetPtr ws);
extern void WriteSetOnExit(int dummy, WriteSetPtr ws);

//#ifdef WS_PRINT
void printWriteSet(WriteSetPtr ws, char *comment);
//#endif

#endif
