require_relative 'spec_helper'

class TestPtrIn < Minitest::Test

  include Dnsruby

  # Tests that message raises no error when decoded, encoded, and decoded again.
  def verify(message_data_as_hex_string, canonical = false)
    # Dnsruby.log.level = Logger::DEBUG
    message = Message.decode([message_data_as_hex_string].pack('H*').force_encoding("ASCII-8BIT"))
    Message.decode(message.encode(canonical))
  end


  #example.com A IN (2-byte size removed at beginning)
  def test_non_canonical
    verify(
        'f8f681900001000200030005076578616d706c6503636f6d0000010001c00c0001000100014f0c00045db8d822c00c002e000100014f0c00a20001080200015180580ec93f57f38df906a8076578616d706c6503636f6d006ae1882b1536a15c44f5813671af57bf9cae0366cff2ec085d6dedfddff0c469fa827ceec953de7cc1eee634f4cf695dc2caa2074f95199a5582e51e63b336d8f091d18c0c1a307ae3f5508ec650c4085a95e54e2c2451d9fc9ae04b4e62f3d1a1689e9507c3692fb84817a70afd3e9cdf066f73cc4ac11ed080a30d2af31510b457b5c04b0002000100014f0c001401620c69616e612d73657276657273036e657400c04b0002000100014f0c00040161c0e9c04b002e000100014f0c00a2000208020001518058109f4c57f56c1906a8076578616d706c6503636f6d006d8dd0fdbd0a0b0bfe7e4306a4a001bb7a13df2faedb1702a329243c326b915191335e99e16a236de99360547efa96ec6ee547a6dcfab94b57de6f7891bcaf99a2ef5d3c72d5bc18d1bf05ff4473f527bd8f2e6621489ab531dfb6a973e37e0f0be52740a362599058b204097a04c96492e527bfca6a22338eb865b51156c2ab0e6940c10700010001000004940004c72b8735c107001c00010001e209001020010500008f00000000000000000053c0e700010001000004940004c72b8535c0e7001c00010001e209001020010500008d000000000000000000530000291000000080000000')
  end


  #32.197.46.207.in-addr.arpa: type PTR, class IN (2-byte size removed at beginning)
  def test_canonical
    verify(
        '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',
    true)
  end
end
