/* 
 * Copyright (c) 2000 Gregory D. Troxel <gdt@lexort.com>
 * Permission is granted to copy this file under the following terms,
 * subtituting Gregory D. Troxel for MIT.
 * Note that these are the same terms as the rest of the
 * xplot distribution.
 */
/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/

#include "xplot.h"

#include <stdio.h>
#include <math.h>

/* define to get debug printfs */
/* #define CONSTRAINT_DEBUG */

/*
 * This routine takes
 *   a constraint object
 *   a representation of the current view:
 *     {x, y} * { coord type, left, right }
 *   a representation of the window's size
 *     size_x, size_y
 *   a representation of the user's selected rectangle
 *     ux_1, ux_2, uy_1, uy_2
 * This routine returns
 *   1 if successful, 0 if declined to specify output values
 *   new {x, y} * { left, right } coords
 *   new {x, y} * { left, right } pixel values for the adjusted dragged region
 */

int
xplot_constraint(constraint_t *constraint,
		 coord_type x_type, coord_type y_type,
		 coord x_left, coord y_bottom,
		 coord x_right, coord y_top,
		 coord *new_x_left, coord *new_y_bottom,
		 coord *new_x_right, coord *new_y_top,
		 int x_size, int y_size,
		 int ux_left, int uy_bottom,
		 int ux_right, int uy_top,
		 int *new_ux_left, int *new_uy_bottom,
		 int *new_ux_right, int *new_uy_top,
		 int *zoomed_x, int *zoomed_y)
{
  coord xl, yb, xr, yt;

  coord xdelta, ydelta;
  double xupp, yupp;

  coord xc, yc;

  *zoomed_x = *zoomed_y = 0;

  /* If this is in one axis only, punt for now */
  if ( ux_left == ux_right || uy_bottom == uy_top ) {
#ifdef CONSTRAINT_DEBUG
    printf("1-axis zoom, no constraint\n");
#endif /* CONSTRAINT_DEBUG */
    return 0;
  }

  /* If coord types differ, punt for now */
  if ( x_type != y_type ) {
#ifdef CONSTRAINT_DEBUG
    printf("coord types differ, no constraint\n");
#endif /* CONSTRAINT_DEBUG */
    return 0;
  }

  /* If necessary, swap L/R T/B to make this ordered */
  if ( ux_left >= ux_right ) {
    int tmp;
    tmp = ux_left; ux_left = ux_right; ux_right = tmp;
#ifdef CONSTRAINT_DEBUG
    printf("x screen coords reordered.\n");
#endif /* CONSTRAINT_DEBUG */
  }
  if ( uy_bottom >= uy_top ) {
    int tmp;
    tmp = uy_bottom; uy_bottom = uy_top; uy_top = tmp;
#ifdef CONSTRAINT_DEBUG
    printf("y screen coords reordered\n");
#endif /* CONSTRAINT_DEBUG */
  }

#ifdef CONSTRAINT_DEBUG
  printf("constraint (%d,%d) to (%d, %d):\n",
	 ux_left, uy_bottom, ux_right, uy_top);
#endif /* CONSTRAINT_DEBUG */

  /* calculate new coords with no constraints */
  xl = unmap_coord(x_type, x_left, x_right, x_size, ux_left);
  xr = unmap_coord(x_type, x_left, x_right, x_size, ux_right);
  yb = unmap_coord(y_type, y_bottom, y_top, y_size, uy_bottom);
  yt = unmap_coord(y_type, y_bottom, y_top, y_size, uy_top);

#ifdef CONSTRAINT_DEBUG
  printf("new trial coords (%s, ", unparse_coord(x_type, xl));
  printf(" %s), (", unparse_coord(y_type, yb));
  printf("%s, ", unparse_coord(x_type, xr));
  printf("%s)\n", unparse_coord(y_type, yt));
#endif /* CONSTRAINT_DEBUG */

  /* compute units/pixel in x and y */
  xdelta = subtract_coord(x_type, xr, xl);
  ydelta = subtract_coord(y_type, yb, yt);
  xupp = fabs(scale_coord(x_type, xdelta) / x_size);
  yupp = fabs(scale_coord(y_type, ydelta) / y_size);

#ifdef CONSTRAINT_DEBUG
  printf("ORIG: xupp %g, yupp %g\t", xupp, yupp);
#endif /* CONSTRAINT_DEBUG */

  /* Center of new graph will be the center of the zoomed area. */
  xc = unmap_coord(x_type, xl, xr, 1, 0.5);
  yc = unmap_coord(y_type, yb, yt, 1, 0.5);

  /* Adjust scale to fit aspect ratio constraint */
  switch ( constraint->constraint_type ) {
  case NONE:
#ifdef CONSTRAINT_DEBUG
    printf("no constraint requested\n");
#endif /* CONSTRAINT_DEBUG */
    break;

  case GEODETIC:
    /* compute aspect ratio for lat/lon at the center lat of new graph */
    constraint->aspect_ratio =
      1.0 / cos( scale_coord(y_type, yc) * M_PI / 180.0 );
    /* fallthrough */
  case ASPECT:
  case RASTER:
    /*
     * We want the same/constrained ratio upp for both dimensions.
     * If they don't match, we either have to zoom further in one,
     * or less far in the other.  We choose the "larger" value (adjusted
     * for desired aspect ratio), so the zoom is less rather than more.
     */
    if ( yupp * constraint->aspect_ratio > xupp ) {
      xupp = yupp * constraint->aspect_ratio;
    } else {
      yupp = xupp / constraint->aspect_ratio;
    }
    break;

  case SCHEME:
    printf("Scheme constraints not yet supported\n");
    break;
  }

  if ( constraint->constraint_type == RASTER ) {
    /*
     * Adjust scale to fit RASTER base*2^k constraint
     * Find smallest valid scale that is not smaller than
     * current chosen values.
     * Example: raster base = 2.5 m/pixel
     *          input is 8 m/pixel
     *          output is 10 m/pixel
     */

    double trial_upp = constraint->coord_per_pixel;
    /* first shrink */
    while ( trial_upp > xupp ) {
      trial_upp /= 2.0;
    }
    while ( trial_upp < xupp ) {
      trial_upp *= 2.0;
      if ( trial_upp >= 80 && trial_upp <= 85)
	trial_upp = 50.8;
    }
    xupp = yupp = trial_upp;
  }

#ifdef CONSTRAINT_DEBUG
  printf("NEW: xupp %g, yupp %g\n", xupp, yupp);
#endif /* CONSTRAINT_DEBUG */

  /* Have middle - now compute delta around middle from scale */
  xdelta = unscale_coord(x_type, xupp * x_size / 2);
  ydelta = unscale_coord(y_type, yupp * y_size / 2);
  *new_x_left = subtract_coord(x_type, xc, xdelta);
  *new_y_bottom = subtract_coord(y_type, yc, ydelta);
  *new_x_right = add_coord(x_type, xc, xdelta);
  *new_y_top = add_coord(y_type, yc, ydelta);

  if ( constraint->constraint_type == RASTER) {
    /* adjust to grid */

    /* XXX adjust left, bottom to multiple of current scale */
    /* XXX adjust right, top to left, bottom + scale * npixels */

    (void) 0;			/* dummy statement for compiler */
  }

  /* compute screen coords in *current view* from new coords */
  *new_ux_left = map_coord(x_type, x_left, x_right, x_size, *new_x_left);
  *new_uy_bottom = map_coord(y_type, y_bottom, y_top, y_size, *new_y_bottom);
  *new_ux_right = map_coord(x_type, x_left, x_right, x_size, *new_x_right);
  *new_uy_top = map_coord(y_type, y_bottom, y_top, y_size, *new_y_top);
  *zoomed_x = 1;
  *zoomed_y = 1;

#ifdef CONSTRAINT_DEBUG
  printf("final zoom box (%d, %d) to (%d, %d)\n",
	 *new_ux_left, *new_uy_bottom, *new_ux_right, *new_uy_top);
#endif /* CONSTRAINT_DEBUG */
  return 1;
}

