//---------------------------------------------------------------------------
// Copyright (C) 2000 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  PARSETAG.C   - source file containing the Object-Based Messaging
//                 parse functions.  Also functions to check and set
//                 MIB (Message Information Block).
//
//  Version: 2.00
//  Revisions:
//

#define MIB_DECL

// Includes
#include "1wsensor.h"
#include "msginc.h"
#include "mib.h"

// local function prototypes
int ParseTAG(int,TAGType *,int,int);
int TAGVector(int,uchar,ushort, TAGType *);
int InMIB(ushort);
int SkipGroup(TAGType *);
MIBEntry *GetMIB(ushort);
int SavePayloadToMIB(TAGType *, ushort, uchar);
int ReadFromMIB(ushort, uchar *, int);
int WriteToMIB(ushort, void *, int);
void InitializeMIB(void);
void *MIBPayload(ushort);
int ZeroMIBVariable(ushort);
int MIBZero(ushort);

//--------------------------------------------------------------------------
//  Parse the Object Based Message.  Reset the parse if 'reset_parse'
//  is TRUE (1).  Only parse to the excecution of a vector if 'parse_all'
//  is FALSE (0).
//
//  Returns:    TRUE (1)  -  TAG parsed without errors
//              FALSE (0) -  problem parsing
//              NOT_DONE (-1) - parse_all is false and not done yet
//
int ParseTAG(int portnum, TAGType *TAG, int reset_parse, int parse_all)
{
   static ushort obmstack[50];
   static int init_mib=FALSE;
   ushort obmdesc;
   uchar lsbyte,msbyte,ln;
   int vect_rslt;

   // check if MIB needs to be initialized
   if (!init_mib)
   {  
      InitializeMIB();
      init_mib = TRUE;
   }

   // peek ahead to make sure this is parse-ble data
   if ((TAG->msg[0] != (GD_ParseData & 0xFF)) ||
       (TAG->msg[1] != (GD_ParseData >> 8))) 
      return FALSE;

   // reset parse counters
   if (reset_parse)
   {
      TAG->parse = 0;
      TAG->depth = 0;
   }

   // loop to parse entire TAG message
   while(TAG->parse < TAG->msglen)
   {
      // get the first byte of the object descriptor (LSB)
      lsbyte = TAG->msg[TAG->parse++]; 
      // check for group discriptor
      if (lsbyte & 0x80)
      {
         // group end?
         if (lsbyte == 0xFF)
         {
            // ending a group so decrease the depth
            TAG->depth--;
            // check for not end of depth   
            if (TAG->depth > 0)
            {
               // execute the vector from this object
               if (TAGVector(portnum, POST_EX,obmstack[TAG->depth],TAG) 
                     && !parse_all)
                  return NOT_DONE;
               // start loop over               
               continue; 
            }
            else
               // done with parse
               return TRUE; 
         }
         else
         {
            // get msbyte
            msbyte = TAG->msg[TAG->parse++]; 
            // construct the object descriptor
            obmdesc = (ushort)((msbyte << 8) | lsbyte);
            // check if this is a valid descriptor (in the MIB)
            if (InMIB(obmdesc))
            {
               // execute in case this is a pre-exectute 
               vect_rslt = TAGVector(portnum, PRE_EX, obmdesc, TAG);

               // push this group onto the stack
               obmstack[TAG->depth] = obmdesc;
               TAG->depth++;

               // check if need to return early 
               if (vect_rslt && !parse_all)
                  return NOT_DONE;

               // start loop over 
               continue;               
            }
            // object not in MIB
            else 
            {  
               // skip all of this object and sub-objects
               if (!SkipGroup(TAG))
                  return FALSE;
               // start loop over               
               continue; 
            }   
         }
      }
      // data descriptor
      else
      {
         // get msbyte
         msbyte = TAG->msg[TAG->parse++]; 
         // construct the object descriptor
         obmdesc = (ushort)((msbyte << 8) | lsbyte);
         // get the length byte
         ln = TAG->msg[TAG->parse++]; 
         // check if descriptor not in the MIB
         if (!InMIB(obmdesc))
         {  
            // skip all of this data object 
            TAG->parse += ln;
            // start loop over               
            continue; 
         }  
         // check if length non-zero data payload
         if (ln != 0)
         {
            // extract the data
            if (!SavePayloadToMIB(TAG,obmdesc,ln))
               return FALSE;

            continue;
         }

      }
   }

   // ran out of bytes 
   return FALSE;
}


//--------------------------------------------------------------------------
//  Retrieve the payload data and place it in the MIB
//
//  Returns:  TRUE  - object found, parse points to right at OD
//            FALSE - object not found
//
int SavePayloadToMIB(TAGType *TAG, ushort obmdesc, uchar ln)
{
   MIBEntry *mibe;
   int i;
   uchar *puchar;
   ushort check; // Odd BUG in compiler, need to use temp variable

   // get the mib entry
   mibe = GetMIB(obmdesc); 
 
   // verify not null
   if (mibe != NULL)
   {
      // make sure not longer then max
      if (mibe->maxlen < ln)
         return FALSE;

      // get a convenient pointer   
      puchar = mibe->payload;

      // set the payload length
      puchar[0] = ln;
      
      // loop to copy the data 
      for (i = 0; i < ln; i++)
         puchar[i + 1] = TAG->msg[TAG->parse + i];  

      // check if this is the Cluster ID or Revision
      // extract for tag tracking purposes
      check = DD_ClusterID;
      if (obmdesc == check)
      {
         TAG->tagid[0] = ln; 
         for (i = 0; i < ln; i++)
            TAG->tagid[i + 1] = TAG->msg[TAG->parse + i]; 
         for (; i < (mibe->maxlen - 1); i++) 
            TAG->tagid[i + 1] = 0;           
      }
      check = DD_ClusterRev;
      if (obmdesc == check)
      {
         TAG->tagrev = 0;
         for (i = ln - 1; i >= 0; i--)
            TAG->tagrev |= TAG->msg[TAG->parse + i] << (8 * i);
      }

      // correct parse count
      TAG->parse += ln;
 
      return TRUE;
   }
   else
      return FALSE;
}

//--------------------------------------------------------------------------
// Vector to the Mib entry group. 'attrib' indicates the pre or post
// aspect of this call.  'obmdesc' is the group identifier to call.
// '*Tag' is the pointer to the current tag being parsed.  The tag
// is used to pass the current 'depth' to the vector function for
// nicer printing.
//
// Returns:  TRUE  - mib entry found and called
//           FALSE - mib entry not found or vector not called due
//                   to incorrect time (pre or post call)
//
int TAGVector(int portnum, uchar attrib, ushort obmdesc, TAGType *TAG)
{
   MIBEntry *mibe;
   GroupVect func;

   // get the mib entry
   mibe = GetMIB(obmdesc); 
 
   // verify not null
   if (mibe != NULL)
   {
      // check if correct time to call (PRE_EX or PST_EX)
      if ((mibe->attrib & attrib) == attrib)
      {   
         // get a pointer to the function
         func = (GroupVect)(mibe->payload);
         // call the function 
         func(portnum,attrib,TAG->depth);
         // return TRUE indicating vector was called
         return TRUE;
      }
   }

   // not in MIB or not time for vector to be called
   return FALSE;
}

//--------------------------------------------------------------------------
// Check to see if object type is in MIB (Message Information Block)
//
// Returns:   TRUE  - object found in MIB
//            FALSE - object not found in MIB
//
int InMIB(ushort obmdesc)
{
   return (GetMIB(obmdesc) != NULL);
}

//--------------------------------------------------------------------------
// Skip the current group and any objects it contains while parsing.
// This function is used when the group object is not know.
// 
// Returns:  TRUE  - object skipped
//           FALSE - error occured while skipping
//
int SkipGroup(TAGType *TAG)
{
   int desired_depth;
   uchar lsbyte,ln;

   // already past the descriptor of group to skip

   // find next end marker with the depth at the current level
   desired_depth = TAG->depth;

   while(TAG->parse < TAG->msglen)
   {
      // get the first byte of the object descriptor (LSB)
      lsbyte = TAG->msg[TAG->parse++]; 
      // check for group discriptor
      if (lsbyte & 0x80)
      {
         // group end?
         if (lsbyte == 0xFF)
         {
            // check if this is the end marker we are looking for
            if (TAG->depth == desired_depth)
               return TRUE;

            // ending a group so decrease the depth
            TAG->depth--;
         }
         else
         {
            // skip get msbyte
            TAG->parse++; 
            // increase the depth
            TAG->depth++;
         }

      }
      // data descriptor
      else
      {
         // skip get msbyte of data descriptor
         TAG->parse++; 
         // get the length byte
         ln = TAG->msg[TAG->parse++]; 
         // skip all of this data object 
         TAG->parse += ln;
      }   
   }      

   // ran out of bytes 
   return FALSE;
}

//--------------------------------------------------------------------------
// Search for the MIB entry described by the 'find_obmdesc' descriptor
//
// Returns:   pointer to MIBEntry
//            NULL: entry not found
//
MIBEntry *GetMIB(ushort find_tagdesc)
{
   int count=0;

   // loop while still have entries in the MIB
   while (MIB[count].tagdesc != 0)
   {
      if (find_tagdesc == MIB[count].tagdesc)
         return &MIB[count];

      count++;
   }

   // entry not found
   return NULL;
}

//--------------------------------------------------------------------------
// Search for the MIB entry described by the 'find_obmdesc' descriptor
//
// Returns:    pointer to MIBEntry's payload
//             NULL: entry not found or payload points to NULL
//
void *MIBPayload(ushort find_tagdesc)
{
   MIBEntry *mibe;
   
   // get the mib entry
   mibe = GetMIB(find_tagdesc); 

   // check for no entry
   if (mibe == NULL)
      return NULL;

   return mibe->payload;
}

//--------------------------------------------------------------------------
// Search for the MIB entry described by the 'find_obmdesc' descriptor.
// Copy the payload into the provided buffer up to a maxlen size.
//
// Returns:  number of bytes copied from mib entry payload into buf
//
int ReadFromMIB(ushort find_tagdesc, uchar *buf, int maxlen)
{
   MIBEntry *mibe;
   uchar *puchar;
   int i,cpylen;

   // get the mib entry
   mibe = GetMIB(find_tagdesc); 

   // check for no entry
   if (mibe == NULL)
      return 0;

   // make convenient pointer
   puchar = (uchar *)mibe->payload;

   // check size, only go up to max size
   if (puchar[0] > maxlen)
      cpylen = maxlen;
   else
      cpylen = puchar[0];

   // copy 
   for (i = 0; i < cpylen; i++)
      buf[i] = puchar[i + 1];

   // return number of bytes read
   return cpylen;  
}

//--------------------------------------------------------------------------
// Search for the MIB entry described by the 'find_obmdesc' descriptor.
// Copy the provided buffer of data to the payload of the MIB entry.
// provided size is ok.
//
// Returns:  number of bytes copied from buffer to the payload mib entry
//           0 could indicate that the MIB entry was not found or did
//           not contain payload space.
//
int WriteToMIB(ushort find_tagdesc, void *buf, int len)
{
   MIBEntry *mibe;
   uchar *toptr,*fromptr;
   int i,cpylen;

   // get the mib entry
   mibe = GetMIB(find_tagdesc);

   // check for no entry
   if (mibe == NULL)
      return 0;
   if (mibe->payload == NULL)
      return 0;

   // make convenient pointers
   toptr = (uchar *)mibe->payload;
   fromptr = (uchar *)buf;

   // check size, only go up to max size
   if (mibe->maxlen > len)
      cpylen = len;
   else
      cpylen = mibe->maxlen;

   // sent the length
   toptr[0] = (uchar)cpylen;

   // copy 
   for (i = 0; i < cpylen; i++)
      toptr[i + 1] = fromptr[i];

   // return number of bytes read
   return cpylen;  
}

//--------------------------------------------------------------------------
// Loop through the MIB and allocate data space for each of the 
// entries that need it.  This function should only be called once.
//
void InitializeMIB(void)
{
   int count=0;
   uchar *puchar;

   // loop while still have entries in the MIB
   while (MIB[count].tagdesc != 0)
   {
      // check if memory not allocated for mib payload
      if ((MIB[count].payload == NULL) && (MIB[count].maxlen != 0))
      {
         // get the memory
         MIB[count].payload = malloc(MIB[count].maxlen + 2);
         if (MIB[count].payload == NULL)
            exit(100);
         // set initial length to zero
         puchar = MIB[count].payload;
         *puchar = 0;
      }

      count++;
   }
}

//--------------------------------------------------------------------------
// Search for the MIB entry described by the 'find_obmdesc' descriptor.
// If possible set the payload length to zero.
//
// Returns:  TRUE  - MIB entry found and payload length set to zero
//           FALSE - MIB entry not found or payload is NULL or
//                   max payload length is zero.
//
int ZeroMIBVariable(ushort find_tagdesc)
{
   MIBEntry *mibe;
   uchar *puchar;

   // get the mib entry
   mibe = GetMIB(find_tagdesc); 

   // check for no entry
   if (mibe == NULL)
      return FALSE;

   // check for zero max length or null payload
   if ((mibe->maxlen == 0) || (mibe->payload == NULL))
      return FALSE;

   // make convenient pointer
   puchar = (uchar *)mibe->payload;

   // zero out the payload length
   puchar[0] = 0;

   return TRUE;
}

//--------------------------------------------------------------------------
// Search for the MIB entry described by the 'find_obmdesc' descriptor.
// Detect if the current length of the MIB entry payload is zero.
//
// Returns:  TRUE - MIB entry found and payload length is set to zero
//                  or MIB entry not found or payload is NULL or
//                  max payload length is zero.
//           FALSE - MIB entry found and payload length is more then zero
//
int MIBZero(ushort find_tagdesc)
{
   MIBEntry *mibe;
   uchar *puchar;

   // get the mib entry
   mibe = GetMIB(find_tagdesc); 

   // check for no entry
   if (mibe == NULL)
      return TRUE;

   // check for zero max length or null payload
   if ((mibe->maxlen == 0) || (mibe->payload == NULL))
      return TRUE;

   // make convenient pointer
   puchar = (uchar *)mibe->payload;

   // check zero on the payload length
   if (puchar[0] == 0)
      return TRUE;

   return FALSE;
}