/**
 * A layout manager for ncurses.
 */

#include <string.h>

#include "ui-layout.h"

extern void set_coords(struct coords *co, int top, int left, int bottom, int right)
{
    co->top = top, co->left = left, co->bottom = bottom, co->right = right;
    co->height = bottom - top, co->width = right - left;
}

extern void mvwhsplit(WINDOW *win, int y, int x, int len)
{
    mvwaddch(win, y, x, ACS_LTEE);
    mvwhline(win, y, x + 1, ACS_HLINE, len - 2);
    mvwaddch(win, y, x + len - 1, ACS_RTEE);
}

extern void mvwvsplit(WINDOW *win, int y, int x, int len)
{
    mvwaddch(win, y, x, ACS_TTEE);
    mvwvline(win, y + 1, x, ACS_VLINE, len - 2);
    mvwaddch(win, y + len - 1, x, ACS_BTEE);
}

extern void mvwaddlstr(WINDOW *win, int y, int x, const char *s, int maxlen)
{
    int len = (int) strlen(s);
    wmove(win, y, x);
    if (len > maxlen) {
        while (maxlen > 3 && *s != '\0') {
            waddch(win, *s++);
            maxlen--;
        }
        while (maxlen > 0) {
            waddch(win, '.');
            maxlen--;
        }
    } else {
        waddstr(win, s);
    }
}

static int int_min(int a, int b)
{
    return (a < b) ? a : b;
}

extern int input_box(char *buf, size_t bufsize, const char *title)
{
    struct coords input;
    int hlen, maxy, maxx, result;

    getmaxyx(stdscr, maxy, maxx);
    set_coords(&input, maxy / 2 - 1, 10, maxy / 2 + 2, maxx - 10);
    hlen = int_min(strlen(title), (input.right - input.left) - 4);

    cbreak(), echo(), leaveok(stdscr, FALSE);

    mvaddch(input.top, input.left, ACS_ULCORNER);
    hline(ACS_HLINE, input.width - 2);
    mvaddch(input.top, input.right - 1, ACS_URCORNER);
    mvaddch(input.top + 1, input.left, ACS_VLINE);
    hline(' ', input.width - 2);
    mvaddch(input.top + 1, input.right - 1, ACS_VLINE);
    mvaddch(input.bottom - 1, input.left, ACS_LLCORNER);
    mvhline(input.bottom - 1, input.left + 1, ACS_HLINE, input.width - 2);
    mvaddch(input.bottom - 1, input.right - 1, ACS_LRCORNER);
    mvaddnstr(input.top, input.left + (input.width - hlen) / 2, title, hlen);

    curs_set(1);
    result = (mvgetnstr(input.top + 1, input.left + 1, buf, bufsize) == ERR) ? -1 : 0;
    curs_set(0);

    noecho(), halfdelay(5), leaveok(stdscr, TRUE);

    return result;
}
