/* $Id$ */

/*
 * Copyright (c) 2015 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>
#include <err.h>
#include <string.h>
#include <sysexits.h>
#include <ctype.h>
#include <sys/stat.h>

#include "syncffsd.h"

#ifndef STATE_OK
#define STATE_OK        0
#define STATE_WARNING   1
#define STATE_CRITICAL  2
#define STATE_UNKNOWN   3
#endif

#define DEFAULT_WARN_DELAY 300
#define DEFAULT_CRIT_DELAY 600
#define DEFAULT_TIMEOUT 3

void 
usage(void)
{
	fprintf(stderr, "check_syncffsd [options]\n");
	fprintf(stderr, "Options:\n");
	fprintf(stderr, "  -c crit_delay Critical sync lag, default %ds\n", 
			DEFAULT_CRIT_DELAY);
	fprintf(stderr, "  -c warn_delay Warning sync lag, default %ds\n", 
			DEFAULT_WARN_DELAY);
	fprintf(stderr, "  -p pid_file   Location of PID file, "
			"default %s\n", DEFAULT_PIDFILE);
	fprintf(stderr, "  -S state_file Location of state file, "
			"default %s\n", DEFAULT_STATEFILE);
	fprintf(stderr, "  -t timeout    Tiemout awaiting status, "
			"default %ds\n", DEFAULT_TIMEOUT);

	exit(EX_USAGE);
}

int
main(argc, argv)
	int argc;
	char **argv;
{
	const char *pidfile = DEFAULT_PIDFILE;
	const char *statefile = DEFAULT_STATEFILE;
	int warn = DEFAULT_WARN_DELAY;
	int crit = DEFAULT_CRIT_DELAY;
	int timeout = DEFAULT_TIMEOUT;
	int fl;
	FILE *p;
	pid_t pid;
	time_t ts;
	FILE *s;
	struct stat st;
	int timestamp;
	int sync_delay = -1;
	int dump_age;
	int total_delay;
	char *prefix = NULL;

	while ((fl = getopt(argc, argv, "c:p:t:S:u:w:")) != -1) {
		switch(fl) {
		case 'c':
			crit = atoi(optarg);
			break;
		case 'p':
			pidfile = optarg;
			break;
		case 't':
			timeout = atoi(optarg);
			break;
		case 'S':
			statefile = optarg;
			break;
		case 'w':
			warn = atoi(optarg);
			break;
		default:
			usage();
			break;
		}
	}

	argc -= optind;

	if (argc != 0)
		usage();

	if ((p = fopen(pidfile, "r")) == NULL) {
		printf("UNKNOWN cannot open %s\n", pidfile);
		exit(STATE_UNKNOWN);
	}

	if (fscanf(p, "%d\n", &pid) != 1) {
		printf("UNKNOWN cannot get PID from %s\n", pidfile);
		exit(STATE_UNKNOWN);
	}

	(void)fclose(p);

	ts = time(NULL);

	/* 
	 * Try to obtain a recent file by sending SIGINFO,
	 * but fallback on using an older one if we cannot,
	 * for instance because we run on different UID)
	 */ 
	if (kill(pid, SIGINFO) == 0) {
		do {
			if (time(NULL) > ts + timeout) {
				printf("UNKNOWN timeout\n");
				exit(STATE_UNKNOWN);
			}

			if (stat(statefile, &st) == 0 && (st.st_mtime >= ts))
				break;

			sleep(1);
		} while (1 /* CONSTCOND */);
	}

	if ((s = fopen(statefile, "r")) == NULL) {
		printf("UNKNOWN cannot open %s\n", statefile);
		exit(STATE_UNKNOWN);
	}

	do {
		char line[512];
		char *cp;

		if (fgets(line, sizeof(line), s) == NULL)
			break;	

		if (strstr(line, "timestamp:") == line) {
			for (cp = line; *cp && !isdigit((int)*cp); cp++);

			if (*cp)
				timestamp = atoi(cp);

			continue;
		}

		if (strstr(line, "sync_delay:") == line) {
			for (cp = line; *cp && !isdigit((int)*cp); cp++);

			if (*cp)
				sync_delay = atoi(cp);

			continue;
		}

		if (strstr(line, "source_prefix:") == line) {
			for (cp = line; *cp && !isblank((int)*cp); cp++);

			if (*++cp) {
				prefix = strdup(cp);
				if ((cp = strchr(prefix, (int)'\n')) != NULL)
					*cp = '\0';
			}

			continue;
		}
	} while (1 /* CONSTCOND */);

	if (sync_delay == -1) {
		printf("UNKNOWN no report from daemon\n");
		exit(STATE_UNKNOWN);
	}

	/*
	 * Add the age of the state dump to the sync delay
	 */
	dump_age = time(NULL) - timestamp;
	total_delay = sync_delay + dump_age;
	
	if (total_delay > crit) {
		printf("CRITICAL syncffsd %s delay %ds (%ds readen %ds ago)\n",
		       prefix, total_delay, sync_delay, dump_age);
		exit(STATE_CRITICAL);
	}

	if (total_delay > warn) {
		printf("WARNING syncffsd %s delay %ds (%ds readen %ds ago)\n",
		       prefix, total_delay, sync_delay, dump_age);
		exit(STATE_WARNING);
	}

	printf("OK syncffsd %s delay %ds (%ds readen %ds ago)\n",
	       prefix, total_delay, sync_delay, dump_age);
	return STATE_OK;
}

