/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
 
#ifndef _PROTO_SIM_AGENT
#define _PROTO_SIM_AGENT

// An "Agent" in a network simulation environment (such as "ns" or "opnet")
// can typically send/recv messages to/from the simulation environment.
// The "ProtoSimAgent" class allows us to bind a UdpSocket instance
// to a simulation environment agent by creating a replacement 
// implementation of the UdpSocket class.

// Our "ProtoSimAgent" also has hooks for binding ProtocolTimerMgr
// instances to timing mechanisms provided by simulation environments

// Simulation agents can for different simulation environments can
// be derived from this base class.

#include "udpSocket.h"
#include "protocolTimer.h"

class ProtoSimAgent
{
    public:
        // Simulation socket functions which must be overridden
        virtual bool OpenSocket(UdpSocket* theSocket, unsigned short* thePort) = 0;
        virtual void CloseSocket(UdpSocket* theSocket) = 0;       
        virtual bool SendTo(UdpSocket* theSocket, 
                            SIMADDR theAddr, 
                            unsigned short thePort,
                            char* buffer, unsigned int len) = 0;
        virtual bool RecvFrom(UdpSocket*        theSocket,
                              char*             buffer,
                              unsigned int*     buflen,
                              SIMADDR*          srcAddr,
                              unsigned short*   srcPort) = 0; 
        virtual bool JoinGroup(UdpSocket* theSocket, SIMADDR theGroup) = 0;
        virtual void LeaveGroup(UdpSocket* theSocket, SIMADDR theGroup) = 0;
        virtual void SetSocketTTL(UdpSocket* theSocket, unsigned char theTTL) = 0;
        virtual void SetSocketLoopback(UdpSocket* theSocket, bool state) = 0;
        
        // Simple installer functions for binding ProtoTimerMgr and
        // ProtoSocket classes to the simulation environment
        static bool TimerInstaller(ProtocolTimerInstallCmd cmd,
                                   double                  delay,
                                   ProtocolTimerMgr*       timerMgr, 
                                   const void*             installData); 
        static bool SocketInstaller(UdpSocketCmd        cmd,
                                    class UdpSocket*    theSocket,
                                    const void*         installData);
        
    protected:
        // Simulation timer mgmt functions (for a single ProtocolTimerMgr)
        virtual void InstallTimer(double delay) = 0;
        virtual void ModifyTimer(double delay) = 0;
        virtual void RemoveTimer() = 0;
        
};

#endif // _PROTO_SIM_AGENT
