/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/

#ifndef _EVENT_DISPATCHER
#define _EVENT_DISPATCHER

#include "protocolTimer.h"
#include "udpSocket.h"

// Asynchronous event dispatcher class
class EventDispatcher
{
    public:
    
        // Asynchronous event types
        enum EventType {EVENT_NULL, EVENT_INPUT, EVENT_DETACH};
    
#ifdef WIN32
//#include <Windows.h>
        typedef SOCKET Descriptor;  // WIN32 uses "SOCKET" type for descriptors
#endif // WIN32
#ifdef UNIX
        typedef int Descriptor;      // UNIX uses "int" type for descriptors
#endif // UNIX
        static const Descriptor INVALID_DESCRIPTOR;

        // For arbitrary asynchronous file descriptor monitoring
        typedef void (*Callback)(Descriptor descriptor, EventType theEvent, const void* userData);

        class Item
        {
            friend class EventDispatcher;
            
            public:
                EventDispatcher::Descriptor Descriptor() {return descriptor;}
                
            private:
                void* operator new(size_t objSize, Item* item);
                Item();
                Item(EventDispatcher::Descriptor theDescriptor, 
                     Callback func, const void* userData);
                Item(UdpSocket* theSocket);
                Item(ProtocolTimerMgr* timerMgr);
                ~Item();
                
                void OnInputEvent();
            
                enum Type {GENERIC, SOCKET, TIMER};
                enum Flag {NONE = 0x00, INPUT = 0x01, OUTPUT = 0x02, EXCEPTION = 0x04};
                
				bool IsActive() {return (0 != flags);}
				bool IsInput() {return (0 != (flags & INPUT));}
				bool IsOutput() {return (0 != (flags & OUTPUT));}
                void SetFlag(Flag f) {flags |= f;}
				void UnsetFlag(Flag f) {flags &= ~f;}
                
                ProtocolTimer* Timer() {return &timer;}
                bool OnTimeout();
                void SetTimeout(double delay);
                void DeactivateTimer() {timer.Deactivate();}
                
                
                
                Type                         type;
                int							 flags;
				EventDispatcher::Descriptor  descriptor;
                const void*					 data;
                Callback					 callback;
                ProtocolTimer                timer;
                
                Item*      prev;
                Item*      next;
        };  // end class EventDispatcher::Item
                 
    // Construction
	    EventDispatcher();
        ~EventDispatcher();
        void Destroy();
        
    // Methods to add remove inputs, timers, etc
        const EventDispatcher::Item* AddGenericInput(EventDispatcher::Descriptor descriptor, 
                                                     EventDispatcher::Callback inputCallback, 
                                                     const void* clientData);        
        const EventDispatcher::Item* AddSocketInput(UdpSocket* theSocket);
		void RemoveGenericInput(EventDispatcher::Descriptor descriptor);
		void RemoveSocketInput(UdpSocket* theSocket);
        void RemoveInput(EventDispatcher::Item* item);
        
        void InstallTimer(ProtocolTimer* theTimer)
            {timer_mgr.InstallTimer(theTimer);}

        // If this optional socket installer is used instead of
        // explicit calls to "AddSocketInput()", the UdpSocket "installData"
        // should be inited to a pointer to the EventDispatcher instance.
        static bool EventDispatcher::SocketInstaller(UdpSocketCmd  cmd,
                                                     UdpSocket*    theSocket,
                                                     const void*   installData);

        // (TBD) Provide a static TimerInstaller so EventDispatcher
        // users can use their own ProtocolTimerMgrs if desired
        static bool EventDispatcher::TimerInstaller(
                        ProtocolTimerInstallCmd     cmd, 
                        double                      delay,
                        ProtocolTimerMgr*           timerMgr, 
                        const void*                 installData);
        
        
    // Methods to set up and control EventDispatcher execution
#ifdef WIN32
		// WIN32 apps need to call this first thing before installing 
        // timers, running, etc
		bool Win32Init();
#endif // WIN32        
        int Run();
	    void Stop(int exitCode) 
		{
            exit_code = exitCode;
			run = false;
#ifdef WIN32
			PostMessage(msg_window, WM_DESTROY, NULL, NULL);
#endif // WIN32
		}
                   
	private:
    // Methods
        Item* GetItemFromPool();
        void ReturnItemToPool(Item* item);
		void RemoveItem(EventDispatcher::Item* theItem);
        EventDispatcher::Descriptor FindMaxDescriptor();
		Item* FindItemByDescriptor(EventDispatcher::Descriptor theDescriptor);
        Item* FindItemByData(const void* data);
        void SetTimeout(double value) {delay = value;}
        bool InstallTimerMgr(ProtocolTimerMgr* timerMgr, double delay);
        bool ModifyTimerMgr(ProtocolTimerMgr* timerMgr, double delay);
        bool RemoveTimerMgr(ProtocolTimerMgr* timerMgr);
        static bool EventDispatcher::PrivateTimerInstaller(ProtocolTimerInstallCmd    cmd, 
                                                           double                    theDelay,
                                                           ProtocolTimerMgr*         timerMgr,
                                                           const void*               installData);
#ifdef WIN32
		enum {SOCKET_MSG = WM_USER+1, TIMER_MSG};
		static LRESULT CALLBACK MessageHandler(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam);
		static void CALLBACK TimerCallback(UINT uID, UINT uMsg, DWORD dwUser, DWORD dw1, DWORD dw2);
		bool Win32Install(EventDispatcher::Item* item);
#endif // WIN32

    // Members
        ProtocolTimerMgr			timer_mgr;
	    double						delay;
	    Item*						top_item;
        Item*                       item_pool;
		Descriptor	                max_descriptor;
	    bool						run;
		int							exit_code;

#ifdef WIN32
		HWND						msg_window;
		UINT						mm_timer_resolution;
		MMRESULT					mm_timer_id;
		bool						mm_timer_msg_pending;
#endif // WIN32
};  // end class EventDispatcher

#endif // _EVENT_DISPATCHER
