---
authors: "azz"
title: "Prettier 1.8: Markdown Support"
---

This release adds Markdown support, a new `--insert-pragma` flag, fixes a number of formatting issues, adds support for some new _experimental_ operators, and improves our editor integration support.

<!-- truncate -->

## Highlights

### Markdown Support

#### Support markdown ([#2943](https://github.com/prettier/prettier/pull/2943)) by [@ikatyang](https://github.com/ikatyang)

You can now run Prettier on Markdown files! 🎉

The implementation is highly compliant with the [CommonMark spec](http://commonmark.org/), and backed by the excellent [`remark-parse`](https://github.com/wooorm/remark/tree/master/packages/remark-parse) package.

**Word Wrap**

One of Prettier's core features is its ability to wrap code at a specified line length. This applies to Markdown too, which means you can maintain nice and clean 80-character-wide Markdown files without having to re-adjust line breaks manually when you add or delete words.

Input:

```
Voilà! In view, a humble vaudevillian veteran cast vicariously as both victim and villain by the vicissitudes of Fate. This visage, no mere veneer of vanity, is a vestige of the vox populi, now vacant, vanished. However, this valourous visitation of a bygone vexation stands vivified and has vowed to vanquish these venal and virulent vermin vanguarding vice and vouchsafing the violently vicious and voracious violation of volition! The only verdict is vengeance; a vendetta held as a votive, not in vain, for the value and veracity of such shall one day vindicate the vigilant and the virtuous. Verily, this vichyssoise of verbiage veers most verbose, so let me simply add that it's my very good honour to meet you and you may call me V.
```

Output:

```
Voilà! In view, a humble vaudevillian veteran cast vicariously as both victim
and villain by the vicissitudes of Fate. This visage, no mere veneer of vanity,
is a vestige of the vox populi, now vacant, vanished. However, this valourous
visitation of a bygone vexation stands vivified and has vowed to vanquish these
venal and virulent vermin vanguarding vice and vouchsafing the violently vicious
and voracious violation of volition! The only verdict is vengeance; a vendetta
held as a votive, not in vain, for the value and veracity of such shall one day
vindicate the vigilant and the virtuous. Verily, this vichyssoise of verbiage
veers most verbose, so let me simply add that it's my very good honour to meet
you and you may call me V.
```

> ~~Note: We're considering adding an option for this, check [#3183](https://github.com/prettier/prettier/pull/3183) for discussion.~~
> Update: We've added an option in `1.8.2` called [`--no-prose-wrap`](https://prettier.io/docs/options#prose-wrap)

> Note for CJK users: If your markdown renderer does not support CJK line ending, you'll have to use plugin like [markdown-it-perfect-newline-for-cjk](https://www.npmjs.com/package/markdown-it-perfect-newline-for-cjk), [hexo-filter-fix-cjk-spacing](https://www.npmjs.com/package/hexo-filter-fix-cjk-spacing), etc. to remove additional spaces.
>
> ```
> // Source
> 一二三
> 四五六
> 七八九
>
> // Rendered content with unsupported renderer
> 一二三 四五六 七八九
>
> // Rendered content with supported renderer or via plugin
> 一二三四五六七八九
> ```

**Code Formatting**

Powered by Prettier's generic "multiparser", Prettier will format code blocks in Markdown! We use the language code provided with the code block to determine which language it is, and thus we can format any language that Prettier supports (including Markdown itself, if you're into that).

Input:

<!-- prettier-ignore -->
````md
```js
reallyUgly    (
javascript
  )
```

```css
.h1 {     color : red }
```
````

Output:

````md
```js
reallyUgly(javascript);
```

```css
.h1 {
  color: red;
}
```
````

<!-- prettier-ignore -->
> Note: In some cases you may not want to format your code in Markdown, and just like in other languages, in Markdown you can use an ignore comment before the code block to ignore it from formatting:
>
> ````md
> <!-- prettier-ignore -->
> ```js
> ugly   ( code ) ;
> ```
> ````

**Lists**

When rearranging list items, after running Prettier all the numbers will be fixed!

![Markdown Lists](./assets/markdown-lists.gif)

> Note: you can actually opt out of this by using `1.` for all list items if you want to optimize for cleaner diffs.

**Tables**

Tables will also automatically be adjusted to fit their contents. This could be completely unmaintainable without an automated tool.

![Markdown Tables](./assets/markdown-tables.gif)

**Markdown-in-JS**

By using either `md` or `markdown` tagged template literals, you can format markdown code inside JavaScript.

<!-- prettier-ignore -->
```js
const markdown = md`
  # heading

  1. list item
`;
```

### CLI

#### Add option to insert `@format` to first docblock if absent ([#2865](https://github.com/prettier/prettier/pull/2865)) by [@samouri](https://github.com/samouri)

In 1.7, we added an option called `--require-pragma` to require files contain an `/** @format */` pragma to be formatted. In order to add this pragma to a large set of files you can now use [`--insert-pragma`](https://prettier.io/docs/cli#insert-pragma) flag.

```
prettier --write "folder/**/*.js" --insert-pragma
```

#### Add `--loglevel` option ([#2992](https://github.com/prettier/prettier/pull/2992)) by [@ikatyang](https://github.com/ikatyang)

This [nifty feature](https://prettier.io/docs/cli#loglevel) allows you to opt in (or out) of Prettier's logging. We've also cleaned up the logging substantially since 1.7.

<!-- prettier-ignore -->
```bash
$ prettier --loglevel=debug blarg
$ ./bin/prettier.js --loglevel=debug blarg
[debug] normalized argv: {"_":["blarg"],"bracket-spacing":false,"color":true,"debug-check":false,"debug-print-doc":false,"flow-parser":false,"insert-pragma":false,"jsx-bracket-same-line":false,"list-different":false,"require-pragma":false,"semi":false,"single-quote":false,"stdin":false,"use-tabs":false,"version":false,"with-node-modules":false,"write":false,"loglevel":"debug","ignore-path":".prettierignore","config-precedence":"cli-override"}
[error] No matching files. Patterns tried: blarg !**/node_modules/** !./node_modules/**

```

### JavaScript

#### Fix indentation for JSDoc comments ([#2470](https://github.com/prettier/prettier/pull/2470)) by [@maxdeviant](https://github.com/maxdeviant)

This has been a long-time known issue with Prettier. When formatting code that results in a change of indentation level, the JSDoc comments would end up being out of alignment. We're happy to report this is now fixed!

<!-- prettier-ignore -->
```js
// Before
function theFunction2(action$, store) {
  /*
     * comments
     */
  return true;
}

// After
function theFunction2(action$, store) {
  /*
   * comments
   */
  return true;
}
```

#### Print pipeline and nullish-coalescing operators ([#3036](https://github.com/prettier/prettier/pull/3036)) by [@azz](https://github.com/azz)

We've added support for two new proposed operators to Prettier: the _pipeline operator_ and the _nullish coalescing operator_.

The [pipeline operator](https://github.com/tc39/proposal-pipeline-operator/) is currently a stage one proposal.

> This proposal introduces a new operator |> similar to F#, OCaml, Elixir, Elm, Julia, Hack, and LiveScript, as well as UNIX pipes. It's a backwards-compatible way of streamlining chained function calls in a readable, functional manner, and provides a practical alternative to extending built-in prototypes.

<!-- prettier-ignore -->
```js
// Before
let result = exclaim(capitalize(doubleSay("hello")));

// After
let result = "hello"
  |> doubleSay
  |> capitalize
  |> exclaim;
```

The [nullish coalescing operator](https://github.com/tc39-transfer/proposal-nullish-coalescing) is another stage one proposal.

> When performing optional property access in a nested structure in conjunction with the optional chaining operator, it is often desired to provide a default value if the result of that property access is null or undefined.

This operator is similar to `||` except it only evaluates the right-hand-side if the left is `undefined` or `null`, not `""`, `0`, `NaN`, etc.

<!-- prettier-ignore -->
```js
const foo = object.foo ?? "default";
```

#### Improved template literal expressions line breaks ([#3124](https://github.com/prettier/prettier/pull/3124)) by [@duailibe](https://github.com/duailibe)

This was another known issue with Prettier, when printing a template literal string with expressions inside that went over the print width, it would wrap the code in weird places inside the expressions. Now, if Prettier needs to insert a line break, it should happen right between `${` and `}`.

<!-- prettier-ignore -->
```jsx
// Before
const description = `The value of the ${cssName} css of the ${this
  ._name} element`;

const foo = `mdl-textfield mdl-js-textfield ${className} ${content.length > 0
  ? "is-dirty"
  : ""} combo-box__input`;

// After
const description = `The value of the \${cssName} css of the \${
  this._name
} element`;

const foo = `mdl-textfield mdl-js-textfield ${className} ${
  content.length > 0 ? 'is-dirty' : ''
} combo-box__input`
```

### JSX

#### Don't inline trailing `}` for arrow functions attributes ([#3110](https://github.com/prettier/prettier/pull/3110)) by [@duailibe](https://github.com/duailibe)

In order to align closer to the [Airbnb style guide](https://github.com/airbnb/javascript/), and since it was never intentionally printed this way, we've moved the `}` from to the next line in JSX. This is more diff friendly, and makes it easier to move code around by shifting lines in your editor.

<!-- prettier-ignore -->
```jsx
// Before
<BookingIntroPanel
  logClick={data =>
    doLogClick("long_name_long_name_long_name", "long_name_long_name_long_name", data)}
/>;

// After
<BookingIntroPanel
  logClick={data =>
    doLogClick("long_name_long_name_long_name", "long_name_long_name_long_name", data)
  }
/>;
```

## Other Changes

### JavaScript

#### Make the factory detection handle multiple elements ([#3112](https://github.com/prettier/prettier/pull/3112)) by [@vjeux](https://github.com/vjeux)

There was a bug in the heuristic that Prettier uses to determine whether an expression is a factory or not. It now works correctly with longer member expressions.

<!-- prettier-ignore -->
```js
// Before
window.FooClient
  .setVars({
    locale: getFooLocale({ page }),
    authorizationToken: data.token
  })
  .initVerify("foo_container");

// After
window.FooClient.setVars({
  locale: getFooLocale({ page }),
  authorizationToken: data.token
}).initVerify("foo_container");
```

#### Handle comments between function name and open paren ([#2979](https://github.com/prettier/prettier/pull/2979)) by [@azz](https://github.com/azz)

Printing comments in the right place is an endless challenge 😉. This fix ensures that comments next to function names are re-printed correctly.

<!-- prettier-ignore -->
```js
// Before
function f(/* comment*/ promise) {}

// After
function f /* comment*/(promise) {}
```

#### Support sequential CallExpressions in member chains ([#2990](https://github.com/prettier/prettier/pull/2990)) by [@chrisvoll](https://github.com/chrisvoll)

Member chains are one of the most complex parts of Prettier. This PR fixes an issue where repeated calls lead to the next method not being pushed to the next line.

<!-- prettier-ignore -->
```js
// Before
wrapper
  .find("SomewhatLongNodeName")
  .prop("longPropFunctionName")().then(function() {
  doSomething();
});

// After
wrapper
  .find("SomewhatLongNodeName")
  .prop("longPropFunctionName")()
  .then(function() {
    doSomething();
  });
```

#### Account for empty lines in long member call chain ([#3035](https://github.com/prettier/prettier/pull/3035)) by [@jackyho112](https://github.com/jackyho112)

Previously, Prettier would delete all newlines within a member chain. Now we keep up to one if it's in the source. This is nice for fluent APIs that you want to break up over multiple lines.

<!-- prettier-ignore -->
```js
angular
  .module("AngularAppModule")

  // Constants.
  .constant("API_URL", "http://localhost:8080/api")

  // App configuration.
  .config(appConfig)
  .run(appRun);
```

#### Fix issue where first argument is left behind when line breaks ([#3079](https://github.com/prettier/prettier/pull/3079)) by [@mutdmour](https://github.com/mutdmour)

This addresses an issue where due to our special object inline behaviour, the indentation missing from the function call.

<!-- prettier-ignore -->
```js
// Before
db.collection("indexOptionDefault").createIndex({ a: 1 },
{
  indexOptionDefaults: true
},
function(err) {
  // code
});

// After
db.collection("indexOptionDefault").createIndex(
  { a: 1 },
  {
    indexOptionDefaults: true
  },
  function(err) {
    // code
  }
);
```

#### Break parens for binaries in member expression ([#2958](https://github.com/prettier/prettier/pull/2958)) by [@duailibe](https://github.com/duailibe)

Similarly, there was another edge case where indentation was missing from logical expressions. This is fixed, too.

<!-- prettier-ignore -->
```js
// Before
const someLongVariable = (idx(
  this.props,
  props => props.someLongPropertyName
) || []
).map(edge => edge.node);

// After
const someLongVariable = (
  idx(this.props, props => props.someLongPropertyName) || []
).map(edge => edge.node);
```

#### Prevent breaking MemberExpression inside NewExpression ([#3075](https://github.com/prettier/prettier/pull/3075)) by [@duailibe](https://github.com/duailibe)

There are so many ways to break a line. Some of them look much worse than others. Breaking between in this case looked really weird, so it has been fixed!

<!-- prettier-ignore -->
```js
// Before
function functionName() {
  if (true) {
    this._aVeryLongVariableNameToForceLineBreak = new this
      .Promise((resolve, reject) => {
        // do something
      });
  }
}

// After
function functionName() {
  if (true) {
    this._aVeryLongVariableNameToForceLineBreak = new this.Promise(
      (resolve, reject) => {
        // do something
      }
    );
  }
}
```

#### Fix array accessors in method chains ([#3137](https://github.com/prettier/prettier/pull/3137)) by [@duailibe](https://github.com/duailibe)

In a method chain we split lines by grouping elements together and accessing an array should be printed in the end of a group instead of the beginning.

<!-- prettier-ignore -->
```js
// Before
find('.org-lclp-edit-copy-url-banner__link')
  [0].getAttribute('href')
  .indexOf(this.landingPageLink)

// After
find('.org-lclp-edit-copy-url-banner__link')[0]
  .getAttribute('href')
  .indexOf(this.landingPageLink)
```

### Flow and TypeScript

#### Fix indentation of intersection object types ([#3074](https://github.com/prettier/prettier/pull/3074)) by [@duailibe](https://github.com/duailibe)

This was a minor alignment bug in intersection types, and has now been fixed.

<!-- prettier-ignore -->
```js
// Before
type intersectionTest = {
  propA: X
} & {
  propB: X
} & {
    propC: X
  } & {
    propD: X
  };

// After
type Props = {
  propA: X
} & {
  propB: X
} & {
  propC: X
} & {
  propD: X
};
```

#### Keep parens around TSAsExpression in ConditionalExpression ([#3053](https://github.com/prettier/prettier/pull/3053)) by [@azz](https://github.com/azz)

We missed a case where we need to keep the parenthesis with TypeScript's `as` assertions. This is now fixed.

<!-- prettier-ignore -->
```ts
// Before
aValue as boolean ? 0 : -1;

// After
(aValue as boolean) ? 0 : -1;
```

### JSX

#### Collapse multiple JSX whitespaces ([#2973](https://github.com/prettier/prettier/pull/2973)) by [@karl](https://github.com/karl)

This fixes up the issue where JSX formatting occasionally needed to be run twice to become stable. This occurred when you had multiple JSX whitespace elements or JSX whitespace followed by a space.

<!-- prettier-ignore -->
```jsx
// Before
<div>
    {" "} <Badge src={notificationIconPng} />
</div>;

// After
<div>
  {" "}
  <Badge src={notificationIconPng} />
</div>
```

#### Don't print JSX bracket on same line when it has trailing comments ([#3088](https://github.com/prettier/prettier/pull/3088)) by [@azz](https://github.com/azz)

This was an issue with the `--jsx-bracket-same-line` option. Turns out you can't _always_ put the bracket on the same line...

<!-- prettier-ignore -->
```jsx
// Input
<div
  // comment
>
  {foo}
</div>

// Before
<div>
// comment
  {foo}
</div>;

// After
<div
// comment
>
  {foo}
</div>;
```

### CSS

#### Preserve line breaks in grid declarations ([#3133](https://github.com/prettier/prettier/pull/3133)) by [@duailibe](https://github.com/duailibe)

Prettier will now preserve line breaks included in the source code when formatting the `grid` and `grid-template-*` rules, since those are important to keep in separate lines, but still applies the formatting like other rules (e.g., numbers and quotes).

<!-- prettier-ignore -->
```css
/* Original Input */
div {
  grid:
    [wide-start] 'header header header' 200.000px
    [wide-end] "footer footer footer" .50fr
    / auto 50.000px auto;
}

/* Before */
div {
  grid: [wide-start] "header header header" 200px [wide-end]
    "footer footer footer" 0.5fr / auto 50px auto;
}

/* After */
div {
  grid:
    [wide-start] "header header header" 200px
    [wide-end] "footer footer footer" 0.5fr
    / auto 50px auto;
}
```

### SCSS

#### Format SCSS maps like CSS rules ([#3070](https://github.com/prettier/prettier/pull/3070)) by [@asmockler](https://github.com/asmockler)

Turns out SCSS maps are much prettier when printed over multiple lines.

<!-- prettier-ignore -->
```scss
// Before
$map: (color: [#111111](https://github.com/prettier/prettier/pull/111111), text-shadow: 1px 1px 0 salmon)

// After
$map: (
  color: [#111111](https://github.com/prettier/prettier/pull/111111),
  text-shadow: 1px 1px 0 salmon
);
```

### CSS-in-JS

#### Fix formatting styled(Foo).attrs(...)`` ([#3073](https://github.com/prettier/prettier/pull/3073)) by [@existentialism](https://github.com/existentialism)

Prettier will now format the CSS in styled-components code that looks like this:

<!-- prettier-ignore -->
```js
styled(Component).attrs({})`
  color: red;
`;
```

### GraphQL

#### Prevent formatting GraphQL template literals with expressions ([#2975](https://github.com/prettier/prettier/pull/2975)) by [@duailibe](https://github.com/duailibe)

Prettier doesn't support formatting JavaScript expressions in GraphQL. See [#2640](https://github.com/prettier/prettier/pull/2640) for tracking. There was a bug where formatting an expression lead to invalid code, so we've completely disabled formatting GraphQL when it contains JavaScript expressions until we fully support it.

<!-- prettier-ignore -->
```js
// Before
(invalid code)

// After
graphql(schema, `{ query test { id }} ${fragment}`)
```

### CLI

#### Don't use ANSI codes if stdout isn't a TTY ([#2903](https://github.com/prettier/prettier/pull/2903)) by [@Narigo](https://github.com/Narigo)

Previously, piping the output of `--list-different` to other tools was troublesome due to the ANSI color codes we use to show whether a file was modified or not. This PR disables the use of color when Prettier is piped to a different process.

### Configuration

#### Use relative paths with CLI ([#2969](https://github.com/prettier/prettier/pull/2969)) by [@ahmedelgabri](https://github.com/ahmedelgabri)

This fixes a bug where passing a path starting with `./` to the CLI wouldn't match patterns used in `.prettierignore`.

```
## .prettierignore
path/to/*.js
```

After this fix, no files will be written to when executing:

<!-- prettier-ignore -->
```bash
$ prettier --write ./path/to/*.js
```

#### Resolve file paths relative to config file ([#3037](https://github.com/prettier/prettier/pull/3037)) by [@azz](https://github.com/azz)

This fixes an issue where `.prettierrc` overrides, under certain conditions, were not being respected for absolute paths with the `resolveConfig` API.

### Core

#### Respect CJK width and Combined Characters ([#3003](https://github.com/prettier/prettier/pull/3003), [#3015](https://github.com/prettier/prettier/pull/3015)) by [@ikatyang](https://github.com/ikatyang)

Chinese, Japanese and Korean characters are now considered two characters wide.

<!-- prettier-ignore -->
```js
// Before (exceeds print width when CJK characters are 2x monospace chars)
const x = ["中文", "中文", "中文", "中文", "中文", "中文", "中文", "中文", "中文", "中文", "中文"];

// After
const x = [
  "中文",
   // ...
  "中文"
];
```

#[#3015](https://github.com/prettier/prettier/pull/3015) also ensures that combining characters (e.g. `Á`) are counted as one character.

### Editor Support

#### Implement getSupportInfo() and use it for inference ([#3033](https://github.com/prettier/prettier/pull/3033)) by [@azz](https://github.com/azz)

We've added a new function to the API (`prettier.getSupportInfo([version])`), and the CLI `--support-info`. This can be used to interrogate Prettier to find out which languages the current version, or an older version, supports. It also provides useful information such as CodeMirror IDs, tmScopes, etc, which can be used to automate some of the work done with lookup tables in text editor integrations.

Internally, we use this information to drive which extensions trigger which parsers, and support some common files that don't have extensions, like `.prettierrc`, `Jakefile`, etc.

<!-- prettier-ignore -->
```bash
## prettier knows that this file is JSON now.
$ prettier --write .prettierrc
```

#### Split source elements relative to their language. ([#3069](https://github.com/prettier/prettier/pull/3069)) by [@CiGit](https://github.com/CiGit)

This fixes an issue in editors that support range formatting, where formatting an object would cause Prettier to crash.

---

## Thanks! ❤️

Thanks to everyone who contributed to this release, as well as those who raised issues! Prettier has become a highly stable piece of software that a large amount of people trust with their code. We take that trust seriously, and fix rare issues that break code with the highest priority. We can't fix these issues if we don't know about them, so never be afraid to [create an issue](https://github.com/prettier/prettier/issues/new)!
