---
authors: "vjeux"
title: "Prettier 1.6: Config File, JSX"
---

This release adds configuration file support to Prettier, as well as some significant enhancements to JSX printing.

<!-- truncate -->

I want to give a special shout out to [@azz](https://github.com/azz) who has been maintaining the repository and implementing a bunch of the changes in this release as I had less time to devote to prettier due to vacation and switching team :)

## Highlights

### Configuration

#### Implement cosmiconfig for workspace configuration ([#2434](https://github.com/prettier/prettier/pull/2434)) by [@azz](https://github.com/azz)

Since the very first release of prettier, people have asked for a `.prettierrc` file. We've been trying to have as few options as possible and tried to avoid being one more `.dotfile` that you have to have when starting a new project.

But, the truth is, we need to have some way to configure prettier that can be kept in sync with all the integrations. By not having one, we pushed the problem to them and saw a bunch of incompatible ways of handling the problem. So now, it's handled by prettier itself.

<!-- prettier-ignore -->
```js
// .prettierrc
{
  "trailingComma": "es5",
  "singleQuote": true
}
```

For more information on configuration file support, see the [README](https://github.com/prettier/prettier/blob/master/README.md#configuration-file).

#### Support .prettierignore files ([#2412](https://github.com/prettier/prettier/pull/2412)) by [@evilebottnawi](https://github.com/evilebottnawi)

Along with telling what configuration to use, you can write a file `.prettierignore` to tell which files not to convert.

```
## .prettierignore
dist/
package.json
```

### JSX

#### Improve JSX Formatting ([#2398](https://github.com/prettier/prettier/pull/2398)) by [@suchipi](https://github.com/suchipi)

The last big friction point from people trying to adopt prettier was around how JSX was being printed. We went through all the issues that were raised and made a bunch of changes:

- Arrow Function Expressions returning JSX will now add parens when the JSX breaks

<!-- prettier-ignore -->
```jsx
// Before
const Component = props =>
  <div>
    Hello {props.name}!
  </div>;

// After
const Component = props => (
  <div>
    Hello {props.name}!
  </div>
);
```

- Conditional expressions within (or containing) JSX are formatted in a different way using parenthesis

<!-- prettier-ignore -->
```jsx
// Before
<div>
  {props.isVisible
    ? <BaseForm
        url="/auth/google"
        method="GET"
      />
    : <Placeholder />}
</div>;

// After
<div>
  {props.isVisible ? (
    <BaseForm
      url="/auth/google"
      method="GET"
    />
  ) : (
    <Placeholder />
  )}
</div>
```

- JSX in logical expressions (|| or &&) is always wrapped in parens when the JSX breaks

<!-- prettier-ignore -->
```jsx
// Before
<div>
  {props.isVisible &&
    <BaseForm
      url="/auth/google"
      method="GET"
    />}
</div>;

// After
<div>
  {props.isVisible && (
    <BaseForm
      url="/auth/google"
      method="GET"
    />
  )}
</div>
```

Hopefully this is going to be more in line with how the majority of the community is writing JSX and we can have prettier be used in more place ;)

#### Inline single expressions in JSX ([#2442](https://github.com/prettier/prettier/pull/2442)) by [@karl](https://github.com/karl)

With JSX, we started by respecting a lot of line breaks that were in the original source. This had the advantage of doing fewer changes to your codebase but chipped away the value of a consistent pretty printer as the same semantic code could be written in two ways.

During each new release we've tightened this and made decisions around how to always print a piece of code. The latest of those is what happens if there's a single child in a JSX object, we're now always going to inline it.

<!-- prettier-ignore -->
```jsx
// Before
return (
  <div>
    {this.props.test}
  </div>
);
return <div>{this.props.test}</div>;

// After
return <div>{this.props.test}</div>;
return <div>{this.props.test}</div>;
```

#### Ensure there is a line break after leading JSX white space ([#2348](https://github.com/prettier/prettier/pull/2348)) by [@karl](https://github.com/karl)

Leading JSX empty spaces are now on their own line. It looked weird to have them before a tag as it "indented" it differently compared to the rest.

<!-- prettier-ignore -->
```jsx
// Before
<span className="d1">
  {' '}<a
    href="https://github.schibsted.io/finn/troika"
    className="link"
  />
</span>

// After
<span className="d1">
  {' '}
  <a
    href="https://github.schibsted.io/finn/troika"
    className="link"
  />
</span>
```

## Other Changes

### JSON

#### Use babylon.parseExpression for JSON ([#2476](https://github.com/prettier/prettier/pull/2476)) by [@josephfrazier](https://github.com/josephfrazier)

We used to use a strict JSON parser that would throw if there was a comment or a trailing comma. This was inconvenient as many JSON files in practice are parsed using JavaScript or json5 that are not as strict. Now, we have relaxed this and are using the JavaScript parser to parse and print JSON. This means that comments will be maintained if there were some.

Note that this is purely additive, if your original file was JSON compliant, it will keep printing a valid JSON.

<!-- prettier-ignore -->
```js
// Before
Syntax error

// After
{ /* some comment */ "a": 1 }
```

### JavaScript

#### Print 3 or more chained calls on multiple lines ([#2673](https://github.com/prettier/prettier/pull/2673)) by [@azz](https://github.com/azz)

This was a longstanding issue with the way we print long member chains. Prettier would try and cram all but the final callback onto one line, which reduced the readability. The solution we landed on was to always break over multiple lines if there are three or more function calls in a method chain.

<!-- prettier-ignore -->
```js
// Before
Promise.resolve(0).catch(function(err) {}).then(function(res) {
  //
});

// After
Promise.resolve(0)
  .catch(function(err) {})
  .then(function(res) {
    //
  });
```

#### Add more supervisory parens ([#2423](https://github.com/prettier/prettier/pull/2423)) by [@azz](https://github.com/azz)

Parenthesis are a hot topic because they are not part of the AST, so prettier ignores all the ones you are putting and re-creating them from scratch. We went through all the things that people reported and came up with a few edge cases that were very confusing when comparisons were chained and `%` was mixed with `*` or `/`.

One thing that we are not changing is the fact that we remove extra parenthesis around combinations of basic arithmetic operators: `+-*/`.

<!-- prettier-ignore -->
```js
// Before
x !== y === z;
x * y % z;

// After
(x !== y) === z;
(x * y) % z;
```

#### Implement prettier-ignore inside JSX ([#2487](https://github.com/prettier/prettier/pull/2487)) by [@azz](https://github.com/azz)

It's useful to be able to ignore pieces of JSX, it's now possible to add a comment inside of a JSX expression to ignore the formatting of the next element.

<!-- prettier-ignore -->
```jsx
// Before
<Component>
  {/*prettier-ignore*/}
  <span ugly format="" />
</Component>

// Before
<Component>
  {/*prettier-ignore*/}
  <span     ugly  format=''   />
</Component>
```

#### Do not swallow prettier-ignore comments ([#2664](https://github.com/prettier/prettier/pull/2664))

In order to support some edge cases, in the internals, we have the ability to avoid printing comments in a generic way and print them in the call site instead. It turns out that when we used `prettier-ignore`, we didn't print the comments at all! This is now fixed.

<!-- prettier-ignore -->
```jsx
// Before
push(
  <td> :)
  </td>,
);

// After
push(
  // prettier-ignore
  <td> :)
  </td>,
);
```

#### Fix indentation of a do-while condition ([#2359](https://github.com/prettier/prettier/pull/2359)) by [@jsnajdr](https://github.com/jsnajdr)

It took 6 months for someone to report that do-while were broken when the while condition is multi-line, it confirms my hunch that this construct is not widely used in practice.

<!-- prettier-ignore -->
```js
// Before
do {} while (
  someVeryLongFunc(
  someVeryLongArgA,
  someVeryLongArgB,
  someVeryLongArgC
)
);

// After
do {} while (
  someVeryLongFunc(
    someVeryLongArgA,
    someVeryLongArgB,
    someVeryLongArgC
  )
);
```

#### Break sequence expressions ([#2388](https://github.com/prettier/prettier/pull/2388)) by [@bakkot](https://github.com/bakkot)

Another underused feature of JavaScript is sequence expressions. We used to do a bad job at printing them when they would go multi-line, this has been corrected :)

<!-- prettier-ignore -->
```js
// Before
(a = b ? c : "lllllllllllllllllllllll"), (a = b
  ? c
  : "lllllllllllllllllllllll"), (a = b ? c : "lllllllllllllllllllllll"), (a = b
  ? c
  : "lllllllllllllllllllllll"), (a = b ? c : "lllllllllllllllllllllll");

// After
(a = b ? c : 'lllllllllllllllllllllll'),
  (a = b ? c : 'lllllllllllllllllllllll'),
  (a = b ? c : 'lllllllllllllllllllllll'),
  (a = b ? c : 'lllllllllllllllllllllll'),
  (a = b ? c : 'lllllllllllllllllllllll')
```

#### Trim trailing whitespace from comments ([#2494](https://github.com/prettier/prettier/pull/2494)) by [@azz](https://github.com/azz)

We took the stance with prettier to remove all the trailing whitespaces. We used to not touch comments because it's user generated, but that doesn't mean that they should have whitespace :)

<!-- prettier-ignore -->
```js
// Before
// There is some space here ->______________

// After
// There is some space here ->
```

#### Fix interleaved comments in class decorators ([#2660](https://github.com/prettier/prettier/pull/2660), [#2661](https://github.com/prettier/prettier/pull/2661))

Our handling for comments inside of the class declaration was very naive, we would just move all the comments to the top. We now are more precise and respect the comments that are interleaved inside of decorators and around `extends`.

<!-- prettier-ignore -->
```js
// Before
// A
// B
// C
@Foo()
@Bar()
class Bar {}

// After
// A
@Foo()
// B
@Bar()
// C
class Bar {}
```

#### Improve bind expression formatting ([#2493](https://github.com/prettier/prettier/pull/2493)) by [@azz](https://github.com/azz)

Bind expressions are being discussed at TC39 and we figured we could print it with prettier. We used to be very naive about it and just chain it. Now, we use the same logic as we have for method chaining with the `.` operator for it. We also fixed some edge cases where it would output invalid code.

<!-- prettier-ignore -->
```js
// Before
observable::filter(data => data.someTest)::throttle(() =>
  interval(10)::take(1)::takeUntil(observable::filter(data => someOtherTest))
)::map(someFunction);

// After
observable
  ::filter(data => data.someTest)
  ::throttle(() =>
    interval(10)::take(1)::takeUntil(observable::filter(data => someOtherTest))
  )
  ::map(someFunction);
```

#### Add support for printing optional catch binding ([#2570](https://github.com/prettier/prettier/pull/2570)) by [@existentialism](https://github.com/existentialism)

It's being discussed at TC39 to be able to make the argument of a `catch(e)` optional. Let's make sure we can support it in prettier if people use it.

<!-- prettier-ignore -->
```js
// Before
Syntax error

// After
try {} catch {}
```

#### Add support for printing optional chaining syntax ([#2572](https://github.com/prettier/prettier/pull/2572)) by [@azz](https://github.com/azz)

Another new proposal being discussed at TC39 is an optional chaining syntax. This is currently a [stage 1 proposal](https://github.com/tc39/proposal-optional-chaining), so the syntax may change at any time.

<!-- prettier-ignore -->
```js
obj?.prop       // optional static property access
obj?.[expr]     // optional dynamic property access
func?.(...args) // optional function or method call
```

#### Handle Closure Compiler type cast syntax correctly ([#2484](https://github.com/prettier/prettier/pull/2484)) by [@yangsu](https://github.com/yangsu)

Comments are tricky to get right, but especially when they have meaning based on where they are positioned. We're now special casing the way we deal with comments used as type cast for Closure Compiler such that they keep having the same semantics.

<!-- prettier-ignore -->
```js
// Before
let assignment /** @type {string} */ = getValue();

// After
let assignment = /** @type {string} */ (getValue());
```

#### Inline first computed property lookup in member chain ([#2670](https://github.com/prettier/prettier/pull/2670)) by [@azz](https://github.com/azz)

It looks kind of odd to have a computed property lookup on the next line, so we added a special case to inline it.

<!-- prettier-ignore -->
```js
// Before
data
  [key]('foo')
  .then(() => console.log('bar'))
  .catch(() => console.log('baz'));

// After
data[key]('foo')
  .then(() => console.log('bar'))
  .catch(() => console.log('baz'));
```

### Flow

#### Support opaque types and export star ([#2543](https://github.com/prettier/prettier/pull/2543), [#2542](https://github.com/prettier/prettier/pull/2542)) by [@existentialism](https://github.com/existentialism)

The flow team introduced two very exciting features under a new syntax. We now support them in prettier. I've personally been waiting for [opaque types](https://medium.com/flow-type/hiding-implementation-details-with-flows-new-opaque-type-aliases-feature-40e188c2a3f9) for a veerrryyyy long time!

<!-- prettier-ignore -->
```ts
// Before
Syntax error

// After
opaque type ID = string;
export type * from "module";
```

#### Strip away unnecessary quotes in keys in type objects and interfaces ([#2643](https://github.com/prettier/prettier/pull/2643)) by [@jackyho112](https://github.com/jackyho112)

We've been doing this on JavaScript objects since the early days of prettier but forgot to apply the same thing to Flow and TypeScript types.

<!-- prettier-ignore -->
```ts
// Before
type A = {
  "string": "A";
}

// After
type A = {
  string: "A";
}
```

#### Print TypeParameter even when unary function type ([#2406](https://github.com/prettier/prettier/pull/2406)) by [@danwang](https://github.com/danwang)

Oopsy, we were dropping the generic in this very specific case.

<!-- prettier-ignore -->
```ts
// Before
type myFunction = A => B;

// After
type myFunction = <T>(A) => B;
```

#### Keep parens around FunctionTypeAnnotation inside ArrayTypeAnnotation ([#2561](https://github.com/prettier/prettier/pull/2561)) by [@azz](https://github.com/azz)

Parenthesis... someday we'll get all of them fixed :)

<!-- prettier-ignore -->
```ts
// Before
const actionArray: () => void[] = [];

// After
const actionArray: (() => void)[] = [];
```

### TypeScript

#### Support TypeScript 2.5 RC ([#2672](https://github.com/prettier/prettier/pull/2672)) by [@azz](https://github.com/azz)

[TypeScript 2.5 RC](https://blogs.msdn.microsoft.com/typescript/2017/08/17/announcing-typescript-2-5-rc/) was recently announced, allowing you to use the upcoming "optional catch binding" syntax in TypeScript, too. 🎉

#### Don't add namespace keyword to global declaration ([#2329](https://github.com/prettier/prettier/pull/2329)) by [@azz](https://github.com/azz)

<!-- prettier-ignore -->
```ts
// Before
namespace global {
  export namespace JSX {  }
}

// After
global {
  export namespace JSX {}
}
```

#### Fix \<this.Component /> ([#2472](https://github.com/prettier/prettier/pull/2472)) by [@backus](https://github.com/backus)

Thanks to the untyped and permissive nature of JavaScript, we've been able to concat undefined to a string and get some interesting code as a result. Now fixed for this case :)

<!-- prettier-ignore -->
```jsx
// Before
<undefined.Author />

// After
<this.Author />
```

#### Allow type assertions to hug ([#2439](https://github.com/prettier/prettier/pull/2439)) by [@azz](https://github.com/azz)

We want to make sure that all the special cases that we added for JavaScript and Flow also work for TypeScript constructs. In this case, objects should also hug if they are wrapped in a `as` operator.

<!-- prettier-ignore -->
```js
// Before
const state = JSON.stringify(
  {
    next: window.location.href,
    nonce,
  } as State
);

// After
const state = JSON.stringify({
  next: window.location.href,
  nonce,
} as State);
```

#### Remove parens for type assertions in binary expressions ([#2419](https://github.com/prettier/prettier/pull/2419)) by [@azz](https://github.com/azz)

Most of the time we add parenthesis for correctness but in this case, we added them for nothing, so we can just get rid of them and have a cleaner code :)

<!-- prettier-ignore -->
```ts
// Before
(<x>a) || {};

// After
<x>a || {};
```

#### Print parens around type assertion as LHS in assignment ([#2525](https://github.com/prettier/prettier/pull/2525)) by [@azz](https://github.com/azz)

Yet another case of missing parenthesis. Thankfully we're getting very few of them nowadays and they are for extremely rare edge cases.

<!-- prettier-ignore -->
```ts
// Before
foo.bar as Baz = [bar];

// After
(foo.bar as Baz) = [bar];
```

#### Print declare for TSInterfaceDeclaration ([#2574](https://github.com/prettier/prettier/pull/2574)) by [@existentialism](https://github.com/existentialism)

The `declare` keyword doesn't do anything for `interface` so we never put it there. However, it felt weird if you were in a declaration file and seeing everything have `declare` before it except for interfaces. So now we reprint `declare` if it was there in the first place.

<!-- prettier-ignore -->
```ts
// Before
interface Dictionary<T> {
  [index: string]: T
}

// After
declare interface Dictionary<T> {
  [index: string]: T
}
```

### CSS

#### Normalize quotes in CSS ([#2624](https://github.com/prettier/prettier/pull/2624)) by [@lydell](https://github.com/lydell)

In order to get a first version of CSS to ship, we kept string quotes as is. We are now respecting the `singleQuote` option of prettier. The difficulty here was to make sure that we output correct code for all the crazy escapes, unicode characters, emoji, special rules like charset which only work with double quotes...

<!-- prettier-ignore -->
```css
// Before
div {
  content: "abc";
}

// After
div {
  content: 'abc';
}
```

#### Normalize numbers in CSS ([#2627](https://github.com/prettier/prettier/pull/2627)) by [@lydell](https://github.com/lydell)

Another place where we can reuse the logic we've done for JavaScript to improve CSS printing.

<!-- prettier-ignore -->
```css
// Before
foo {
  border: 1px solid rgba(0., 0.0, .0, .3);
}

// After
foo {
  border: 1px solid rgba(0, 0, 0, 0.3);
}
```

#### Quote unquoted CSS attribute values in selectors ([#2644](https://github.com/prettier/prettier/pull/2644)) by [@lydell](https://github.com/lydell)

I can never quite remember the rules behind quotes around attributes so we're now always putting quotes there.

<!-- prettier-ignore -->
```js
// Before
a[id=test] {}

// After
a[id="test"] {}
```

#### Add support for css keyword ([#2337](https://github.com/prettier/prettier/pull/2337)) by [@zanza00](https://github.com/zanza00)

<!-- prettier-ignore -->
```js
// Before
const header = css`.top-bar {background: black;margin: 0;position: fixed;}`

// After
const header = css`
  .top-bar {
    background: black;
    margin: 0;
    position: fixed;
  }
`;
```

#### Support styled-components with existing component ([#2552](https://github.com/prettier/prettier/pull/2552), [#2619](https://github.com/prettier/prettier/pull/2619)) by [@azz](https://github.com/azz)

styled-components has a lot of different variants for tagging template literals as CSS. It's not ideal that we've got to encode all those ways inside of prettier but since we started, might as well do it for real.

<!-- prettier-ignore -->
```js
styled(ExistingComponent)`
  css: property;
`;

styled.button.attr({})`
  border: rebeccapurple;
`;
```

#### Trim whitespace in descendant combinator ([#2411](https://github.com/prettier/prettier/pull/2411)) by [@azz](https://github.com/azz)

The CSS parsers we use do not give us a 100% semantic tree: in many occasions they bail and just give us what is being entered. It's up to us to make sure we clean this up while maintaining correctness. In this case, we just printed spaces between selectors as is but we know it's correct to always replace it by a single space.

<!-- prettier-ignore -->
```cs
// Before
.hello

            .how-you-doin {
  height: 42;
}

// After
.hello .how-you-doin {
  height: 42;
}
```

#### Strip BOM before parsing ([#2373](https://github.com/prettier/prettier/pull/2373)) by [@azz](https://github.com/azz)

I still have nightmares from dealing with [BOM](https://en.wikipedia.org/wiki/Byte_order_mark) in a previous life. Thankfully, in 2017 it's no longer a big issue as most tooling is now aware of it. Thanks [@azz](https://github.com/azz) for fixing an edge cases related to CSS parsing.

<!-- prettier-ignore -->
```css
// Before
[BOM]/* Block comment *
html {
  content: "#{1}";
}
```

<!-- prettier-ignore -->
```css
// After
[BOM]/* Block comment */
html {
  content: "#{1}";
}
```

### GraphQL

#### Add support for range-formatting GraphQL ([#2319](https://github.com/prettier/prettier/pull/2319)) by [@josephfrazier](https://github.com/josephfrazier)

If you tried to use the range formatting feature in a GraphQL file, it would throw an exception, now it properly worked again and only reformats the piece you selected.

#### Add `.gql` file extension to be parsed as GraphQL ([#2357](https://github.com/prettier/prettier/pull/2357)) by [@rrdelaney](https://github.com/rrdelaney)

At Facebook, we use `.graphql` extension but it looks like it's common to have `.gql` as well, doesn't cost a lot to support it in the heuristic that figures out what parser to use.

### CLI

#### Support multiple patterns with ignore pattern ([#2356](https://github.com/prettier/prettier/pull/2356)) by [@evilebottnawi](https://github.com/evilebottnawi)

It was already possible to have multiple glob patterns but they would be additive, with this change, you can add a glob pattern to ignore some files. It should be very handy to ignore folders that are deeply nested.

<!-- prettier-ignore -->
```sh
prettier --write '{**/*,*}.{js,jsx,json}' '!vendor/**'
```

#### Make --list-different to work with --stdin ([#2393](https://github.com/prettier/prettier/pull/2393)) by [@josephfrazier](https://github.com/josephfrazier)

This is a handy way of knowing if prettier would print a piece of code in a different way. We already had all the concepts in place, we just needed to wire them up correctly.

<!-- prettier-ignore -->
```shellsession
$ echo 'call ( ) ;' | prettier --list-different
(stdin)
$ echo $?
1
```
