/* $Id: pfstatd.c,v 1.2 2008/06/10 17:16:46 dhartmei Exp $ */

/*
 * Copyright (c) 2002-2006, Daniel Hartmeier
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer. 
 *    - Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

static const char rcsid[] = "$Id: pfstatd.c,v 1.2 2008/06/10 17:16:46 dhartmei Exp $";

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "pf.h"

static int debug = 0;
static FILE *f = NULL;

static void
print(int type, const char *arg, int idx, double val)
{
	if (f == NULL)
		return;
	if (debug)
		printf("%d %s %d %.0f\n", type, arg[0] ? arg : "-", idx, val);
	fprintf(f, "%d %s %d %.0f\n", type, arg[0] ? arg : "-", idx, val);
}

static void
usage(void)
{
	extern char *__progname;

	fprintf(stderr, "usage: %s [-d] [-a addr] [-p port]\n", __progname);
	exit(1);
}

int
main(int argc, char *argv[])
{
	const char *user = "nobody";
	const char *addr = "0.0.0.0";
	unsigned port = 9999;
	int ch, fdp = -1, fdl = -1, fdc = -1, val, len;
	struct sockaddr_in sa;

	while ((ch = getopt(argc, argv, "da:p:u:")) != -1) {
		switch (ch) {
		case 'd':
			debug = 1;
			break;
		case 'a':
			addr = optarg;
			break;
		case 'p':
			port = atoi(optarg);
			break;
		case 'u':
			user = optarg;
			break;
		default:
			usage();
		}
	}
	if (argc != optind)
		usage();
	signal(SIGPIPE, SIG_IGN);
	if ((fdp = open("/dev/pf", O_RDONLY)) < 0) {
		fprintf(stderr, "open: /dev/pf: %s\n", strerror(errno));
		return (1);
	}
	/* if root, drop privileges */
	if (!getuid()) {
		struct passwd *pw;

		if ((pw = getpwnam(user)) == NULL) {
			fprintf(stderr, "getpwnam: %s: %s\n", user,
			    strerror(errno));
			return (1);
		}
		setgroups(1, &pw->pw_gid);
		if (setegid(pw->pw_gid) || setgid(pw->pw_gid)) {
			fprintf(stderr, "setgid: %s\n", strerror(errno));
			return (1);
		}
		if (seteuid(pw->pw_uid) || setuid(pw->pw_uid)) {
			fprintf(stderr, "setuid: %s\n", strerror(errno));
			return (1);
		}
		if (debug)
			printf("dropped privileges to user %s\n", user);
	}
	if ((fdl = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		fprintf(stderr, "socket: %s\n", strerror(errno));
		return (1);
	}
	val = 1;
	if (setsockopt(fdl, SOL_SOCKET, SO_REUSEADDR, (const char *)&val,
	    sizeof(val))) {
		fprintf(stderr, "setsockopt: %s\n", strerror(errno));
		return (1);
	}
	memset(&sa, 0, sizeof(sa));
	sa.sin_family = AF_INET;
	sa.sin_addr.s_addr = inet_addr(addr);
	sa.sin_port = htons(port);
	if (bind(fdl, (const struct sockaddr *)&sa, sizeof(sa))) {
		fprintf(stderr, "bind: %s:%u: %s\n", addr, (unsigned)port,
		    strerror(errno));
		return (1);
	}
	if (listen(fdl, 1)) {
		fprintf(stderr, "listen: %s\n", strerror(errno));
		return (1);
	}
	/* run once while not yet detached to print errors, if any */
	if (pf_query(fdp, print))
		return (1);
	if (!debug && daemon(0, 0)) {
		fprintf(stderr, "daemon: %s\n", strerror(errno));
		return (1);
	}
	if (debug)
		printf("listening for incoming connections on %s:%u...\n",
		    addr, port);
	while (1) {
		len = sizeof(sa);
		if ((fdc = accept(fdl, (struct sockaddr *)&sa, &len)) < 0) {
			sleep(1);
			continue;
		}
		if (debug)
			printf("connection from %s:%u\n",
			    inet_ntoa(sa.sin_addr), ntohs(sa.sin_port));
		if ((f = fdopen(fdc, "w")) == NULL) {
			sleep(1);
			continue;
		}
		pf_query(fdp, print);
		fclose(f);
		sleep(1);
	}
	return (0);
}
