/* $Id: pf.c,v 1.1.1.1 2007/01/11 16:01:58 dhartmei Exp $ */

/*
 * Copyright (c) 2002-2006, Daniel Hartmeier
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer. 
 *    - Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

static const char rcsid[] = "$Id: pf.c,v 1.1.1.1 2007/01/11 16:01:58 dhartmei Exp $";

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <net/pfvar.h>
#include <altq/altq.h>
#include <altq/altq_cbq.h>
#include <altq/altq_priq.h>
#include <altq/altq_hfsc.h>
#include <arpa/inet.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "pf.h"

union altq_stats {
	class_stats_t		 cbq;
	struct priq_classstats	 priq;
	struct hfsc_classstats	 hfsc;
};

static int
query_queues(int fd, void (*cb)(int, const char *, int, double))
{
	struct pfioc_altq pa;
	unsigned mnr, nr;

	/* first, find out how many queues there are */
	memset(&pa, 0, sizeof(pa));
	if (ioctl(fd, DIOCGETALTQS, &pa))
		/* this is expected when ALTQ is disabled (i.e. FreeBSD) */
		return (0);
	mnr = pa.nr;

	/* fetch each of those queues */
	for (nr = 0; nr < mnr; ++nr) {
		struct pfioc_qstats pq;
		union altq_stats s;
		u_int64_t c[8];
		int i;

		pa.nr = nr;
		if (ioctl(fd, DIOCGETALTQ, &pa)) {
			fprintf(stderr, "ioctl: DIOCGETALTQ: %s\n",
			    strerror(errno));
			return (1);
		}
		if (pa.altq.qid <= 0)
			continue;
		/* fetch stats of this queue */
		memset(&pq, 0, sizeof(pq));
		pq.nr = nr;
		pq.ticket = pa.ticket;
		pq.buf = &s;
		pq.nbytes = sizeof(s);
		if (ioctl(fd, DIOCGETQSTATS, &pq)) {
			fprintf(stderr, "ioctl: DIOCGETQSTATS: %s\n",
			    strerror(errno));
			return (1);
		}
		switch (pa.altq.scheduler) {
		case ALTQT_CBQ:
			c[0] = s.cbq.xmit_cnt.packets;
			c[1] = s.cbq.xmit_cnt.bytes;
			c[2] = s.cbq.drop_cnt.packets;
			c[3] = s.cbq.drop_cnt.bytes;
			c[4] = s.cbq.qcnt;
			c[5] = s.cbq.qmax;
			c[6] = s.cbq.borrows;
			c[7] = s.cbq.delays;
			break;
		case ALTQT_PRIQ:
			c[0] = s.priq.xmitcnt.packets;
			c[1] = s.priq.xmitcnt.bytes;
			c[2] = s.priq.dropcnt.packets;
			c[3] = s.priq.dropcnt.bytes;
			c[4] = s.priq.qlength;
			c[5] = s.priq.qlimit;
			c[6] = 0;
			c[7] = 0;
			break;
		case ALTQT_HFSC:
			c[0] = s.hfsc.xmit_cnt.packets;
			c[1] = s.hfsc.xmit_cnt.bytes;
			c[2] = s.hfsc.drop_cnt.packets;
			c[3] = s.hfsc.drop_cnt.bytes;
			c[4] = s.hfsc.qlength;
			c[5] = s.hfsc.qlimit;
			c[6] = 0;
			c[7] = 0;
			break;
		default:
			/* skip unsupported queue type */
			continue;
		}
		for (i = 0; i < 8; ++i)
			(*cb)(COL_TYPE_QUEUE, pa.altq.qname, i, c[i]);
	}
	return (0);
}

static int
query_ifaces(int fd, void (*cb)(int, const char *, int, double))
{
	struct pfioc_iface io;
	struct pfi_kif ifs[256];
	int i, j;

	memset(&io, 0, sizeof(io));
	io.pfiio_buffer = ifs;
	io.pfiio_esize = sizeof(ifs[0]);
	io.pfiio_size = sizeof(ifs) / sizeof(ifs[0]);
	if (ioctl(fd, DIOCIGETIFACES, &io)) {
		fprintf(stderr, "ioctl: DIOCIGETIFACES: %s\n", strerror(errno));
		return (1);
	}
	for (i = 0; i < io.pfiio_size; ++i)
		for (j = 0; j < 16; ++j)
			(*cb)(COL_TYPE_IFACE, ifs[i].pfik_name,
			    j, j & 4 ?
			    ifs[i].pfik_packets[j&1?0:1][j&2?0:1][j&8?0:1] :
			    ifs[i].pfik_bytes[j&1?0:1][j&2?0:1][j&8?0:1]);
	/* bytes/packets[af][dir][op] */
	return (0);
}

static int
query_counters(int fd, void (*cb)(int, const char *, int, double))
{
	struct pf_status s;
	int i;

	memset(&s, 0, sizeof(s));
	if (ioctl(fd, DIOCGETSTATUS, &s)) {
		fprintf(stderr, "ioctl: DIOCGETSTATUS: %s\n", strerror(errno));
		return (1);
	}
	(*cb)(COL_TYPE_SINCE, "", 0, s.since);
	(*cb)(COL_TYPE_GLOBAL, "", 0, s.states);
	for (i = 0; i < FCNT_MAX; ++i)
		(*cb)(COL_TYPE_GLOBAL, "", 1 + i, s.fcounters[i]);
	for (i = 0; i < PFRES_MAX; ++i)
		(*cb)(COL_TYPE_GLOBAL, "", 1 + FCNT_MAX + i, s.counters[i]);
	return (0);
}

int
pf_query(int fd, void (*cb)(int, const char *, int, double))
{
	if (query_counters(fd, cb)) {
		fprintf(stderr, "pf_query: query_counters() failed\n");
		return (1);
	}
	if (query_ifaces(fd, cb)) {
		fprintf(stderr, "pf_query: query_ifaces() failed\n");
		return (1);
	}
	if (query_queues(fd, cb)) {
		fprintf(stderr, "pf_query: query_queues() failed\n");
		return (1);
	}
	return (0);
}
