#include <fstream>
#include <sstream>
#include <algorithm>
#include <map>
#include <assert.h>
#include <ctype.h>
#include "ordTop.h"


using namespace ord;
using namespace std;
#include "intfc.h"
#include "parse.tab.h"

#include "version.h"

static const int lineLim = 68 ;

static bool outputtingTeX = false ;


struct Functions {
    const string name ;
    const Ordinal * (*evaluate) (list<const ParseContainer *> * params);
};


static void exitCleanup()
{
    pS->log(ParseSemantics::exitClean);
}

static void abortDocError()
{
    if (!ord::interactiveMode) assert(0);
}

ParserState * ord::parserState = NULL ;


void ParseSemantics::waitForEnter(list<const string *> * strList)
{
    assert(promptUser);
    
    if (strList) 
        for (list<const string *>::iterator iter = strList->begin();
            iter != strList->end();++iter)  *ParseSemantics::out << **iter << " " ;
    *ParseSemantics::out << "Type ENTER to Continue:";
    ParseSemantics::out->flush();
    if (ord::interactiveMode) {
        static const int bufSz=8102;
        char buf[bufSz] ;
        cin.getline(buf,bufSz);
        pS->setPromptLine(0);
    }

}



int ParseContainer::size(list<const ParseContainer *> *lst)
{
    int size = 0 ;
    list<const ParseContainer * >::iterator iter ;
    for (iter = lst->begin(); iter != lst->end(); ++iter) {
         const ParseContainer &c = **iter ;
         if (c.type &  ParseContainer::ord) size++;
         else if (c.type & ParseContainer::ordLst)
         size+= c.getList()->size();
    }
    return size ;
}

const Ordinal * ParseContainer::getNth(list<const ParseContainer *> *lst,int n)
{
    int count = 0 ;
    list<const ParseContainer * >::iterator iter ;
    for (iter = lst->begin(); iter != lst->end(); ++iter) {
         const ParseContainer &c = **iter ;
         if (c.type & ParseContainer::ord) {
            if (++count == n) return c.getOrd() ;
         }
         else if (c.type & ParseContainer::ordLst) {
            list<const Ordinal * >& l = *(c.getList());
            list<const Ordinal * >::iterator i2 ;
            for (i2 = l.begin(); i2 != l.end(); ++i2) {
                if (++count == n) return *i2 ;
            }
        }
    }
    return NULL ;
}




const Ordinal * ParseContainer::front() const 
{
    if (type & ord) return getOrd();
    if (type & ordLst) {
        list<const Ordinal *>* lst = getList();
        return lst->front();
    }
    assert(0) ;
    return NULL ;
}

const Ordinal * ParseContainer::getFirstOrdinal() const
{
    return ParseSemantics::getFirstOrdinal(this);
}

list<const Ordinal *>* ParseContainer::linearList(
    list<const ParseContainer *> *lst, int skip) {
    int count = 0 ;
    list<const Ordinal *>* ret = new list<const Ordinal *> ;
    list<const ParseContainer * >::iterator iter ;
    for (iter = lst->begin(); iter != lst->end(); ++iter) {
         const ParseContainer &c = **iter ;
         if (c.type & ParseContainer::ord) {
            if (++count > skip) ret->push_back(c.getOrd());
         } else if (c.type & ParseContainer::ordLst) {
            list<const Ordinal *>&l= *(c.getList());
                for (list<const Ordinal *>::iterator l2 = l.begin();
                    l2!=l.end();l2++) {
                        if (++count > skip)
                        ret->push_back(*l2);
                    }
         }
    }
    return ret ;
}


void ParseContainer::out(ostream& out) 
{
    if (type == ord) {
        out << "PCord{" << getOrd()->normalForm() << "}\n" ;
        pS->promptCheck();
    }
    else if (type == ordLst) {
        out << "PCordLst{" ;
        bool first = true ;
        for (list<const Ordinal *>::iterator iter = getList()->begin();
            iter != getList()->end(); ++iter) {
                if (first) out << " " ;
                else {
                    out << ",\n" ;
                    pS->promptCheck();
                    out << "         " ;
                }
                out << (*iter)->normalForm(); 
        }
        out << "}\n" ;
        pS->promptCheck();
    } else assert(0);

}



void ParseSemantics::setPromptLine(int n)
{
    if (ord::interactiveMode) promptLine = n ;
}

void ParseSemantics::addChar(char c, EscTeXlevel skipEsc)
{
    char tmp[2] ;
    tmp[0]=c;
    tmp[1]= '\0' ;
    const char * toAdd = tmp ;
    switch (skipEsc) {
default:
        assert(0);
case escAll:
        toAdd = escTeX(c);
        break ;
case escScript:
        toAdd = escTeXscr(c);
        break ;
case escNone :
        break ;
    }
    int inc = 1 ;
    if (toAdd) inc = strlen(toAdd) ;
    assert(inc+formatBufIndex<formatBufSize);
    if (toAdd) {
        strcpy(formatBuf+formatBufIndex,toAdd);
        formatBufIndex+=inc ;
        formatCharIndex++;
    } else {
        formatBuf[formatBufIndex++] = c ;
        formatCharIndex++;
    }
    
}

const char * ParseSemantics::chTtyTeX(const char * in)
{
    return checkTeX(in,"\\protect{\\tt ","}");
}

static const char * dup(const char * x)
{
    assert(x);
    char * buf = new char [strlen(x)+1] ;
    strcpy(buf,x);
    return buf ;
}


const char * ParseSemantics::checkTeX(const char * in, const char *pre,
    const char *post)
{
    static const int bufSize=1023 ;
    static char buf[bufSize+1] ;
    int index = 0 ;
    if (pre) {
        strcpy(buf,pre);
        index += strlen(pre);
    }
    if (!TeXdocMode) return in ;
    for (const char *pt = in; *pt; pt++) {
        const char * toAdd = escTeX(*pt);
        if (!toAdd) {
            assert(index < bufSize);
            buf[index++] = *pt ;
        } else {
            int length = strlen(toAdd);
            assert(index+length<bufSize);
            strcpy(buf+index,toAdd);
            index+=length ;
        }
    }
    if (post) {
        int tailLength = strlen(post);
        assert(index+tailLength<bufSize);
        strcpy(buf+index,post);
        return dup(buf) ;
    }
    assert(index<bufSize);
    buf[index]='\0' ;
    return dup(buf) ;
}

static void vsp()
{
    if (ParseSemantics::TeXdocMode) *(ParseSemantics::out) <<
        "\n\n\\addvspace{.05in}\n"  ;
}

static const char * blockCommands[] = {

    "\\mindextt{",
    "\\mindex{",
    "\\indextt{",
    "\\index{",
    "\\cite{",
    NULL
};



static const char * checkBlockIndexSkip(const char * pt,
    ParseSemantics::EscTeXlevel& level)
{
    
    bool texDoc = ParseSemantics::TeXdocMode ;
    const char * savePt = pt ;
    static const char * ttStart = "{\\tt " ;
    static const int ttLen = strlen(ttStart);
    if (!strncmp(pt,ttStart,ttLen)) {
        level = ParseSemantics::escScript ;
        pt += ttLen ;
        const char * ptr = pt ;
        
        
        int count = 1 ;
        for (; *ptr; ptr++) {
            if (*ptr == '}') {
                count-- ;
                if (texDoc && (count>0)) *(char *)ptr = 127 ;
            }
            else if (*ptr == '{') {
                count++ ;
                if (texDoc) *(char *)ptr = 126 ;
            }
            if (!count) break ;
        }
        assert(*ptr == '}') ;
        if (texDoc) return ptr+1 ;
        char * chptr = (char *) ptr ;
        for (;chptr > pt;chptr--) *chptr = chptr[-1] ;
        
        return pt+1 ;
        assert(0);
    }
    for (const char ** bcmd = blockCommands; *bcmd; bcmd++) {
       if (!strncmp(pt,*bcmd,strlen(*bcmd))) {
            level = ParseSemantics::escNone ;
            pt += strlen(*bcmd);
            int left = 1 ;
            int right = 0 ;
            while (*pt) {
                if (*pt == '}') right++ ;
                else if (*pt == '{') left++ ;
                if (left == right) return pt+1;
                else pt++;
            }
            *ParseSemantics::out<<"ERROR: Internal error 1 in documentation:\n\"" <<
                savePt << "\"\n" ;
            abortDocError();
            pS->promptCheck();
            return savePt ;
        }
    }
    return savePt ;
}

void ParseSemantics::promptCheck()
{
    if (promptUser)
    if (ord::interactiveMode && (promptLineLimit > 4))
        if (++promptLine >= promptLineLimit) waitForEnter();

}


void ParseSemantics::writeFormatted(const char ** text, bool skipnl)
{
    formatCharLength = 0 ;
    formatBufLength = 0 ;
    formatCharIndex = 0 ;
    formatBufIndex = 0 ;
    formatResetQ();
    bool lastBreak = false ;
    for (const char ** lines = text ;*lines; lines++) {
        
        static const int maxLine = 8191 ;
        char lineBuf[maxLine+1] ;
        assert(strlen(*lines) < maxLine);
        strcpy(lineBuf,*lines) ;

        
        const char * line = lineBuf ;
        const char * protect = line-1;
        EscTeXlevel escLevel = escAll ;
        for (const char *pt = line ; *pt; pt++) {
            
            const char * savePt = pt ;
            if (!TeXdocMode) pt = checkBlockIndexSkip(pt,escLevel);
            else {
                const char * tmp = checkBlockIndexSkip(pt,escLevel) ;
                if (tmp > pt) protect = tmp ;
            }
            
            if (!*pt) break ;
            if (*pt == '\n') {
                bool doBreak = false ;
                if (TeXdocMode) if  (!lastBreak) {
                    doBreak = true ;
                    lastBreak = true ;
                } else {
                    vsp();
                } else if ((formatBufIndex) < formatBufSize) {
                    formatBuf[formatBufIndex++] = *pt ;
                } else assert(0);
                formatBuf[formatBufIndex]= 0 ;
                *out << formatBuf ;
                promptCheck();
                formatBufLength = 0 ;
                formatCharLength = 0 ;
                formatCharIndex = 0 ;
                formatBufIndex = 0 ;
                if (doBreak) brTeX();
            } else if (*pt != ' ') {
                lastBreak = false ;
                
                addChar(*pt,escLevel);
                
                
            } else { 
                if (!formatBufIndex) continue ;
                formatBuf[formatBufIndex] = 0 ;
                int newCharLength = formatCharLength + formatCharIndex + 1;
                int newBufLength = formatBufLength + formatBufIndex + 1;
                if (newCharLength > lineLim) {
                    *out << "\n";
                    promptCheck();
                    *out << formatBuf << " ";
                    formatBufLength = formatBufIndex+1 ;
                    formatCharLength = formatCharIndex+1 ;

                } else {
                    *out << formatBuf << " " ;
                    formatBufLength = newBufLength ;
                    formatCharLength = newCharLength ;
                }
                formatBufIndex = 0 ;
                formatCharIndex = 0 ;
            }
        }
        
        if (formatBufIndex) {
             formatBuf[formatBufIndex] = 0 ;
             int newBufLength = formatBufLength + formatBufIndex + 1;
             int newCharLength = formatCharLength + formatCharIndex + 1;
             if (newCharLength > lineLim) {
                *out << "\n" ;
                promptCheck();
                *out << formatBuf  ;
                formatBufLength = formatBufIndex ;
                formatCharLength = formatCharIndex ;

             } else {
                 *out << formatBuf ;
                  formatBufLength = newBufLength ;
                  formatCharLength = newCharLength ;
             }
            if (formatBuf[formatBufIndex-1] != ' ') {
                *out << " " ;
                formatCharLength++;
                formatBufLength++;
            }
            formatBufIndex = 0 ;
            formatCharIndex = 0 ;
        } 
    }
    if (!skipnl) {
        *out << "\n";
        promptCheck();
        
    }
}

static void writeFormatted(const char ** text)
{
    pS->writeFormatted(text);
}

static const Ordinal * evalEps(list<const ParseContainer *> * params)
{
    if (!params) {
        *ParseSemantics::out <<
            "ERROR: Function epsilon must have 1 argument.\n" ;
        abortDocError();
        pS->promptCheck();
        return &(Ordinal::zero);
    }
    
    int size = ParseContainer::size(params);
    if (size !=1) {
        *ParseSemantics::out << "ERROR: " << size <<
            " parameters given for epsilon. Exactly 1 is required.\n";
        abortDocError();
        pS->promptCheck();
        return &(Ordinal::zero);
    }
    const Ordinal * p1 = ParseContainer::getFirst(params) ;
    assert(p1);

    return &(psi(1,*p1));

}


static  const Ordinal ** listToArray(list<const ParseContainer *> * params)
{
    if (!params) return NULL;
    int size = 0 ;
    list<const ParseContainer * >::iterator iter ;
    for (iter = params->begin(); iter != params->end(); ++iter) {
           const ParseContainer &c = **iter ;
           if (c.type & ParseContainer::ord) size++;
           else if (c.type & ParseContainer::ordLst)
            size+= c.getList()->size();
    }
 
    if (size <  1) return NULL ;
    const Ordinal ** parameters = new const Ordinal * [size+1] ;
    parameters[size] = NULL ;
    int count = 0 ;
    int index = 0 ;
    
    for (iter = params->begin(); iter != params->end(); ++iter,count++) {
        const ParseContainer &c = **iter ;
        if (c.type & ParseContainer::ord) {
            assert(index<size);
            parameters[index++] = (*iter)->getOrd();
        } else if (c.type & ParseContainer::ordLst) {
            list<const Ordinal *>& ordList = *(c.getList());
            list<const Ordinal *>::iterator iterO ;
            for (iterO = ordList.begin(); iterO != ordList.end(); ++iterO) {
                assert(index<size);
                parameters[index++] = *iterO;
            }

        }
    }
    return parameters ;
}

static const Ordinal * evalGamma(list<const ParseContainer *> * params)
{
    const Ordinal ** parameters = listToArray(params);
    int size = 0 ;
    if (parameters) size = params->size();
    if ((size!=1)) { 
        abortDocError();
        *ParseSemantics::out <<
            "ERROR: Function gamma must have 1 argument.\n" ;
        pS->promptCheck();
        return &(Ordinal::zero);
    }
    const Ordinal * arg = params->front()->front();
    
    if (arg->isFinite()) arg =
        new Ordinal(arg->getImpl().addLoc(OrdinalImpl::one));
    return &(finiteFunctional(*arg,Ordinal::zero, Ordinal::zero));
}

static const Ordinal * evalPsi(list<const ParseContainer *> * params)
{
    const Ordinal ** parameters = listToArray(params);
    if (!parameters) {
        abortDocError();
        *ParseSemantics::out <<
            "ERROR: Function psi must have 1 or more arguments.\n" ;
        pS->promptCheck();
        return &(Ordinal::zero);
    }
    
    return &(finiteFunctional(parameters));


}


const Ordinal * ParseSemantics::getFirstOrdinal(const ParseContainer * contain)
{
    const Ordinal * ret = contain->checkOrd();
    if (ret) return ret ;
        
    list<const Ordinal *>* lst = contain->checkList();
    assert(lst);
    ret = lst->front();
    assert(ret);
    return ret ;

}


const ParseContainer * ParseSemantics::iterOrd(const ParseContainer * contain,
    list<const ParseContainer *> * lst)
{
    assert(contain);
    const Ordinal * iter = getFirstOrdinal(contain);
    const Ordinal ** parameters = NULL;
    if (lst) parameters = listToArray(lst);
    return new ParseContainer (&(iterativeFunctional(*iter,parameters)));
}




const ParseContainer * ParseSemantics::admisOrd(
    const ParseContainer * embedContain, const ParseContainer * cadmis1,
    const ParseContainer * citer2, list<const ParseContainer *> * lst,
    const ParseContainer *cord)
{
    assert(cadmis1);
    assert(citer2);
    const Ordinal * embedOrd = &Ordinal::zero ;
    if (embedContain) {
        embedOrd = embedContain->getFirstOrdinal();
        assert(embedOrd);
    }
    const Ordinal * admis1 = cadmis1->getFirstOrdinal();
    const Ordinal * iter2 = citer2->getFirstOrdinal();
    const Ordinal * ord =  &Ordinal::zero ;

    
    if (cord) {
        ord = cord->getFirstOrdinal();
        
    }

    const Ordinal ** parameters = NULL;
    if (lst) parameters = listToArray(lst);
    return new ParseContainer(
        &(admisLevelFunctional(*admis1,*iter2,parameters,*ord,*embedOrd)));
}




static const Functions theFunctions[] = 
    {
        {"epsilon",evalEps},
        {"gamma",evalGamma},
        {"psi",evalPsi},
      
        {"",NULL}
    };

static bool no_args(const char *cmd, list<const string *> * params)
{
    if (params) if(params->size() > 0) {
        abortDocError();
        *ParseSemantics::out << "ERROR: Command " << cmd <<
            " takes no arguments.\n" ;
        pS->promptCheck();
        return false;
    }
    return true ;
}

const string * getOneArgument(const char *cmd, list<const string *> * params,
    const char * type = "destination file name")
{
    if (params) {
       int size = params->size() ;
       if (size > 1) {
            abortDocError();
            *ParseSemantics::out << "ERROR: Command '" << cmd <<
                "' has the " << type << " as its only parameter.\n" ;
            pS->promptCheck();
        }
       if (size == 1) {
        const string * ret = params->front();
        return ret ;
       }
    }
    return NULL ;
}

static void commandHelp(const char * str);


static void helpCmd(const char *cmd, list<const string *> * params);



static struct InitPair {
   const string name;
   const Ordinal* ord;
}  initMapPairs[] = {
    {"omega", &Ordinal::omega},
    {"w", &Ordinal::omega},
    {"omega1CK", &(ord::omega1CK)},
    {"w1", &(ord::omega1CK)},
    {"w1CK", &(ord::omega1CK)},
    {"eps0", &(ord::eps0)},
    {"",NULL}
};

static void helpPredefinedTex(list<const string *> * lst)
{
    *ParseSemantics::out << "The predefined ordinal variables are:\\\\\n" ;
    for (const InitPair * pr = initMapPairs; pr->ord; pr++) {
        *ParseSemantics::out << "{\\tt " <<  pr->name << "} $ = " <<
            pr->ord->texNormalForm() << "$\\\\" ;
        *ParseSemantics::out << "\\mindextt{" << pr->name << "}\n" ;
    }

}
static void helpPredefined(list<const string *> * lst)
{
    *ParseSemantics::out << "The predefined ordinal variables are:\n" ;
    pS->promptCheck();
    if (pS->outText()) {
        for (const InitPair * pr = initMapPairs; pr->ord; pr++)
        *ParseSemantics::out << pr->name << " = " <<  pr->ord->normalForm() <<
        "\n" ;
        pS->promptCheck();
    }
    if (pS->outTeX()) {
        for (const InitPair * pr = initMapPairs; pr->ord; pr++)
        *ParseSemantics::out << pr->name << " = " <<
            pr->ord->texNormalForm() << "\n" ;
        pS->promptCheck();
    }
}




static const Ordinal * predefinedVariable(const string& str)
{
    for (const InitPair * pr = initMapPairs; pr->ord; pr++) 
        if (pr->name == str) return pr->ord ;
    return NULL;
}

void ParserState::initMap()
{
    for (const InitPair * pr = initMapPairs; pr->ord; pr++) {
       list<const Ordinal *> *lst = new list<const Ordinal *> ();
        lst->push_back(pr->ord);
        assignments[pr->name] = *lst ;
    }
}

void ParseSemantics::listAssignments(ostream& strOut,bool textOnly)
{
    int saveOpts = formatOptions;
    if (textOnly) formatOptions = formatText ;
    for (map<const string, list<const Ordinal *> >::iterator
        iter= parserState->assignments.begin();
        iter != parserState->assignments.end(); ++iter) {
            if (predefinedVariable(iter->first)) continue ;
            if (pS->outText()) {
                strOut << iter->first << " = " ;
                bool anyOut = false ;
                for (list <const Ordinal *>::iterator
                    ordIter=iter->second.begin();
                        ordIter != iter->second.end(); ++ordIter) {
                  if (anyOut) strOut << ", " ;
                strOut << (*ordIter)->normalForm() ;
                anyOut= true ;
            }
            strOut << "\n" ;
            pS->promptCheck();
            
        }
        if (pS->outTeX()) {
            strOut << iter->first << " = " ;
            bool anyOut = false ;
            for (list <const Ordinal *>::iterator
                ordIter=iter->second.begin();
                     ordIter != iter->second.end(); ++ordIter) {
                    if (anyOut) strOut << ", " ;
                    strOut << (*ordIter)->texNormalForm() ;
                    anyOut= true ;
                }
                strOut << "\n" ;
                pS->promptCheck();
            }
    }
    if (textOnly) formatOptions = saveOpts ;
}

static void listCmd(const char * cmd, list<const string *> * params)
{
    no_args(cmd,params);
    pS->listAssignments();
    
}

static void listTexCmd(const char * cmd, list<const string *> * params)
{
    no_args(cmd,params);
    for (map<const string, list<const Ordinal *> >::iterator
        iter= parserState->assignments.begin();
        iter != parserState->assignments.end(); ++iter) {
        *ParseSemantics::out << iter->first << " = "  ;
        bool anyOut = false ;
        for (list <const Ordinal *>::iterator ordIter=iter->second.begin();
            ordIter != iter->second.end(); ++ordIter) {
            if (anyOut) *ParseSemantics::out << ", " ;
            *ParseSemantics::out << (*ordIter)->texNormalForm() ;
        }
        *ParseSemantics::out << "\n" ;
        pS->promptCheck();
    }

}


static void saveCmd(const char * cmd, list<const string *> * params)
{
    const string * dest = getOneArgument(cmd,params);
    if (!dest) dest = &(ParseSemantics::defaultSaveFile) ;
    else ParseSemantics::defaultSaveFile = *dest ;
    ofstream saveFile(dest->c_str());
    if (!saveFile.good()) {
        abortDocError();
        *ParseSemantics::out << "ERROR: Cannot write file " << *dest << "'.\n" ;
        pS->promptCheck();
        return ;
    }

    bool sPrompt = pS->promptUser ;
    pS->promptUser = false ;
    pS->listAssignments(saveFile,true);
    pS->promptUser = sPrompt ;

    if (!saveFile.good()) {
        abortDocError();
        *ParseSemantics::out << "ERROR: Cannot finish writing file '"
                << *dest << "'.\n" ;
         pS->promptCheck();
         return ;
     }
    saveFile << "\n" ;


}

static void readCmd(const char * cmd, list<const string *> * params)
{
    const string * dest = getOneArgument(cmd,params, "input file name");
    

    if (!dest) dest = &(ParseSemantics::defaultSaveFile) ;
    
    
    pS->readFromFile(dest);
    
}
static void readAllCmd(const char * cmd, list<const string *> * params)
{
    bool saveMode = pS->promptUser ;
    pS->promptUser = false ;
    readCmd(cmd,params);
    pS->promptUser = saveMode ;
}


static void exportTexCmd(const char * cmd, list<const string *> * params)
{
    const string * dest = getOneArgument(cmd,params);
    if (!dest) dest = &(ParseSemantics::defaultExportTexFile) ;
    else ParseSemantics::defaultExportTexFile = *dest ;
    ofstream saveFile(dest->c_str());
    if (!saveFile.good()) {
        abortDocError();
        *ParseSemantics::out << "ERROR: Cannot write file " << *dest << "'.\n" ;
        pS->promptCheck();
        return ;
    }
    saveFile << "\n\\begin{math}\n" ;
    for (map<const string, list<const Ordinal *> >::iterator
        iter= parserState->assignments.begin(); iter != parserState->assignments.end(); ++iter) {
        saveFile << iter->first << " = " ;
        bool anyOut = false ;
        for (list <const Ordinal *>::iterator ordIter=iter->second.begin();
            ordIter != iter->second.end(); ++ordIter) {
            if (anyOut) saveFile << ", " ;
            saveFile << (*ordIter)->texNormalForm() ;
        }
        if (!saveFile.good()) {
            abortDocError();
            *ParseSemantics::out << "ERROR: Cannot finish writing file '" <<
                *dest << "'.\n" ;
            pS->promptCheck();
            return ;
        }
        saveFile << "\n";
    }
    saveFile << "\\end{math}\n" ;
    if (!saveFile.good()) 
        abortDocError();
        *ParseSemantics::out << "ERROR: Cannot finish writing file '" << *dest << "'.\n" ;

}

extern int yydebug ;
static void yydebugCmd(const char * cmd, list<const string *> * params)
{
    const string * offFlag = getOneArgument(cmd,params,"off command");
    if (offFlag) if (*offFlag == "off") { yydebug = 0 ; return ;}
    yydebug = 1 ;
}


static void quitCmd(const char * cmd, list<const string *> * params)
{
    int ext = 0 ;
    if (!no_args(cmd,params)) ext = 1 ;
    exit(ext);

}

static void logCmd(const char * cmd, list<const string *> * params)
{
    const string * dest = getOneArgument(cmd,params,"log file name");
    if (!dest) dest = &ParseSemantics::defaultLogFile ;
    pS->log(ParseSemantics::createLog,dest);
}



static void logOptStop(list<const string *> * lst)
{
    pS->log(ParseSemantics::stopLog);

}


static void logOptFlush(list<const string *> * lst)
{
    pS->log(ParseSemantics::flushLog);

}

static const CommandOptions logOptOptions[] =
    {
        CommandOptions("flush",logOptFlush, "flush log file"),
        CommandOptions("stop",logOptStop,"stop logging"),
    
        CommandOptions(NULL,NULL),
    };


static void helpCmdsCmd(list<const string *> * lst);
static void helpSyntaxCmd(list<const string *> * lst);

static void helpMemberCmd(list<const string *> * lst);

static const char *purposeOrdCalc[] = {
        "The Ordinal Calculator is an interactive tool for understanding",
        "the hierarchies of recursive and countable",
        "ordinals\\cite{Veblen08,Miller76,Gallier91}.",
        "It is also a research tool to help to expand these hierarchies.",
        "Its motivating goal is ultimately to expand the foundations of",
        "mathematics by using computer technology to manage the",
        "combinatorial explosion in complexity that comes",
        "with explicitly defining the recursive ordinals implicitly defined",
        "by the axioms of Zermelo-Frankel set theory\\cite{CohenCH,Budnik06}.",
        "The underlying philosophy focuses on what formal systems tell us",
        "about physically realizable combinatorial",
        "processes\\cite{Budnik09}.\n",
        NULL
    };


static void helpPurpose(list<const string *> * lst)
{
    writeFormatted(purposeOrdCalc);
}

static void helpList(list<const string *> * lst)
{
    const char * text[] = {
        "Lists are a sequence of ordinals. An assignment statement can",
        "name a single ordinal or a list of them separated",
        "by commas. In most circumstances only the first element in the list",
        "is used, but some functions (such as member function 'limitOrdLst')",
        "use the full list. Type 'help members' to learn more about 'limitOrdLst'.",
        NULL
        
     };
    writeFormatted(text);

}



static void helpVeblenCmd(list<const string *> * lst)
{
    const char * text[] = {
        "Ordinals are displayed in TeX and plain text format. "
        "(Enter 'help opts' to control this.) "
        "The finite ordinals are the nonnegative integers.",
        "The ordinal operators are +, * and ^ for addition,",
        "multiplication and exponentiation. Exponentiation has the highest",
        "precedence. Parenthesis can be used to group subexpressions.",

        "\n\nThe ordinal of the integers, {\\tt omega}\\indextt{omega},",
        "is represented by the single lower"
        "case letter: '{\\tt w}'\\indextt{w}.",
        "The Veblen\\index{Veblen function}",
        "function is specified as",
        "'{\\tt psi(p1,p2,...,pn)}\\indextt{psi}' where n is",
        "any integer > 0. Special notations are displayed in some cases.",
        "Specifically {\\tt psi(x)}\\indextt{psi(x)} is displayed",
        "as {\\tt w^x}\\index{w$\\UseVerb{Exp}$x}.",
        "{\\tt psi(1,x)}\\indextt{psi(1,x)} is displayed",
        "as {\\tt epsilon(x)}. \\indextt{epsilon(x)}"
        "{\\tt psi(1,x,0)}\\indextt{psi(1,x,0)}",
        "is displayed as {\\tt gamma(x)}.\\indextt{gamma(x)}",
        "In all cases the displayed version can be used as input.",

        "\n\nLarger ordinals are",
        "specified as {\\tt psi_{px}(p1,p2,...,pn)}\\index{psi_{px}(p1,p2,...,pn)}.",
        "The first parameter is enclosed in brackets not parenthesis.",
        "{\\tt psi_{1}} is defined as the union of ",
        "{\\tt w, epsilon(0), gamma(0),} ",
        "{\\tt psi(1, 0, 0, 0), psi(1, 0, 0, 0, 0), psi(1, 0, 0, 0, 0, 0),} ",
        "{\\tt psi(1, 0, 0, 0, 0, 0, 0),} ... You can access the",
        "sequence whose union is a specific ordinal using member functions.",
        "Type {\\tt help members} to learn more about this.",
        "Larger notations beyond the recursive ordinals are also available",
        "in this implementation. See the documentation 'A Computational",
        "Approach to the Ordinal Numbers' to learn about",
        "`Countable admissible ordinals'.",
        "\n\nThere are several predefined ordinals.",
        "'{\\tt w}' and '{\\tt omega}'\\indextt{omega w} can be",
        "be used interchangeably for the ordinal of the integers and in",
        "other contexts. '{\\tt eps0}' and ",
        "'{\\tt omega1CK}' are also predefined.",
        "Type 'help defined'",
        "to learn more.",
        NULL
    };
    writeFormatted(text);
}
static void helpBooleanCmd(list<const string *> * lst)
{
    const char * text[] = {
        "Any two ordinals or ordinal expressions can be compared",
        "using the operators: <, <=, >, >= and ==. The result of the",
        "comparison is the text either TRUE\\index{TRUE}",
        "or FALSE\\index{FALSE}. Comparison operators have",
        "lower precedence\\index{precedence} than ordinal operators." ,
        NULL
    };

    writeFormatted(text);
}

static void helpVersionCmd(list<const string *> * lst)
{
    const char * text[] = {
        "The program version is " , version ,
        NULL
    };
    writeFormatted(text);
}



static void showExampleCmd(const char *cmd, list<const string *> * params);

static void toggleCmpCheck(const char *cmd, list<const string *> * params)
{
    if (params) if (params->size()) {
        abortDocError();
        *ParseSemantics::out << "ERROR: toggleCmpCheck has no parameters.\n";
        pS->promptCheck();
        return ;
    }
    parserState->runChecks^= true ;
    *ParseSemantics::out << "CmpCheck (comparison based consistency checking) is "
        << (parserState->runChecks? "enabled": "disabled") << ".\n" ;
    pS->promptCheck();

}


static const CommandOptions helpOptions[] = {
   CommandOptions("cmds",helpCmdsCmd,"lists commands"),
   CommandOptions("defined",helpPredefined,"list predefined ordinal variables"),
   CommandOptions("compare",helpBooleanCmd,"describes comparison operators"),
   CommandOptions("members",helpMemberCmd,"describes member functions"),
   CommandOptions("ordinal",helpVeblenCmd,"describes available ordinal notations"),
   CommandOptions("ordlist",helpList,"describes ordinal lists and their use"),
   CommandOptions("purpose",helpPurpose,
        "describes the purpose and philosophy of this project"),
   CommandOptions("syntax",helpSyntaxCmd,"describes syntax"),
   CommandOptions("version",helpVersionCmd,"displays program version"),
        
        CommandOptions(NULL,NULL)
    };

static void optsTextCmd(list<const string *> * lst)
{
    pS->setFormatText();
}

static void optsTexCmd(list<const string *> * lst)
{
    pS->setFormatTeX();

}

static void optsPsiCmd(list<const string *> * lst)
{
    pS->addFormatPsi();

}

static void optsBothCmd(list<const string *> * lst)
{
    pS->setFormatBoth();

}

static const char * firstCharLower(const char  *nm)
{
    if (!isupper(nm[0])) return nm ;
    const char * veb = "Veblen" ;
    static const int ln = strlen(veb);
    if (!strncmp(nm,veb,ln)) return nm ;
    static const int sz = 8192 ;
    static char buf[sz+1] ;
    assert(strlen(nm)<sz);
    strcpy(buf,nm);
    buf[0]=tolower(buf[0]);
    return buf ;
}

static const char * titleToSecRef(const char * title)
{
    int length = strlen(title) ;
    for (const char * pt = title; *pt; pt++) if ((*pt == '\'') ||
        (*pt == ' ')) length -- ;
    char * ret = new char[length+1] ;
    char * dest = ret ;

    for (const char * pt = title; *pt; pt++) {
        if ((*pt == '\'') || (*pt == ' ')) continue;
        *dest++ = *pt ;
    }
    *dest = '\0' ;
    return ret ;
}

    

void ParseSemantics::exampleOutFile(const ExampleScripts * entry)
{
    assert(entry);
    const char * title = entry->title ;
    const char ** text = entry->script ;
    const char * exampleName = title ;
    if (TeXdocMode) {
        const char *examp = firstCharLower(exampleName);
        *out <<"\\sub" << sectionTeX << "{"
            << exampleName << "}\n" <<
            "\\label{sec" << titleToSecRef(exampleName) << "}\n" <<
            "\\mindex{" << examp << " example}\n" <<
            "\\mindex{example of " << examp << "}\n" ;
    } else *out << "\n" << exampleName << " example\n\n" ;

    if (const char * what = entry->whatItDoes) {
        
        *out << "The following " << what ;
        if (TeXdocMode) *out << ".\\\\[.15in]\n";
        else *out << ".\n" ;

        if (const char ** text = entry->descript) {
            
            pS->writeFormatted(text) ;
            if (!TeXdocMode) waitForEnter();
            else *out << "\\\\[.15in]\n";
        }
    }
    
    static const char *tmpFile = "tmpOrdCalc.tmp";
    ostream * saveOut = NULL;

    if (TeXdocMode) {
        *out << "{\\tt " ;
        saveOut = out ;
        out = new ofstream(tmpFile);
        assert(out->good());
    }


    exampleInput = text;

    ParserState * saveState = parserState ;
    parserState = new ParserState ;
    pS = &(parserState->getParseSemantics());

    pS->exampleParse();


    if (TeXdocMode) {
        delete out ;
        out = saveOut ;
        ifstream * inf = new ifstream(tmpFile) ;
    
        while (inf->good()) {
            const int bufsz=8191 ;
            char buf[bufsz+1] ;
            const char * arys[] = {buf,NULL} ;
            inf->getline(buf,bufsz);
            int count = inf->gcount();
            buf[count-1]= '\n' ;
            buf[count] = '\0' ;
            if (!count) {
                if (!inf->good()) break ;
                buf[0]='\0' ;
            }
            pS->writeFormatted(arys,true);
        }
        delete inf ;
        unlink(tmpFile);
        if (TeXdocMode) *out <<"}\n" ;
    } 
    delete parserState ;
    parserState = saveState ;
    pS = &(saveState->getParseSemantics());
}


void ParseSemantics::exampleOut(const char * title,  const char * text[])
{
    const char * exampleName = title ;
    if (ParseSemantics::TeXdocMode) {
        const char *examp = firstCharLower(exampleName);
        *ParseSemantics::out <<"\\sub" << ParseSemantics::sectionTeX << "{"
            << exampleName << "}\n" <<
            "\\mindex{" << examp << " example}\n" <<
            "\\mindex{example of " << examp << "}\n" ;
        *ParseSemantics::out << "{\\tt " ;
    } else *ParseSemantics::out << "\n" << exampleName << " example\n\n" ;
    
    ::writeFormatted(text);
    if (ParseSemantics::TeXdocMode) *ParseSemantics::out <<"}\n" ;


}

const char * memberExampCommands[]= {
        "a=psi(1,0,0,0,0)",
        "a.listElts(3)",
        "b=a.limitElt(6)",
        NULL
 };



static const char * comparisonExampCommands[] = {
        "psi(1,0,0) == gamma(0)",
        "psi(1,w) == epsilon(w)",
        "w^w < psi(1)",
        "psi(1)",
        NULL    
};





static const char * displayOptsExampCommands[]  = {
    "a=w^(w^w)",
    "b=epsilon(a)",
    "c=gamma(b)",
    "list",
    "opts tex",
    "list",
    "opts both",
    "list",
        NULL
};


static const char * arithExampCommands[] = {
    "a=w^w",
        "b=w*w",
        "c=a+b",
        "d=b+a",
        NULL
};

static const char * veblenFiniteExampCommands[] = {
    "a=psi(w,w)",
    "b=psi(a,3,1)",
    "b.listElts(3)",
    "c=psi(a,a,b,1,3)",
    "c.listElts(3)",
    NULL
} ;


static const char * veblenFiniteExplanation[] = {
    "The Veblen function with a finite number of parameters, psi(x1,x2,...xn)",
    "is built up from",
    "the function omega^x. psi(x) = omega^x. psi(1,x) enumerates the fixed points",
    "of omega^x. This is epsilon(x). Each additional variable diagonalizes the",
    "functions definable with existing variables. These functions can have any finite",
    "number of parameters.",
    NULL
} ;

static const char * veblenExtendedExampCommands[] = {
    "a=psi_{1}(1)",
    "a.listElts(4)",
    "b=psi_{w+1}(3)",
    "b.listElts(4)",
    NULL
} ;

static const char * veblenExtendedExplanation[] = {
    "The extended Veblen function, psi_{a}(x1,x2,...,xn),",
    "iterates the idea of the Veblen function",
    "up to any recursive ordinal. The first parameter is",
    "the recursive ordinal of this iteration.",
    NULL
} ;

static const char * admissibleExampCommands[] = {
    "a=w_{1}(1)",
    "a.listElts(4)",
    NULL
} ;

static const char * admissibleExplanation[] = {
    "Countable admissible level notations, omega_{k,g}(x1,x3,..,xn) extend the",
    "idea of recursive notation to larger countable ordinals. The first parameter",
    "is the admissible level. omega_{1} is the",
    "Church-Kleene ordinal\\index{Church-Kleene ordinal}. The",
    "remaining parameters are similar to those defined for smaller ordinal",
    "notations.",
    NULL
} ;


static const char * admissibleDropDownExampCommands[] = {
    "a=w_{1}[1]",
    "a.listElts(4)",
    "b=w_{1}",
    "c=b.limitOrd(w^3)",
    "c.listElts(4)",
    "d=w_{5,c}(3,0)",
    "d.listElts(4)",
    NULL
} ;
static const char * admissibleDropDownExplanation[] = {
    "Admissible level ordinals have the a limit sequence defined",
    "in terms of lower levels. The lowest admissible level is that",
    "of recursive ordinals. To implement this definition of limit",
    "sequence, a trailing parameter in square brackets is used.",
    "This parameter (if present)",
    "defines an ordinal at one admissible level lower than indicated",
    "by other parameters.",
    NULL
};

static const char * admissibleContextExampCommands[] = {
    "a=[[1]]w_{1}",
    "a.listElts(4)",
    NULL
} ;
static const char * admissibleContextExplanation[] = {
    "The context parameter in admissible level ordinals allows one to use",
    "any notation at any admissible level to define notations at any lower",
    "admissible level or to define recursive ordinals.",
    NULL
};

static const char * listsExampCommands[] = {
    "lst = 1, 12, w, gamma(w^w), w1",
    "a=w1.limitOrdLst(lst)",
    "bg = w_{w+33}",
    "c=bg.limitOrdLst(lst)",
    NULL
};

static const char * listExplanation[] = {

    "Lists are a sequence of ordinals (including ",
    "integers).\\index{calculator list}",
    "A list can be",
    "assigned to a variable just as a single ordinal can be. In most circumstances",
    "lists are evaluated as the first ordinal in the list.",
    "In 'limitOrdLst'\\indextt{limitOrdLst}",
    "all of the list entries are used. These member functions",
    "return a list with an input list",
    NULL
};

static const char * desLimitOrdLstExampCommands[] = {
    
    "lst = 1, 5, w, psi(2,3)",
    "bg = w_{3}",
    "d= bg.desLimitOrdLst(2,lst)",
    
    NULL
};


static const char * desLimitOrdLstExplanation[] = {

    "'desLimitOrdLst'\\indextt{desLimitOrdLst} iterates 'limitOrdLst' to",
    "a specified 'depth'.",
    "The first parameter is the integer depth of iteration",
    "and the second is the list of parameters to be used.",
    "This routine first takes'limitOrd'\\indextt{limitOrd} of ",
    "each element in the second parameter",
    "creating a list",
    "of outputs. It then takes this list and evaluates 'limitOrd' for",
    "each of these values at each entry in the original parameter list.",
    "All of these results are combined in a new list",
    "and the process is iterated",
    "'depth' times. The number of results grows exponentially",
    "with 'depth'.",
    NULL
};






static void dummyExamp(list<const string *> * lst) 
{
    
    
    assert(0);
}

static void dbgCtrOrd(list<const string *> * lst)
{
    OrdinalImpl::debugControl.setCtrOrd(true);
}

static void dbgAll(list<const string *> * lst)
{
    OrdinalImpl::debugControl.setAll();
}

static void dbgArith(list<const string *> * lst)
{
    OrdinalImpl::debugControl.setArith(true);
}

static void dbgClear(list<const string *> * lst)
{
    OrdinalImpl::debugControl.clearAll();
}

static void dbgCompare(list<const string *> * lst)
{
    OrdinalImpl::debugControl.setCompare(true);
}

static void dbgexp(list<const string *> * lst)
{
    OrdinalImpl::debugControl.setExpon(true);
}

static void dbgLimArith(list<const string *> * lst)
{
    OrdinalImpl::debugControl.setArith(true);
}

static void dbgLimit(list<const string *> * lst)
{
    OrdinalImpl::debugControl.setLimit(true);
}

static void optsPromptLinesCmd(list<const string *> * lst)
{
    int count = 1 ;
    const string * lineCount = NULL ;
    for (list<const string *>::iterator iter = lst->begin(); iter != lst->end();
        ++iter)  if (count++ == 2) {lineCount = *iter; break;}
    if (!lineCount) {
        abortDocError();
        *ParseSemantics::out << 
        "ERROR: Missing line count argument to 'opts promptLines'.\n" ;
        pS->promptCheck();
    }
    stringstream ssm(*lineCount);
    int lcount ;
    ssm >> lcount ;
    
    pS->setPromptLimit(lcount);
}


static const CommandOptions optsOptions[] = 
    {
        CommandOptions("both", optsBothCmd,
            "display ordinals in both plain text and TeX formats"),
        CommandOptions("tex", optsTexCmd, "display ordinals in TeX format only"),
        CommandOptions("text", optsTextCmd,
            "display ordinals in plain text format only (default)"),
        CommandOptions("psi", optsPsiCmd,
        "additionally display ordinals in Psi format (turned off by the above options)"),
        CommandOptions("promptLimit",optsPromptLinesCmd,
            "lines to display before pausing, < 4 disables pause"),
        CommandOptions(NULL,NULL)
     };

static const ExampleScripts exampleOptions[] =
    {
        ExampleScripts("arith",dummyExamp,
            "demonstrates ordinal arithmetic",
            "Simple ordinal arithmetic", arithExampCommands),
        ExampleScripts("compare",dummyExamp,"shows compare examples",
            "Comparison operators", comparisonExampCommands),
        ExampleScripts("display",dummyExamp,
            "shows how display options work",
            "Display options", displayOptsExampCommands),
        ExampleScripts("member",dummyExamp,"demonstrates member functions",
            "Member functions",memberExampCommands),
        ExampleScripts("VeblenFinite",dummyExamp,
            "demonstrates Veblen functions of a finite number of ordinals",
            "Veblen function of N ordinals",
            veblenFiniteExampCommands,veblenFiniteExplanation),
        ExampleScripts("VeblenExtend",dummyExamp,
            "demonstrates Veblen functions iterated up to a recursive ordinal",
            "Extended Veblen function",
            veblenExtendedExampCommands,veblenExtendedExplanation),
        ExampleScripts("admissible",dummyExamp,
            "demonstrates countable admissible level ordinal notations",
            "Admissible countable ordinal notations",
            admissibleExampCommands,admissibleExplanation),
        ExampleScripts("admissibleDrillDown",dummyExamp,
            "demonstrates admissible notations dropping down one level",
            "Admissible notations drop down parameter",
            admissibleDropDownExampCommands,admissibleDropDownExplanation),
        ExampleScripts("admissibleContext",dummyExamp,
            "demonstrates admissible ordinal context parameters",
            "Admissible notations context parameter",
            admissibleContextExampCommands,admissibleContextExplanation),
        ExampleScripts("list",dummyExamp,"shows how lists work",
            "Lists of ordinals",listsExampCommands,listExplanation),
        ExampleScripts("desLimitOrdLst",dummyExamp,
        "shows how to construct a list of descending trees",
            "List of descending trees",desLimitOrdLstExampCommands,
                desLimitOrdLstExplanation),
        ExampleScripts(NULL,NULL)
     };

static const CommandOptions setDbgOptions[] =
    {
        CommandOptions("all",dbgAll,"turn on all debugging"),
        CommandOptions("arith",dbgArith,"debug ordinal arithmetic"),
        CommandOptions("clear",dbgClear,"turn off all debugging"),
        CommandOptions("compare",dbgCompare,"debug compare"),
        CommandOptions("exp",dbgexp,"debug exponential"),
        CommandOptions("limArith",dbgLimArith,"limited debugging of arithmetic"),
        CommandOptions("limit",dbgLimit,"debug limit element functions"),
        CommandOptions("construct",dbgCtrOrd,"debug constructors"),
        
        CommandOptions(NULL,NULL)
        
    };


static void optsCmd(const char * cmd, list<const string *> * params)
{
    if(!params || !params->size()) {
        abortDocError();
        *ParseSemantics::out << "ERROR: command '" << cmd <<
            "' requires one argument.\n";
        pS->promptCheck();
        return ;
    }
    abortDocError();
    *ParseSemantics::out << "ERROR: command '" << *(params->front()) <<
        "' is not a parameter of command '" << cmd << "'.\n" ;
     pS->promptCheck();
}

static void logOptCmd(const char * cmd, list<const string *> * params)
{
    optsCmd(cmd,params);
}

static void promptCmd (const char *cmd,  list<const string *> * params)
{
    pS->waitForEnter(params);
}



static const Commands theCommands[]=
    {
        {"cmpCheck",toggleCmpCheck,0,"toggle comparison checking for debugging"},
        {"examples",showExampleCmd,exampleOptions,"shows examples"},
        {"exit",quitCmd,0,"exits the program"},
        {"exportTeX",exportTexCmd,0,
            "exports assignments statements in TeX format"},
        {"help",helpCmd,helpOptions,"displays information on various topics"},
        {"list",listCmd,0,"lists assignment statements"},
        {"log",logCmd,0,"writes a log file (ord.log default)"},
        {"listTeX",listTexCmd,0,
            "lists assignment statements in TeX format"},
        {"logopt",logOptCmd,logOptOptions,"controls the log file"},
        {"opts",optsCmd,optsOptions,
            "controls display format and other options"},
        {"prompt",promptCmd,0,"prompts for ENTER with optional string argument"},
        {"quit",quitCmd,0,"exits the program"},
        {"read",readCmd,0,"read \"input file\" (ord_calc.ord default)"},
        {"readall",readAllCmd,0,"same as read but no 'wait for ENTER' prompt"},
        {"save",saveCmd,0,
            "saves assignment statements to a file (ord_calc.ord default)"},
        {"setDbg",optsCmd,setDbgOptions,"set debugging options"},
        {"yydebug",yydebugCmd,0,"enables parser debugging (off option)"},
        {NULL,NULL}
    };



static void showExampleCmd(const char *cmd, list<const string *> * params)
{
    const char * text[] = {
        
        "In the examples a line that begins "
        "with the standard prompt 'ordCalc> '",
        "contains user input. All other lines contain program output",
        "\n\nTo select an examples type '{\\tt examples}' followed by one of",
        "the following options.",
        NULL
    };
    writeFormatted(text);
    pS->brTeX();

    for (const ExampleScripts * opts = exampleOptions; opts->name ;
        opts=(const ExampleScripts*)opts->getNext()){
         *ParseSemantics::out << ParseSemantics::sqTeX() ;
         if (ParseSemantics::TeXdocMode) {
            *ParseSemantics::out << "\\Index{" <<
                ParseSemantics::chTtyTeX(opts->name) << "}" ;
         } else *ParseSemantics::out << ParseSemantics::chTtyTeX(opts->name) ;
       *ParseSemantics::out << ParseSemantics::sqTeX(false);
          *ParseSemantics::out << " -- "<<ParseSemantics::checkTeX(opts->whatItDoes) ;
        *ParseSemantics::out << ".\n" ;
        pS->promptCheck();
        pS->brTeX();
    }
}




static void helpCmdsCmd(list<const string *> * lst)
{
    *ParseSemantics::out << "The following commands are available:\n" ;
    pS->promptCheck();
    pS->brTeX();
    for (const Commands * cmd = theCommands; cmd->name; cmd++) {
        *ParseSemantics::out << ParseSemantics::sqTeX();
        if (ParseSemantics::TeXdocMode) {
            *ParseSemantics::out << "\\Index{" <<
                ParseSemantics::chTtyTeX(cmd->name) << "}" ;
        } else *ParseSemantics::out << ParseSemantics::chTtyTeX(cmd->name) ;
        *ParseSemantics::out << ParseSemantics::sqTeX(false);
        if (cmd->whatItDoes) *ParseSemantics::out << " -- " <<
            ParseSemantics::checkTeX(cmd->whatItDoes) ;
        *ParseSemantics::out << ".\n" ;
        pS->promptCheck();
        pS->brTeX();
    }
    
    pS->promptCheck();
 }



static void helpSyntaxCmd(list<const string *> * lst)
{
    const char * text[] = {
    "The syntax is that of restricted arithmetic expressions",
    "and assignment statements. The tokens are variable names,",
    "nonnegative integers and the operators: +, * and ^ (addition,",
    "multiplication and exponentiation). Comparison operators are",
    "also supported. Type '{\\tt help comparison}' to learn about them.",
    "The letter 'w' is" ,
    "predefined as omega, the ordinal of the integers.",
    "Type '{\\tt help defined}' for a list of all predefined variables." ,
    "To learn more about ordinals type '{\\tt help ordinal}'." ,
    "C{\\tt ++} style member functions are supported with a '.' separating",
    "the variable name (or expression enclosed in parenthesis) from the",
    "member function name. Enter '{\\tt help members}' for the list of member",
    "functions.",
    "\n\nAn assignment statement or ordinal expression can be entered",
    "and it will be evaluated and displayed in normal form.",
    "Typing '{\\tt help opts}' lists the display options.",
    "Assignment statements are stored. They can be listed (command 'list') ",
    "and their value can be used in subsequent expressions. ",
    "All statements end at the end of a line unless the last character is",
    "'\\'. Lines can be continued indefinitely. Comments must be preceded",
    "by either '{\\tt \%}' or '{\\tt //}'.",
    "\n\nCommands can be entered as one or more names separated by",
    "white space. File names should be enclosed in double quotes ({\\tt \"}) if",
    "they contain any non alphanumeric characters such as dot, '.'.",
    "Command names can be used as variables.",
    "Enter '{\\tt help cmds}' to get a list of commands and their functions.",
    NULL
    };
    writeFormatted(text);
}




static void helpCmdOpt(const char *cmd, list<const string *> * params,
    bool listCmds=true)
{
    int size = 0 ;
    bool valid = false ;
    if (params) if (size = params->size()) {
        const string * param = params->front();
        for (const Commands * cmds = theCommands; cmds->name; cmds++) 
            if (*param == cmds->name) {
            valid = true ;
            if (listCmds) {
                *ParseSemantics::out << "Command " << ParseSemantics::sqTeX() ;
                if (ParseSemantics::TeXdocMode){
                    *ParseSemantics::out << "\\Index{" <<
                        ParseSemantics::chTtyTeX(cmds->name) << "}" ;
                } else *ParseSemantics::out << 
                    ParseSemantics::chTtyTeX(cmds->name);
                *ParseSemantics::out << ParseSemantics::sqTeX(false) << " -- " 
                    << ParseSemantics::checkTeX(cmds->whatItDoes) << ".\n" ;
                pS->promptCheck();
                pS->brTeX();
            }
            if (cmds->options) {
                if (listCmds) {
                *ParseSemantics::out <<
                "It has one parameter with the following options.\n";
                pS->promptCheck();
                pS->brTeX();
                if (*param == cmd) {
                    for (const Commands * cmds2 = theCommands;
                        cmds2->name;cmds2++) {
                        *ParseSemantics::out << 
                            ParseSemantics::sqTeX()  ;
                        if (ParseSemantics::TeXdocMode){
                            *ParseSemantics::out << "\\Indextt{" <<
                            (cmds2->name) << "}" ;
                        } else *ParseSemantics::out << cmds2->name ;
                        *ParseSemantics::out << ParseSemantics::sqTeX(false)
                            << " get help on "
                            << ParseSemantics::checkTeX(cmds2->whatItDoes) << ".\n" ;
                        pS->promptCheck();
                        pS->brTeX();
                    }
                }
                }
                for (const CommandOptions *opts = cmds->options;opts->name;
                    opts=opts->getNext()) {
                
                    *ParseSemantics::out << ParseSemantics::sqTeX()  ;
                     if (ParseSemantics::TeXdocMode){
                        *ParseSemantics::out << "\\Index{" <<
                            ParseSemantics::chTtyTeX(opts->name) << "}"  ;

                    } else  *ParseSemantics::out <<
                        ParseSemantics::chTtyTeX(opts->name) ;
                    *ParseSemantics::out << ParseSemantics::sqTeX(false) <<
                        " -- " << ParseSemantics::checkTeX(opts->whatItDoes)<<".\n" ;
                    pS->promptCheck();
                    pS->brTeX();
                }
            }

        }
        if (size > 1) {
            abortDocError();
            *ParseSemantics::out << cmd << "ERROR: '" <<
            cmd << "' has 0 or 1 parameters.\n" ;
            pS->promptCheck();
        }
        if (!valid) {
            abortDocError();
            *ParseSemantics::out << "ERROR: '" << *(params->front()) <<
            "' is not a valid parameter for '" << cmd << "'.\n" ;
            pS->promptCheck();
        }
        return ;
    }
    commandHelp(cmd);
}

static void helpCmd(const char *cmd, list<const string *> * params)
{
    helpCmdOpt(cmd,params);
}


ParseSemantics * ord::pS = new ParseSemantics;

void parseInput(const char * file=NULL)
{
    parserState = new ParserState ;
    if (file) ord::pS->setPromptLimit(0);
    atexit(exitCleanup);
    ParseSemantics::out = &cout ;
    ord::pS->parseInput(file);
}

void writeTeXdoc(const char *fileName,const char * sec)
{
    outputtingTeX = true ;
    const char * saveSec = ParseSemantics::sectionTeX;
    if (sec) ParseSemantics::sectionTeX = sec ;
    ord::pS->createTeXDocumentation(fileName);
    if (sec) ParseSemantics::sectionTeX = saveSec ;
    outputtingTeX = false ;
}


void ParseSemantics::found(const char * what)
{
    *out << "found " << what << "\n" ;
    pS->promptCheck();

}

const ParseContainer * ParseSemantics::number(int val)
{
    
    Ordinal * ret =  new Ordinal(val) ;
    
    return new ParseContainer(ret) ;
}


static void something()
{
    const char * text[] = {
     "This interpreter converts ordinal expressions to",
    "a normal form based on Cantor noraml form.",
    "Statements end at the end of a line unless the last ",
    "character in the line is '\\', the continuation character.",
    "Lines can be continued indefinitely.",
    "The basic ordinal operators are + * ^ for addition,",
    "multiplication and exponentiation.",
    "The lower case letter `w` is omega or the ordinal",
    "of the integers. The name `psi' is reserved for the",
    "Veblen hiearchy. psi(0,n) = w^n; psi(1,n) = epsilon-n.",
    "An expression or an assignment statement can be entered.",
    "Once defined in an assignment, the ordinal can be used",
    "in subsequent statements.",
    NULL
    };
}

static void commandHelp(const char * str)
{
    ostream * out = ParseSemantics::out ;
    if (!outputtingTeX) *out << "                   " << version << "\n\n" ;
    const char * text[] = {
    "This is a command line interactive interface to a program",
    "for exploring the ordinals.",
    "It supports recursive ordinals up to and beyond the",
    "the Bachmann-Howard ordinal\\cite{Howard72}.",
    "It defines notations",
    "for the Church-Kleene ordinal and some larger countable ordinals.",
    "We refer to these as admissible level ordinals.",
    "They are used in a form of ordinal collapsing",
    "to define large recursive ordinals.\n\n",
     NULL
    };
    writeFormatted(text);
    if (outputtingTeX) *out << "\\input{ordCalcCmdLine}\n\n" <<
    "\\subsection{Help topics}\n\n" ;
    const char *text2[] = {
    "Following are topics you",
    "can get more information about by entering '{\\tt help} topic'.",
    NULL
    };
    writeFormatted(text2);
    vsp();
    list<const string *> lst ;
    lst.push_back(new string("help"));
    helpCmdOpt("help",&lst,false);
    if (outputtingTeX) {
        *out << "\nThis program supports " <<
        "\\href{http://tiswww.case.edu/php/chet/readline/rluserman.html}" <<
        "{GNU `\\Index{readline}' input line editing.}" << 
        "\n\nYou can download the program and documentation at: " <<
        "\\href{http://www.mtnmath.com/ord}"
        << "{Mountain Math Software} or at " <<
        "\\href{https://sourceforge.net/projects/ord}{SourceForge.net}\n" <<
        "({\\tt http://www.mtnmath.com/ord} or\n" <<
        "{\\tt https://sourceforge.net/projects/ord}).\n" ;
    } else {
        const char *text[]= {
        "\nThis program supports GNU 'readline' input line editing.\n",
        "\nYou can download the program and documentation at:\n",
        "http://www.mtnmath.com/ord (Mountain Math Software) or at\n",
        "https://sourceforge.net/projects/ord (SourceForge.net).\n",
        "Send comments and problem reports to 'ord@mtnmath.com'.\n",
        NULL
        };
        writeFormatted(text);

    }
    
}

void ParseSemantics::outPsiList(list<const Ordinal *>* ordLst, ostream * out,
    int charCount)
{
    bool anyOut = false ;
    for (list <const Ordinal *>::iterator ordIter= ordLst->begin();
          ordIter != ordLst->end(); ++ordIter) {
            if (anyOut) *out << ", " ;
            *out << (*ordIter)->psiNormalForm() ;
            anyOut = true ;
   }
}



void ParseSemantics::outTeXlist(list<const Ordinal *>* ordLst, ostream * out,
    int charCount)
{
    bool anyOut = false ;
    for (list <const Ordinal *>::iterator ordIter= ordLst->begin();
          ordIter != ordLst->end(); ++ordIter) {
            if (anyOut) *out << ", " ;
            *out << (*ordIter)->texNormalForm() ;
            anyOut = true ;
   }
}

void ParseSemantics::outList(list<const Ordinal *>* ordLst, ostream * out,
    int charCount)
{
    const int lineLimit = 72 ;
    bool anyOut = false ;
    for (list <const Ordinal *>::iterator ordIter= ordLst->begin();
          ordIter != ordLst->end(); ++ordIter) {
            if (anyOut) *out << ", " ;

            *out << (*ordIter)->normalForm() ;
            anyOut = true ;
   }
}



const ParseContainer * ParseSemantics::ddMark(const ParseContainer * exp)
{
    const Ordinal * theOrd = exp->checkOrd();
    if (!theOrd) theOrd = exp->getFirstOrdinal();

    
    return new ParseContainer(theOrd,
        (ParseContainer::Type)
        (ParseContainer::drillDown|ParseContainer::ord));
}

void ParseSemantics::outList(list<const ParseContainer *> *ordLst,
    ostream * out, int charCount)
{
    bool anyOut = false ;
    for (list<const ParseContainer *>::iterator containIter= ordLst->begin();
          containIter != ordLst->end(); ++containIter) {
            const ParseContainer & c = **containIter ;
            if (c.type & ParseContainer::ord) {
                if (anyOut) *out << ", " ;
                *out << (c.getOrd())->normalForm() ;
                anyOut = true ;
            } else if (c.type & ParseContainer::ordLst) {
                list<const Ordinal *>&l = *(c.getList());
                for(list<const Ordinal *>::iterator ordIter= l.begin();
                    ordIter != l.end(); ++ordIter) {
                    if (anyOut) *out << ", " ;
                    *out << (*ordIter)->normalForm();
                    anyOut = true ;
                }

            } else continue ;
        
   }
}

void ParseSemantics::outTeXlist(list<const ParseContainer *> * ordLst,
    ostream * out, int charCout)
{
  bool anyOut = false ;
    for (list<const ParseContainer *>::iterator containIter= ordLst->begin();
          containIter != ordLst->end(); ++containIter) {
            const ParseContainer & c = **containIter ;
            if (c.type & ParseContainer::ord) {
                if (anyOut) *out << ", " ;
                *out << (c.getOrd())->texNormalForm() ;
                anyOut = true ;
            } else if (c.type & ParseContainer::ordLst) {
                list<const Ordinal *>&l = *(c.getList());
                for(list<const Ordinal *>::iterator ordIter= l.begin();
                    ordIter != l.end(); ++ordIter) {
                    if (anyOut) *out << ", " ;
                    *out << (*ordIter)->texNormalForm();
                    anyOut = true ;
                }

            } else continue ;
        
   }
}


void ParseSemantics::outPsiList(list<const ParseContainer *> * ordLst,
    ostream * out, int charCout)
{
  bool anyOut = false ;
    for (list<const ParseContainer *>::iterator containIter= ordLst->begin();
          containIter != ordLst->end(); ++containIter) {
            const ParseContainer & c = **containIter ;
            if (c.type & ParseContainer::ord) {
                if (anyOut) *out << ", " ;
                *out << (c.getOrd())->psiNormalForm() ;
                anyOut = true ;
            } else if (c.type & ParseContainer::ordLst) {
                list<const Ordinal *>&l = *(c.getList());
                for(list<const Ordinal *>::iterator ordIter= l.begin();
                    ordIter != l.end(); ++ordIter) {
                    if (anyOut) *out << ", " ;
                    *out << (*ordIter)->psiNormalForm();
                    anyOut = true ;
                }

            } else continue ;
        
   }
}




void ParseSemantics::assign(const string * name,
    list<const ParseContainer *> * ordLst)
{
    if (const Ordinal * preDef = predefinedVariable(*name)) {
        abortDocError();
        *out << "ERROR: '" << *name  << "' is predefined as " <<
           preDef->normalForm() << " and cannot be changed.\n" <<
           "Type 'help defined' to list all predefined variables.\n" ;
        return ;
    }

    list<const Ordinal *> *linList = (ParseContainer::linearList(ordLst));
    list<const Ordinal *> * dupDef = findListValue(name);
    if (dupDef) {
        if (outText()) {
            *out << "Redefining '" << *name << "' from " ;
            outList(dupDef);
            *out << " to "  ;
            outList(linList);
            *out << ".\n" ;
            pS->promptCheck();
        }
        if (outTeX()) {
            *out << "Redefining '" << *name << "' from " ;
            outTeXlist(dupDef);
             *out << " to " ;
             outTeXlist(linList);
             *out <<".\n";
            pS->promptCheck();
        }
    }
    assert(outText() || outTeX());
    parserState->assignments[*name]= *linList ;
    if (!dupDef) {
        if (outText()) {
            *out << "Assigning " ;
            outList(ordLst) ;
            *out << " to '" << *name << "'.\n" ;
            pS->promptCheck();
        }
        if (outTeX()) {
            *out << "Assigning " ;
            outTeXlist(ordLst);
            *out << " to '" << *name << "'.\n" ;
            pS->promptCheck();
        }
        if (outPsi()) {
            *out << "Assigning " ;
            outPsiList(ordLst);
            *out << " to '" << *name << "'.\n" ;
            pS->promptCheck();

        }        
    }
}



const Ordinal * ParseSemantics::findValue(const string * name) const
{
    list<const Ordinal *>* ordList = findListValue(name);
    if (ordList) return ordList->front();
    return NULL ;
}

list<const Ordinal *> * ParseSemantics::findListValue(const string * name) const
{
    map<const string, list<const Ordinal *> >::iterator
        iter=parserState->assignments.find(*name);
            if( iter != parserState->assignments.end()) {
       list<const Ordinal *>& ordLst = iter->second ;
       return &ordLst;
    }
    return NULL ;
}

static void doBreak()
{
}

const ParseContainer * ParseSemantics::evaluate(const string * name)
{
    

    list<const Ordinal *>* ret = findListValue(name);
    if (ret) return new ParseContainer(ret) ;
    const Ordinal *ord = findValue(name) ;
    if (ord) return new ParseContainer(ord);
    doBreak();
    *out << "ERROR: Variable '" << *name <<
        "' is not defined. It is set to 0.\n" ;
    pS->promptCheck();
    return new ParseContainer(&zero) ;
}

const Ordinal * ParseSemantics::subtract(const Ordinal& ord1,
    const Ordinal& ord2) const
{
    int err = 0 ;
    if (!ord2.isFinite()) err |= 1 ;
    if (!ord1.hasFiniteTerm()) err |= 2 ;
    if (!err) {
        Int subFrom = ord1.getImpl().getFiniteTerm();
        Int toSub = ord2.getImpl().getInt();
        if (subFrom < toSub) err |= 4 ;
        else {
            return new Ordinal(ord1.getImpl().subtract(toSub));
        }
    }
    assert(err);
    *out << "ERROR: " ;
    if (err&1) {*out << "The value subtracted must be finite.\n" ;
        pS->promptCheck();
        }
    if (err&2) {*out << "The value subtracted from must have a finite term.\n";
        pS->promptCheck();
        }
    if (err&4) {*out << "The finite term of the first operand must\n"  <<
        "be >= the second operand.\n" ;
        pS->promptCheck();
        }
    stz();
    abortDocError();
    return &Ordinal::zero ;
}

void ParseSemantics::stz() const
{
    *out << "Value set to zero.\n" ;
        pS->promptCheck();
}

const ParseContainer * ParseSemantics::operation(const ParseContainer * cop1, char op,
    const ParseContainer * cop2)
{
    assert(cop1);
    assert(cop2);
    const Ordinal * op1 = cop1->getFirstOrdinal();
    const Ordinal * op2 = cop2->getFirstOrdinal();
    assert(op1);
    assert(op2);
    
    switch (op) {
case '+' :
        return new ParseContainer( new Ordinal(*op1 + *op2)) ;
case '-' :
        return new ParseContainer(subtract(*op1, *op2));
case '*' :
        return new ParseContainer( new Ordinal(*op1 * *op2)) ;
case '^' :
        return new ParseContainer( new Ordinal(*op1 ^ *op2)) ;
default :
        *out << "ERROR: Internal error 2 ParseSemantics::operation\n" ;
        abortDocError();
        pS->promptCheck();
        return new ParseContainer( &zero) ;
    }
}

void ParseSemantics::expr(const ParseContainer * cexp)
{
    
    list<const Ordinal *>* lexp = cexp->checkList();
    if (lexp) {
        if (outText()) {
           *out << "Normal form: " ;
           outList(lexp) ;
           *out << "\n" ;
            pS->promptCheck();
        }
        if (outTeX()) {
           *out << "TeX normal form: " ;
           outTeXlist(lexp);
           *out << "\n" ;
            pS->promptCheck();
        }
        if (outPsi()) {
           *out << "Psi normal form: " ;
           outPsiList(lexp);
           *out << "\n" ;
            pS->promptCheck();
        }
        return ;
    }
    const Ordinal * exp = cexp->getFirstOrdinal();
    assert(exp);
   if (outText()) {*out << "Normal form: " <<  exp->normalForm() << "\n" ;
        pS->promptCheck();
        }
   if (outTeX()) {*out << "TeX normal form: " << exp->texNormalForm()
        << "\n" ;
        pS->promptCheck();
        }
   if (outPsi()) {*out << "Psi normal form: " << exp->psiNormalForm()
        << "\n" ;
        pS->promptCheck();
        }

}


static const ParseContainer * limitOrdLst(const Ordinal * theOrd,
        list<const ParseContainer *> *ordList) 
{
    const ParseContainer * pc = pS->limitOrdLst(theOrd,ordList);
    
    return pc ;
}


static const ParseContainer * desLimitOrdLst(const Ordinal * theOrd,
        list<const ParseContainer *> *ordList)
{
    return pS->desLimitOrdLst(theOrd,ordList);
}

const ParseContainer * ParseSemantics::limitOrdLst(const Ordinal * theOrd,
        list<const ParseContainer *> *ordList) const 
{
    list<const Ordinal *> * oLst = ParseContainer::linearList(ordList,1);
    const ParseContainer * ret = NULL ;
    if (outText()) ret = limitOrdLst(theOrd,oLst,&Ordinal::normalForm);
    if (outTeX()) ret = limitOrdLst(theOrd,oLst,&Ordinal::texNormalForm);
    assert(ret);
    return ret ;
}



const ParseContainer * ParseSemantics::limitOrdLst(
    const Ordinal *base, list<const Ordinal *>*ordList,
    const string (Ordinal::*nf)() const ) const
{
        list<const Ordinal *> * nextLev = new  list<const Ordinal *> ;
        const Ordinal * ret = &Ordinal::zero ;
        for (list <const Ordinal *>::iterator ordIter=ordList->begin();
        ordIter != ordList->end(); ++ordIter) {
            const Ordinal& o = **ordIter ;
            if (base->limitType().compare(o.maxLimitType())<=0) continue ;
            ret = &(base->limitOrd(o));
            nextLev->push_back(ret) ;
            *ParseSemantics::out << "( " << (base->*nf)() <<
                " ).limitOrd( " << (o.*nf)() << " ) = " <<
                (ret->*nf)() << "\n" ;
            pS->promptCheck();
        }
        const ParseContainer * ck = new ParseContainer(nextLev);
        
        return ck ;
}

const ParseContainer * ParseSemantics::desLimitOrdLst(int depth, int index,
    const Ordinal *base, list<const Ordinal *>*ordList,
    const string (Ordinal::*nf)() const ) const
{

    if (index >= depth) return new ParseContainer(base) ;
    const ParseContainer * contain = limitOrdLst(base,ordList,nf);
    assert(contain->type & ParseContainer::ordLst);
    list<const Ordinal *> * nextLev = contain->getList();

    bool retNow = false ;
    if (nextLev->size() < 1) {
        nextLev->push_back(&Ordinal::zero);
        retNow = true ;
    }
    if (++index >= depth) retNow = true ;
    if (retNow) return new ParseContainer(nextLev) ;
    *ParseSemantics::out << "Descending to " << index << " for " <<
        (base->*nf)() << "\n" ;
    pS->promptCheck();
    list<const Ordinal *>* retList = new list<const Ordinal *>(*nextLev);
    for (list <const Ordinal *>::iterator iterBase=nextLev->begin();
        iterBase != nextLev->end(); ++iterBase) {
            const ParseContainer * contain = desLimitOrdLst(depth,index,*iterBase,ordList,nf);
            assert(contain->type & ParseContainer::ordLst);
            retList->splice(retList->end(), *(contain->getList()));
           delete contain ;
     }
    delete nextLev ;
    return new ParseContainer(retList);
}
    


const ParseContainer * ParseSemantics::desLimitOrdLst(const Ordinal * theOrd,
        list<const ParseContainer *> *ordList) const
{
    const Ordinal * first = ParseContainer::getFirst(ordList);
    if (!first->isFinite()) {
        abortDocError();
        *ParseSemantics::out <<
            "ERROR: The first parameter of limitOrdLst is the depth of nesting.\n";
        pS->promptCheck();
        *out << "It must be a finite integer.\n";
        pS->promptCheck();
        *out << "Value set to 0.\n" ;
        pS->promptCheck();
        return new ParseContainer(&Ordinal::zero) ;
    }
    int depth = first->getImpl().get_int();
    if (depth < 1) {
        abortDocError();
        *ParseSemantics::out <<
        "ERROR: Parameter 'depth' must be > 0 and fit in a signed integer.\n" ;
        pS->promptCheck();
        *out << "Value set to 0.\n" ;
        pS->promptCheck();
        return new ParseContainer(&Ordinal::zero) ;
    }
    list<const Ordinal *> * oLst = ParseContainer::linearList(ordList,1);
    
    const ParseContainer * ret = new ParseContainer(&Ordinal::zero) ;
    if (outText()) ret = desLimitOrdLst(depth,0,theOrd,oLst,&Ordinal::normalForm);
    if (outTeX()) ret = desLimitOrdLst(depth,0,theOrd,oLst,&Ordinal::texNormalForm);
    return ret ;
}


static const ParseContainer * listLimitElts(const Ordinal * theOrd,
    list<const ParseContainer *> *ordList)
{
    int size = 0 ;
    if (ordList) if ((size=ParseContainer::size(ordList)) > 1) {
        *ParseSemantics::out <<
            "ERROR: Member function listLimitElts limited to at most 1 parameter.\n";
        abortDocError();
        pS->promptCheck();
        return  new ParseContainer(theOrd) ;
    }
    int n = 10 ;
    if (ordList) if (size == 1) {
        const Ordinal * ord = ParseContainer::getFirst(ordList);
        assert(ord);
        Int v = ord->getImpl().getFiniteTerm();
        const int lim = 10000 ;
        n  = lim ;
        if (v > lim) { *ParseSemantics::out <<
            "ERROR: Member function listLimitElts limited to listing " << lim 
            << " values.\n" ;
            abortDocError();
            pS->promptCheck();
        } else n = v.get_si() ; 

    } 
    if (pS->outText()) theOrd->listiElts(n,parserState->runChecks);
    if (pS->outTeX()) theOrd->listiEltsTex(n);
    return new ParseContainer (theOrd) ;
}

static const ParseContainer * descend(const Ordinal * theOrd,
    list<const ParseContainer *> *ordList)
{
    int lsize = 0;
    if (ordList) lsize= ParseContainer::size(ordList);
    int err = 0 ;
    int limit = 10 ;
    int index = -1 ;
    if ((lsize < 1) || (lsize > 2)) err |= 1 ;
    else  {
        const Ordinal * param1 = ParseContainer::getFirst(ordList);
        if (param1->isFinite()) index = param1->getImpl().get_int();
        else err |= 2 ;
        if (index < 0) err |= 4 ;
        if (lsize ==2) {
            const Ordinal * param2 = ParseContainer::getSecond(ordList);
            if (param2->isFinite()) limit = param2->getImpl().get_int();
            else err |= 8 ;
            if (limit < 0) err |= 0x10 ;
        }

    }
    if (err) {
        *ParseSemantics::out 
            << "ERROR: " << err <<
                " 'descend' has one or two 'int' type parameters.\n";
        abortDocError();
        pS->promptCheck();
        pS->stz();
        return new ParseContainer(&(Ordinal::zero));
    }
    *ParseSemantics::out << "index = " << index << ". limit = "<<limit<<"\n";
    pS->promptCheck();
   return new ParseContainer(theOrd->descend(index,limit));
   
}

static const ParseContainer * descendFull(const Ordinal * theOrd,
    list<const ParseContainer *> *ordList)
{
   
    int err = 0 ;
    int lsize = 0;
    if (ordList) lsize = ParseContainer::size(ordList);
    int limit = 10 ;
    int index = -1 ;
    int sel = -1 ;
    if ((lsize < 1) || (lsize > 3)) err |= 1 ;
    else  {
        const Ordinal * param1 = ParseContainer::getFirst(ordList);
        if (param1->isFinite()) index = param1->getImpl().get_int();
        else err |= 2 ;
        if (index < 0) err |= 4 ;
        if (lsize > 1) {
            const Ordinal * param2 = ParseContainer::getSecond(ordList);
            if (param2->isFinite()) limit = param2->getImpl().get_int();
            else err |= 8 ;
            if (limit < 0) err |= 0x10 ;
        }
        if (lsize == 3) {
            const Ordinal * param3 = ParseContainer::getThird(ordList);
            if (param3->isFinite()) sel = param3->getImpl().get_int();
            else err |= 0x20 ;
            if (limit < 0) err |= 0x40 ;
        }

    }
    if (err) {
        *ParseSemantics::out 
            << "ERROR: " << err <<
                " 'descendFull' has one to three 'int' type parameters.\n";
        abortDocError();
        pS->promptCheck();
        pS->stz();
        return new ParseContainer (&(Ordinal::zero));
    }
    *ParseSemantics::out << "index = " << index << ". limit = "<<limit<<"\n";
    pS->promptCheck();
   return new ParseContainer (theOrd->descendFull(index,limit,sel));
   
}

static const ParseContainer * getCompare(const Ordinal * theOrd,
    list<const ParseContainer *> *ordList)
{
    int lsize = 0;
    if (ordList) lsize= ParseContainer::size(ordList);
    if (lsize) {
        *ParseSemantics::out 
            << "ERROR: command 'getCompare' has no parameters.\n" <<
            "The parameter(s) given are ignored.\n" ;
        abortDocError();
    }
    const Ordinal  &ix = * new
        Ordinal( theOrd->getImpl().getCompareIndexCK(false)) ;
    return new ParseContainer(&ix);
}



static const ParseContainer * limitElt(const Ordinal * theOrd,
    list<const ParseContainer *> *ordList)
{
    bool err = false ;
    if (!ordList) err = true ;
    else if (ParseContainer::size(ordList) != 1) err = true ;
    if (err) {
        *ParseSemantics::out 
            << "ERROR: Member function 'limitElt' requires exactly one parameter.\n";
        abortDocError();
        pS->promptCheck();
        pS->stz();
        return new ParseContainer (&(Ordinal::zero));
    }
    const Ordinal * param = ParseContainer::getFirst(ordList);
    if (!param->isFinite()) {
        *ParseSemantics::out <<
    "ERROR: The argument to member function 'limitElt' must be an integer.\n" ;
        abortDocError();
        pS->promptCheck();
        pS->stz();
        return new ParseContainer(&(Ordinal::zero));
    }
    if (!theOrd->isLimit()) {
        *ParseSemantics::out <<
            "ERROR: The ordinal 'limitElt' is called from must be a limit.\n" ;
        abortDocError();
        pS->promptCheck();
        pS->stz();
        return new ParseContainer(&(Ordinal::zero));
    }
    Int intParam = param->getImpl().getInt();
    const Ordinal &ret = theOrd->limitElement(intParam);
    
    return new ParseContainer(&ret) ;
}



static const ParseContainer * limitOrd(const Ordinal * theOrd,
    list<const ParseContainer *> *ordList)
{
    bool err = false ;
    if (!ordList) err = true;
    else if (ParseContainer::size(ordList) != 1) err = true ;
    if (err) {
        *ParseSemantics::out <<
            "ERROR: 'limitOrd' requires exactly one parameter.\n";
        abortDocError();
        pS->promptCheck();
        pS->stz();
        return new ParseContainer(&(Ordinal::zero));
    }
    const Ordinal * param = ParseContainer::getFirst(ordList);
    if (param->isFinite()) {
        Int intParam = param->getImpl().getInt();
        
        return limitElt(theOrd, ordList);
    }
    if (param->maxLimitType().compare(theOrd->limitType())>=0) {
         *ParseSemantics::out 
            << "ERROR: The 'limitOrd' argument for " << 
            theOrd->normalForm() << " must have maxLimitType() < " <<
            theOrd->limitType().normalForm() << ".\n" ;
        abortDocError();
        pS->promptCheck();
        pS->stz();
        return new ParseContainer(&(Ordinal::zero));
    }
    return new ParseContainer(&(theOrd->limitOrd(*param)));
}

static const ParseContainer * limitType(const Ordinal * theOrd,
    list<const ParseContainer *> *ordList)
{
    if (ordList) if (ParseContainer::size(ordList)) {
        *ParseSemantics::out <<
            "ERROR: limitType takes no arguments.\n" ;
        abortDocError();
        pS->promptCheck();
        pS->stz();
        return new ParseContainer(&(Ordinal::zero));
    }
    return new ParseContainer(new Ordinal(theOrd->limitType()));
}

static const ParseContainer * maxParameter(const Ordinal * theOrd,
    list<const ParseContainer *> *ordList)
{
    if (ordList) if (ParseContainer::size(ordList)) {
        *ParseSemantics::out <<
            "ERROR: maxParameter takes no arguments.\n" ;
        abortDocError();
        pS->promptCheck();
        pS->stz();
        return new ParseContainer (&(Ordinal::zero));
    }
    return new ParseContainer(new Ordinal(theOrd->getImpl().getMaxParameter()));
}
static const ParseContainer * maxLimitType(const Ordinal * theOrd,
    list<const ParseContainer *> *ordList)
{
    if (ordList) if (ParseContainer::size(ordList)) {
        *ParseSemantics::out <<
            "ERROR: maxLimitType takes no arguments.\n" ;
        abortDocError();
        pS->promptCheck();
        pS->stz();
        return new ParseContainer (&(Ordinal::zero));
    }
    return new ParseContainer( new Ordinal(theOrd->maxLimitType()));
}





struct MemberFunctions {
    const char * name ;
    const ParseContainer *(*memberFunction) (const Ordinal *o,
        list<const ParseContainer *> *);
    const char* whatItDoes ;
};

static const MemberFunctions memberFunctions[] = {
    
    {"descend",descend,"(n,m) iteratively (up to m) take nth limit element"},
    {"descendFull",descendFull,
        "(n,m,k) iteratively (up to m) take n limit elements with root k"},
    {"getCompareIx",getCompare,"display admissible compare index"},
    {"limitElt",limitElt,"evaluates to specified finite limit element"},
    {"limitElement",limitElt,"an alias for 'limitElt'"},
    {"listLimitElts",listLimitElts,
        "lists specified (default 10) limit elements"},
    {"listElts",listLimitElts,
        "alias for listLimitElts"},
    {"limitOrd",limitOrd,
        "evaluates to specified (may be infinite) limit element"},
    {"limitType",limitType,"return limitType"},
    {"limitOrdLst",limitOrdLst,
        "apply each input from list to limitOrd and return that list"},
    {"desLimitOrdLst",desLimitOrdLst,
        "(depth, list) does limitOrdLst iteratively on all outputs depth times"}, 
    {"maxLimitType",maxLimitType,"return maxLimitType"},
    {"maxParameter",maxParameter,"return maxParameter (for debugging)"},
    {NULL,NULL}
};


static void helpMemberCmd(list<const string *> * lst)
{
    const char * text[] = {
        "Every ordinal (except 0) is the union of smaller ordinals.",
        "Every limit ordinal is the union of an infinite sequence",
        "of smaller ordinals. Member functions allow access to",
        "to these smaller ordinals.",
        "One can specify how many elements of this sequence",
        "to display or get the value of a ",
        "specific instance of the sequence.",
        "For a limit ordinal, the sequence displayed, were it",
        "extended to infinity and its union taken, ",
        "that union would equal the original ordinal.",
        "\n\nThe syntax for a member function begins with either an ordinal ",
        "name (from an assignment statement) or an ordinal expression ",
        "enclosed in parenthesis. This is followed by a dot (.) and then the ",
        "member function name and its parameters enclosed in parenthesis.",
        "The format is 'ordinal_name.memberFunction(p)' where p may",
        "be optional. Functions 'limitOrdLst' and 'desLimitOrdLst' return a list",
        "all other member functions return a scalar value. Unless specified",
        "otherwise, the returned value is that of the",
        "ordinal the function was called from.",
        NULL
    } ;
    writeFormatted(text);
    vsp();
    *ParseSemantics::out << "The member functions are:\n" ,
    pS->brTeX();
    for (const MemberFunctions * mem = memberFunctions; mem->name; mem++) {
        pS->brTeX();
        *ParseSemantics::out << ParseSemantics::sqTeX() ;
        if (ParseSemantics::TeXdocMode) {
            *ParseSemantics::out << "\\Index{" <<
                ParseSemantics::chTtyTeX(mem->name) << "}" ;
        } else *ParseSemantics::out << ParseSemantics::chTtyTeX(mem->name) ;
        *ParseSemantics::out << ParseSemantics::sqTeX(false)
            << " -- " << 
            ParseSemantics::checkTeX(mem->whatItDoes) << ".\n" ;
            
            pS->promptCheck();
    }

}


const ParseContainer * ParseSemantics::memberFunction(const ParseContainer * cordinal,
        const string * str, list<const ParseContainer * > * arguments)
{
    const Ordinal * ordinal = cordinal->getFirstOrdinal();
    for (const MemberFunctions * mf = memberFunctions; mf->name;mf++) {
        if (*str == mf->name) {
           return (mf->memberFunction(ordinal,arguments));
        }
    }
    *out << "ERROR: " << *str << " is not a member function of " <<
        ordinal->normalForm() << ".\n" ;
    abortDocError();
    pS->promptCheck();
    return new ParseContainer(ordinal) ;
}



const ParseContainer * ParseSemantics::evaluateFunction(const string * name,
   list<const ParseContainer *> * arguments)
{
    const Functions * funcs = theFunctions;
    for (; funcs->name.size(); funcs++) 
        if (*name == funcs->name) break ;
    if (funcs->name.size()) return
        new ParseContainer((*(funcs->evaluate))(arguments)) ;
    *out << "ERROR: " <<  *name << " is not a function name.\n" ;
    abortDocError();
    pS->promptCheck();
    return new ParseContainer( &(Ordinal::zero));
}


list<const string *> * ParseSemantics::makeStrList(const string * ordinal)
{
    list<const string *> *ret = new list<const string *> ();
    ret->push_back(ordinal);
    return ret ;
}

list<const string *> * ParseSemantics::makeStrList(const string * ordinal,
   const string * ordinal2)
{
    list<const string *> *ret = makeStrList(ordinal);
    ret->push_back(ordinal2);
    return ret ;
}

list<const string *> * ParseSemantics::makeQuoteList(const string * ordinal,
   const string * quote)
{
    list<const string *> *ret = makeStrList(ordinal);
    string * edited = new string(quote->substr(1,quote->length()-2)) ;
    
    ret->push_back(edited);
    return ret ;
}



list<const string *> * ParseSemantics::appendStrList(
    list<const string *> * lst, const string * ordinal)
{
    lst->push_back(ordinal);
    return lst ;
}

list<const string *> * ParseSemantics::appendQuoteList(
    list<const string *> * lst, const string * quote)
{
    string * edited = new string( quote->substr(1,quote->length()-2)) ;
    

    lst->push_back(edited);
    return lst ;
}




list<const ParseContainer *> * ParseSemantics::makeList(const Ordinal * ordinal)
{
    list<const ParseContainer *> *ret = new list<const ParseContainer *> ;

    ret->push_back(new ParseContainer(ordinal));
    return ret ;
}

list<const ParseContainer *> * ParseSemantics::makeContainList(const ParseContainer*contain) 
{
    list<const ParseContainer *> *ret = new list<const ParseContainer *> ;

    ret->push_back(contain);
    return ret ;
}



list<const ParseContainer *> * ParseSemantics::appendContainList(
    list<const ParseContainer *> * lst, const ParseContainer * cordinal)
{
    lst->push_back(cordinal);
    return lst ;
}

list<const ParseContainer *> * ParseSemantics::appendList(
    list<const ParseContainer *> * lst, const Ordinal * ordinal)
{
    lst->push_back(new ParseContainer(ordinal));
    return lst ;
}

list<const ParseContainer *> * ParseSemantics::makeParamList(
    const string * nm)
{
    const ParseContainer * c = NULL ;
    list<const ParseContainer *> *ret = new list<const ParseContainer *> ;
    list<const Ordinal *>* ordList = findListValue(nm);
    if (!ordList) {
        const Ordinal * ord = findValue(nm) ;
        if (ord) {
            c = new ParseContainer(ord) ;
        } else {
            c = new ParseContainer(&Ordinal::zero);
            *ParseSemantics::out << "ERROR: Variable '" << *nm <<
                "' is not defined. It is set to 0.\n" ;
            abortDocError();
            pS->promptCheck();
        }
    } else {
        if (ordList->size() > 1) c = new ParseContainer(ordList);
        else c = new ParseContainer(ordList->front());
    }
    ret->push_back(c);
    return ret ;
}



list<const ParseContainer *> * ParseSemantics::appendParamList(
    list<const ParseContainer *> * lst,list<const ParseContainer *> * lst2 )
{
    for (list<const ParseContainer *>::iterator iter = lst2->begin();
        iter != lst2->end();++iter) {
        lst->push_back(*iter);
    }
    return lst ;
}


const string * ParseSemantics::checkCommand(list<const string *> * lst)
{
    

     
    list<const string *>::iterator iter = lst->begin();
    const string * cmds = *iter ;
    ++iter;
    const string * param = NULL;
    if (iter != lst->end()) param = *iter ;
    lst->pop_front();
    

    for (const Commands * cmd = theCommands;cmd->name; cmd++) {
        if (cmd->name == *cmds) {
            if (param) if (cmd->options) 
            for (const CommandOptions * opts = cmd->options;
               opts->name;opts= opts->getNext()) {
               if (opts->name == *param) {
                    if (opts->type == CommandOptions::cmd) opts->action(lst);
                    else {
                        assert(opts->type == CommandOptions::scr);
                        exampleOutFile((const ExampleScripts *) opts);
                    }
                    return NULL ;
                }
            }
            cmd->action(cmd->name,lst);
            return NULL;
       }
    }
    return cmds ;
}
bool ParseSemantics::checkCmd(const string * cmds)
{

    for (const Commands * cmd = theCommands;cmd->name; cmd++) {
        if (cmd->name == *cmds) {
            cmd->action(cmd->name,NULL);
            return true;
        }
    }
    return false ;

}


bool ParseSemantics::compare(const ParseContainer * cordA, int oper,
    const ParseContainer * cordB)
{
    assert(cordA);
    assert(cordB);
    const Ordinal * ordA = cordA->getFirstOrdinal();
    const Ordinal * ordB = cordB->getFirstOrdinal();
    assert(ordA);
    assert(ordB);
    bool ret ;
    switch (oper) {
case LESS :
        ret = *ordA < *ordB ;
        break ;
case GTR :
        ret = *ordA > *ordB ;
        break;
case LEQ :
        ret = *ordA <= *ordB ;
        break ;
case GEQ :
        ret = *ordA >= *ordB ;
        break;
case EQUAL :
        ret = *ordA == *ordB ;
        break;

default:
        *out << "ERROR: Internal error 3 ParseSemantics::compare\n" ;
        abortDocError();
        pS->promptCheck();
        return false ;
    }
    *out << (ret ? "TRUE" : "FALSE") << "\n" ; 
    pS->promptCheck();
    return ret ;
}

void ParseSemantics::createTeXDocumentation(const char * fileName)
{
    assert(fileName);
    ofstream texFile(fileName);
    if (!texFile.good()) {
        cerr << "ERROR: cannot create output file '" << fileName << "'.\n" ;
        abortDocError();
        return ;
    }
    bool delState = false ;
    if (!parserState) {
        parserState = new ParserState ;
        delState = true ;
    }
    ostream *saveOut = out ;
    out = &texFile ;
    TeXdocMode = true ;
    const char *text2[] = {
        "The source code and documentation is licensed for use and",
        "and distribution under the Gnu General Public License, Version 2,",
        "June 1991. A copy of this license must be distributed with the",
        "program. It is also at:", 
        "\\href{http://www.gnu.org/licenses/gpl-2.0.html}",
        "{\\tt http://www.gnu.org/licenses/gpl-2.0.html}.",
        "The ordinal calculator source code, documentation and some",
        "executables can be downloaded from:",
        "\\href{http://www.mtnmath.com/ord}",
        "{\\tt http://www.mtnmath.com/ord} or ",
        "\\href{https://sourceforge.net/projects/ord}",
        "{\\tt https://sourceforge.net/projects/ord}.\n",

        "Most of this manual is automatically extracted from the online",
        "documentation.\n" ,
        NULL
    };
    const char ** lists[] = {purposeOrdCalc,text2,NULL};

    *out << "\\" << sectionTeX << "{Introduction}\n";
    for (const char ***txts = lists; *txts; txts++) 
        for (const char** txt = *txts; *txt;txt++) {
        *out << *txt << "\n" ;
        int len = strlen(*txt);
        assert(len > 0);
        if ((*txt)[len-1] == '\n') {
            *out << "\\addvspace{.1in}\n\n" ;
        }
    }
    
    commandHelp("help") ;

    *out << "\\" << sectionTeX <<
        "{\\Index{Ordinal}s}\\indextt{*}\\indextt{+}\\indextt{\\UseVerb{Exp}}\n";
    helpVeblenCmd(NULL);

    *out << "\\" << sectionTeX <<
        "{Predefined ordinals\\mindex{predefined ordinals}}\n";
    helpPredefinedTex(NULL);


    *out << "\\" << sectionTeX << "{Syntax}\n\\mindex{calculator syntax}\n" ;
    *out << "\\mindex{assignment statement}\n" ;
    helpSyntaxCmd(NULL);
    
    *out << "\\" << sectionTeX << "{Ordinal lists}\n\\mindex{ordinal lists}\n" ;
    helpList(NULL);
    

    *out << "\\" << sectionTeX << "{Commands}\n\\mindex{calculator commands}\n";
    *out << "\\sub" << sectionTeX << "{All commands}\n";
    helpCmdsCmd(NULL);

    *out << "\\sub" << sectionTeX <<
        "{Commands with options}\n\\mindex{calculator command options}" <<
        "\\label{SecCmdsOptions}\n";

    *out << "\n\nFollowing are the \\MIndex{commands with options}.\\\\\n" ;
    outTeXcmdsWithOpts();

    *out << "\\" <<  sectionTeX <<
        "{Member functions}\n\\mindex{member functions}\n";
    helpMemberCmd(NULL);

    *out << "\\" << sectionTeX << 
        "{Comparison operators}\n\\mindex{comparison operators}\\mindex{$<$}\n";
    *out << "\\mindex{$>$}\\mindex{$<=$}\\mindex{$>=$}\\mindex{$==$}\n" ;
    helpBooleanCmd(NULL);

    *out << "\\" << sectionTeX << "{Examples}\n\\mindex{calculator examples}\n" ;
    showExampleCmd("examples",NULL);

    for (const ExampleScripts * opts = exampleOptions; opts->name ;
        opts= (const ExampleScripts *)opts->getNext()){
        exampleOutFile(opts);
    }

    out = saveOut ;
    TeXdocMode=false ;
    if (delState) {
        delete parserState ;
        parserState = NULL ;
    }
}

const char * ParseSemantics::escTeXscr(char c)
{
    static char buf[32] ;
    if (!TeXdocMode) return NULL;
    if (c ==126) {
        buf[0]= '\\' ;
        buf[1]= '{' ;
        buf[2] = '\0' ;
        return buf ;
    }
    if (c ==127) {
        buf[0]= '\\' ;
        buf[1]= '}' ;
        buf[2] = '\0' ;
        return buf ;
    }

    if ((c == '_') || (c == '%')) {
        buf[0] = '\\' ;
        buf[1] = c ;
        buf[2] = '\0' ;
        return buf ;
    }
    if (c == '^') {
        strcpy(buf,"\\UseVerb{Exp}");
        return buf ;
    }
    return NULL ;
}


const char * ParseSemantics::escTeX(char c)
{
    static char buf[32] ;
    if (!TeXdocMode) return NULL;
    static const char *requireEscape = "#$%&_{}";
    static const char *requireVerb = "~^\\" ;
    static const char *requireMath = "<>" ;
    
    if (c == '\'') {
        if (((++formatSq)&1)) {
            int ix = 0 ;
            buf[ix++] = '`' ;
            buf[ix++] ='\0' ;
            return buf ;
        }
        return NULL ;
    }
    if (c == '"') {
        char use = '`' ;
        if (!((++formatDq)&1)) use = '\'' ;
        int ix = 0 ;
        buf[ix++] = use ;
        buf[ix++] = use ;
        buf[ix++] ='\0' ;
        return buf ;
    }
    for (const char * pt = requireEscape; *pt; pt++) {
        if (*pt == c) {
            int ix=0 ;
            buf[ix++]='\\' ;
            buf[ix++]=c ;
            buf[ix++]= '\0' ;
            return buf ;
        }
    }
    for (const char * pt = requireVerb; *pt; pt++) {
        if (*pt == c) {
            if (*pt == '^') {
                strcpy(buf,"\\UseVerb{Exp}") ;
                return buf ;
            }
            strcpy(buf,"\\verb\"");
            int ix = strlen(buf);
            buf[ix++]=c ;
            buf[ix++]= '"' ;
            buf[ix++]= '\0' ;
            return buf;
        }
    }
    for (const char * pt = requireMath; *pt; pt++) {
        if (*pt == c) {
            int ix = 0 ;
            buf[ix++] = '$' ;
            buf[ix++]=c ;
            buf[ix++] = '$' ;
            buf[ix++]= '\0' ;
            return buf;
        }
    }

    return NULL ;
}

const char *ParseSemantics::sqTeX(bool left)
{
    static const char * sqr = "'" ;
    static const char * sql = "`" ;

    if (!TeXdocMode) return sqr ;
    const char * use = sqr ;
    if (left) use = sql ;
    return use ;
}

const char *ParseSemantics::dqTeX(bool left)
{
    static const char * dqr = "''" ;
    static const char * dql = "``" ;
    static const char * dq = "\"" ;

    if (!TeXdocMode) return dq ;
    const char * use = dqr ;
    if (left) use = dql ;
    return use ;
}


void ParseSemantics::outTeXcmdsWithOpts()
{
    for (const Commands * cmds = theCommands; cmds->name ; cmds++)
        if (cmds->options) {
        if (!strcmp("help",cmds->name)) continue ;
        *out << "\n\\vspace{.1in}\n" ;
        
        
        list<const string *> lst ;
        lst.push_back(new string(cmds->name));
        helpCmdOpt("help",&lst);
    }
}

string * ord::intToString(int v)
{
    std::stringstream out;
    out << v;
    string * s = new string(out.str());
    return s ;
}
 
string ParseSemantics::defaultSaveFile = "ord_calc.ord";
string ParseSemantics::defaultExportTexFile = "ord_calc.tex";
ostream * ParseSemantics::out = NULL ;
string ParseSemantics::defaultLogFile = "log.ord" ;
ofstream * ParseSemantics::logFileStream = NULL ;

bool ParseSemantics::TeXdocMode = false ;
int ParseSemantics::formatBufLength = 0;
int ParseSemantics::formatCharLength = 0;
int ParseSemantics::formatCharIndex = 0;
int ParseSemantics::formatBufIndex = 0;
char ParseSemantics::formatBuf[formatBufSize+1] ;
int ParseSemantics::formatSq = 0 ;
int ParseSemantics::formatDq = 0;
const char * ParseSemantics::sectionTeX = "section" ;
