/* codec_jpeg.c
 * Copyright (C) 2001 QT4Linux and OpenQuicktime Teams
 *
 * This file is part of OpenQuicktime, a free QuickTime library.
 *
 * Based on QT4Linux by Adam Williams.
 *
 * OpenQuicktime is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation.
 *
 * OpenQuicktime is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <jpeglib.h>

#include "openquicktime.h"
#include "colormodels.h"
#include "private.h"

#ifdef __cplusplus
extern "C" {
#endif

/*************************************************************/
/* Some Global Variables */
/*************************************************************/
static int              Initialised   = 0;      
  /* Initialising counter */

static char             Signature[5]  = "jpeg"; 
  /* Codec Signature */

#define JPEG_DEFAULT_QUALITY 85

/* Each instance of the codec should have its own Param structure allocated */
typedef struct {
  /* Error structure */
  struct jpeg_error_mgr jerr;
  /* Working buffer */
  unsigned char **line[3];
  /* quality */
  int quality;
  /* decompression structure */
  struct jpeg_decompress_struct decinfo;
  struct jpeg_source_mgr jsrc;

  /* compression structure */
  struct jpeg_compress_struct encinfo;
  struct jpeg_destination_mgr jdest;

} Param;  

int set_param_JPEG (quicktime_t* file,
		    int          track,
		    const char*  param,
		    const void*  data)
{
  Param *p = (Param*)(((quicktime_codec_t*)(file->vtracks[track].codec))->priv);

  if (!strcmp(param,"quality")) {
    p->quality = *(( int*)data);
    jpeg_set_quality(&p->encinfo, p->quality, TRUE);

    return 0; 
  }

  fprintf(stderr,"set_param_JPEG: unknown parameter named '%s'\n",param);
  return 1;
}

int get_param_JPEG (quicktime_t* file,
		    int          track,
		    const char*  param,
		    void*        data)
{
  Param *p = (Param*)(((quicktime_codec_t*)(file->vtracks[track].codec))->priv);

  if (!strcmp(param,"quality")) { *((int*)data) = p->quality; return QUICKTIME_INTEGER_PARAMETER; }
  
  fprintf(stderr,"set_param_JPEG: unknown parameter named '%s'\n",param);
  return 1;
}


static int delete_JPEG(quicktime_video_map_t *vtrack)
{
  Param *p = (Param*)(((quicktime_codec_t*)(vtrack->codec))->priv);

  printf("Deleting external video codec %s\n", Signature);
  
  if(p)
    {
      free(p);
    }
  printf("External video codec %s deleted\n", Signature);

  return --Initialised;
}

/*************************************************************/
/* Decoding function
/*************************************************************/
static int decode_JPEG(quicktime_t *file,
		       int track,
		       unsigned long inputsize,
		       unsigned char *input,
		       unsigned char **output)
{
  int width, width2, height;
  int i, j, k, r_h, r_v;
  unsigned char *base[3];

  Param *p = (Param*)(((quicktime_codec_t*)(file->vtracks[track].codec))->priv);

  base[0] = output[0];
  base[1] = output[1];
  base[2] = output[2];

  switch(file->color_model) {
  case BC_YUV420P:
    p->jsrc.next_input_byte = input;
    p->jsrc.bytes_in_buffer = inputsize;
    
    jpeg_read_header(&p->decinfo, TRUE);
    
    r_h = p->decinfo.cur_comp_info[0]->h_samp_factor;
    r_v = p->decinfo.cur_comp_info[0]->v_samp_factor;
    
    p->decinfo.do_fancy_upsampling = FALSE;
    p->decinfo.do_block_smoothing  = FALSE;
    p->decinfo.out_color_space     = JCS_YCbCr;
    p->decinfo.dct_method          = JDCT_IFAST;
    p->decinfo.raw_data_out        = TRUE;
    
    jpeg_start_decompress(&p->decinfo);
    width = p->decinfo.output_width;
    height = p->decinfo.output_height;
    width2 = width >> 1;
    
    for (i = 0; i < height; i += r_v*DCTSIZE) {
      for (j=0, k=0; j< (r_v*DCTSIZE); j += r_v, k++) {
	p->line[0][j]   = base[0]; base[0] += width;
	if (r_v == 2) {
	  p->line[0][j+1] = base[0]; base[0] += width;
	}
	p->line[1][k]   = base[1]; 
	p->line[2][k]   = base[2];
	if (r_v == 2 || k&1) {
	  base[1] += width2; base[2] += width2;
	}
      }
      jpeg_read_raw_data(&p->decinfo, p->line, r_v*DCTSIZE);
    }
    
    jpeg_finish_decompress(&p->decinfo);
    break;
  case BC_NONE:
    break;
  }

  return 0;
}


/*************************************************************/
/* Encoding function
/*************************************************************/
static int encode_JPEG(quicktime_t *file, 
		       int track,
		       unsigned char **input,
		       unsigned char *output,
		       int *IsAKeyFrame)
{
  Param *p = (Param*)(((quicktime_codec_t*)(file->vtracks[track].codec))->priv);
  quicktime_video_map_t *vtrack = &(file->vtracks[track]);
  int width2, i, j, k;
  int width 	= vtrack->track->tkhd.track_width;
  int height 	= vtrack->track->tkhd.track_height;
  unsigned char *base[3];

  int outsize = width * height * 4;
 
  p->jdest.next_output_byte = output;

  p->jdest.free_in_buffer = outsize;
  jpeg_start_compress(&p->encinfo, TRUE);
  width2 = width>>1;

  base[0] = input[0];
  base[1] = input[1];
  base[2] = input[2];

  for (i = 0; i < height; i += 2*DCTSIZE) {
    for (j=0, k=0; j<2*DCTSIZE;j+=2, k++) {

      p->line[0][j]   = base[0]; base[0] += width;
      p->line[0][j+1] = base[0]; base[0] += width;
      p->line[1][k]   = base[1]; base[1] += width2;
      p->line[2][k]   = base[2]; base[2] += width2;
    }
    jpeg_write_raw_data(&p->encinfo, p->line, 2*DCTSIZE);
  }
  jpeg_finish_compress(&p->encinfo);


  *IsAKeyFrame = 1;

  return (((outsize - p->jdest.free_in_buffer)+3)&~3);
}

static int reads_colormodel_JPEG(quicktime_t *file, 
		int colormodel, 
		int track)
{
  return ( colormodel == BC_YUV420P );
}

static int writes_colormodel_JPEG(quicktime_t *file, 
		int colormodel, 
		int track)
{ 
  return ( colormodel == BC_YUV420P );
}

/* Dummy decoding functions */
static void jpegdec_init_source (j_decompress_ptr cinfo){}
static int jpegdec_fill_input_buffer (j_decompress_ptr cinfo){return 1;}
static void jpegdec_skip_input_data (j_decompress_ptr cinfo, long num_bytes){}
static int jpegdec_resync_to_restart (j_decompress_ptr cinfo, int desired){return 1;}
static void jpegdec_term_source (j_decompress_ptr cinfo){}
/* Dummy encoding functions */
static void jpegenc_init_destination (j_compress_ptr cinfo){}
static int jpegenc_flush_destination (j_compress_ptr cinfo){return 1;}
static void jpegenc_term_destination (j_compress_ptr cinfo){}


void *init_JPEG(quicktime_video_map_t *vtrack)
{
  /* Here we go, now we support as many initializatin as you want ;)  */
  Param *p = NULL;  
  int height = vtrack->track->tkhd.track_height;
  int width = vtrack->track->tkhd.track_width;

  Initialised++;
        
  /* Init public items, should be called just once */
  printf("Initializing new instance of the PHOTO_JPEG video codec\n");

  /* Init params, should be called for every instance */
  p = (Param*) malloc(sizeof(Param));

  /* setup jpeglib */
  memset(p, 0, sizeof(Param));
  p->decinfo.err = p->encinfo.err = jpeg_std_error(&p->jerr);

  p->line[0] = malloc(height*sizeof(char*));
  p->line[1] = malloc(height*sizeof(char*)/2);
  p->line[2] = malloc(height*sizeof(char*)/2);
  
  /* Decoder init */
  jpeg_create_decompress(&p->decinfo);
  p->jsrc.init_source       = jpegdec_init_source;
  p->jsrc.fill_input_buffer = jpegdec_fill_input_buffer;
  p->jsrc.skip_input_data   = jpegdec_skip_input_data;
  p->jsrc.resync_to_restart = jpegdec_resync_to_restart;
  p->jsrc.term_source       = jpegdec_term_source;
  p->decinfo.src = &p->jsrc;
  
  /* Encoder init */
  jpeg_create_compress(&p->encinfo);

  p->encinfo.image_width = width;
  p->encinfo.image_height = height;

  p->encinfo.input_components = 3;
  jpeg_set_defaults(&p->encinfo);
  p->encinfo.dct_method = JDCT_FASTEST;
  jpeg_set_quality(&p->encinfo, JPEG_DEFAULT_QUALITY, TRUE);
  p->quality = JPEG_DEFAULT_QUALITY;

  p->encinfo.raw_data_in = TRUE;
  p->encinfo.in_color_space = JCS_YCbCr;

  p->encinfo.comp_info[0].h_samp_factor = 2;
  p->encinfo.comp_info[0].v_samp_factor = 2;
  p->encinfo.comp_info[1].h_samp_factor = 1;
  p->encinfo.comp_info[1].v_samp_factor = 1;
  p->encinfo.comp_info[2].h_samp_factor = 1;
  p->encinfo.comp_info[2].v_samp_factor = 1;

  jpeg_suppress_tables(&p->encinfo, TRUE);

  p->jdest.init_destination    = jpegenc_init_destination;
  p->jdest.empty_output_buffer = jpegenc_flush_destination;
  p->jdest.term_destination    = jpegenc_term_destination;
  p->encinfo.dest              = &p->jdest;
  return p;
}


/*************************************************************/
/* Registering function 
/*************************************************************/
int quicktime_codec_register(quicktime_extern_video_t *codec)
{
  strcpy(codec->fourcc,Signature);
  printf("Registering video codec %s\n", codec->fourcc);
  codec->init   = init_JPEG;
  codec->encode = encode_JPEG;
  codec->decode = decode_JPEG;
  codec->delete_codec = delete_JPEG;
  codec->reads_colormodel = reads_colormodel_JPEG;
  codec->writes_colormodel = writes_colormodel_JPEG;
  codec->set_param = set_param_JPEG;
  codec->get_param = get_param_JPEG;

  return 1;
}
