//
// $Id: HashSetTest.m,v 1.21 2007/03/28 03:16:52 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "HashSetTest.h"
#import "Random.h"
#import <ObjectiveLib/HashSet.h>
#import <ObjectiveLib/Vector.h>
#import <ObjectiveLib/DataOutStream.h>
#import <ObjectiveLib/DataInStream.h>
#import <ObjectiveLib/ObjectOutStream.h>
#import <ObjectiveLib/ObjectInStream.h>
#import "Number.h"
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Text.h>
#import <ObjectiveLib/Reaper.h>
#else
#import <Foundation/NSString.h>
#import <Foundation/NSValue.h>
#import <Foundation/NSData.h>
#import <Foundation/NSArchiver.h>
#if defined(HAVE_KEYED_ARCHIVES)
#import <Foundation/NSKeyedArchiver.h>
#endif
#endif
#if defined(__NEXT_RUNTIME__)
#import <objc/objc-class.h>
#endif
#include <stdlib.h>

@implementation HashSetTest

#if !defined(OL_NO_OPENSTEP)
- (void) testCoding
{
    NSMutableData* data;
    NSArchiver* archiver;
    NSData* archData;
    OLHashSet* s1;
    OLHashSet* s2;
    OLNumber* v;
    int i;


    s1 = [[OLHashSet alloc] init];
    for (i = 0; i< 1000; i++)
    {
        v = [[OLNumber alloc] initWithInt: i];
        [s1 insert: v];
        [v RELEASE];
    }
    data = [[NSMutableData alloc] initWithCapacity: 1000];
    archiver = [[NSArchiver alloc] initForWritingWithMutableData: data];
    [archiver encodeRootObject: s1];
    [archiver RELEASE];
    s2 = [NSUnarchiver unarchiveObjectWithData: data];
    [data RELEASE];
    if (![s1 isEqual: s2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The hash sets should be equal"];
    }
#if defined(HAVE_KEYED_ARCHIVES)
    archData = [NSKeyedArchiver archivedDataWithRootObject: s1];
    s2 = [NSKeyedUnarchiver unarchiveObjectWithData: archData];
    if (![s1 isEqual: s2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The hash sets should be equal"];
    }
#endif
    [s1 RELEASE];
}
#endif

- (void) testConvenienceAllocators
{
    OLHashSet* s;
    OLHashSet* s2;
    OLNumber* num;

    s = REAP([OLHashSet hashSet]);
    if (![s IS_MEMBER_OF: [OLHashSet class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLHashSet class])->name, ((Class)[s class])->name];
    }
    if (![s empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The map should be empty"];
    }
    num = [[OLNumber alloc] initWithInt: 1];
    REAP([s insert: num]);
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 2];
    REAP([s insert: num]);
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 3];
    REAP([s insert: num]);
    [num RELEASE];
    s2 = REAP([OLHashSet hashSetFrom: REAP([s begin]) to: REAP([s end])]);
    if (![s isEqual: s2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The maps should be equal"];
    }
    num = [[OLNumber alloc] initWithInt: 4];
    REAP([s insert: num]);
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 5];
    REAP([s insert: num]);
    [num RELEASE];
    s2 = REAP([OLHashSet hashSetWithHashSet: s]);
    if (![s isEqual: s2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The maps should be equal"];
    }
}

- (void) testEqualRange
{
    OLHashSet* s;
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    OLPair* p;
    OLHashIterator* end;
    int i;

    p = REAP([OLArrayIterator pairWithPointer: strs andPointer: strs + 5]);
    s = [[OLHashSet alloc] initFrom: [p first] to: [p second]];
    end = REAP([s end]);
    p = REAP([s equalRange: strs[1]]);
    if ([[p first] isEqual: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should have been found", [strs[1] cString]];
    }
    if (![[[p first] dereference] isEqual: strs[1]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got\"%s\"", [strs[1] cString]];
    }
    i = [OLIterator distanceFrom: [p first] to: [p second]];
    if (i != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", i];
    }
    p = REAP([s equalRange: @"doggy"]);
    if (![[p first] isEqual: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"joe\" should have been found"];
    }
    if (![[p second] isEqual: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"joe\" should have been found"];
    }
    [s RELEASE];
}

- (void) testErase
{
    OLHashSet* s;
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    OLPair* p;
    OLHashIterator* end;
    OLHashIterator* itor;
    int i;
    unsigned erased;

    p = REAP([OLArrayIterator pairWithPointer: strs andPointer: strs + 5]);
    s = [[OLHashSet alloc] initFrom: [p first] to: [p second]];
    end = REAP([s end]);
    for (i = 0; i < 5; i++)
    {
        itor = REAP([s find: strs[i]]);
        if ([itor isEqual: end])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "\"%s\" should have been found", [strs[i] cString]];
        }
        [s erase: itor];
        itor = REAP([s find: strs[i]]);
        if (![itor isEqual: end])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "\"%s\" should not have been found", [strs[i] cString]];
        }
        if ([s size] != 4 - i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", 4 - i, [s size]];
        }
    }
    [s insertFrom: [p first] to: [p second]];
    erased = [s eraseValue: strs[0]];
    if (erased != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", erased];
    }
    erased = [s eraseValue: strs[0]];
    if (erased != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", erased];
    }
    [s clear];
    if ([s size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", [s size]];
    }
    [s RELEASE];
}

- (void) testFind
{
    OLHashSet* s;
    OLNumber* nums[10000];
    OLPair* p;
    OLHashIterator* itor;
    unsigned count;
    int i;

    for (i = 0; i < 10000; i++)
        nums[i] = [[OLNumber alloc] initWithInt: i];
    p = REAP([OLArrayIterator pairWithPointer: nums andPointer: nums + 10000]);
    s = [[OLHashSet alloc] initFrom: [p first] to: [p second]];
    itor = REAP([s find: nums[7574]]);
    if ([itor isEqual: REAP([s end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "7574 should have been found"];
    }
    count = [s count: nums[9935]];
    if (count != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", count];
    }
    [s erase: itor];
    itor = REAP([s find: nums[7574]]);
    if (![itor isEqual: REAP([s end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "7574 should not have been found"];
    }
    count = [s count: nums[7574]];
    if (count != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", count];
    }
    [s RELEASE];
    for (i = 0; i < 10000; i++)
        [nums[i] RELEASE];
}

- (void) testInsert
{
    OLHashSet* s;
    OLPair* p;
    OLNumber* num;
    OLText* strs[5000];
    int i;

    s = [[OLHashSet alloc] init];
    p = REAP([s insert: @"one"]);
    if (![[p second] boolValue])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The insert should have been successful"];
    }
    if (![[[p first] dereference] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The expected \"one\", but got \"%s\"",
            [[[p first] dereference] cString]];
    }
    p = REAP([s insert: @"one"]);
    if ([[p second] boolValue])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The insert should not have been successful"];
    }
    if (![[[p first] dereference] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The expected \"one\", but got \"%s\"",
            [[[p first] dereference] cString]];
    }
    for (i = 0; i < 5000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        strs[i] = REAP([num toText]);
        [num RELEASE];
    }
    p = REAP([OLArrayIterator pairWithPointer: strs andPointer: strs + 5000]);
    [s insertFrom: [p first] to: [p second]];
    if ([s size] != 5001)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5001, but got %u", [s size]];
    }
    for (i = 0; i < 5000; i++)
    {
        if ([s count: strs[i]] != 1)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but didn't find it",
                [strs[i] cString]];
        }
    }
    [s RELEASE];
}

- (void) testInitializers
{
    OLHashSet* s;
    OLHashSet* s2;
    OLVector* v;
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    OLHashIterator* begin;
    OLHashIterator* end;
    int i;

    s = [[OLHashSet alloc] init];
    if ([s size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The expected 0, but got %u", [s size]];
    }
    [s RELEASE];
    v = [[OLVector alloc] init];
    for (i = 0; i < 5; i++)
        [v pushBack: strs[i]];
    s = [[OLHashSet alloc] initFrom: REAP([v begin]) to: REAP([v end])];
    [v RELEASE];
    [self logMessage: "The set contains:"];
    for (begin = REAP([s begin]), end = REAP([s end]); ![begin isEqual: end]; [begin advance])
        [self logMessage: "%s", [[begin dereference] cString]];
    for (i = 0; i < 5; i++)
    {
        if ([s count: strs[i]] != 1)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "\"%s\" was not found", [strs[i] cString]];
        }
    }
    s2 = [[OLHashSet alloc] initWithHashSet: s];
    if ([s2 size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The expected 5, but got %u", [s2 size]];
    }
    for (i = 0; i < 5; i++)
    {
        if ([s2 count: strs[i]] != 1)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "\"%s\" was not found", [strs[i] cString]];
        }
    }
    [s RELEASE];
    [s2 RELEASE];
}

- (void) testIsEqual
{
    OLHashSet* s;
    OLHashSet* s2;
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    OLPair* p;

    p = REAP([OLArrayIterator pairWithPointer: strs andPointer: strs + 5]);
    s = [[OLHashSet alloc] initFrom: [p first] to: [p second]];
    s2 = [[OLHashSet alloc] initFrom: [p first] to: [p second]];
    if (![s isEqual: s2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sets should be equal"];
    }
    [s2 RELEASE];
    s2 = [s copy];
    if (![s isEqual: s2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sets should be equal"];
    }
    [s2 erase: REAP([s2 find: strs[0]])];
    if ([s isEqual: s2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sets should not be equal"];
    }
    p = REAP([s2 insert: @"joe"]);
    if ([s isEqual: s2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sets should be equal"];
    }
    [s2 erase: [p first]];
    REAP([s2 insert: strs[0]]);
    if (![s isEqual: s2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sets should be equal"];
    }
    [s RELEASE];
    [s2 RELEASE];
}

- (void) testIterators
{
    OLHashSet* s;
    OLNumber* nums[10000];
    OLPair* p;
    OLHashIterator* begin;
    OLHashIterator* end;
    int foundNums[10000];
    int found;
    int i;

    s = [[OLHashSet alloc] init];
    [self logMessage: "Allocating numbers"];
    for (i = 0; i < 10000; i++)
        nums[i] = [[OLNumber alloc] initWithInt: i];
    [self logMessage: "Done"];
    p = REAP([OLArrayIterator pairWithPointer: nums andPointer: nums + 10000]);
    [self logMessage: "Inserting"];
    [s insertFrom: [p first] to: [p second]];
    [self logMessage: "Done"];
    [self logMessage: "Releasing numbers"];
    for (i = 0; i < 10000; i++)
    {
        [nums[i] RELEASE];
        foundNums[i] = -1;
    }
    [self logMessage: "Done"];
    [self logMessage: "Traversing set"];
    for (begin = REAP([s begin]), end = REAP([s end]); ![begin isEqual: end]; [begin advance])
    {
        found = [[begin dereference] intValue];
        if (found < 0 || found >= 10000)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected a number from 0 to 9999, but got %i", found];
        }
        if (foundNums[found] != -1)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Found %i more than once in the collection", found];
        }
        foundNums[found] = found;
    }
    [self logMessage: "Done"];
    [self logMessage: "Verifying results"];
    for (i = 0; i < 10000; i++)
    {
        if (foundNums[i] == -1)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The number %i was not iterated over", foundNums[i]];
        }
    }
    [self logMessage: "Done"];
    [self logMessage: "Releasing set"];
    [s RELEASE];
    [self logMessage: "Done"];
}

- (void) testProperties
{
    OLHashSet* s;
    OLHashSet* s2;
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    OLPair* p;

    s = [[OLHashSet alloc] init];
    if (![s empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The set is empty"];
    }
    p = REAP([OLArrayIterator pairWithPointer: strs andPointer: strs + 5]);
    [s insertFrom: [p first] to: [p second]];
    if ([s empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The set is not empty"];
    }
    if ([[s keyEqual] class] != [OLEqualTo class])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"OLEqualTo\" but got \"%s\"", ((Class)[[s keyEqual] class])->name];
    }
    s2 = [[OLHashSet alloc] init];
    if ([s maxSize] != [s2 maxSize])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "All hash sets must have the same max size"];
    }
    [self logMessage: "The max size of a hash set is %u", [s maxSize]];
    [s RELEASE];
    [s2 RELEASE];
}

- (void) testStreaming
{
    OLDataOutStream* dout;
    OLObjectOutStream* oout;
    OLObjectInStream* oin;
    OLHashSet* set;
    OLHashSet* readSet;
    OLNumber* v;
    int i;

    dout = REAP([OLDataOutStream stream]);
    oout = REAP([OLObjectOutStream streamWithOutStream: dout]);
    set = [[OLHashSet alloc] init];
    for (i = 0; i< 10000; i++)
    {
        v = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
        REAP([set insert: v]);
        [v RELEASE];
    }
    [oout writeObject: set];
    oin = REAP([OLObjectInStream streamWithInStream:
        REAP([OLDataInStream streamWithBytes: [dout bytes] count: [dout count]])]);
    readSet = REAP([oin readObject]);
    if (![readSet IS_MEMBER_OF: [OLHashSet class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected OLHashSet class, but got %s", ((Class)[readSet class])->name];
    }
    if (![readSet isEqual: set])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The hash sets should be equal"];
    }
    [set RELEASE];
}

- (void) testSwap
{
    OLHashSet* s;
    OLHashSet* s2;
    OLHashSet* s3;
    OLHashSet* s4;
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    CONSTSTR* strs2[] = { @"six", @"seven" };
    OLPair* p;

    p = REAP([OLArrayIterator pairWithPointer: strs andPointer: strs + 5]);
    s = [[OLHashSet alloc] initFrom: [p first] to: [p second]];
    p = REAP([OLArrayIterator pairWithPointer: strs2 andPointer: strs2 + 2]);
    s2 = [[OLHashSet alloc] initFrom: [p first] to: [p second]];
    s3 = [s copy];
    s4 = [s2 copy];
    [s swap: s2];
    if (![s isEqual: s4])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sets should be equal"];
    }
    if (![s2 isEqual: s3])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sets should be equal"];
    }
    [s RELEASE];
    [s2 RELEASE];
    [s3 RELEASE];
    [s4 RELEASE];
}

@end
