//
// $Id: HashMapTest.m,v 1.18 2007/03/28 03:16:52 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "HashMapTest.h"
#import "Random.h"
#import <ObjectiveLib/HashMap.h>
#import <ObjectiveLib/Vector.h>
#import <ObjectiveLib/DataOutStream.h>
#import <ObjectiveLib/DataInStream.h>
#import <ObjectiveLib/ObjectOutStream.h>
#import <ObjectiveLib/ObjectInStream.h>
#import "Number.h"
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Text.h>
#import <ObjectiveLib/Reaper.h>
#else
#import <Foundation/NSString.h>
#import <Foundation/NSValue.h>
#import <Foundation/NSData.h>
#import <Foundation/NSArchiver.h>
#if defined(HAVE_KEYED_ARCHIVES)
#import <Foundation/NSKeyedArchiver.h>
#endif
#endif
#include <stdlib.h>
#if defined(__NEXT_RUNTIME__)
#import <objc/objc-class.h>
#else
#import <objc/objc.h>
#endif

@implementation HashMapTest

- (void) testAssignKey
{
    OLHashMap* m;
    OLHashIterator* b;

    m = [[OLHashMap alloc] init];
    [m assignKey: @"doggy" value: @"boy"];
    if ([m size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", [m size]];
    }
    b = REAP([m begin]);
    if (![[[b dereference] first] isEqual: @"doggy"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"doggy\", but got \"%s\"",
            [[[b dereference] first] cString]];
    }
    if (![[[b dereference] second] isEqual: @"boy"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"boy\", but got \"%s\"",
            [[[b dereference] second] cString]];
    }
    [m assignKey: @"doggy" value: @"chunk"];
    if ([m size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", [m size]];
    }
    b = REAP([m begin]);
    if (![[[b dereference] first] isEqual: @"doggy"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"doggy\", but got \"%s\"",
            [[[b dereference] first] cString]];
    }
    if (![[[b dereference] second] isEqual: @"chunk"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"chunk\", but got \"%s\"",
            [[[b dereference] second] cString]];
    }
    [m RELEASE];
}

#if !defined(OL_NO_OPENSTEP)
- (void) testCoding
{
    NSMutableData* data;
    NSArchiver* archiver;
    NSData* archData;
    OLHashMap* m1;
    OLHashMap* m2;
    OLNumber* k;
    OLNumber* v;
    int i;


    m1 = [[OLHashMap alloc] init];
    [self logMessage: "Building hash map"];
    for (i = 0; i< 10000; i++)
    {
        k = [[OLNumber alloc] initWithInt: i];
        v = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
        [m1 assignKey: k value: v];
        [k RELEASE];
        [v RELEASE];
    }
    data = [[NSMutableData alloc] initWithCapacity: 10000];
    archiver = [[NSArchiver alloc] initForWritingWithMutableData: data];
    [self logMessage: "Encoding hash map"];
    [archiver encodeRootObject: m1];
    [archiver RELEASE];
    [self logMessage: "Decoding hash map"];
    m2 = [NSUnarchiver unarchiveObjectWithData: data];
    [data RELEASE];
    if (![m1 isEqual: m2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The hash maps should be equal"];
    }
#if defined(HAVE_KEYED_ARCHIVES)
    [self logMessage: "Encoding hash map with keys"];
    archData = [NSKeyedArchiver archivedDataWithRootObject: m1];
    [self logMessage: "Decoding hash map with keys"];
    m2 = [NSKeyedUnarchiver unarchiveObjectWithData: archData];
    if (![m1 isEqual: m2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The hash maps should be equal"];
    }
#endif
    [m1 RELEASE];
    [self logMessage: "Done"];
}
#endif

- (void) testConvenienceAllocators
{
    OLHashMap* m;
    OLHashMap* m2;
    OLNumber* num;

    m = REAP([OLHashMap hashMap]);
    if (![m IS_MEMBER_OF: [OLHashMap class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLHashMap class])->name, ((Class)[m class])->name];
    }
    if (![m empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The map should be empty"];
    }
    num = [[OLNumber alloc] initWithInt: 1];
    [m assignKey: num value: REAP([num toText])];
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 2];
    [m assignKey: num value: REAP([num toText])];
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 3];
    [m assignKey: num value: REAP([num toText])];
    [num RELEASE];
    m2 = REAP([OLHashMap hashMapFrom: REAP([m begin]) to: REAP([m end])]);
    if (![m isEqual: m2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The maps should be equal"];
    }
    num = [[OLNumber alloc] initWithInt: 4];
    [m assignKey: num value: REAP([num toText])];
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 5];
    [m assignKey: num value: REAP([num toText])];
    [num RELEASE];
    m2 = REAP([OLHashMap hashMapWithHashMap: m]);
    if (![m isEqual: m2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The maps should be equal"];
    }
}

- (void) testEqualRange
{
    OLHashMap* m;
    OLPair* p;
    unsigned dist;

    m = [[OLHashMap alloc] init];
    [m assignKey: @"doggy" value: @"boy"];
    p = REAP([m equalRange: @"doggy"]);
    dist = [OLIterator distanceFrom: [p first] to: [p second]];
    if (dist != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", dist];
    }
    if (![[[[p first] dereference] first] isEqual: @"doggy"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"doggy\", but got \"%s\"",
            [[[[p first] dereference] first] cString]];
    }
    if (![[[[p first] dereference] second] isEqual: @"boy"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"boy\", but got \"%s\"",
            [[[[p first] dereference] second] cString]];
    }
    p = REAP([m equalRange: @"chunks"]);
    dist = [OLIterator distanceFrom: [p first] to: [p second]];
    if (dist != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", dist];
    }
    if (![[p first] isEqual: REAP([m end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The key should not have been found"];
    }
    if (![[p second] isEqual: REAP([m end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The key should not have been found"];
    }
    [m RELEASE];
}

- (void) testErase
{
    OLHashMap* m;
    CONSTSTR* vals[] = { @"one", @"two", @"three", @"four", @"five" };
    OLNumber* key;
    OLHashIterator* b;
    unsigned count;
    int i;

    m = [[OLHashMap alloc] init];
    key = [[OLNumber alloc] initWithInt: 0];
    REAP([m insertKey: key value: vals[0]]);
    [key RELEASE];
    [m erase: REAP([m begin])];
    if ([m size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", [m size]];
    }
    for (i = 0; i < 5; i++)
    {
        key = [[OLNumber alloc] initWithInt: i];
        REAP([m insertKey: key value: vals[i]]);
        [key RELEASE];
    }
    [m eraseFrom: [REAP([m begin]) advance] to: REAP([m end])];
    if ([m size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", [m size]];
    }
    b = REAP([m begin]);
    if ([[[b dereference] first] intValue] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", [[[b dereference] first] intValue]];
    }
    if (![[[b dereference] second] isEqual: vals[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"",
            [[[b dereference] second] cString]];
    }
    count = [m eraseKey: [[b dereference] first]];
    if ([m size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", [m size]];
    }
    if (count != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", count];
    }
    for (i = 0; i < 5; i++)
    {
        key = [[OLNumber alloc] initWithInt: i];
        REAP([m insertKey: key value: vals[i]]);
        [key RELEASE];
    }
    [m clear];
    if ([m size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", [m size]];
    }
    [m RELEASE];
}

- (void) testFind
{
    OLHashMap* m;
    CONSTSTR* vals[] = { @"one", @"two", @"three", @"four", @"five" };
    OLNumber* key;
    OLHashIterator* itor;
    id val;
    int i;
    unsigned count;

    m = [[OLHashMap alloc] init];
    for (i = 0; i < 5; i++)
    {
        key = [[OLNumber alloc] initWithInt: i];
        [m assignKey: key value: vals[i]];
        [key RELEASE];
    }
    key = [[OLNumber alloc] initWithInt: 0];
    count = [m count: key];
    if (count != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", count];
    }
    itor = REAP([m find: key]);
    if ([[[itor dereference] first] intValue] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i",
            [[[itor dereference] first] intValue]];
    }
    if (![[[itor dereference] second] isEqual: vals[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            [vals[0] cString], [[[itor dereference] second] cString]];
    }
    val = [m valueForKey: key];
    if (![val isEqual: vals[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            [vals[0] cString], [val cString]];
    }
    [key RELEASE];
    key = [[OLNumber alloc] initWithInt: 700];
    count = [m count: key];
    if (count != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", count];
    }
    itor = REAP([m find: key]);
    if (![itor isEqual: REAP([m end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The key should not have been found"];
    }
    val = [m valueForKey: key];
    if (val != nil)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The key should not have been found"];
    }
    [key RELEASE];
    [m RELEASE];
}

- (void) testInsert
{
    OLHashMap* m;
    OLPair* p;
    OLPair* r;
    OLVector* v;
    CONSTSTR* vals[] = { @"one", @"two", @"three", @"four", @"five" };
    OLNumber* key;
    OLHashIterator* itor;
    int i;

    m = [[OLHashMap alloc] init];
    p = [[OLPair alloc] initWithFirst: @"my" second: @"dog"];
    r = REAP([m insert: p]);
    [p RELEASE];
    if ([m size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", [m size]];
    }
    if (![[r second] boolValue])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, but got NO"];
    }
    if (![[[[r first] dereference] first] isEqual: @"my"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"my\", but got \"%s\"",
            [[[[r first] dereference] first] cString]];
    }
    if (![[[[r first] dereference] second] isEqual: @"dog"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"dog\", but got \"%s\"",
            [[[[r first] dereference] second] cString]];
    }
    p = [[OLPair alloc] initWithFirst: @"my" second: @"flea"];
    r = REAP([m insert: p]);
    [p RELEASE];
    if ([m size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", [m size]];
    }
    if ([[r second] boolValue])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected NO, but got YES"];
    }
    if (![[[[r first] dereference] first] isEqual: @"my"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"my\", but got \"%s\"",
            [[[[r first] dereference] first] cString]];
    }
    if (![[[[r first] dereference] second] isEqual: @"dog"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"dog\", but got \"%s\"",
            [[[[r first] dereference] second] cString]];
    }
    r = REAP([m insertKey: @"your" value: @"cat"]);
    if ([m size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [m size]];
    }
    if (![[r second] boolValue])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, but got NO"];
    }
    if (![[[[r first] dereference] first] isEqual: @"your"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"your\", but got \"%s\"",
            [[[[r first] dereference] first] cString]];
    }
    if (![[[[r first] dereference] second] isEqual: @"cat"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"cat\", but got \"%s\"",
            [[[[r first] dereference] second] cString]];
    }
    r = REAP([m insertKey: @"your" value: @"girlfriend"]);
    if ([m size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [m size]];
    }
    if ([[r second] boolValue])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected NO, but got YES"];
    }
    if (![[[[r first] dereference] first] isEqual: @"your"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"your\", but got \"%s\"",
            [[[[r first] dereference] first] cString]];
    }
    if (![[[[r first] dereference] second] isEqual: @"cat"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"cat\", but got \"%s\"",
            [[[[r first] dereference] second] cString]];
    }
    [m RELEASE];
    v = [[OLVector alloc] init];
    for (i = 0; i < 5; i++)
    {
        key = [[OLNumber alloc] initWithInt: i];
        p = [[OLPair alloc] initWithFirst: key second: vals[i]];
        [key RELEASE];
        [v pushBack: p];
        [p RELEASE];
    }
    m = [[OLHashMap alloc] init];
    [m insertFrom: REAP([v begin]) to: REAP([v end])];
    [v RELEASE];
    if ([m size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %u", [m size]];
    }
    for (itor = REAP([m begin]), i = 0; i < 5; [itor advance], i++)
    {
        if (![[[itor dereference] second] isEqual: vals[[[[itor dereference] first] intValue]]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [vals[i] cString], [[[itor dereference] second] cString]];
        }
    }
    [m RELEASE];
}

- (void) testInitializers
{
    OLHashMap* m;
    OLVector* v;
    CONSTSTR* vals[] = { @"one", @"two", @"three", @"four", @"five" };
    OLNumber* key;
    OLPair* p;
    int i;
    unsigned count;

    m = [[OLHashMap alloc] init];
    if ([m size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", [m size]];
    }
    [m RELEASE];
    m = [[OLHashMap alloc] initWithTableSize: 7000];
    if ([m size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", [m size]];
    }
    [m RELEASE];
    v = [[OLVector alloc] init];
    for (i = 0; i < 5; i++)
    {
        key = [[OLNumber alloc] initWithInt: i];
        p = [[OLPair alloc] initWithFirst: key second: vals[i]];
        [key RELEASE];
        [v pushBack: p];
        [p RELEASE];
    }
    m = [[OLHashMap alloc] initFrom: REAP([v begin]) to: REAP([v end])];
    if ([m size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %u", [m size]];
    }
    for (i = 0; i < 5; i++)
    {
        count = [m count: [[v at: i] first]];
        if (count != 1)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 1, but got %u", count];
        }
        if (![[m valueForKey: [[v at: i] first]] isEqual: vals[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [vals[i] cString], [m valueForKey: [[v at: i] first]]];
        }
    }
    [v RELEASE];
    [m RELEASE];
}

- (void) testIterators
{
    OLHashMap* m;
    CONSTSTR* vals[] = { @"one", @"two", @"three", @"four", @"five" };
    OLNumber* key;
    OLHashIterator* itor;
    OLHashIterator* end;
    int i;

    m = [[OLHashMap alloc] init];
    for (i = 0; i < 5; i++)
    {
        key = [[OLNumber alloc] initWithInt: i];
        REAP([m insertKey: key value: vals[i]]);
        [key RELEASE];
    }
    for (itor = REAP([m begin]), end = REAP([m end]); ![itor isEqual: end]; [itor advance])
    {
        if (![[[itor dereference] second] isEqual: vals[[[[itor dereference] first] intValue]]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [vals[i] cString], [[[itor dereference] second] cString]];
        }
    }
    [m RELEASE];
}

- (void) testProperties
{
    OLHashMap* m;
    OLHashMap* m2;

    m = [[OLHashMap alloc] init];
    if ([m size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", [m size]];
    }
    if (![m empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The hash map is empty"];
    }
    [m assignKey: @"doggy" value: @"boy"];
    if ([m size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", [m size]];
    }
    if ([m empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The hash map is not empty"];
    }
    m2 = [[OLHashMap alloc] init];
    if ([m isEqual: m2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The maps are not equal"];
    }
    if (![m2 size] == 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", [m2 size]];
    }
    [m2 insertFrom: REAP([m begin]) to: REAP([m end])];
    if (![m isEqual: m2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The maps are equal"];
    }
    if ([m maxSize] != [m2 maxSize])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "All hash maps must have the same max size"];
    }
    [self logMessage: "The max size of a hash map is %u", [m maxSize]];
    [m RELEASE];
    [m2 RELEASE];
}

- (void) testStreaming
{
    OLDataOutStream* dout;
    OLObjectOutStream* oout;
    OLObjectInStream* oin;
    OLHashMap* map;
    OLHashMap* readMap;
    OLNumber* k;
    OLNumber* v;
    int i;

    dout = REAP([OLDataOutStream stream]);
    oout = REAP([OLObjectOutStream streamWithOutStream: dout]);
    map = [[OLHashMap alloc] init];
    for (i = 0; i< 10000; i++)
    {
        k = [[OLNumber alloc] initWithInt: i];
        v = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
        [map assignKey: k value: v];
        [k RELEASE];
        [v RELEASE];
    }
    [oout writeObject: map];
    oin = REAP([OLObjectInStream streamWithInStream:
        REAP([OLDataInStream streamWithBytes: [dout bytes] count: [dout count]])]);
    readMap = REAP([oin readObject]);
    if (![readMap IS_MEMBER_OF: [OLHashMap class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected OLBitSet class, but got %s", ((Class)[readMap class])->name];
    }
    if (![readMap isEqual: map])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The hash maps should be equal"];
    }
    [map RELEASE];
}

- (void) testSwap
{
    OLHashMap* m;
    OLHashMap* m2;
    CONSTSTR* vals[] = { @"one", @"two" };
    CONSTSTR* vals2[] = { @"three", @"four" };
    OLNumber* key;
    OLHashIterator* itor;
    OLHashIterator* end;
    int i;

    m = [[OLHashMap alloc] init];
    m2 = [[OLHashMap alloc] init];
    for (i = 0; i < 2; i++)
    {
        key = [[OLNumber alloc] initWithInt: i];
        [m assignKey: key value: vals[i]];
        [key RELEASE];
    }
    for ( ; i < 4; i++)
    {
        key = [[OLNumber alloc] initWithInt: i];
        [m2 assignKey: key value: vals2[i - 2]];
        [key RELEASE];
    }
    [m swap: m2];
    if ([m size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [m size]];
    }
    if ([m2 size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [m2 size]];
    }
    for (itor = REAP([m2 begin]), end = REAP([m2 end]), i = 0;
         ![itor isEqual: end]; [itor advance], i++)
    {
        if ([[[itor dereference] first] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[[itor dereference] first] intValue]];
        }
        if (![[[itor dereference] second] isEqual: vals[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [vals[i] cString], [[[itor dereference] second] cString]];
        }
    }
    for (itor = REAP([m begin]), end = REAP([m end]);
         ![itor isEqual: end]; [itor advance], i++)
    {
        if ([[[itor dereference] first] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[[itor dereference] first] intValue]];
        }
        if (![[[itor dereference] second] isEqual: vals2[i - 2]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [vals2[i - 2] cString], [[[itor dereference] second] cString]];
        }
    }
    [m RELEASE];
    [m2 RELEASE];
}

@end
