//
// $Id: BitSetTest.m,v 1.16 2007/03/28 03:16:52 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "BitSetTest.h"
#import "Random.h"
#import <ObjectiveLib/BitSet.h>
#import <string.h>
#import <limits.h>
#include <stdlib.h>
#import <ObjectiveLib/DataOutStream.h>
#import <ObjectiveLib/DataInStream.h>
#import <ObjectiveLib/ObjectOutStream.h>
#import <ObjectiveLib/ObjectInStream.h>
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Reaper.h>
#else
#import <Foundation/NSArchiver.h>
#if defined(HAVE_KEYED_ARCHIVES)
#import <Foundation/NSKeyedArchiver.h>
#endif
#import <Foundation/NSString.h>
#import <Foundation/NSData.h>
#endif
#if defined(__NEXT_RUNTIME__)
#import <objc/objc-class.h>
#else
#import <objc/objc.h>
#import <objc/objc-api.h>
#endif

#define OL_BITS_PER_WORD (CHAR_BIT * sizeof(unsigned long))

@implementation BitSetTest

#if !defined(OL_NO_OPENSTEP)
- (void) testCoding
{
    char buf[51];
    OLBitSet* bs1;
    OLBitSet* bs2;
    NSMutableData* data;
    NSArchiver* archiver;
    NSData* archData;
    int i;


    for (i = 0; i < 50; i++)
    {
        if ((OLRandom() & 1) != 0)
            buf[i] = '1';
        else
            buf[i] = '0';
    }
    buf[i] = 0;
    [self logMessage: "Creating bit set: %s", buf];
    bs1 = [[OLBitSet alloc] initWithString: buf position: 0 count: 50];
    data = [[NSMutableData alloc] initWithCapacity: 50];
    archiver = [[NSArchiver alloc] initForWritingWithMutableData: data];
    [archiver encodeRootObject: bs1];
    [archiver RELEASE];
    bs2 = [NSUnarchiver unarchiveObjectWithData: data];
    [data RELEASE];
    if (![bs1 isEqual: bs2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The bit sets should be equal"];
    }
#if defined(HAVE_KEYED_ARCHIVES)
    archData = [NSKeyedArchiver archivedDataWithRootObject: bs1];
    bs2 = [NSKeyedUnarchiver unarchiveObjectWithData: archData];
    if (![bs1 isEqual: bs2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The bit sets should be equal"];
    }
#endif
    [bs1 RELEASE];
}
#endif

- (void) testConvenienceAllcoators
{
    OLBitSet* bs;
    OLBitSet* bs2;
    char* str;

    bs = REAP([OLBitSet bitSetWithSetSize: 73]);
    if (![bs IS_MEMBER_OF: [OLBitSet class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLBitSet class])->name, ((Class)[bs class])->name];
    }
    if ([bs size] != 73)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 73, but got %u", [bs size]];
    }
    if ([bs any])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The bit set should be clear"];
    }
    [bs flip: 0];
    bs2 = REAP([OLBitSet bitSetWithBitSet: bs]);
    if (![bs isEqual: bs2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The bit sets should be equal"];
    }
    bs = REAP([OLBitSet bitSetWithString: "0101" position: 0 count: 4]);
    if (![bs IS_MEMBER_OF: [OLBitSet class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLBitSet class])->name, ((Class)[bs class])->name];
    }
    str = [bs toString];
    if (strcmp(str, "0101") != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"0101\", but got \"%s\"", str];
    }
    objc_free(str);
    bs = REAP([OLBitSet bitSetWithValue: 3]);
    str = [bs toString];
    if (strcmp(str, "11000000000000000000000000000000") != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"11000000000000000000000000000000\", but got \"%s\"", str];
    }
    objc_free(str);
}

- (void) testFlip
{
    OLBitSet* bs;
    OLBitSet* bs2;
    unsigned i;

    bs = [[OLBitSet alloc] initWithSetSize: OL_BITS_PER_WORD + 1];
    [bs flip: 0];
    [bs flip: OL_BITS_PER_WORD];
    if ([bs count] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [bs count]];
    }
    if (![bs test: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 0 should be set"];
    }
    if (![bs test: OL_BITS_PER_WORD])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position %u should be set", OL_BITS_PER_WORD];
    }
    for (i = 1; i < OL_BITS_PER_WORD; i++)
    {
        if ([bs test: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Position %u should not be set", i];
        }
    }
    [bs flip: 0];
    [bs flip: OL_BITS_PER_WORD];
    if ([bs count] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", [bs count]];
    }
    [bs flip];
    if ([bs count] != OL_BITS_PER_WORD + 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", OL_BITS_PER_WORD + 1, [bs count]];
    }
    if ([bs count] != [bs size])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", [bs size], [bs count]];
    }
    [bs flip];
    if ([bs count] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", [bs count]];
    }
    [bs flip: 0];
    [bs flip: OL_BITS_PER_WORD];
    bs2 = REAP([bs bitSetFlipped]);
    [bs RELEASE];
    if ([bs2 count] != OL_BITS_PER_WORD - 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", OL_BITS_PER_WORD - 1, [bs2 count]];
    }
    if ([bs2 test: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 0 should not be set"];
    }
    if ([bs2 test: OL_BITS_PER_WORD])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position %u should not be set", OL_BITS_PER_WORD];
    }
    for (i = 1; i < OL_BITS_PER_WORD; i++)
    {
        if (![bs2 test: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Position %u should be set", i];
        }
    }
}

- (void) testInitializers
{
    OLBitSet* bs;
    OLBitSet* bs2;

    bs = [[OLBitSet alloc] initWithSetSize: 4];
    if ([bs size] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i", [bs size]];
    }
    if ([bs any])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The bitset must not have any bits set"];
    }
    [bs RELEASE];
    bs = [[OLBitSet alloc] initWithSetSize: 72];
    if ([bs size] != 72)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 72, but got %i", [bs size]];
    }
    if ([bs any])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The bitset must not have any bits set"];
    }
    [bs RELEASE];
    bs = [[OLBitSet alloc] initWithValue: 5];
    if ([bs size] != OL_BITS_PER_WORD)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", OL_BITS_PER_WORD, [bs size]];
    }
    if ([bs count] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [bs count]];
    }
    [bs RELEASE];
    bs = [[OLBitSet alloc] initWithString: "100110111" position: 0 count: 9];
    if ([bs size] != 9)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 9, but got %u", [bs size]];
    }
    if ([bs count] != 6)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 6, but got %u", [bs count]];
    }
    bs2 = [[OLBitSet alloc] initWithBitSet: bs];
    if ([bs2 size] != 9)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 9, but got %u", [bs2 size]];
    }
    if ([bs2 count] != 6)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 6, but got %u", [bs2 count]];
    }
    if (![bs isEqual: bs2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The bitsets must be equal"];
    }
    [bs RELEASE];
    [bs2 RELEASE];
}

- (void) testLogical
{
    OLBitSet* bs;
    OLBitSet* bs2;
    const char* bsStr = "0101";

    bs = [[OLBitSet alloc] initWithString: bsStr position: 0 count: 4];
    bs2 = [[OLBitSet alloc] initWithString: "11" position: 0 count: 2];
    [bs logicalAnd: bs2];
    if ([bs count] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [bs count]];
    }
    if ([bs test: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 0 should not be set"];
    }
    if (![bs test: 1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 1 should be set"];
    }
    if ([bs test: 2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 2 should not be set"];
    }
    if (![bs test: 3])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 3 should be set"];
    }
    [bs RELEASE];
    bs = [[OLBitSet alloc] initWithString: bsStr position: 0 count: 4];
    [bs logicalOr: bs2];
    if ([bs count] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %u", [bs count]];
    }
    if (![bs test: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 0 should be set"];
    }
    if (![bs test: 1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 1 should be set"];
    }
    if ([bs test: 2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 2 should not be set"];
    }
    if (![bs test: 3])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 3 should be set"];
    }
    [bs RELEASE];
    bs = [[OLBitSet alloc] initWithString: bsStr position: 0 count: 4];
    [bs logicalXor: bs2];
    if ([bs count] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [bs count]];
    }
    if (![bs test: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 0 should be set"];
    }
    if ([bs test: 1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 1 should not be set"];
    }
    if ([bs test: 2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 2 should not be set"];
    }
    if (![bs test: 3])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 3 should be set"];
    }
    [bs RELEASE];
    [bs2 RELEASE];
}

- (void) testProperties
{
    OLBitSet* bs;

    bs = [[OLBitSet alloc] initWithString: "1010" position: 0 count: 4];
    if (![bs any])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "There must be something showing up"];
    }
    if ([bs count] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [bs count]];
    }
    if ([bs size] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %u", [bs size]];
    }
    [bs reset];
    if (![bs none])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "There must not be anything showing up"];
    }
    [bs RELEASE];
}

- (void) testReset
{
    OLBitSet* bs;
    unsigned i;

    bs = [[OLBitSet alloc] initWithSetSize: OL_BITS_PER_WORD + 1];
    [bs set];
    [bs reset: 0];
    [bs reset: OL_BITS_PER_WORD];
    if ([bs count] != OL_BITS_PER_WORD - 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", OL_BITS_PER_WORD -1, [bs count]];
    }
    if ([bs test: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 0 should not be set"];
    }
    if ([bs test: OL_BITS_PER_WORD])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position %u should not be set", OL_BITS_PER_WORD];
    }
    for (i = 1; i < OL_BITS_PER_WORD; i++)
    {
        if (![bs test: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Position %u should be set", i];
        }
    }
    [bs reset];
    if ([bs count] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", [bs count]];
    }
    [bs RELEASE];
}

- (void) testSet
{
    OLBitSet* bs;
    unsigned i;

    bs = [[OLBitSet alloc] initWithSetSize: OL_BITS_PER_WORD + 1];
    [bs set: 0];
    [bs set: OL_BITS_PER_WORD];
    if ([bs count] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [bs count]];
    }
    if (![bs test: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 0 should be set"];
    }
    if (![bs test: OL_BITS_PER_WORD])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position %u should be set", OL_BITS_PER_WORD];
    }
    for (i = 1; i < OL_BITS_PER_WORD; i++)
    {
        if ([bs test: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Position %u should not be set", i];
        }
    }
    [bs set];
    if ([bs count] != OL_BITS_PER_WORD + 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", OL_BITS_PER_WORD + 1, [bs count]];
    }
    if ([bs count] != [bs size])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", [bs size], [bs count]];
    }
    [bs RELEASE];
}

- (void) testShiftLeft
{
    OLBitSet* bs;
    OLBitSet* bs2;
    unsigned i;

    bs = [[OLBitSet alloc] initWithSetSize: OL_BITS_PER_WORD + 2];
    [bs set: 0];
    [bs set: OL_BITS_PER_WORD];
    bs2 = REAP([bs bitSetShiftedLeft: 1]);
    [bs shiftLeft: 1];
    if ([bs count] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [bs count]];
    }
    if ([bs test: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 0 should not be set"];
    }
    if (![bs test: 1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 1 should be set"];
    }
    for (i = 2; i < OL_BITS_PER_WORD + 1; i++)
    {
        if ([bs test: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Position %u should not be set", i];
        }
    }
    if (![bs test: OL_BITS_PER_WORD + 1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position %u should be set", OL_BITS_PER_WORD + 1];
    }
    if (![bs isEqual: bs2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The bitsets must be equal"];
    }
    [bs RELEASE];
}

- (void) testShiftRight
{
    OLBitSet* bs;
    OLBitSet* bs2;
    unsigned i;

    bs = [[OLBitSet alloc] initWithSetSize: OL_BITS_PER_WORD + 2];
    [bs set: 1];
    [bs set: OL_BITS_PER_WORD + 1];
    bs2 = REAP([bs bitSetShiftedRight: 1]);
    [bs shiftRight: 1];
    if ([bs count] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [bs count]];
    }
    if (![bs test: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 0 should be set"];
    }
    for (i = 1; i < OL_BITS_PER_WORD; i++)
    {
        if ([bs test: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Position %u should not be set", i];
        }
    }
    if (![bs test: OL_BITS_PER_WORD])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position %u should be set", OL_BITS_PER_WORD];
    }
    if ([bs test: OL_BITS_PER_WORD + 1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position %u should not be set", OL_BITS_PER_WORD + 1];
    }
    if (![bs isEqual: bs2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The bitsets must be equal"];
    }
    [bs RELEASE];
}

- (void) testStreaming
{
    OLDataOutStream* dout;
    OLObjectOutStream* oout;
    OLObjectInStream* oin;
    OLBitSet* bitSet;
    OLBitSet* readBitSet;
    char buf[50];
    int i;
    char* bitSetString;
    char* readBitSetString;

    dout = REAP([OLDataOutStream stream]);
    oout = REAP([OLObjectOutStream streamWithOutStream: dout]);
    for (i = 0; i < 50; i++)
    {
        if ((OLRandom() & 1) != 0)
            buf[i] = '1';
        else
            buf[i] = '0';
    }
    bitSet = [[OLBitSet alloc] initWithString: buf position: 0 count: 50];
    [oout writeObject: bitSet];
    oin = REAP([OLObjectInStream streamWithInStream:
        REAP([OLDataInStream streamWithBytes: [dout bytes] count: [dout count]])]);
    readBitSet = REAP([oin readObject]);
    if (![readBitSet IS_MEMBER_OF: [OLBitSet class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected OLBitSet class, but got %s", ((Class)[readBitSet class])->name];
    }
    if (![readBitSet isEqual: bitSet])
    {
        bitSetString = [bitSet toString];
        readBitSetString = [readBitSet toString];
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            bitSetString, readBitSetString];
        objc_free(bitSetString);
        objc_free(readBitSetString);
    }
    [bitSet RELEASE];
}

- (void) testTest
{
    OLBitSet* bs;
    unsigned i;

    bs = [[OLBitSet alloc] initWithSetSize: OL_BITS_PER_WORD + 1];
    [bs set: 1];
    [bs set: OL_BITS_PER_WORD];
    if ([bs test: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 0 must not be set"];
    }
    if (![bs test: 1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position 1 must be set"];
    }
    for (i = 2; i < OL_BITS_PER_WORD; i++)
    {
        if ([bs test: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Position %u must not be set", i];
        }
    }
    if (![bs test: OL_BITS_PER_WORD])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Position %u must be set", OL_BITS_PER_WORD];
    }
    [bs RELEASE];
}

- (void) testToString
{
    OLBitSet* bs;
    char* str;

    bs = [[OLBitSet alloc] initWithSetSize: 4];
    [bs set: 0];
    [bs set: 2];
    str = [bs toString];
    if (strcmp("1010", str))
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"1010\", but got %s", str];
    }
    objc_free(str);
    [bs RELEASE];
}

@end
