//
// $Id: AlgorithmTest.m,v 1.28 2007/04/15 23:33:01 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "AlgorithmTest.h"
#import "Random.h"
#import <ObjectiveLib/Algorithm.h>
#import <ObjectiveLib/Vector.h>
#import <ObjectiveLib/List.h>
#import <ObjectiveLib/Deque.h>
#import <ObjectiveLib/Arithmetic.h>
#import <ObjectiveLib/Set.h>
#import <ObjectiveLib/Text.h>
#import "Number.h"
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Reaper.h>
#else
#import <Foundation/NSString.h>
#endif
#include <stdlib.h>
#import <math.h>
#import <strings.h>

static int __indexOf(OLArrayIterator* itor, OLVector* v)
{
    int i;
    int end = [v size];
    int num = [[itor dereference] intValue];

    for (i = 0; i < end; i++)
    {
        if ([[v at: i] intValue] == num)
            break;
    }
    return i;
}

@interface LessThanFive :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
    <OLBoolUnaryFunction>
{
}

- (BOOL) performUnaryFunctionWithArg: (id)arg;

@end

@implementation LessThanFive

- (BOOL) performUnaryFunctionWithArg: (id)arg
{
    OLNumber* five = [[OLNumber alloc] initWithInt: 5];
    BOOL result = [arg compare: five] < 0 ? YES : NO;

    [five RELEASE];
    return result;
}

@end

@interface AddDoggy : OLStreamableFunctor <OLUnaryFunction>
{
}

- (id) performUnaryFunctionWithArg: (id)arg;

@end

@implementation AddDoggy

- (id) performUnaryFunctionWithArg: (id)arg
{
    OLText* doggy = [[OLText alloc] initWithCString: "doggy"];

    [arg insertText: doggy atOffset: 0];
    [doggy RELEASE];
    return nil;
}

@end

@interface RandomGen : OLStreamableFunctor <OLUnaryFunction>
{
}

- (id) init;
#if defined(OL_NO_OPENSTEP)
- (id) free;
#else
- (void) dealloc;
#endif
- (id) performUnaryFunctionWithArg: (id)arg;

@end

@implementation RandomGen

- (id) init
{
    FILE* dev = fopen("/dev/random", "r");
    unsigned seed;

    [super init];
    if (dev != NULL)
    {
        fread(&seed, sizeof(unsigned), 1, dev);
        fclose(dev);
        srand(seed);
    }
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    SUPER_FREE;
}

- (id) performUnaryFunctionWithArg: (id)arg
{
    return AUTORELEASE([[OLNumber alloc] initWithUnsignedInt: rand() % [arg unsignedIntValue]]);
}

@end

@interface PartitionFunction :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
    <OLBoolUnaryFunction>
{
@private
    int value;
}

- (id) initWithValue: (int)val;
- (BOOL) performUnaryFunctionWithArg: (id)arg;

@end


@implementation PartitionFunction

- (id) initWithValue: (int)val
{
    [super init];
    value = val;
    return self;
}

- (BOOL) performUnaryFunctionWithArg: (id)arg
{
    return [arg intValue] < value ? YES : NO;
}

@end

@interface FindIfFunction :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
    <OLBoolUnaryFunction>
{
@private
    int value;
}

- (id) initWithValue: (int)val;
- (BOOL) performUnaryFunctionWithArg: (id)arg;

@end


@implementation FindIfFunction

- (id) initWithValue: (int)val
{
    [super init];
    value = val;
    return self;
}

- (BOOL) performUnaryFunctionWithArg: (id)arg
{
    return [arg intValue] == value ? YES : NO;
}

@end

@interface StrICmp :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
    <OLBoolBinaryFunction>
{
}

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2;

@end

@implementation StrICmp

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    if ([arg1 IS_MEMBER_OF: [OLText class]] &&
        [arg2 IS_MEMBER_OF: [OLText class]])
    {
        return strcasecmp([arg1 cString], [arg2 cString]) == 0 ? YES : NO;
    }
    return strcasecmp([arg1 cString], [arg2 cString]) == 0 ? YES : NO;
}

@end

@interface IntCompare :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
    <OLBoolBinaryFunction>
{
}

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2;

@end

@implementation IntCompare

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return floor([arg1 doubleValue]) == floor([arg2 doubleValue]);
}

@end

@interface AppendDoggy : OLStreamableFunctor <OLUnaryFunction>
{
}

- (id) performUnaryFunctionWithArg: (id)arg;

@end

@implementation AppendDoggy

- (id) performUnaryFunctionWithArg: (id)arg
{
    OLTextBuffer* buf = [[OLTextBuffer alloc] initWithText: arg];
    OLText* doggy = [[OLText alloc] initWithCString: "doggy"];

    [buf appendText: doggy];
    [doggy RELEASE];
    doggy = [buf text];
    [buf RELEASE];
    return doggy;
}

@end

@interface AppendOther : OLStreamableFunctor <OLBinaryFunction>
{
}

-(id) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2;

@end

@implementation AppendOther

-(id) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    OLTextBuffer* buf = [[OLTextBuffer alloc] initWithText: arg1];
    OLText* result;

    [buf appendText: arg2];
    result = [buf text];
    [buf RELEASE];
    return result;
}

@end

@interface ComparablePair : OLPair
{
}

- (int) compare: (id)other;
#if defined(OL_NO_OPENSTEP)
- (id) copy;
#else
- (id) copyWithZone: (NSZone*)zone;
#endif

@end

@implementation ComparablePair

- (int) compare: (id)other
{
    int result = -1;

    if ([other IS_KIND_OF: [ComparablePair class]])
        result = [first compare: ((ComparablePair*)other)->first];
    return result;
}

#if defined(OL_NO_OPENSTEP)

- (id) copy
{
    return [[ComparablePair alloc] initWithFirst: first second: second];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[ComparablePair allocWithZone: zone] initWithFirst: first second: second];
}

#endif

@end

@implementation AlgorithmTest

- (void) testAdjacentFind
{
    OLList* l;
    OLNumber* num;
    OLForwardIterator* r;
    int nums[] = { 1, 2, 3, 4, 5, 5, 6, 7 };
    int i;

    l = [[OLList alloc] init];
    for (i = 0; i < 8; i++)
    {
        num = [[OLNumber alloc] initWithInt: nums[i]];
        [l pushBack: num];
        [num RELEASE];
    }
    r = REAP([OLAlgorithm adjacentFindFrom: REAP([l begin]) to: REAP([l end])]);
    if ([[r dereference] intValue] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [[r dereference] intValue]];
    }
    if ([[[r advance] dereference] intValue] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [[r dereference] intValue]];
    }
    REAP([l erase: (OLListIterator*)r]);
    r = REAP([OLAlgorithm adjacentFindFrom: REAP([l begin]) to: REAP([l end])]);
    if (![r isEqual: REAP([l end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The iterator should be at the end"];
    }
    [l RELEASE];
}

- (void) testBinarySearch
{
    OLList* l;
    OLNumber* num;
    int i;
    OLGreater* greater;
    OLListIterator* b;
    OLListIterator* e;

    l = [[OLList alloc] init];
    for (i = 0; i < 100; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [num RELEASE];
    }
    [l sort];
    b = REAP([l begin]);
    e = REAP([l end]);
    num = [[OLNumber alloc] initWithInt: 30];
    if (![OLAlgorithm binarySearchFrom: b to: e value: num])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number 30 should have been found"];
    }
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 70];
    if (![OLAlgorithm binarySearchFrom: b to: e value: num])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number 70 should have been found"];
    }
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 99];
    if (![OLAlgorithm binarySearchFrom: b to: e value: num])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number 99 should have been found"];
    }
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 0];
    if (![OLAlgorithm binarySearchFrom: b to: e value: num])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number 0 should have been found"];
    }
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: -100];
    if ([OLAlgorithm binarySearchFrom: b to: e value: num])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number -100 should not have been found"];
    }
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 1000];
    if ([OLAlgorithm binarySearchFrom: b to: e value: num])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number 1000 should not have been found"];
    }
    [num RELEASE];
    greater = [[OLGreater alloc] init];
    [l sortWith: greater];
    b = REAP([l begin]);
    e = REAP([l end]);
    num = [[OLNumber alloc] initWithInt: 30];
    if (![OLAlgorithm binarySearchFrom: b to: e value: num predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number 30 should have been found"];
    }
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 70];
    if (![OLAlgorithm binarySearchFrom: b to: e value: num predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number 70 should have been found"];
    }
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 99];
    if (![OLAlgorithm binarySearchFrom: b to: e value: num predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number 99 should have been found"];
    }
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 0];
    if (![OLAlgorithm binarySearchFrom: b to: e value: num predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number 0 should have been found"];
    }
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: -100];
    if ([OLAlgorithm binarySearchFrom: b to: e value: num predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number -100 should not have been found"];
    }
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 1000];
    if ([OLAlgorithm binarySearchFrom: b to: e value: num predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number 1000 should not have been found"];
    }
    [num RELEASE];
    [greater RELEASE];
    [l RELEASE];
}

- (void) testCopy
{
    CONSTSTR* strs[] = { @"zero", @"zero", @"one", @"two", @"three", @"four", @"five", @"six", @"six" };
    int i;
    OLVector* v;
    OLPair* p;
    OLList* l;
    OLForwardIterator* r;
    OLListIterator* end;
    OLListIterator* itor;

    v = [[OLVector alloc] init];
    p = REAP([OLArrayIterator pairWithPointer: strs + 2 andPointer: strs + 7]);
    [v insertAt: REAP([v begin]) from: [p first] to: [p second]];
    l = [[OLList alloc] initWithSize: 5 filledWith: @"doggy"];
    r = REAP([OLAlgorithm copyFrom: REAP([v begin]) to: REAP([v end])
        destination: REAP([l begin])]);
    end = REAP([l end]);
    if (![r isEqual: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The resulting iterator should be the end of the list"];
    }
    for (itor = REAP([l begin]), i = 2; ![itor isEqual: end]; [itor advance], i++)
    {
        if (![[itor dereference] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[itor dereference] cString]];
        }
    }
    [l pushBack: @"six"];
    r = REAP([OLAlgorithm copyFrom: REAP([[l begin] advance]) to: REAP([l end])
        destination: REAP([l begin])]);
    end = REAP([l end]);
    if (![r isEqual: [end reverse]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The resulting iterator is wrong"];
    }
    [end advance];
    for (itor = REAP([l begin]), i = 3; ![itor isEqual: end]; [itor advance], i++)
    {
        if (![[itor dereference] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[itor dereference] cString]];
        }
    }
    [l assign: 5 filledWith: @"doggy"];
    r = REAP([OLAlgorithm copyBackwardFrom: REAP([v begin]) to: REAP([v end])
        destination: REAP([l end])]);
    if (![r isEqual: REAP([l begin])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The resulting iterator should be at the beginning of the list"];
    }
    end = REAP([l end]);
    for (itor = REAP([l begin]), i = 2; ![itor isEqual: end]; [itor advance], i++)
    {
        if (![[itor dereference] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[itor dereference] cString]];
        }
    }
    [l pushFront: @"zero"];
    end = REAP([l end]);
    r = REAP([OLAlgorithm copyBackwardFrom: REAP([l begin]) to: REAP([[l end] reverse])
        destination: end]);
    if (![r isEqual: REAP([[l begin] advance])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The resulting iterator is wrong"];
    }
    end = REAP([l end]);
    for (itor = REAP([l begin]), i = 0; ![itor isEqual: end]; [itor advance], i++)
    {
        if (![[itor dereference] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[itor dereference] cString]];
        }
    }
    [l RELEASE];
    [v RELEASE];
}

- (void) testCounting
{
    OLList* l;
    OLNumber* num;
    OLNumber* target;
    LessThanFive* lessThanFive;
    OLListIterator* lb;
    OLListIterator* le;
    int nums[] = { 5, 1, 2, 3, 4, 5, 6, 7, 8, 5 };
    int i;
    unsigned count;

    l = [[OLList alloc] init];
    for (i = 0; i < 10; i++)
    {
        num = [[OLNumber alloc] initWithInt: nums[i]];
        [l pushBack: num];
        [num RELEASE];
    }
    lb = REAP([l begin]);
    le = REAP([l end]);
    target = [[OLNumber alloc] initWithInt: 5];
    count = [OLAlgorithm countFrom: lb to: le value: target];
    [target RELEASE];
    if (count != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %i", count];
    }
    target = [[OLNumber alloc] initWithInt: 1];
    count = [OLAlgorithm countFrom: lb to: le value: target];
    [target RELEASE];
    if (count != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", count];
    }
    target = [[OLNumber alloc] initWithInt: -27];
    count = [OLAlgorithm countFrom: lb to: le value: target];
    [target RELEASE];
    if (count != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", count];
    }
    lessThanFive = [[LessThanFive alloc] init];
    count = [OLAlgorithm countFrom: lb to: le predicate: lessThanFive];
    [lessThanFive RELEASE];
    if (count != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i", count];
    }
    [l RELEASE];
}

- (void) testEqual
{
    OLVector* v;
    OLList* l;
    OLNumber* num;
    IntCompare* comp;
    int i;
    double i2;

    v = [[OLVector alloc] init];
    l = [[OLList alloc] init];
    for (i = 0; i < 3000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [l pushBack: num];
        [num RELEASE];
    }
    if (![OLAlgorithm equalFrom: REAP([v begin]) to: REAP([v end]) with: REAP([l begin])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequences should be equal"];
    }
    num = [[OLNumber alloc] initWithInt: 75];
    REAP([v insertAt: [REAP([v begin]) advanceBy: 2001] value: num]);
    [num RELEASE];
    [v popBack];
    if ([OLAlgorithm equalFrom: REAP([v begin]) to: REAP([v end]) with: REAP([l begin])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequences should not be equal"];
    }
    [v clear];
    [l clear];
    for (i2 = 0.0; i2 < 3000.0; i2++)
    {
        num = [[OLNumber alloc] initWithDouble: i2];
        [v pushBack: num];
        [num RELEASE];
        num = [[OLNumber alloc] initWithDouble: i2 + 0.4];
        [l pushBack: num];
        [num RELEASE];
    }
    comp = [[IntCompare alloc] init];
    if (![OLAlgorithm equalFrom: REAP([v begin]) to: REAP([v end])
        with: REAP([l begin]) predicate: comp])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequences should be equal"];
    }
    num = [[OLNumber alloc] initWithDouble: 75.0];
    REAP([v insertAt: [REAP([v begin]) advanceBy: 2001] value: num]);
    [num RELEASE];
    [v popBack];
    if ([OLAlgorithm equalFrom: REAP([v begin]) to: REAP([v end])
        with: REAP([l begin]) predicate: comp])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequences should not be equal"];
    }
    [comp RELEASE];
    [v RELEASE];
    [l RELEASE];
}

- (void) testEqualRange
{
    OLDeque* d;
    OLPair* p;
    OLNumber* num;
    OLGreater* greater;
    int nums[] = { 0, 1, 2, 3, 3, 3, 3, 4, 5, 6, 7, 8, 9, 10 };
    int i;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 13; i++)
    {
        num = [[OLNumber alloc] initWithInt: nums[i]];
        [d pushBack: num];
        [num RELEASE];
    }
    num = [[OLNumber alloc] initWithInt: 3];
    p = REAP([OLAlgorithm equalRangeFrom: REAP([d begin]) to: REAP([d end]) value: num]);
    if ([[[p second] dereference] intValue] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i", [[[p second] dereference] intValue]];
    }
    if ([OLIterator distanceFrom: [p first] to: [p second]] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i",
            [OLIterator distanceFrom: [p first] to: [p second]]];
    }
    while (![[p first] isEqual: [p second]])
    {
        if ([[[p first] dereference] intValue] != 3)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 3, but got %i", [[[p first] dereference] intValue]];
        }
        [[p first] advance];
    }
    greater = [[OLGreater alloc] init];
    [OLAlgorithm sortFrom: REAP([d begin]) to: REAP([d end]) predicate: greater];
    p = REAP([OLAlgorithm equalRangeFrom: REAP([d begin]) to: REAP([d end])
        value: num predicate: greater]);
    [greater RELEASE];
    [num RELEASE];
    if ([[[p second] dereference] intValue] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [[[p second] dereference] intValue]];
    }
    if ([OLIterator distanceFrom: [p first] to: [p second]] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i",
            [OLIterator distanceFrom: [p first] to: [p second]]];
    }
    while (![[p first] isEqual: [p second]])
    {
        if ([[[p first] dereference] intValue] != 3)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 3, but got %i", [[[p first] dereference] intValue]];
        }
        [[p first] advance];
    }
    [d RELEASE];
}

- (void) testFill
{
    OLDeque* d;
    OLNumber* num;
    CONSTSTR* filled = @"doggy";
    OLDequeIterator* fillTo;
    int i;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 20; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    fillTo = [REAP([d begin]) advanceBy: 7];
    [OLAlgorithm fillFrom: REAP([d begin]) to: fillTo value: filled];
    for (i = 0; i < 7; i++)
    {
        if (![[d at: i] isEqual: filled])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"doggy\", but got \"%s\"",
                [[d at: i] cString]];
        }
    }
    for ( ; i < 20; i++)
    {
        if ([[d at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[d at: i] intValue]];
        }
    }
    [OLAlgorithm fillFrom: [fillTo advanceBy: 3] count: 4 value: filled];
    for (i = 10; i < 14; i++)
    {
        if (![[d at: i] isEqual: filled])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"doggy\", but got \"%s\"",
                [[d at: i] cString]];
        }
    }
    for ( ; i < 20; i++)
    {
        if ([[d at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[d at: i] intValue]];
        }
    }
    [d RELEASE];
}

- (void) testFind
{
    OLList* l;
    OLForwardIterator* r;
    OLListIterator* begin;
    OLListIterator* end;
    OLNumber* num;
    FindIfFunction* func;
    int i;

    l = [[OLList alloc] init];
    for (i = 0; i < 1000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [num RELEASE];
    }
    num = [[OLNumber alloc] initWithInt: 726];
    begin = REAP([l begin]);
    end = REAP([l end]);
    r = REAP([OLAlgorithm findFrom: begin to: end value: num]);
    [num RELEASE];
    if ([r isEqual: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The value should have been found"];
    }
    if ([[r dereference] intValue] != 726)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 726, but got %i", [[r dereference] intValue]];
    }
    num = [[OLNumber alloc] initWithInt: -324];
    r = REAP([OLAlgorithm findFrom: begin to: end value: num]);
    [num RELEASE];
    if (![r isEqual: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The value should not have been found"];
    }
    func = [[FindIfFunction alloc] initWithValue: 524];
    r = REAP([OLAlgorithm findFrom: begin to: end if: func]);
    [func RELEASE];
    if ([r isEqual: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The value should have been found"];
    }
    if ([[r dereference] intValue] != 524)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 524, but got %i", [[r dereference] intValue]];
    }
    func = [[FindIfFunction alloc] initWithValue: 6024];
    r = REAP([OLAlgorithm findFrom: begin to: end if: func]);
    [func RELEASE];
    if (![r isEqual: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The value should not have been found"];
    }
    [l RELEASE];
}

- (void) testFindEnd
{
    OLDeque* d;
    OLVector* v;
    OLNumber* num;
    OLForwardIterator* r;
    IntCompare* comp;
    OLDequeIterator* dbegin;
    OLDequeIterator* dend;
    int i;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 1000; i++)
    {
        num = [[OLNumber alloc] initWithDouble: i + .5];
        [d pushBack: num];
        [num RELEASE];
    }
    v = [[OLVector alloc] init];
    for (i = 784; i < 899; i++)
    {
        num = [[OLNumber alloc] initWithDouble: i + 0.5];
        [v pushBack: num];
        [d pushBack: num];
        [num RELEASE];
    }
    dbegin = REAP([d begin]);
    dend = REAP([d end]);
    r = REAP([OLAlgorithm findEndFrom: dbegin to: dend
        subFrom: REAP([v begin]) subTo: REAP([v end])]);
    if ([r isEqual: dend])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The value should have been found"];
    }
    if ([[r dereference] doubleValue] != 784.5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 784, but got %i", [[r dereference] intValue]];
    }
    if ([OLIterator distanceFrom: dbegin to: r] != 1000)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Search found the last one, not the first one (distance %u)",
            [OLIterator distanceFrom: dbegin to: r]];
    }
    [v clear];
    for (i = 784; i < 899; i++)
    {
        num = [[OLNumber alloc] initWithDouble: i];
        [v pushBack: num];
        [num RELEASE];
    }
    comp = [[IntCompare alloc] init];
    r = REAP([OLAlgorithm findEndFrom: dbegin to: dend
        subFrom: REAP([v begin]) subTo: REAP([v end]) predicate: comp]);
    [comp RELEASE];
    if ([r isEqual: dend])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The value should have been found"];
    }
    if ([[r dereference] doubleValue] != 784.5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 784, but got %i", [[r dereference] intValue]];
    }
    if ([OLIterator distanceFrom: dbegin to: r] != 1000)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Search found the last one, not the first one"];
    }
    [d RELEASE];
    [v RELEASE];
}

- (void) testFindFirst
{
    OLDeque* d;
    OLVector* v;
    OLNumber* num;
    OLForwardIterator* r;
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    CONSTSTR* strs2[] = { @"SIX", @"SEVEN", @"EIGHT", @"FIVE", @"TEN" };
    StrICmp* cmp;
    int i;
    int rand;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 1000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    v = [[OLVector alloc] init];
    for ( ; i < 1999; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    rand = OLRandom() % 1000;
    num = [[OLNumber alloc] initWithInt: rand];
    [v pushBack: num];
    [num RELEASE];
    r = REAP([OLAlgorithm findFirstFrom: REAP([d begin]) to: REAP([d end])
        ofFrom: REAP([v begin]) ofTo: REAP([v end])]);
    if ([r isEqual: REAP([d end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The value should have been found"];
    }
    if ([[r dereference] intValue] != rand)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %i, but got %i",
            rand, [[r dereference] intValue]];
    }
    [d clear];
    for (i = 0; i < 5; i++)
        [d pushBack: strs[i]];
    [v clear];
    for (i = 0; i < 5; i++)
        [v pushBack: strs2[i]];
    cmp = [[StrICmp alloc] init];
    r = REAP([OLAlgorithm findFirstFrom: REAP([d begin]) to: REAP([d end])
        ofFrom: REAP([v begin]) ofTo: REAP([v end]) predicate: cmp]);
    [cmp RELEASE];
    if ([r isEqual: REAP([d end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The value should have been found"];
    }
    if (![[r dereference] isEqual: @"five"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"five\", but got \"%i\"", [[r dereference] cString]];
    }
    REAP([v erase: [REAP([v end]) advanceBy: -2]]);
    r = REAP([OLAlgorithm findFirstFrom: REAP([d begin]) to: REAP([d end])
        ofFrom: REAP([v begin]) ofTo: REAP([v end])]);
    if (![r isEqual: REAP([d end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The value should not have been found"];
    }
    [v RELEASE];
    [d RELEASE];
}

- (void) testForEach
{
    OLVector* v;
    AddDoggy* func;
    OLNumber* num;
    OLTextBuffer* tbuf;
    OLText* text;
    double i;
    int j;

    v = [[OLVector alloc] init];
    for (i = 0.0; i < 10000.0; i++)
    {
        num = [[OLNumber alloc] initWithDouble: i];
        text = REAP([num toText]);
        [num RELEASE];
        tbuf = [[OLTextBuffer alloc] initWithText: text];
        [v pushBack: tbuf];
        [tbuf RELEASE];
    }
    func = [[AddDoggy alloc] init];
    [OLAlgorithm forEachFrom: REAP([v begin]) to: REAP([v end]) function: func];
    [func RELEASE];
    text = [[OLText alloc] initWithCString: "doggy"];
    for (j = 0; j < 10000; j++)
    {
        if ([[v at: j] findText: text fromOffset: 0] != 0)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The string \"%s\" was not transformed",
                [REAP([[v at: j] text]) cString]];
        }
    }
    [text RELEASE];
    [v RELEASE];
}

- (void) testIncludes
{
    OLDeque* d;
    OLVector* v;
    OLNumber* num;
    OLGreater* greater;
    OLForwardIterator* ue;
    int i;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 20; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    v = [[OLVector alloc] init];
    for (i = 11; i < 15; i += 2)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    if (![OLAlgorithm includesFrom: REAP([d begin]) to: REAP([d end])
        subFrom: REAP([v begin]) subTo: REAP([v end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequence should have been found"];
    }
    [v clear];
    for (i = 18; i < 25; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    if ([OLAlgorithm includesFrom: REAP([d begin]) to: REAP([d end])
        subFrom: REAP([v begin]) subTo: REAP([v end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequence should not have been found"];
    }
    [d clear];
    [v clear];
    for (i = 9999; i >= 0; i--)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    for (i = 0; i < 500; i++)
    {
        num = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
        [v pushBack: num];
        [num RELEASE];
    }
    greater = [[OLGreater alloc] init];
    [OLAlgorithm sortFrom: REAP([v begin]) to: REAP([v end]) predicate: greater];
    ue = REAP([OLAlgorithm uniqueFrom: REAP([v begin]) to: REAP([v end])]);
    if (![OLAlgorithm includesFrom: REAP([d begin]) to: REAP([d end])
            subFrom: REAP([v begin]) subTo: ue predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequence should have been found"];
    }
    [v clear];
    for (i = 0; i < 5000; i++)
    {
        num = [[OLNumber alloc] initWithInt: (OLRandom() % 10000) + 8000];
        [v pushBack: num];
        [num RELEASE];
    }
    [OLAlgorithm sortFrom: REAP([v begin]) to: REAP([v end]) predicate: greater];
    ue = REAP([OLAlgorithm uniqueFrom: REAP([v begin]) to: REAP([v end])]);
    if ([OLAlgorithm includesFrom: REAP([d begin]) to: REAP([d end])
            subFrom: REAP([v begin]) subTo: ue predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequence should not have been found (there is a slight chance that this test will fail due to the random numbers involved)"];
    }
    [v RELEASE];
    [d RELEASE];
    [greater RELEASE];
}

- (void) testInPlaceMerge
{
    OLVector* v;
    OLNumber* num;
    OLGreater* greater;
    OLArrayIterator* mid;
    int i;
    int next;

    v = [[OLVector alloc] init];
    for (i = 0; i < 20; i += 2)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    for (i = 1; i < 20; i += 2)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    [OLAlgorithm inPlaceMergeFrom: REAP([v begin])
        middle: [REAP([v begin]) advanceBy: 10] to: REAP([v end])];
    for (i = 0; i < 20; i++)
    {
        if ([[v at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[v at: i] intValue]];
        }
    }
    [v clear];
    for (i = 0; i < 20; i += 2)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    for ( ; i < 40; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    for (i = 1; i < 20; i += 2)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    [OLAlgorithm inPlaceMergeFrom: REAP([v begin])
        middle: [REAP([v begin]) advanceBy: 30] to: REAP([v end])];
    for (i = 0; i < 40; i++)
    {
        if ([[v at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[v at: i] intValue]];
        }
    }
    [v clear];
    for (i = 0; i < 20000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    [OLAlgorithm randomShuffleFrom: REAP([v begin]) to: REAP([v end])];
    mid = [REAP([v begin]) advanceBy: OLRandom() % 20000];
    greater = [[OLGreater alloc] init];
    [OLAlgorithm sortFrom: REAP([v begin]) to: mid predicate: greater];
    [OLAlgorithm sortFrom: mid to: REAP([v end]) predicate: greater];
    [OLAlgorithm inPlaceMergeFrom: REAP([v begin]) middle: mid
        to: REAP([v end]) predicate: greater];
    [greater RELEASE];
    for (i = 0, next = 1; next < 20000; i++, next++)
    {
        if ([[v at: next] intValue] > [[v at: i] intValue])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "[%i] %i should not be greater than [%i] %i",
                next, [[v at: next] intValue], i, [[v at: i] intValue]];
        }
    }
    [v RELEASE];
}

- (void) testLexicographicalCompare
{
    OLList* l;
    OLVector* v;
    OLNumber* num;
    OLGreater* greater;
    int i;

    l = [[OLList alloc] init];
    v = [[OLVector alloc] init];
    for (i = 0; i < 5; i++)
    {
        if (i == 3)
        {
            num = [[OLNumber alloc] initWithInt: i + 1];
            [v pushBack: num];
            [num RELEASE];
            num = [[OLNumber alloc] initWithInt: i];
            [l pushBack: num];
            [num RELEASE];
        }
        else
        {
            num = [[OLNumber alloc] initWithInt: i];
            [l pushBack: num];
            [v pushBack: num];
            [num RELEASE];
        }
    }
    greater = [[OLGreater alloc] init];
    if (![OLAlgorithm lexicographicalCompareFrom: REAP([l begin]) to: REAP([l end])
            andFrom: REAP([v begin]) andTo: REAP([v end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "l should be less than v"];
    }
    if (![OLAlgorithm lexicographicalCompareFrom: REAP([v begin]) to: REAP([v end])
            andFrom: REAP([l begin]) andTo: REAP([l end]) predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "v should be greater than l"];
    }
    REAP([v eraseFrom: [REAP([v begin]) advanceBy: 3] to: REAP([v end])]);
    if (![OLAlgorithm lexicographicalCompareFrom: REAP([v begin]) to: REAP([v end])
            andFrom: REAP([l begin]) andTo: REAP([l end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "v should be less than l"];
    }
    if ([OLAlgorithm lexicographicalCompareFrom: REAP([l begin]) to: REAP([l end])
            andFrom: REAP([v begin]) andTo: REAP([v end]) predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "l should not be greater than v"];
    }
    if (![OLAlgorithm lexicographicalCompareFrom: REAP([v begin]) to: REAP([v end])
            andFrom: REAP([l begin]) andTo: REAP([l end]) predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "v should be greater than l"];
    }
    [v RELEASE];
    [l RELEASE];
    [greater RELEASE];
}

- (void) testLowerBound
{
    OLList* l;
    OLNumber* num;
    OLNumber* target;
    OLForwardIterator* r;
    OLGreater* greater;
    double i;

    l = [[OLList alloc] init];
    for (i = 0.0; i < 1000.0; i++)
    {
        num = [[OLNumber alloc] initWithDouble: i];
        [l pushBack: num];
        [num RELEASE];
    }
    target = [[OLNumber alloc] initWithDouble: 39.5];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end]) value: target]);
    [target RELEASE];
    if ([[r dereference] doubleValue] != 40.0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 40.0, but got %G", [[r dereference] doubleValue]];
    }
    target = [[OLNumber alloc] initWithDouble: 700];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end]) value: target]);
    [target RELEASE];
    if ([[r dereference] doubleValue] != 700.0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 700.0, but got %G", [[r dereference] doubleValue]];
    }
    target = [[OLNumber alloc] initWithDouble: 10000.0];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end]) value: target]);
    [target RELEASE];
    if (![r isEqual: REAP([l end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected to be at the end"];
    }
    target = [[OLNumber alloc] initWithDouble: -10000.0];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end]) value: target]);
    [target RELEASE];
    if (![r isEqual: REAP([l begin])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected to be at the beginning"];
    }
    greater = [[OLGreater alloc] init];
    [l sortWith: greater];
    target = [[OLNumber alloc] initWithDouble: 39.5];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end])
        value: target predicate: greater]);
    [target RELEASE];
    if ([[r dereference] doubleValue] != 39.0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 39.0, but got %G", [[r dereference] doubleValue]];
    }
    target = [[OLNumber alloc] initWithDouble: 700];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end])
        value: target predicate: greater]);
    [target RELEASE];
    if ([[r dereference] doubleValue] != 700.0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 700.0, but got %G", [[r dereference] doubleValue]];
    }
    target = [[OLNumber alloc] initWithDouble: 10000.0];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end])
        value: target predicate: greater]);
    [target RELEASE];
    if (![r isEqual: REAP([l begin])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected to be at the beginning"];
    }
    target = [[OLNumber alloc] initWithDouble: -10000.0];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end])
        value: target predicate: greater]);
    [target RELEASE];
    if (![r isEqual: REAP([l end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected to be at the end"];
    }
    [greater RELEASE];
    [l RELEASE];
}

- (void) testMakeHeap
{
    OLVector* v;
    OLNumber* num;
    OLGreater* greater;
    OLArrayIterator* begin;
    OLArrayIterator* end;
    int i;

    v = [[OLVector alloc] init];
    for (i = 0; i < 10000; i++)
    {
        num = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
        [v pushBack: num];
        [num RELEASE];
    }
    begin = REAP([v begin]);
    end = REAP([v end]);
    [OLAlgorithm makeHeapFrom: begin to: end];
    if (![OLAlgorithm isHeapFrom: begin to: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not a heap"];
    }
    [OLAlgorithm randomShuffleFrom: begin to: end];
    greater = [[OLGreater alloc] init];
    [OLAlgorithm makeHeapFrom: begin to: end predicate: greater];
    if (![OLAlgorithm isHeapFrom: begin to: end predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not a heap"];
    }
    [greater RELEASE];
    [v RELEASE];
}

- (void) testMerge
{
    OLList* l;
    OLDeque* d;
    OLVector* v;
    OLBackInsertIterator* bii;
    OLNumber* num;
    OLGreater* greater;
    int i;
    int count;
    int next;

    l = [[OLList alloc] init];
    d = [[OLDeque alloc] init];
    for (i = 0; i < 20; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        if (i & 1)
            [l pushBack: num];
        else
            [d pushBack: num];
        [num RELEASE];
    }
    v = [[OLVector alloc] init];
    bii = [[OLBackInsertIterator alloc] initWithBackInserter: v];
    REAP([OLAlgorithm mergeFrom: REAP([l begin]) to: REAP([l end])
        andFrom: REAP([d begin]) andTo: REAP([d end]) destination: bii]);
    for (i = 0; i < 20; i++)
    {
        if ([[v at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[v at: i] intValue]];
        }
    }
    [l clear];
    [d clear];
    [v clear];
    count = OLRandom() % 10000;
    for (i = count; i > 0; i--)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [num RELEASE];
    }
    count = OLRandom() % 10000;
    for (i = count; i > 0; i--)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    greater = [[OLGreater alloc] init];
    REAP([OLAlgorithm mergeFrom: REAP([l begin]) to: REAP([l end])
        andFrom: REAP([d begin]) andTo: REAP([d end]) destination: bii predicate: greater]);
    [greater RELEASE];
    count = [v size];
    for (i = 0, next = 1; next < count; i++, next++)
    {
        if ([[v at: next] intValue] > [[v at: i] intValue])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "%i is less than %i",
                [[v at: next] intValue], [[v at: i] intValue]];
        }
    }
    [l RELEASE];
    [d RELEASE];
    [v RELEASE];
    [bii RELEASE];
}

- (void) testMinMax
{
    OLNumber* one = [[OLNumber alloc] initWithInt: 1];
    OLNumber* two = [[OLNumber alloc] initWithInt: 2];
    OLGreater* greater;
    id result;

    result = [OLAlgorithm min: one or: two];
    if ([result intValue] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [result intValue]];
    }
    if (result != one)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The same id must be returned"];
    }
    result = [OLAlgorithm max: one or: two];
    if ([result intValue] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [result intValue]];
    }
    if (result != two)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The same id must be returned"];
    }
    greater = [[OLGreater alloc] init];
    result = [OLAlgorithm min: one or: two predicate: greater];
    if ([result intValue] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [result intValue]];
    }
    if (result != two)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The same id must be returned"];
    }
    result = [OLAlgorithm max: one or: two predicate: greater];
    if ([result intValue] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [result intValue]];
    }
    if (result != one)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The same id must be returned"];
    }
    [greater RELEASE];
    [one RELEASE];
    [two RELEASE];
}

- (void) testMinMaxElement
{
    OLVector* v;
    OLNumber* num;
    OLGreater* greater;
    OLForwardIterator* r;
    OLArrayIterator* begin;
    OLArrayIterator* end;
    int i;

    v = [[OLVector alloc] init];
    for (i = 0; i < 1000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    begin = REAP([v begin]);
    end = REAP([v end]);
    [OLAlgorithm randomShuffleFrom: begin to: end];
    r = REAP([OLAlgorithm minElementFrom: begin to: end]);
    if ([[r dereference] intValue] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", [[r dereference] intValue]];
    }
    [self logMessage: "Found 0 at index %u", [OLIterator distanceFrom: begin to: r]];
    r = REAP([OLAlgorithm maxElementFrom: begin to: end]);
    if ([[r dereference] intValue] != 999)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 999, but got %i", [[r dereference] intValue]];
    }
    [self logMessage: "Found 999 at index %u", [OLIterator distanceFrom: begin to: r]];
    greater = [[OLGreater alloc] init];
    [OLAlgorithm randomShuffleFrom: begin to: end];
    r = REAP([OLAlgorithm minElementFrom: begin to: end predicate: greater]);
    if ([[r dereference] intValue] != 999)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 999, but got %i", [[r dereference] intValue]];
    }
    [self logMessage: "Found 999 at index %u", [OLIterator distanceFrom: begin to: r]];
    r = REAP([OLAlgorithm maxElementFrom: begin to: end predicate: greater]);
    [greater RELEASE];
    if ([[r dereference] intValue] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", [[r dereference] intValue]];
    }
    [self logMessage: "Found 0 at index %u", [OLIterator distanceFrom: begin to: r]];
    [v RELEASE];
}

- (void) testMismatch
{
    OLVector* v;
    OLDeque* d;
    OLPair* p;
    OLNumber* num;
    OLGreaterEqual* ge;
    int i;

    v = [[OLVector alloc] init];
    d = [[OLDeque alloc] init];
    for (i = 0; i < 20; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [d pushBack: num];
        [num RELEASE];
        if (i == 13)
        {
            num = [[OLNumber alloc] initWithInt: i];
            [d pushBack: num];
            [num RELEASE];
        }
    }
    p = REAP([OLAlgorithm mismatchFrom: REAP([v begin]) to: REAP([v end])
        with: REAP([d begin])]);
    if ([[[p first] dereference] intValue] != 14)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 14, but got %i", [[[p first] dereference] intValue]];
    }
    if ([[[p second] dereference] intValue] != 13)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 13, but got %i", [[[p second] dereference] intValue]];
    }
    ge = [[OLGreaterEqual alloc] init];
    p = REAP([OLAlgorithm mismatchFrom: REAP([d begin]) to: REAP([d end])
        with: REAP([v begin]) predicate: ge]);
    [ge RELEASE];
    if ([[[p first] dereference] intValue] != 13)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 13, but got %i", [[[p first] dereference] intValue]];
    }
    if ([[[p second] dereference] intValue] != 14)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 14, but got %i", [[[p second] dereference] intValue]];
    }
    [v RELEASE];
    [d RELEASE];
}

- (void) testNextPermutation
{
    OLVector* v;
    OLNumber* num;
    OLArrayIterator* begin;
    OLArrayIterator* end;
    OLSet* set;
    OLVector* array;
    OLGreater* greater;
    OLPair* ptrs;
    typedef struct { int one; int two; int three; } Perm;
    Perm perms[] =
    {
        { 1, 2, 3 },
        { 1, 3, 2 },
        { 2, 1, 3 },
        { 2, 3, 1 },
        { 3, 1, 2 },
        { 3, 2, 1 }
    };
    id objs[5];
    int i;
    int j;
    int val;

    v = [[OLVector alloc] init];
    for (i = 1; i < 4; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    begin = REAP([v begin]);
    end = REAP([v end]);
    for (i = 0; i < 6; i++)
    {
        [self logMessage: "[%i] %i %i %i",
            i, [[v at: 0] intValue], [[v at: 1] intValue], [[v at: 2] intValue]];
        for (j = 0; j < 3; j++)
        {
            val = *((int*)(((int*)&perms[i])) + j);
            if ([[v at: j] intValue] != val)
            {
                [self errInFile: __FILE__ line: __LINE__
                    format: "Expected %i, but got %i",
                    val, [[v at: j] intValue]];
            }
        }
        if (![OLAlgorithm nextPermutationFrom: begin to: end] && i < 5)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected a successful run"];
        }
    }
    [v clear];
    for (i = 4; i >= 0; i--)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    set = [[OLSet alloc] init];
    j = 0;
    greater = [[OLGreater alloc] init];
    begin = REAP([v begin]);
    end = REAP([v end]);
    do
    {
        for (i = 0; i < 5; i++)
            objs[i] = [v at: i];
        ptrs = REAP([OLArrayIterator pairWithPointer: objs distance: 5]);
        array = [[OLVector alloc] initFrom: [ptrs first] to: [ptrs second]];
        if (![[REAP([set insert: array]) second] boolValue])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The array was already found"];
        }
        [array RELEASE];
        j++;
    } while ([OLAlgorithm nextPermutationFrom: begin to: end predicate: greater]);
    [self logMessage: "Found %i permutations", j];
    [set RELEASE];
    [greater RELEASE];
    [v RELEASE];
}

- (void) testNthElement
{
    OLVector* v;
    OLNumber* num;
    OLArrayIterator* mid;
    OLArrayIterator* itor;
    OLArrayIterator* end;
    OLGreater* greater;
    int nums[] = { 4, 10, 70, 30, 10, 69, 96, 100 };
    int i;
    int pivot;

    v = [[OLVector alloc] init];
    for (i = 0; i < 8; i++)
    {
        num = [[OLNumber alloc] initWithInt: nums[i]];
        [v pushBack: num];
        [num RELEASE];
    }
    mid = [REAP([v begin]) advanceBy: 4];
    [OLAlgorithm nthElementFrom: REAP([v begin]) nth: mid to: REAP([v end])];
    mid = [REAP([v begin]) advanceBy: 2];
    if ([[mid dereference] intValue] != 10)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 10, but got %i", [[mid dereference] intValue]];
    }
    itor = [mid copy];
    [itor reverse];
    i = [[itor dereference] intValue];
    if (i != 10 && i != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 10 or 4, but got %i", [[itor dereference] intValue]];
    }
    [itor reverse];
    if (i != 10 && i != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 10 or 4, but got %i", [[itor dereference] intValue]];
    }
    [itor RELEASE];
    [mid advance];
    end = REAP([v end]);
    for ( ; ![mid isEqual: end]; [mid advance])
    {
        if ([[mid dereference] intValue] <= 10)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The value %i is greater than 10", [[mid dereference] intValue]];
        }
    }
    [v clear];
    for (i = 0; i < 10000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    [OLAlgorithm randomShuffleFrom: REAP([v begin]) to: REAP([v end])];
    mid = [REAP([v begin]) advanceBy: 5027];
    greater = [[OLGreater alloc] init];
    [OLAlgorithm nthElementFrom: REAP([v begin]) nth: mid
        to: REAP([v end]) predicate: greater];
    [mid reverse];
    pivot = [[mid dereference] intValue];
    [mid advance];
    [greater RELEASE];
    for (itor = REAP([v begin]); ![itor isEqual: mid]; [itor advance])
    {
        if ([[itor dereference] intValue] < pivot)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The value %i is less than %i",
                [[itor dereference] intValue], pivot];
        }
    }
    end = REAP([v end]);
    for ([itor advance]; ![itor isEqual: end]; [itor advance])
    {
        if ([[itor dereference] intValue] >= pivot)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The value %i is not less than %i",
                [[itor dereference] intValue], pivot];
        }
    }
    [v RELEASE];
}

- (void) testPartialSort
{
    OLDeque* d;
    OLNumber* num;
    OLDequeIterator* mid;
    OLGreater* greater;
    int i;
    int j;
    int pivot;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 10; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    [OLAlgorithm randomShuffleFrom: REAP([d begin]) to: REAP([d end])];
    [self logMessage: "Pre partial sort: %i %i %i %i %i %i %i %i %i %i",
       [[d at: 0] intValue], [[d at: 1] intValue], [[d at: 2] intValue], [[d at: 3] intValue],
       [[d at: 4] intValue], [[d at: 5] intValue], [[d at: 6] intValue], [[d at: 7] intValue],
       [[d at: 8] intValue], [[d at: 9] intValue]];
    mid = [REAP([d begin]) advanceBy: 3];
    [OLAlgorithm partialSortFrom: REAP([d begin]) middle: mid to: REAP([d end])];
    [self logMessage: "Post partial sort: %i %i %i %i %i %i %i %i %i %i",
       [[d at: 0] intValue], [[d at: 1] intValue], [[d at: 2] intValue], [[d at: 3] intValue],
       [[d at: 4] intValue], [[d at: 5] intValue], [[d at: 6] intValue], [[d at: 7] intValue],
       [[d at: 8] intValue], [[d at: 9] intValue]];
    for (i = 0; i < 3; i++)
    {
        if ([[d at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The number %i at position %i should be %i",
                [[d at: i] intValue], i, i];
        }
    }
    pivot = [[[mid reverse] dereference] intValue];
    for ( ; i < 10; i++)
    {
        if ([[d at: i] intValue] < pivot)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The number %i at position %i is not less than %i",
                [[d at: i] intValue], i, pivot];
        }
    }
    [d clear];
    for (i = 0; i < 50000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    [OLAlgorithm randomShuffleFrom: REAP([d begin]) to: REAP([d end])];
    mid = [REAP([d begin]) advanceBy: 4528];
    greater = [[OLGreater alloc] init];
    [OLAlgorithm partialSortFrom: REAP([d begin]) middle: mid
        to: REAP([d end]) predicate: greater];
    [greater RELEASE];
    for (i = 0, j = 49999; i < 4528; i++, j--)
    {
        if ([[d at: i] intValue] != j)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The number %i at position %i should be %i",
                [[d at: i] intValue], i, j];
        }
    }
    pivot = [[[mid reverse] dereference] intValue];
    for ( ; i < 50000; i++)
    {
        if ([[d at: i] intValue] > pivot)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The number %i at position %i is not less than %i",
                [[d at: i] intValue], i, pivot];
        }
    }
    [d RELEASE];
}

- (void) testPartialSortCopy
{
    OLDeque* d;
    OLVector* v;
    OLNumber* num;
    OLGreater* greater;
    int i;
    int j;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 10; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    [OLAlgorithm randomShuffleFrom: REAP([d begin]) to: REAP([d end])];
    v = [[OLVector alloc] initWithSize: 5 filledWith: @"junk"];
    REAP([OLAlgorithm partialSortCopyFrom: REAP([d begin]) to: REAP([d end])
        destinationFrom: REAP([v begin]) destinationTo: REAP([v end])]);
    for (i = 0; i < 5; i++)
    {
        if ([[v at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[v at: i] intValue]];
        }
    }
    [v clear];
    [v assign: 20 filledWith: @"junk"];
    REAP([OLAlgorithm partialSortCopyFrom: REAP([d begin]) to: REAP([d end])
        destinationFrom: REAP([v begin]) destinationTo: REAP([v end])]);
    for (i = 0; i < 10; i++)
    {
        if ([[v at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[v at: i] intValue]];
        }
    }
    for (; i < 20; i++)
    {
        if (![[v at: i] isEqual: @"junk"])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Item %i should be junk", i];
        }
    }
    [d clear];
    for (i = 0; i < 75000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    [OLAlgorithm randomShuffleFrom: REAP([d begin]) to: REAP([d end])];
    greater = [[OLGreater alloc] init];
    REAP([OLAlgorithm partialSortCopyFrom: REAP([d begin]) to: REAP([d end])
        destinationFrom: REAP([v begin]) destinationTo: REAP([v end]) predicate: greater]);
    for (i = 0, j = 74999; i < 20; i++, j--)
    {
        if ([[v at: i] intValue] != j)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                j, [[v at: i] intValue]];
        }
    }
    [v clear];
    [v assign: 100000 filledWith: @"junk"];
    REAP([OLAlgorithm partialSortCopyFrom: REAP([d begin]) to: REAP([d end])
        destinationFrom: REAP([v begin]) destinationTo: REAP([v end]) predicate: greater]);
    for (i = 0, j = 74999; i < 75000; i++, j--)
    {
        if ([[v at: i] intValue] != j)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                j, [[v at: i] intValue]];
        }
    }
    for ( ; i < 100000; i++)
    {
        if (![[v at: i] isEqual: @"junk"])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Item %i should be junk", i];
        }
    }
    [greater RELEASE];
    [v RELEASE];
    [d RELEASE];
}

- (void) testPartition
{
    OLDeque* d;
    OLNumber* num;
    OLBidirectionalIterator* r;
    PartitionFunction* func;
    OLDequeIterator* itor;
    OLDequeIterator* end;
    int i;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 1000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    itor = REAP([d begin]);
    end = REAP([d end]);
    [OLAlgorithm randomShuffleFrom: itor to: end];
    func = [[PartitionFunction alloc] initWithValue: 673];
    r = REAP([OLAlgorithm partitionFrom: itor to: end predicate: func]);
    for ( ; ![itor isEqual: r]; [itor advance])
    {
        if (![func performUnaryFunctionWithArg: [itor dereference]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i to be less than 673", [[itor dereference] intValue]];
        }
    }
    for ( ; ![itor isEqual: end]; [itor advance])
    {
        if ([func performUnaryFunctionWithArg: [itor dereference]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i to be not less than 673", [[itor dereference] intValue]];
        }
    }
    [func RELEASE];
    [d RELEASE];
}

- (void) testPopHeap
{
    OLVector* v;
    OLNumber* num;
    OLGreater* greater;
    OLArrayIterator* begin;
    OLArrayIterator* end;
    int i;

    v = [[OLVector alloc] init];
    for (i = 0; i < 10000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    begin = REAP([v begin]);
    end = REAP([v end]);
    [OLAlgorithm makeHeapFrom: begin to: end];
    if (![OLAlgorithm isHeapFrom: begin to: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not a heap"];
    }
    [OLAlgorithm popHeapFrom: begin to: end];
    [end reverse];
    if (![OLAlgorithm isHeapFrom: begin to: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not a heap"];
    }
    if ([[end dereference] intValue] != 9999)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 9999, but got %i", [[end dereference] intValue]];
    }
    [end advance];
    greater = [[OLGreater alloc] init];
    [OLAlgorithm makeHeapFrom: begin to: end predicate: greater];
    if (![OLAlgorithm isHeapFrom: begin to: end predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not a heap"];
    }
    [OLAlgorithm popHeapFrom: begin to: end predicate: greater];
    [end reverse];
    if (![OLAlgorithm isHeapFrom: begin to: end predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not a heap"];
    }
    if ([[end dereference] intValue] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", [[end dereference] intValue]];
    }
    [greater RELEASE];
    [v RELEASE];
}

- (void) testPrevPermutation
{
    OLVector* v;
    OLNumber* num;
    OLArrayIterator* begin;
    OLArrayIterator* end;
    OLSet* set;
    OLVector* array;
    OLGreater* greater;
    OLPair* ptrs;
    typedef struct { int one; int two; int three; } Perm;
    Perm perms[] =
    {
        { 3, 2, 1 },
        { 3, 1, 2 },
        { 2, 3, 1 },
        { 2, 1, 3 },
        { 1, 3, 2 },
        { 1, 2, 3 }
    };
    id objs[5];
    int i;
    int j;
    int val;

    v = [[OLVector alloc] init];
    for (i = 3; i >= 1; i--)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    begin = REAP([v begin]);
    end = REAP([v end]);
    for (i = 0; i < 6; i++)
    {
        [self logMessage: "[%i] %i %i %i",
            i, [[v at: 0] intValue], [[v at: 1] intValue], [[v at: 2] intValue]];
        for (j = 0; j < 3; j++)
        {
            val = *((int*)(((int*)&perms[i])) + j);
            if ([[v at: j] intValue] != val)
            {
                [self errInFile: __FILE__ line: __LINE__
                    format: "Expected %i, but got %i",
                    val, [[v at: j] intValue]];
            }
        }
        if (![OLAlgorithm prevPermutationFrom: begin to: end] && i < 5)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected a successful run"];
        }
    }
    [v clear];
    for (i = 0; i < 5; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    set = [[OLSet alloc] init];
    j = 0;
    greater = [[OLGreater alloc] init];
    begin = REAP([v begin]);
    end = REAP([v end]);
    do
    {
        for (i = 0; i < 5; i++)
            objs[i] = [v at: i];
        ptrs = REAP([OLArrayIterator pairWithPointer: objs distance: 5]);
        array = [[OLVector alloc] initFrom: [ptrs first] to: [ptrs second]];
        if (![[REAP([set insert: array]) second] boolValue])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The array was already found"];
        }
        [array RELEASE];
        j++;
    } while ([OLAlgorithm prevPermutationFrom: begin to: end predicate: greater]);
    [self logMessage: "Found %i permutations", j];
    [set RELEASE];
    [greater RELEASE];
    [v RELEASE];
}

- (void) testPushHeap
{
    OLVector* v;
    OLNumber* num;
    OLGreater* greater;
    OLArrayIterator* begin;
    OLArrayIterator* end;
    int i;

    v = [[OLVector alloc] init];
    for (i = 0; i < 10000; i++)
    {
        num = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
        [v pushBack: num];
        [num RELEASE];
    }
    begin = REAP([v begin]);
    end = REAP([v end]);
    [OLAlgorithm makeHeapFrom: begin to: end];
    if (![OLAlgorithm isHeapFrom: begin to: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not a heap"];
    }
    num = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
    [v pushBack: num];
    [num RELEASE];
    begin = REAP([v begin]);
    end = REAP([v end]);
    [OLAlgorithm pushHeapFrom: begin to: end];
    if (![OLAlgorithm isHeapFrom: begin to: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not a heap"];
    }
    greater = [[OLGreater alloc] init];
    [OLAlgorithm makeHeapFrom: begin to: end predicate: greater];
    if (![OLAlgorithm isHeapFrom: begin to: end predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not a heap"];
    }
    num = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
    [v pushBack: num];
    [num RELEASE];
    begin = REAP([v begin]);
    end = REAP([v end]);
    if ([OLAlgorithm isHeapFrom: begin to: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It shouldn't be a heap"];
    }
    [OLAlgorithm pushHeapFrom: begin to: end predicate: greater];
    if (![OLAlgorithm isHeapFrom: begin to: end predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not a heap"];
    }
    [greater RELEASE];
    [v RELEASE];
}

- (void) testRandomShuffle
{
    OLDeque* d;
    OLDeque* d1;
    OLNumber* num;
    RandomGen* gen;
    int i;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 10; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    d1 = [[OLDeque alloc] initWithDeque: d];
    [OLAlgorithm randomShuffleFrom: REAP([d begin]) to: REAP([d end])];
    if ([d isEqual: d1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequences should not be equal"];
    }
    [self logMessage: "First random shuffle: %i %i %i %i %i %i %i %i %i %i",
       [[d at: 0] intValue], [[d at: 1] intValue], [[d at: 2] intValue], [[d at: 3] intValue],
       [[d at: 4] intValue], [[d at: 5] intValue], [[d at: 6] intValue], [[d at: 7] intValue],
       [[d at: 8] intValue], [[d at: 9] intValue]];
    [d1 assignFrom: REAP([d begin]) to: REAP([d end])];
    [OLAlgorithm randomShuffleFrom: REAP([d begin]) to: REAP([d end])];
    if ([d isEqual: d1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequences should not be equal"];
    }
    [self logMessage: "Second random shuffle: %i %i %i %i %i %i %i %i %i %i",
       [[d at: 0] intValue], [[d at: 1] intValue], [[d at: 2] intValue], [[d at: 3] intValue],
       [[d at: 4] intValue], [[d at: 5] intValue], [[d at: 6] intValue], [[d at: 7] intValue],
       [[d at: 8] intValue], [[d at: 9] intValue]];
    [d1 assignFrom: REAP([d begin]) to: REAP([d end])];
    [OLAlgorithm randomShuffleFrom: REAP([d begin]) to: REAP([d end])];
    if ([d isEqual: d1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequences should not be equal"];
    }
    [self logMessage: "Third random shuffle: %i %i %i %i %i %i %i %i %i %i",
       [[d at: 0] intValue], [[d at: 1] intValue], [[d at: 2] intValue], [[d at: 3] intValue],
       [[d at: 4] intValue], [[d at: 5] intValue], [[d at: 6] intValue], [[d at: 7] intValue],
       [[d at: 8] intValue], [[d at: 9] intValue]];
    [d1 assignFrom: REAP([d begin]) to: REAP([d end])];
    [OLAlgorithm randomShuffleFrom: REAP([d begin]) to: REAP([d end])];
    if ([d isEqual: d1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequences should not be equal"];
    }
    [self logMessage: "Fourth random shuffle: %i %i %i %i %i %i %i %i %i %i",
       [[d at: 0] intValue], [[d at: 1] intValue], [[d at: 2] intValue], [[d at: 3] intValue],
       [[d at: 4] intValue], [[d at: 5] intValue], [[d at: 6] intValue], [[d at: 7] intValue],
       [[d at: 8] intValue], [[d at: 9] intValue]];
    gen = [[RandomGen alloc] init];
    [d clear];
    for (i = 0; i < 10000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    [d1 assignFrom: REAP([d begin]) to: REAP([d end])];
    [OLAlgorithm randomShuffleFrom: REAP([d begin]) to: REAP([d end]) randGen: gen];
    [gen RELEASE];
    if ([d isEqual: d1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequences should not be equal"];
    }
    [d RELEASE];
    [d1 RELEASE];
}

- (void) testRemove
{
    OLList* l;
    OLNumber* num;
    LessThanFive* func;
    OLForwardIterator* r;
    OLListIterator* begin;
    OLListIterator* end;
    int i;

    l = [[OLList alloc] init];
    for (i = 0; i < 10; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [num RELEASE];
    }
    num = [[OLNumber alloc] initWithInt: 5];
    begin = REAP([l begin]);
    end = REAP([l end]);
    r = REAP([OLAlgorithm removeFrom: begin to: end value: num]);
    if ([OLIterator distanceFrom: begin to: r] != 9)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 9, but got %i",
            [OLIterator distanceFrom: begin to: r]];
    }
    if ([OLIterator distanceFrom: r to: end] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i",
            [OLIterator distanceFrom: r to: end]];
    }
    if ([OLAlgorithm binarySearchFrom: begin to: r value: num])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number should not have been found"];
    }
    [num RELEASE];
    func = [[LessThanFive alloc] init];
    REAP([l eraseFrom: (OLListIterator*)r to: end]);
    end = REAP([l end]);
    r = REAP([OLAlgorithm removeFrom: begin to: end if: func]);
    [func RELEASE];
    if ([OLIterator distanceFrom: begin to: r] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i",
            [OLIterator distanceFrom: begin to: r]];
    }
    if ([OLIterator distanceFrom: r to: end] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i",
            [OLIterator distanceFrom: r to: end]];
    }
    REAP([l eraseFrom: (OLListIterator*)r to: end]);
    end = REAP([l end]);
    for ( ; ![begin isEqual: end]; [begin advance])
    {
        if ([[begin dereference] intValue] < 5)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "%i is less than 5", [[begin dereference] intValue]];
        }
    }
    [l RELEASE];
}

- (void) testRemoveCopy
{
    OLDeque* d;
    OLVector* v;
    OLNumber* num;
    OLBackInsertIterator* bii;
    LessThanFive* func;
    int i;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 10; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    v = [[OLVector alloc] init];
    bii = [[OLBackInsertIterator alloc] initWithBackInserter: v];
    num = [[OLNumber alloc] initWithInt: 5];
    REAP([OLAlgorithm removeCopyFrom: REAP([d begin]) to: REAP([d end])
        destination: bii value: num]);
    if ([v size] != 9)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 9, but got %i", [v size]];
    }
    if ([OLAlgorithm binarySearchFrom: REAP([v begin]) to: REAP([v end]) value: num])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The number should not have been found"];
    }
    [num RELEASE];
    func = [[LessThanFive alloc] init];
    [v clear];
    REAP([OLAlgorithm removeCopyFrom: REAP([d begin]) to: REAP([d end])
        destination: bii if: func]);
    [func RELEASE];
    for (i = 0; i < [v size]; i++)
    {
        if ([[v at: i] intValue] < 5)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "%i is less than 5", [[v at: i] intValue]];
        }
    }
    [v RELEASE];
    [d RELEASE];
    [bii RELEASE];
}

- (void) testReplace
{
    OLDeque* d;
    OLNumber* num;
    OLNumber* num2;
    LessThanFive* func;
    int i;
    unsigned count;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 20; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    num = [[OLNumber alloc] initWithInt: 13];
    num2 = [[OLNumber alloc] initWithInt: 7527];
    [OLAlgorithm replaceFrom: REAP([d begin]) to: REAP([d end])
        oldValue: num newValue: num2];
    [num RELEASE];
    if ([[d at: 13] intValue] != 7527)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 7527, but got %i", [[d at: 13] intValue]];
    }
    count = [OLAlgorithm countFrom: REAP([d begin]) to: REAP([d end]) value: num2];
    if (count != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", count];
    }
    func = [[LessThanFive alloc] init];
    [OLAlgorithm replaceFrom: REAP([d begin]) to: REAP([d end])
        if: func newValue: num2];
    [func RELEASE];
    [num2 RELEASE];
    for (i = 0; i < 5; i++)
    {
        if ([[d at: i] intValue] != 7527)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 7527, but got %i", [[d at: i] intValue]];
        }
    }
    for ( ; i < 13; i++)
    {
        if ([[d at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", i, [[d at: i] intValue]];
        }
    }
    if ([[d at: i] intValue] != 7527)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 7527, but got %i", [[d at: i] intValue]];
    }
    for (i++ ; i < 20; i++)
    {
        if ([[d at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", i, [[d at: i] intValue]];
        }
    }
    [d RELEASE];
}

- (void) testReplaceCopy
{
    OLDeque* d;
    OLVector* v;
    OLNumber* num;
    OLNumber* num2;
    OLBackInsertIterator* bii;
    LessThanFive* func;
    int i;
    unsigned count;

    d = [[OLDeque alloc] init];
    v = [[OLVector alloc] init];
    bii = [[OLBackInsertIterator alloc] initWithBackInserter: v];
    for (i = 0; i < 20; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    num = [[OLNumber alloc] initWithInt: 13];
    num2 = [[OLNumber alloc] initWithInt: 7527];
    REAP([OLAlgorithm replaceCopyFrom: REAP([d begin]) to: REAP([d end]) destination: bii
        oldValue: num newValue: num2]);
    [num RELEASE];
    if ([v size] != [d size])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", [d size], [v size]];
    }
    if ([[v at: 13] intValue] != 7527)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 7527, but got %i", [[v at: 13] intValue]];
    }
    if ([[d at: 13] intValue] != 13)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 13, but got %i", [[d at: 13] intValue]];
    }
    count = [OLAlgorithm countFrom: REAP([v begin]) to: REAP([v end]) value: num2];
    if (count != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", count];
    }
    [v clear];
    func = [[LessThanFive alloc] init];
    REAP([OLAlgorithm replaceCopyFrom: REAP([d begin]) to: REAP([d end]) destination: bii
        if: func newValue: num2]);
    [func RELEASE];
    [num2 RELEASE];
    if ([v size] != [d size])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", [d size], [v size]];
    }
    for (i = 0; i < 5; i++)
    {
        if ([[v at: i] intValue] != 7527)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 7527, but got %i", [[v at: i] intValue]];
        }
    }
    for ( ; i < 20; i++)
    {
        if ([[v at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", i, [[v at: i] intValue]];
        }
    }
    for (i = 0; i < 20; i++)
    {
        if ([[d at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", i, [[d at: i] intValue]];
        }
    }
    [d RELEASE];
    [v RELEASE];
    [bii RELEASE];
}

- (void) testReverse
{
    OLVector* v;
    OLNumber* num;
    int i;

    v = [[OLVector alloc] init];
    for (i = 0; i < 100; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    [OLAlgorithm reverseFrom: REAP([v begin]) to: REAP([v end])];
    for (i = 0; i < 100; i++)
    {
        if ([[v at: i] intValue] != 99 - i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got, %i",
                99 - i, [[v at: i] intValue]];
        }
    }
    [v RELEASE];
}

- (void) testReverseCopy
{
    OLVector* v;
    OLList* l;
    OLNumber* num;
    OLBackInsertIterator* bii;
    OLReverseBidiIterator* litor;
    OLReverseBidiIterator* lend;
    OLArrayIterator* vitor;
    OLArrayIterator* vend;
    int i;

    l = [[OLList alloc] init];
    v = [[OLVector alloc] init];
    bii = [[OLBackInsertIterator alloc] initWithBackInserter: v];
    for (i = 0; i < 100; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [num RELEASE];
    }
    REAP([OLAlgorithm reverseCopyFrom: REAP([l begin]) to: REAP([l end]) destination: bii]);
    if ([v size] != [l size])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got, %u", [l size], [v size]];
    }
    for (litor = REAP([l rbegin]), lend = REAP([l rend]),
         vitor = REAP([v begin]), vend = REAP([v end]);
         ![vitor isEqual: vend];
         [vitor advance], [litor advance])
    {
        if ([[vitor dereference] intValue] != [[litor dereference] intValue])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got, %i",
                [[litor dereference] intValue], [[vitor dereference] intValue]];
        }
    }
    [l RELEASE];
    [v RELEASE];
    [bii RELEASE];
}

- (void) testRotate
{
    OLList* l;
    CONSTSTR* strs[] = { @"one", @"two", @"three" };
    OLForwardIterator* r;
    OLListIterator* itor;
    OLListIterator* mid;
    OLListIterator* end;
    OLVector* v;
    OLArrayIterator* vtor;
    OLArrayIterator* vend;
    OLNumber* num;
    int i;

    l = [[OLList alloc] init];
    [l pushBack: strs[0]];
    [l pushBack: strs[1]];
    [l pushBack: strs[2]];
    r = REAP([OLAlgorithm rotateFrom: REAP([l begin])
        middle: [REAP([l begin]) advance] to: REAP([l end])]);
    if (![[r dereference] isEqual: strs[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[r dereference] cString]];
    }
    if (![[l front] isEqual: strs[1]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[l front] cString]];
    }
    itor = [REAP([l begin]) advance];
    if (![[itor dereference] isEqual: strs[2]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"three\", but got \"%s\"", [[itor dereference] cString]];
    }
    if (![[l back] isEqual: strs[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[l back] cString]];
    }
    [l clear];
    for (i = 0; i < 100; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [num RELEASE];
    }
    mid = REAP([l begin]);
    [OLIterator advanceIterator: mid distance: 51];
    r = REAP([OLAlgorithm rotateFrom: REAP([l begin]) middle: mid to: REAP([l end])]);
    if ([[r dereference] intValue] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", [[r dereference] intValue]];
    }
    for (itor = REAP([l begin]), i = 51; ![itor isEqual: r]; [itor advance], i++)
    {
        if ([[itor dereference] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", i, [[itor dereference] intValue]];
        }
    }
    end = REAP([l end]);
    for (i = 0; ![r isEqual: end]; [r advance], i++)
    {
        if ([[r dereference] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", i, [[r dereference] intValue]];
        }
    }
    [l sort];
    v = [[OLVector alloc] initWithSize: [l size] filledWith: @"yo"];
    mid = REAP([l begin]);
    [OLIterator advanceIterator: mid distance: 37];
    r = REAP([OLAlgorithm rotateCopyFrom: REAP([l begin])
        middle: mid to: REAP([l end]) destination: REAP([v begin])]);
    [l RELEASE];
    vend = REAP([v end]);
    if (![r isEqual: vend])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The iterator must be at the end of the sequence"];
    }
    for (vtor = REAP([v begin]), i = 37; i < 100; [vtor advance], i++)
    {
        if ([[vtor dereference] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", i, [[vtor dereference] intValue]];
        }
    }
    for (i = 0; i < 37; [vtor advance], i++)
    {
        if ([[vtor dereference] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", i, [[vtor dereference] intValue]];
        }
    }
    [v RELEASE];
}

- (void) testSearch
{
    OLDeque* d;
    OLVector* v;
    OLNumber* num;
    OLForwardIterator* r;
    IntCompare* comp;
    OLDequeIterator* dbegin;
    OLDequeIterator* dend;
    int i;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 1000; i++)
    {
        num = [[OLNumber alloc] initWithDouble: i + .5];
        [d pushBack: num];
        [num RELEASE];
    }
    v = [[OLVector alloc] init];
    for (i = 784; i < 899; i++)
    {
        num = [[OLNumber alloc] initWithDouble: i + 0.5];
        [v pushBack: num];
        [d pushBack: num];
        [num RELEASE];
    }
    dbegin = REAP([d begin]);
    dend = REAP([d end]);
    r = REAP([OLAlgorithm searchFrom: dbegin to: dend
        subFrom: REAP([v begin]) subTo: REAP([v end])]);
    if ([r isEqual: dend])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The value should have been found"];
    }
    if ([[r dereference] doubleValue] != 784.5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 784, but got %i", [[r dereference] intValue]];
    }
    if ([OLIterator distanceFrom: dbegin to: r] != 784)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Search found the last one, not the first one (distance %u)",
            [OLIterator distanceFrom: dbegin to: r]];
    }
    [v clear];
    for (i = 784; i < 899; i++)
    {
        num = [[OLNumber alloc] initWithDouble: i];
        [v pushBack: num];
        [num RELEASE];
    }
    comp = [[IntCompare alloc] init];
    r = REAP([OLAlgorithm searchFrom: dbegin to: dend
        subFrom: REAP([v begin]) subTo: REAP([v end])
        predicate: comp]);
    [comp RELEASE];
    if ([r isEqual: dend])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The value should have been found"];
    }
    if ([[r dereference] doubleValue] != 784.5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 784, but got %i", [[r dereference] intValue]];
    }
    if ([OLIterator distanceFrom: dbegin to: r] != 784)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Search found the last one, not the first one"];
    }
    [d RELEASE];
    [v RELEASE];
}

- (void) testSearchCount
{
    OLVector* v;
    OLNumber* num;
    OLForwardIterator* r;
    OLArrayIterator* begin;
    OLArrayIterator* end;
    IntCompare* comp;
    int nums[] = { 1, 2, 2, 7, 2, 2, 2, 4, 9, 6 };
    int i;

    v = [[OLVector alloc] init];
    for (i = 0; i < 10; i++)
    {
        num = [[OLNumber alloc] initWithDouble: nums[i] + 0.3];
        [v pushBack: num];
        [num RELEASE];
    }
    num = [[OLNumber alloc] initWithDouble: 2.3];
    begin = REAP([v begin]);
    end = REAP([v end]);
    r = REAP([OLAlgorithm searchFrom: begin to: end count: 3 value: num]);
    [num RELEASE];
    if ([r isEqual: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequence should have been found"];
    }
    if ([OLIterator distanceFrom: begin to: r] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %u",
            [OLIterator distanceFrom: begin to: r]];
    }
    comp = [[IntCompare alloc] init];
    num = [[OLNumber alloc] initWithDouble: 2.0];
    r = REAP([OLAlgorithm searchFrom: begin to: end count: 3 value: num predicate: comp]);
    [num RELEASE];
    [comp RELEASE];
    if ([r isEqual: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The sequence should have been found"];
    }
    if ([OLIterator distanceFrom: begin to: r] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %u",
            [OLIterator distanceFrom: begin to: r]];
    }
    [v RELEASE];
}

- (void) testSetDifference
{
    OLDeque* d;
    OLList* l;
    OLVector* v;
    OLNumber* num;
    OLBackInsertIterator* bii;
    OLGreater* greater;
    int i;

    d = [[OLDeque alloc] init];
    l = [[OLList alloc] init];
    v = [[OLVector alloc] init];
    for (i = 0; i < 50; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [num RELEASE];
    }
    for ( ; i < 100; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [d pushBack: num];
        [num RELEASE];
    }
    for ( ; i < 150; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    bii = [[OLBackInsertIterator alloc] initWithBackInserter: v];
    REAP([OLAlgorithm setDifferenceFrom: REAP([l begin]) to: REAP([l end])
        andFrom: REAP([d begin]) andTo: REAP([d end]) destination: bii]);
    if ([v size] != 50)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 50, but got %i", [v size]];
    }
    for (i = 0; i < 50; i++)
    {
        if ([[v at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[v at: i] intValue]];
        }
    }
    [v clear];
    greater = [[OLGreater alloc] init];
    [l sortWith: greater];
    [OLAlgorithm sortFrom: REAP([d begin]) to: REAP([d end]) predicate: greater];
    REAP([OLAlgorithm setDifferenceFrom: REAP([l begin]) to: REAP([l end])
        andFrom: REAP([d begin]) andTo: REAP([d end]) destination: bii predicate: greater]);
    [greater RELEASE];
    if ([v size] != 50)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 50, but got %i", [v size]];
    }
    for (i = 0; i < 50; i++)
    {
        if ([[v at: i] intValue] != 49 - i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                49 - i, [[v at: i] intValue]];
        }
    }
    [v RELEASE];
    [l RELEASE];
    [d RELEASE];
    [bii RELEASE];
}

- (void) testSetIntersection
{
    OLDeque* d;
    OLList* l;
    OLVector* v;
    OLNumber* num;
    OLBackInsertIterator* bii;
    OLGreater* greater;
    int i;

    d = [[OLDeque alloc] init];
    l = [[OLList alloc] init];
    v = [[OLVector alloc] init];
    for (i = 0; i < 50; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [num RELEASE];
    }
    for ( ; i < 100; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [d pushBack: num];
        [num RELEASE];
    }
    for ( ; i < 150; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    bii = [[OLBackInsertIterator alloc] initWithBackInserter: v];
    REAP([OLAlgorithm setIntersectionFrom: REAP([l begin]) to: REAP([l end])
        andFrom: REAP([d begin]) andTo: REAP([d end]) destination: bii]);
    if ([v size] != 50)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 50, but got %i", [v size]];
    }
    for (i = 50; i < 100; i++)
    {
        if ([[v at: i - 50] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[v at: i - 50] intValue]];
        }
    }
    [v clear];
    greater = [[OLGreater alloc] init];
    [l sortWith: greater];
    [OLAlgorithm sortFrom: REAP([d begin]) to: REAP([d end]) predicate: greater];
    REAP([OLAlgorithm setIntersectionFrom: REAP([l begin]) to: REAP([l end])
        andFrom: REAP([d begin]) andTo: REAP([d end]) destination: bii predicate: greater]);
    [greater RELEASE];
    if ([v size] != 50)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 50, but got %i", [v size]];
    }
    for (i = 0; i < 50; i++)
    {
        if ([[v at: i] intValue] != 99 - i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                99 - i, [[v at: i] intValue]];
        }
    }
    [v RELEASE];
    [l RELEASE];
    [d RELEASE];
    [bii RELEASE];
}

- (void) testSetSymmetricDifference
{
    OLDeque* d;
    OLList* l;
    OLVector* v;
    OLNumber* num;
    OLBackInsertIterator* bii;
    OLGreater* greater;
    int i;

    d = [[OLDeque alloc] init];
    l = [[OLList alloc] init];
    v = [[OLVector alloc] init];
    for (i = 0; i < 50; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [num RELEASE];
    }
    for ( ; i < 100; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [d pushBack: num];
        [num RELEASE];
    }
    for ( ; i < 150; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    bii = [[OLBackInsertIterator alloc] initWithBackInserter: v];
    REAP([OLAlgorithm setSymmetricDifferenceFrom: REAP([l begin]) to: REAP([l end])
        andFrom: REAP([d begin]) andTo: REAP([d end]) destination: bii]);
    if ([v size] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 100, but got %i", [v size]];
    }
    for (i = 0; i < 50; i++)
    {
        if ([[v at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[v at: i] intValue]];
        }
    }
    for (i = 100; i < 150; i++)
    {
        if ([[v at: i - 50] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[v at: i - 50] intValue]];
        }
    }
    [v clear];
    greater = [[OLGreater alloc] init];
    [l sortWith: greater];
    [OLAlgorithm sortFrom: REAP([d begin]) to: REAP([d end]) predicate: greater];
    REAP([OLAlgorithm setSymmetricDifferenceFrom: REAP([l begin]) to: REAP([l end])
        andFrom: REAP([d begin]) andTo: REAP([d end]) destination: bii predicate: greater]);
    [greater RELEASE];
    if ([v size] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 50, but got %i", [v size]];
    }
    for (i = 0; i < 50; i++)
    {
        if ([[v at: i] intValue] != 149 - i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                149 - i, [[v at: i] intValue]];
        }
    }
    for ( ; i < 100; i++)
    {
        if ([[v at: i] intValue] != 49 - (i - 50))
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                49 - (i - 50), [[v at: i] intValue]];
        }
    }
    [v RELEASE];
    [l RELEASE];
    [d RELEASE];
    [bii RELEASE];
}

- (void) testSetUnion
{
    OLDeque* d;
    OLList* l;
    OLVector* v;
    OLNumber* num;
    OLBackInsertIterator* bii;
    OLGreater* greater;
    int i;

    d = [[OLDeque alloc] init];
    l = [[OLList alloc] init];
    v = [[OLVector alloc] init];
    for (i = 0; i < 50; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [num RELEASE];
    }
    for ( ; i < 100; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [d pushBack: num];
        [num RELEASE];
    }
    for ( ; i < 150; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [d pushBack: num];
        [num RELEASE];
    }
    bii = [[OLBackInsertIterator alloc] initWithBackInserter: v];
    REAP([OLAlgorithm setUnionFrom: REAP([l begin]) to: REAP([l end])
        andFrom: REAP([d begin]) andTo: REAP([d end]) destination: bii]);
    if ([v size] != 150)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 150, but got %i", [v size]];
    }
    for (i = 0; i < 150; i++)
    {
        if ([[v at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[v at: i] intValue]];
        }
    }
    [v clear];
    greater = [[OLGreater alloc] init];
    [l sortWith: greater];
    [OLAlgorithm sortFrom: REAP([d begin]) to: REAP([d end]) predicate: greater];
    REAP([OLAlgorithm setUnionFrom: REAP([l begin]) to: REAP([l end])
        andFrom: REAP([d begin]) andTo: REAP([d end]) destination: bii predicate: greater]);
    [greater RELEASE];
    if ([v size] != 150)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 50, but got %i", [v size]];
    }
    for (i = 0; i < 150; i++)
    {
        if ([[v at: i] intValue] != 149 - i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                149 - i, [[v at: i] intValue]];
        }
    }
    [v RELEASE];
    [l RELEASE];
    [d RELEASE];
    [bii RELEASE];
}

- (void) testSort
{
    OLVector* v;
    OLNumber* num;
    OLGreater* greater;
    int i;
    int next;

    v = [[OLVector alloc] init];
    for (i = 0; i < 1000; i++)
    {
        num = [[OLNumber alloc] initWithInt: OLRandom() % 1000];
        [v pushBack: num];
        [num RELEASE];
    }
    [OLAlgorithm sortFrom: REAP([v begin]) to: REAP([v end])];
    for (i = 0, next = 1; next < 1000; i++, next++)
    {
        if ([[v at: next] intValue] < [[v at: i] intValue])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The sequence is not sorted"];
        }
    }
    greater = [[OLGreater alloc] init];
    [OLAlgorithm sortFrom: REAP([v begin]) to: REAP([v end]) predicate: greater];
    [greater RELEASE];
    for (i = 0, next = 1; next < 1000; i++, next++)
    {
        if ([[v at: next] intValue] > [[v at: i] intValue])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The sequence is not sorted"];
        }
    }
    [v RELEASE];
}

- (void) testSortHeap
{
    OLVector* v;
    OLNumber* num;
    OLGreater* greater;
    OLArrayIterator* begin;
    OLArrayIterator* end;
    int i;

    v = [[OLVector alloc] init];
    for (i = 0; i < 50000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    begin = REAP([v begin]);
    end = REAP([v end]);
    [OLAlgorithm makeHeapFrom: begin to: end];
    if (![OLAlgorithm isHeapFrom: begin to: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not a heap"];
    }
    [OLAlgorithm sortHeapFrom: begin to: end];
    if (![OLAlgorithm isSortedFrom: begin to: end])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not sorted"];
    }
    greater = [[OLGreater alloc] init];
    [OLAlgorithm makeHeapFrom: begin to: end predicate: greater];
    if (![OLAlgorithm isHeapFrom: begin to: end predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not a heap"];
    }
    [OLAlgorithm sortHeapFrom: begin to: end predicate: greater];
    if (![OLAlgorithm isSortedFrom: begin to: end predicate: greater])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "It's not sorted"];
    }
    [greater RELEASE];
    [v RELEASE];
}

- (void) testStablePartition
{
    OLVector* v;
    OLVector* v2;
    OLNumber* num;
    OLArrayIterator* itor;
    OLArrayIterator* end;
    OLForwardIterator* r;
    PartitionFunction* func;
    OLArrayIterator* itor2;
    int i;

    v = [[OLVector alloc] init];
    for (i = 0; i < 10000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    itor = REAP([v begin]);
    end = REAP([v end]);
    [OLAlgorithm randomShuffleFrom: itor to: end];
    v2 = [[OLVector alloc] initWithVector: v];
    func = [[PartitionFunction alloc] initWithValue: 4723];
    r = REAP([OLAlgorithm stablePartitionFrom: itor to: end predicate: func]);
    for ( ; ![itor isEqual: r]; [itor advance])
    {
        if (![func performUnaryFunctionWithArg: [itor dereference]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i to be less than 4723", [[itor dereference] intValue]];
        }
    }
    for ( ; ![itor isEqual: end]; [itor advance])
    {
        if ([func performUnaryFunctionWithArg: [itor dereference]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i to be not less than 4723", [[itor dereference] intValue]];
        }
    }
    [func RELEASE];
    for ([itor advanceBy: -[v size]], itor2 = [REAP([v begin]) advance];
         ![itor2 isEqual: r];
         [itor advance], [itor2 advance])
    {
        if (__indexOf(itor2, v2) < __indexOf(itor, v2))
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i to be not less %i",
                [[itor2 dereference] intValue], [[itor dereference] intValue]];
        }
    }
    for ([itor advance], [itor2 advance];
         ![itor2 isEqual: end];
         [itor advance], [itor2 advance])
    {
        if (__indexOf(itor2, v2) < __indexOf(itor, v2))
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i to be not less %i",
                [[itor2 dereference] intValue], [[itor dereference] intValue]];
        }
    }
    [v RELEASE];
    [v2 RELEASE];
}

- (void) testStableSort
{
    OLDeque* d;
    ComparablePair* p;
    OLNumber* num;
    OLNumber* num2;
    OLDequeIterator* itor;
    OLDequeIterator* end;
    OLGreater* greater;
    int i;
    int j;

    d = [[OLDeque alloc] init];
    for (i = 0; i < 1000; i++)
    {
        for (j = 0; j < 10; j++)
        {
            num = [[OLNumber alloc] initWithInt: j];
            p = [[ComparablePair alloc] initWithFirst: num second: nil];
            [num RELEASE];
            [d pushBack: p];
            [p RELEASE];
        }
    }
    [OLAlgorithm randomShuffleFrom: REAP([d begin]) to: REAP([d end])];
    for (i = 0; i < 10; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        for (itor = REAP([d begin]), end = REAP([d end]), j = 0;
             ![itor isEqual: end]; [itor advance])
        {
            if ([[[itor dereference] first] isEqual: num])
            {
                num2 = [[OLNumber alloc] initWithInt: j++];
                [[itor dereference] setSecond: num2];
                [num2 RELEASE];
            }
        }
        [num RELEASE];
    }
    [OLAlgorithm stableSortFrom: REAP([d begin]) to: REAP([d end])];
    for (i = 0, itor = REAP([d begin]); i < 10; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        for (j = 0; j < 1000; j++, [itor advance])
        {
            num2 = [[OLNumber alloc] initWithInt: j];
            if (!([[[itor dereference] first] isEqual: num] &&
                  [[[itor dereference] second] isEqual: num2]))
            {
                [self errInFile: __FILE__ line: __LINE__
                    format: "Expected {%i, %i}, but got {%i, %i}",
                    i, j,
                    [[[itor dereference] first] intValue],
                    [[[itor dereference] second] intValue]];
            }
            [num2 RELEASE];
        }
        [num RELEASE];
    }
    [d RELEASE];
    d = [[OLDeque alloc] init];
    for (i = 0; i < 1000; i++)
    {
        for (j = 0; j < 10; j++)
        {
            num = [[OLNumber alloc] initWithInt: j];
            p = [[ComparablePair alloc] initWithFirst: num second: nil];
            [num RELEASE];
            [d pushBack: p];
            [p RELEASE];
        }
    }
    [OLAlgorithm randomShuffleFrom: REAP([d begin]) to: REAP([d end])];
    for (i = 0; i < 10; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        for (itor = REAP([d begin]), end = REAP([d end]), j = 0;
             ![itor isEqual: end]; [itor advance])
        {
            if ([[[itor dereference] first] isEqual: num])
            {
                num2 = [[OLNumber alloc] initWithInt: j++];
                [[itor dereference] setSecond: num2];
                [num2 RELEASE];
            }
        }
        [num RELEASE];
    }
    greater = [[OLGreater alloc] init];
    [OLAlgorithm stableSortFrom: REAP([d begin]) to: REAP([d end]) predicate: greater];
    [greater RELEASE];
    for (i = 9, itor = REAP([d begin]); i >= 0; i--)
    {
        num = [[OLNumber alloc] initWithInt: i];
        for (j = 0; j < 1000; j++, [itor advance])
        {
            num2 = [[OLNumber alloc] initWithInt: j];
            if (!([[[itor dereference] first] isEqual: num] &&
                  [[[itor dereference] second] isEqual: num2]))
            {
                [self errInFile: __FILE__ line: __LINE__
                    format: "Expected {%i, %i}, but got {%i, %i}",
                    i, j,
                    [[[itor dereference] first] intValue],
                    [[[itor dereference] second] intValue]];
            }
            [num2 RELEASE];
        }
        [num RELEASE];
    }
    [d RELEASE];
}

- (void) testSwap
{
    CONSTSTR* one = @"one";
    CONSTSTR* two = @"two";
    OLVector* v;
    OLList* l;

    [OLAlgorithm swap: &one and: &two];
    if (![one isEqual: @"two"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [one cString]];
    }
    if (![two isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [two cString]];
    }
    [OLAlgorithm swap: &one and: &two];
    v = [[OLVector alloc] init];
    [v pushBack: one];
    [v pushBack: two];
    [OLAlgorithm swapIterators: REAP([v begin]) and: [REAP([v end]) reverse]];
    if (![[v front] isEqual: @"two"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[v front] cString]];
    }
    if (![[v back] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[v back] cString]];
    }
    l = [[OLList alloc] init];
    [l pushBack: one];
    [OLAlgorithm swapIterators: REAP([v begin]) and: REAP([l begin])];
    if (![[v front] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[v front] cString]];
    }
    if (![[l front] isEqual: @"two"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[l front] cString]];
    }
    [v RELEASE];
    [l RELEASE];
}

- (void) testSwapRanges
{
    OLList* l;
    OLVector* v;
    OLNumber* num;
    OLForwardIterator* r;
    OLListIterator* lend;
    OLListIterator* litor;
    int i;

    l = [[OLList alloc] init];
    v = [[OLVector alloc] init];
    [self logMessage: "About to build list"];
    for (i = 0; i < 10000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [l pushBack: num];
        [num RELEASE];
    }
    [self logMessage: "Done building list"];
    [self logMessage: "About to build vector"];
    for ( ; i < 20000; i++)
    {
        num = [[OLNumber alloc] initWithInt: i];
        [v pushBack: num];
        [num RELEASE];
    }
    [self logMessage: "Done building vector"];
    lend = REAP([l end]);
    litor = REAP([l begin]);
    [self logMessage: "About to swap ranges"];
    r = REAP([OLAlgorithm swapRangesFrom: litor to: lend with: REAP([v begin])]);
    [self logMessage: "Done swapping ranges"];
    if (![r isEqual: REAP([v end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The result should be at the end of the vector"];
    }
    for (i = 0; i < 10000; i++)
    {
        if ([[v at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[v at: i] intValue]];
        }
    }
    for ( ; ![litor isEqual: lend]; [litor advance], i++)
    {
        if ([[litor dereference] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                i, [[litor dereference] intValue]];
        }
    }
    [v RELEASE];
    [l RELEASE];
}

- (void) testTransform
{
    OLVector* v;
    OLDeque* d;
    OLVector* v2;
    const char* strs[] = { "one", "two", "three" };
    const char* strs2[] = { "yehudi", "stefanie", "clara" };
    AppendDoggy* func;
    AppendOther* bfunc;
    OLBackInsertIterator* bii;
    OLText* text;
    int i;

    v = [[OLVector alloc] init];
    for (i = 0; i < 3; i++)
    {
        text = [[OLText alloc] initWithCString: strs[i]];
        [v pushBack: text];
        [text RELEASE];
    }
    func = [[AppendDoggy alloc] init];
    d = [[OLDeque alloc] init];
    bii = [[OLBackInsertIterator alloc] initWithBackInserter: d];
    REAP([OLAlgorithm transformFrom: REAP([v begin]) to: REAP([v end])
        destination: bii function: func]);
    [func RELEASE];
    if (![[d at: 0] isEqual: @"onedoggy"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"onedoggy\", but got \"%s\"",
            [[d at: 0] cString]];
    }
    if (![[d at: 1] isEqual: @"twodoggy"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"twodoggy\", but got \"%s\"",
            [[d at: 1] cString]];
    }
    if (![[d at: 2] isEqual: @"threedoggy"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"threedoggy\", but got \"%s\"",
            [[d at: 2] cString]];
    }
    [d clear];
    v2 = [[OLVector alloc] init];
    for (i = 0; i < 3; i++)
    {
        text = [[OLText alloc] initWithCString: strs2[i]];
        [v2 pushBack: text];
        [text RELEASE];
    }
    bfunc = [[AppendOther alloc] init];
    REAP([OLAlgorithm transformFrom: REAP([v begin]) to: REAP([v end])
        withArgsFrom: REAP([v2 begin]) destination: bii function: bfunc]);
    [bfunc RELEASE];
    [v2 RELEASE];
    if (![[d at: 0] isEqual: @"oneyehudi"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"oneyehudi\", but got \"%s\"",
            [[d at: 0] cString]];
    }
    if (![[d at: 1] isEqual: @"twostefanie"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"twostefanie\", but got \"%s\"",
            [[d at: 1] cString]];
    }
    if (![[d at: 2] isEqual: @"threeclara"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"threeclara\", but got \"%s\"",
            [[d at: 2] cString]];
    }
    [v RELEASE];
    [d RELEASE];
    [bii RELEASE];
}

- (void) testUnique
{
    OLVector* v;
    OLNumber* num;
    IntCompare* comp;
    OLArrayIterator* begin;
    OLForwardIterator* r;
    int i;
    int j;
    int jmax;
    double i2;

    v = [[OLVector alloc] init];
    for (i = 0; i < 10; i++)
    {
        jmax = OLRandom() % 1000 + 1;
        for (j = 0; j < jmax; j++)
        {
            num = [[OLNumber alloc] initWithInt: i];
            [v pushBack: num];
            [num RELEASE];
        }
    }
    [self logMessage: "Looking for 10 unique elements from %u", [v size]];
    begin = REAP([v begin]);
    r = REAP([OLAlgorithm uniqueFrom: begin to: REAP([v end])]);
    if ([OLIterator distanceFrom: begin to: r] != 10)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 10, but got %u", [OLIterator distanceFrom: begin to: r]];
    }
    for (i = 0; i < 10; i++)
    {
        if ([[v at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", i, [[v at: i] intValue]];
        }
    }
    [v clear];
    for (i2 = 0.0; i2 < 100.0; i2 += 0.1)
    {
        num = [[OLNumber alloc] initWithDouble: i2];
        [v pushBack: num];
        [num RELEASE];
    }
    comp = [[IntCompare alloc] init];
    begin = REAP([v begin]);
    r = REAP([OLAlgorithm uniqueFrom: begin to: REAP([v end]) predicate: comp]);
    if ([OLIterator distanceFrom: begin to: r] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 100, but got %u", [OLIterator distanceFrom: begin to: r]];
    }
    for (i = 0; i < 100; i++)
    {
        num = [[OLNumber alloc] initWithDouble: (double)i];
        if (![comp performBinaryFunctionWithArg: [v at: i] andArg: num])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Position %i doesn't test", i];
        }
        [num RELEASE];
    }
    [comp RELEASE];
    [v RELEASE];
}

- (void) testUniqueCopy
{
    OLList* l;
    OLVector* v;
    OLBackInsertIterator* bii;
    OLNumber* num;
    IntCompare* comp;
    int i;
    int j;
    int jmax;
    double i2;

    l = [[OLList alloc] init];
    for (i = 0; i < 10; i++)
    {
        jmax = OLRandom() % 1000 + 1;
        for (j = 0; j < jmax; j++)
        {
            num = [[OLNumber alloc] initWithInt: i];
            [l pushBack: num];
            [num RELEASE];
        }
    }
    [self logMessage: "Looking for 10 unique elements from %u", [l size]];
    v = [[OLVector alloc] init];
    bii = [[OLBackInsertIterator alloc] initWithBackInserter: v];
    REAP([OLAlgorithm uniqueCopyFrom: REAP([l begin]) to: REAP([l end]) destination: bii]);
    if ([v size] != 10)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 10, but got %u", [v size]];
    }
    for (i = 0; i < 10; i++)
    {
        if ([[v at: i] intValue] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", i, [[v at: i] intValue]];
        }
    }
    [l clear];
    [v clear];
    for (i2 = 0.0; i2 < 100.0; i2 += 0.1)
    {
        num = [[OLNumber alloc] initWithDouble: i2];
        [l pushBack: num];
        [num RELEASE];
    }
    comp = [[IntCompare alloc] init];
    REAP([OLAlgorithm uniqueCopyFrom: REAP([l begin]) to: REAP([l end])
        destination: bii predicate: comp]);
    if ([v size] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 100, but got %u", [v size]];
    }
    for (i = 0; i < 100; i++)
    {
        num = [[OLNumber alloc] initWithDouble: (double)i];
        if (![comp performBinaryFunctionWithArg: [v at: i] andArg: num])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Position %i doesn't test", i];
        }
        [num RELEASE];
    }
    [comp RELEASE];
    [v RELEASE];
    [l RELEASE];
    [bii RELEASE];
}

- (void) testUpperBound
{
    OLList* l;
    OLNumber* num;
    OLNumber* target;
    OLForwardIterator* r;
    OLGreater* greater;
    double i;

    l = [[OLList alloc] init];
    for (i = 0.0; i < 1000.0; i++)
    {
        num = [[OLNumber alloc] initWithDouble: i];
        [l pushBack: num];
        [num RELEASE];
    }
    target = [[OLNumber alloc] initWithDouble: 39.5];
    r = REAP([OLAlgorithm upperBoundFrom: REAP([l begin]) to: REAP([l end]) value: target]);
    [target RELEASE];
    if ([[r dereference] doubleValue] != 40.0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 40.0, but got %G", [[r dereference] doubleValue]];
    }
    target = [[OLNumber alloc] initWithDouble: 700];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end]) value: target]);
    [target RELEASE];
    if ([[r dereference] doubleValue] != 700.0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 700.0, but got %G", [[r dereference] doubleValue]];
    }
    target = [[OLNumber alloc] initWithDouble: 10000.0];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end]) value: target]);
    [target RELEASE];
    if (![r isEqual: REAP([l end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected to be at the end"];
    }
    target = [[OLNumber alloc] initWithDouble: -10000.0];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end]) value: target]);
    [target RELEASE];
    if (![r isEqual: REAP([l begin])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected to be at the beginning"];
    }
    greater = [[OLGreater alloc] init];
    [l sortWith: greater];
    target = [[OLNumber alloc] initWithDouble: 39.5];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end])
        value: target predicate: greater]);
    [target RELEASE];
    if ([[r dereference] doubleValue] != 39.0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 40.0, but got %G", [[r dereference] doubleValue]];
    }
    target = [[OLNumber alloc] initWithDouble: 700];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end])
        value: target predicate: greater]);
    [target RELEASE];
    if ([[r dereference] doubleValue] != 700.0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 700.0, but got %G", [[r dereference] doubleValue]];
    }
    target = [[OLNumber alloc] initWithDouble: 10000.0];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end])
        value: target predicate: greater]);
    [target RELEASE];
    if (![r isEqual: REAP([l begin])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected to be at the beginning"];
    }
    target = [[OLNumber alloc] initWithDouble: -10000.0];
    r = REAP([OLAlgorithm lowerBoundFrom: REAP([l begin]) to: REAP([l end])
        value: target predicate: greater]);
    [target RELEASE];
    if (![r isEqual: REAP([l end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected to be at the end"];
    }
    [greater RELEASE];
    [l RELEASE];
}

@end
