//
// $Id: Set.m,v 1.24 2007/03/06 20:42:20 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "Set.h"
#import "Tree.h"
#import "Macros.h"
#import "RunTime.h"
#import "ObjectInStream.h"
#import "ObjectOutStream.h"
#import "Utility.h"
#if !defined(OL_NO_OPENSTEP)
#import <Foundation/NSCoder.h>
#import <Foundation/NSAutoreleasePool.h>
#endif
#import <stdlib.h>

#if !defined(OL_NO_OPENSTEP)
extern NSString* const COMPARATOR_KEY;
#endif

@implementation OLSet

+ (id) set
{
    OL_BEGIN_AUTO_CTOR(OLSet)
        init
    OL_END_AUTO_CTOR;
}

+ (id) setFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OL_BEGIN_AUTO_CTOR(OLSet)
        initFrom: first to: last
    OL_END_AUTO_CTOR;
}

+ (id) setWithCompare: (OLStreamableFunctor<OLBoolBinaryFunction>*)comp
{
    OL_BEGIN_AUTO_CTOR(OLSet)
        initWithCompare: comp
    OL_END_AUTO_CTOR;
}

+ (id) setWithOLSet: (OLSet*)right
{
    OL_BEGIN_AUTO_CTOR(OLSet)
        initWithOLSet: right
    OL_END_AUTO_CTOR;
}

- (id) init
{
    [super init];
    tree = [[OLTree alloc] init];
    return self;
}

- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    [self init];
    [self insertFrom: first to: last];
    return self;
}

- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last compare: (OLStreamableFunctor<OLBoolBinaryFunction>*)comp
{
    [self initWithCompare: comp];
    [self insertFrom: first to: last];
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    OLStreamableFunctor<OLBoolBinaryFunction>* comp;

    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        comp = [decoder decodeObjectForKey: COMPARATOR_KEY];
    }
    else
    {
        comp = [decoder decodeObject];
    }
    [self initWithCompare: comp];
    readContainerWithInsertRange(self, decoder, @selector(decodeObject));
    return self;
}
#endif

- (id) initWithCompare: (OLStreamableFunctor<OLBoolBinaryFunction>*)comp
{
    [super init];
    tree = [[OLTree alloc] initWithCompare: comp];
    return self;
}

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    OLStreamableFunctor<OLBoolBinaryFunction>* comp = [stream readObject];
    [self initWithCompare: comp];
    OBJ_FREE_AUTO(comp);
    readContainerWithInsertRange(self, stream, @selector(readObject));
    return self;
}

- (id) initWithOLSet: (OLSet*)set
{
    [super init];
    tree = [[OLTree alloc] initWithTree: set->tree];
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(tree);
    SUPER_FREE;
}

- (OLAssociativeIterator*) begin
{
    return [tree begin];
}

- (void) clear
{
    [tree clear];
}

- (int) compare: (id)other
{
    if (IS_KIND_OF(other, OLSet))
        return [tree compare: ((OLSet*)other)->tree];
    return -1;
}

#if defined(OL_NO_OPENSTEP)

- (id) copy
{
    return [[OLSet alloc] initWithOLSet: self];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLSet allocWithZone: zone] initWithOLSet: self];
}

#endif

- (unsigned) count: (id)key
{
    return [tree count: key];
}

- (BOOL) empty
{
    return [tree empty];
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeObject: [tree keyComp] forKey: COMPARATOR_KEY];
    }
    else
    {
        [encoder encodeObject: [tree keyComp]];
    }
    writeContainer(tree, @selector(beginImpl), @selector(endImpl),
        encoder, @selector(encodeObject:));
}
#endif

- (OLAssociativeIterator*) end
{
    return [tree end];
}

- (OLPair*) equalRange: (id)key
{
    return [tree equalRange: key];
}

- (void) erase: (OLAssociativeIterator*)where
{
    [tree erase: where];
}

- (void) eraseFrom: (OLAssociativeIterator*)first to: (OLAssociativeIterator*)last
{
    [tree eraseFrom: first to: last];
}

- (unsigned) eraseKey: (id)key
{
    return [tree eraseKey: key];
}

- (OLAssociativeIterator*) find: (id)key
{
    return [tree find: key];
}

- (id) insert: (id)object
{
    return [tree insertUnique: object];
}

- (OLAssociativeIterator*) insertAt: (OLAssociativeIterator*)where value: (id)object
{
    return [tree insertUniqueAt: where value: object];
}

- (void) insertFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    [tree insertUniqueFrom: first to: last];
}

- (BOOL) isEqual: (id)object
{
    return IS_KIND_OF(object, OLSet) &&
           [tree isEqualUnique: ((OLSet*)object)->tree];
}

- (OLStreamableFunctor<OLBoolBinaryFunction>*) keyComp
{
    return [tree keyComp];
}

- (OLAssociativeIterator*) lowerBound: (id)key
{
    return [tree lowerBound: key];
}

- (unsigned) maxSize
{
    return [tree maxSize];
}

- (OLReverseBidiIterator*) rbegin
{
    return [tree rbegin];
}

- (OLReverseBidiIterator*) rend
{
    return [tree rend];
}

- (unsigned) size
{
    return [tree size];
}

- (void) swap: (OLSet*)right
{
    [tree swap: right->tree];
}

- (OLAssociativeIterator*) upperBound: (id)key
{
    return [tree upperBound: key];
}

- (OLStreamableFunctor<OLBoolBinaryFunction>*) valueComp
{
    return [tree keyComp];
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [stream writeObject: [tree keyComp]];
    writeContainer(tree, @selector(beginImpl), @selector(endImpl),
        stream, @selector(writeObject:));
}

@end

@implementation OLMultiSet

+ (id) multiSet
{
    OL_BEGIN_AUTO_CTOR(OLMultiSet)
        init
    OL_END_AUTO_CTOR;
}

+ (id) multiSetFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OL_BEGIN_AUTO_CTOR(OLMultiSet)
        initFrom: first to: last
    OL_END_AUTO_CTOR;
}

+ (id) multiSetWithCompare: (OLStreamableFunctor<OLBoolBinaryFunction>*)comp
{
    OL_BEGIN_AUTO_CTOR(OLMultiSet)
        initWithCompare: comp
    OL_END_AUTO_CTOR;
}

+ (id) multiSetWithOLSet: (OLSet*)right
{
    OL_BEGIN_AUTO_CTOR(OLMultiSet)
        initWithOLSet: right
    OL_END_AUTO_CTOR;
}

- (id) insert: (id)object
{
    return [tree insertEqual: object];
}

- (OLAssociativeIterator*) insertAt: (OLAssociativeIterator*)where value: (id)object
{
    return [tree insertEqualAt: where value: object];
}

- (void) insertFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    [tree insertEqualFrom: first to: last];
}

- (BOOL) isEqual: (id)object
{
    return IS_KIND_OF(object, OLMultiSet) &&
           [tree isEqualNonUnique: ((OLMultiSet*)object)->tree];
}

@end
