//
// $Id: Reaper.h,v 1.7 2007/03/06 20:42:20 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(REAPER_OL_GUARD)
#define REAPER_OL_GUARD

#include <ObjectiveLib/ObjectBase.h>

/**
 * Forward declaration of the class used to store the undead objects.
 */
@class OLVectorJunior;

/**
 * @class OLReaper Reaper.h ObjectiveLib/Reaper.h
 *
 * A class for automatically freeing objects. This class is fundamentally
 * equivalent to NSAutoreleasePool from OpenStep. Each thread contains a list
 * of active reapers. Calling #currentReaper will return the head of the
 * current thread's list, thus the most recently defined reaper. When a reaper is
 * freed it will free all of the objects that it has been given to reap.
 *
 * Typcial usage would be to create a reaper at some point in the code at a
 * high level, for example in @c main(), and then to add allocated objects
 * to the reaper instead of trying to manage all the memory by hand. For example,
 * imagine in the following example that a reaper has been created in the
 * application's @c main function.
 * @code
 * OLReaper reaper = [OLReaper currentReaper];
 * OLTextBuffer* buffer = [reaper reap: [OLTextBuffer textBuffer]];
 * [buffer appendChars: charBuf fromOffset: 0 count: 8192];
 * [buffer appendText: [reaper reap: [OLText textWithCString: "Hi, there"]]]
 * // Do something with the text from the buffer
 * ... [reaper reap: [buffer text]] ...
 * @endcode
 *
 * Note that in the above example memory management is handled by the reaper. All of the
 * allocated objects added to the reaper will be cleaned up when the reaper is freed.
 */
@interface OLReaper : Object
{
@protected
    /**
     * The array of as-yet-uncleaned-up objects
     */
    OLVectorJunior* undead;
}

/**
 * Retrieve the reaper currently in use. This reaper will be the most recently defined
 * OLReaper in the current thread.
 *
 * @return the current reaper
 */
+ (OLReaper*) currentReaper;

/**
 * Initialize the class. This for the runtime system to call. Please don't call
 * this method.
 */
+ (id) initialize;

/**
 * @name Initializers and Deallocators
 */
/* @{ */

/**
 * Initialize a new reaper. The reaper contains no objects to free.
 *
 * @return a reference to this reaper
 */
- (id) init;

/**
 * Free all objects that have been added to this reaper and free the reaper. That's
 * free the reaper, not fear the reaper.
 */
- (id) free;
/* @} */

/**
 * Add the given object to this reaper. The object will be freed when the reaper
 * is freed.
 *
 * @param object the object to reap
 * @return a reference to @a object
 */
- (id) reap: (id)object;

/**
 * Return the number of objects currently scheduled for destruction.
 *
 * @return the size of the reapers cache
 */
- (unsigned) size;

@end

#endif
