//
// $Id: PointerRefTable.m,v 1.6 2007/03/06 20:42:20 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "PointerRefTable.h"
#import "Macros.h"
#import "RunTime.h"
#import <stddef.h>
#import <stdlib.h>

typedef struct _OLPointerRefNode
{
    struct _OLPointerRefNode*   next;
    void*                       object;
    uint32_t                    handle;
} OLPointerRefNode;

@interface OLRefBucketHead :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject <NSCopying>
#endif
{
@private
    OLPointerRefNode* node;
}

- (id) initWithNode: (OLPointerRefNode*)nde;

#if defined(OL_NO_OPENSTEP)
- (id) copy;
#else
- (id) copyWithZone: (NSZone*)zone;
#endif
- (OLPointerRefNode*)node;
- (void) setNode: (OLPointerRefNode*)node;

@end

@implementation OLRefBucketHead

- (id) initWithNode: (OLPointerRefNode*)nde
{
    [super init];
    node = nde;
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLRefBucketHead alloc] initWithNode: node];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLRefBucketHead allocWithZone: zone] initWithNode: node];
}
#endif

- (OLPointerRefNode*)node
{
    return node;
}

- (void) setNode: (OLPointerRefNode*)nde
{
    node = nde;
}

@end

@implementation OLPointerRefTable

- (id) init
{
    unsigned i;
    unsigned capacity;
    OLRefBucketHead* buck;

    [super init];
    capacity = [self nextSize: 30];
    buckets = [[OLVector alloc] init];
    [buckets reserve: capacity];
    for (i = 0; i < capacity; i++)
    {
        buck = [[OLRefBucketHead alloc] initWithNode: NULL];
        [buckets pushBack: buck];
        OBJ_RELEASE(buck);
    }
    currentHandle = 0;
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OLPointerRefNode* cur;
    OLPointerRefNode* next;
    unsigned size = [buckets size];
    unsigned i;

    for (i = 0; i < size; i++)
    {
        cur = [[buckets at: i] node];
        while (cur != NULL)
        {
            next = cur->next;
            objc_free(cur);
            cur = next;
        }
    }
    OBJ_RELEASE(buckets);
    SUPER_FREE;
}

- (uint32_t) assign: (void*)object
{
    OLPointerRefNode* first;
    OLPointerRefNode* cur;
    BOOL insertOkay = YES;
    unsigned bucket = [self bucketOfPointer: object];
    uint32_t theHandle = currentHandle;

    first = [[buckets at: bucket] node];
    for (cur = first; cur != NULL; cur = cur->next)
    {
        if (cur->object == object)
        {
            insertOkay = NO;
            break;
        }
    }
    if (insertOkay)
    {
        cur = objc_malloc(sizeof(OLPointerRefNode));
        cur->object = object;
        cur->handle = currentHandle++;
        cur->next = first;
        [[buckets at: bucket] setNode: cur];
    }
    return theHandle;
}

- (unsigned) bucketOfPointer: (void*)object
{
    return [self bucketOfPointer: object size: [buckets size]];
}

- (unsigned) bucketOfPointer: (void*)object size: (unsigned)size
{
    return (unsigned)object % size;
}

- (unsigned) capacity
{
    return [buckets size];
}

- (uint32_t) lookUp: (void*)object
{
    OLPointerRefNode* node = [[buckets at: [self bucketOfPointer: object]] node];

    while (node != NULL && node->object != object)
        node = node->next;
    return (node == NULL) ? UINT32_MAX : node->handle;
}

- (unsigned) nextSize: (unsigned)size
{
    unsigned newSize = 1;
    unsigned oldSize = 1;
    unsigned tempSize;

    while (newSize < size)
    {
        tempSize = oldSize;
        oldSize = newSize;
        newSize += tempSize;
    }
    newSize |= 1;
    return newSize;
}

- (void) resize: (unsigned)hint
{
    OLVector* tmp;
    OLPointerRefNode* node;
    OLRefBucketHead* head;
    unsigned oldSize = [buckets size];
    unsigned newSize;
    unsigned i;
    unsigned bucket;

    if (hint > oldSize)
    {
        newSize = [self nextSize: hint];
        if (newSize > oldSize)
        {
            tmp = [[OLVector alloc] init];
            [tmp reserve: newSize];
            for (i = 0; i < newSize; i++)
            {
                head = [[OLRefBucketHead alloc] initWithNode: NULL];
                [tmp pushBack: head];
                OBJ_RELEASE(head);
            }
            for (i = 0; i < oldSize; i++)
            {
                node = [[buckets at: i] node];
                while (node != NULL)
                {
                    bucket = [self bucketOfPointer: node->object size: newSize];
                    [[buckets at: i] setNode: node->next];
                    node->next = [[tmp at: bucket] node];
                    [[tmp at: bucket] setNode: node];
                    node = [[buckets at: i] node];
                }
            }
            [buckets swap: tmp];
            OBJ_RELEASE(tmp);
        }
    }
}

@end
