//
// $Id: List.m,v 1.34 2007/04/16 01:48:47 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "List.h"
#import "Macros.h"
#import "RunTime.h"
#import "ObjectInStream.h"
#import "ObjectOutStream.h"
#import "Utility.h"
#import <limits.h>
#import <stdlib.h>

@interface OLListNode :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
{
@public
    id          object;
    OLListNode* previous;
    OLListNode* next;
}
@end

@interface OLListIterator (PrivateMethods)

- (id) initWithNode: (OLListNode*)nde;
- (OLListNode*) node;
- (void) setNode: (OLListNode*)n;

@end

@interface OLList (PrivateMethods)

- (OLListIterator*) beginImpl;
- (OLListIterator*) endImpl;
- (OLListNode*) eraseImpl: (OLListIterator*)where;
- (OLListIterator*) eraseImplFrom: (OLListIterator*)first to: (OLListIterator*)last needItor: (BOOL)needItor;
- (OLListNode*) insertImpl: (OLListIterator*)where value: (id)object;
- (void) moveNodesTo: (OLListNode*)pos from: (OLListNode*)first to: (OLListNode*)last;

@end

@implementation OLList

+ (id) list
{
    OL_BEGIN_AUTO_CTOR(OLList)
        init
    OL_END_AUTO_CTOR;
}

+ (id) listFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OL_BEGIN_AUTO_CTOR(OLList)
        initFrom: first to: last
    OL_END_AUTO_CTOR;
}

+ (id) listWithList: (OLList*)right
{
    OL_BEGIN_AUTO_CTOR(OLList)
        initWithList: right
    OL_END_AUTO_CTOR;
}

+ (id) listWithSize: (unsigned)size filledWith: (id)value
{
    OL_BEGIN_AUTO_CTOR(OLList)
        initWithSize: size filledWith: value
    OL_END_AUTO_CTOR;
}

- (id) init
{
    [super init];
    node = [[OLListNode alloc] init];
    node->object = nil;
    node->previous = node;
    node->next = node;
    return self;
}

- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OLListIterator* where;

    [self init];
    where = [self beginImpl];
    [self insertAt: where from: first to: last];
    OBJ_RELEASE(where);
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    [self init];
    readContainerWithPushBack(self, decoder, @selector(decodeObject));
    return self;
}
#endif

- (id) initWithList: (OLList*)list
{
    OLListIterator* itorBegin = [list beginImpl];
    OLListIterator* itorEnd = [list endImpl];

    [self initFrom: itorBegin to: itorEnd];
    OBJ_RELEASE(itorBegin);
    OBJ_RELEASE(itorEnd);
    return self;
}

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    [self init];
    readContainerWithPushBack(self, stream, @selector(readObject));
    return self;
}

- (id) initWithSize: (unsigned)size filledWith: (id)value
{
    OLListIterator* itorBegin;

    [self init];
    itorBegin = [self beginImpl];
    [self insertAt: itorBegin count: size filledWith: value];
    OBJ_RELEASE(itorBegin);
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    [self clear];
    OBJ_RELEASE(node);
    SUPER_FREE;
}

- (void) assign: (unsigned)count filledWith: (id)value
{
    OLListIterator* itorBegin = [self beginImpl];
    OLListIterator* itorEnd = [self endImpl];
    OLListNode* nde;

    for ( ; ![itorBegin isEqual: itorEnd] && count > 0; [itorBegin advance], --count)
    {
        nde = [itorBegin node];
        OBJ_RELEASE(nde->object);
        nde->object = OBJ_RETAIN(value);
    }
    if (count > 0)
    {
        [self insertAt: itorEnd count: count filledWith: value];
    }
    else
    {
        [self eraseImplFrom: itorBegin to: itorEnd needItor: NO];
    }
    OBJ_RELEASE(itorBegin);
    OBJ_RELEASE(itorEnd);
}

- (void) assignFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OLListIterator* myFirst = [self beginImpl];
    OLListIterator* myLast = [self endImpl];
    OLListIterator* firstCopy = [first copy];
    OLListNode* nde;

    for ( ; !([firstCopy isEqual: last] || [myFirst isEqual: myLast]);
            [firstCopy advance], [myFirst advance])
    {
        nde = [myFirst node];
        OBJ_RELEASE(nde->object);
        nde->object = OBJ_RETAIN([firstCopy dereference]);
    }
    if ([firstCopy isEqual: last])
    {
        [self eraseImplFrom: myFirst to: myLast needItor: NO];
    }
    else
    {
        [self insertAt: myLast from: firstCopy to: last];
    }
    OBJ_RELEASE(myFirst);
    OBJ_RELEASE(myLast);
    OBJ_RELEASE(firstCopy);
}

- (id) back
{
    return node->previous->object;
}

- (OLListIterator*) begin
{
    return OBJ_AUTORELEASE([self beginImpl]);
}

- (void) clear
{
    OLListNode* cur = node->next;
    OLListNode* tmp;

    while (cur != node)
    {
        tmp = cur;
        cur = cur->next;
        OBJ_RELEASE(tmp->object);
        OBJ_RELEASE(tmp);
    }
    node->next = node;
    node->previous = node;
}

- (int) compare: (id)other
{
    return compareContainers(self, other, @selector(beginImpl), @selector(endImpl));
}

#if defined(OL_NO_OPENSTEP)

- (id) copy
{
    return [[OLList alloc] initWithList: self];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLList allocWithZone: zone] initWithList: self];
}

#endif

- (BOOL) empty
{
    return (node->next == node) ? YES : NO;
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    writeContainer(self, @selector(beginImpl), @selector(endImpl),
        encoder, @selector(encodeObject:));
}
#endif

- (OLListIterator*) end
{
    return OBJ_AUTORELEASE([self endImpl]);
}

- (OLListIterator*) erase: (OLListIterator*)where
{
    return OBJ_AUTORELEASE([[OLListIterator alloc] initWithNode: [self eraseImpl: where]]);
}

- (OLListIterator*) eraseFrom: (OLListIterator*)first to: (OLListIterator*)last
{
    return OBJ_AUTORELEASE([self eraseImplFrom: first to: last needItor: YES]);
}

- (id) front
{
    return node->next->object;
}

- (OLListIterator*) insertAt: (OLListIterator*)where value: (id)object
{
    return OBJ_AUTORELEASE([[OLListIterator alloc]
        initWithNode: [self insertImpl: where value: object]]);
}

- (void) insertAt: (OLListIterator*)where count: (unsigned)num filledWith: (id)value
{
    for ( ; num > 0; --num)
        [self insertImpl: where value: value];
}

- (void) insertAt: (OLListIterator*)where from: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OLForwardIterator* firstCopy = [first copy];

    for ( ; ![firstCopy isEqual: last]; [firstCopy advance])
        [self insertImpl: where value: [firstCopy dereference]];
    OBJ_RELEASE(firstCopy);
}

- (BOOL) isEqual: (id)object
{
    OLList* other;
    OLListIterator* myFirst;
    OLListIterator* myLast;
    OLListIterator* itsFirst;
    BOOL broken = NO;

    if (!IS_KIND_OF(object, OLList))
        return NO;
    other = (OLList*)object;
    if ([other size] != [self size])
        return NO;
    myFirst = [self beginImpl];
    myLast = [self endImpl];
    itsFirst = [other beginImpl];
    for ( ; ![myFirst isEqual: myLast]; [myFirst advance], [itsFirst advance])
    {
        if (![[myFirst dereference] isEqual: [itsFirst dereference]])
        {
            broken = YES;
            break;
        }
    }
    OBJ_RELEASE(myFirst);
    OBJ_RELEASE(myLast);
    OBJ_RELEASE(itsFirst);
    return broken ? NO : YES;
}

- (unsigned) maxSize
{
    return UINT_MAX;
}

- (void) merge: (OLList*)right
{
    OLLess* less = [[OLLess alloc] init];

    [self merge: right withOrder: less];
    OBJ_RELEASE(less);
}

- (void) merge: (OLList*)right withOrder: (id<OLBoolBinaryFunction>)pred
{
    OLListIterator* thisFirst = [self beginImpl];
    OLListIterator* thisLast = [self endImpl];
    OLListIterator* thatFirst = [right beginImpl];
    OLListIterator* thatLast = [right endImpl];
    OLListNode* next;

    while (!([thisFirst isEqual: thisLast] || [thatFirst isEqual: thatLast]))
    {
        if ([pred performBinaryFunctionWithArg:
                [thatFirst dereference] andArg: [thisFirst dereference]])
        {
            next = [thatFirst node]->next;
            [self moveNodesTo: [thisFirst node] from: [thatFirst node] to: next];
            [thatFirst setNode: next];
        }
        else
        {
            [thisFirst advance];
        }
    }
    if (![thatFirst isEqual: thatLast])
    {
        [self insertAt: thisLast from: thatFirst to: thatLast];
        [right eraseImplFrom: thatFirst to: thatLast needItor: NO];
    }
    OBJ_RELEASE(thisFirst);
    OBJ_RELEASE(thisLast);
    OBJ_RELEASE(thatFirst);
    OBJ_RELEASE(thatLast);
}

- (void) popBack
{
    OLListIterator* last;

    if (![self empty])
    {
        last = [self endImpl];
        [last reverse];
        [self eraseImpl: last];
        OBJ_RELEASE(last);
    }
}

- (void) popFront
{
    OLListIterator* first;

    if (![self empty])
    {
        first = [self beginImpl];
        [self eraseImpl: first];
        OBJ_RELEASE(first);
    }
}

- (void) pushBack: (id)object
{
    OLListIterator* last = [self endImpl];

    [self insertImpl: last value: object];
    OBJ_RELEASE(last);
}

- (void) pushFront: (id)object
{
    OLListIterator* first = [self beginImpl];

    [self insertImpl: first value: object];
    OBJ_RELEASE(first);
}

- (OLReverseBidiIterator*) rbegin
{
    OLListIterator* last = [self endImpl];
    OLReverseBidiIterator* rb = OBJ_AUTORELEASE([[OLReverseBidiIterator alloc]
        initWithIterator: last]);

    OBJ_RELEASE(last);
    return rb;
}

- (void) remove: (id)object
{
    OLListIterator* first = [self beginImpl];
    OLListIterator* last = [self endImpl];
    OLListIterator* tmp;

    while (![first isEqual: last])
    {
        if ([object isEqual: [first dereference]])
        {
            tmp = [[OLListIterator alloc] initWithNode: [self eraseImpl: first]];
            OBJ_RELEASE(first);
            first = tmp;
        }
        else
        {
            [first advance];
        }
    }
    OBJ_RELEASE(first);
    OBJ_RELEASE(last);
}

- (void) removeIf: (id<OLBoolUnaryFunction>)pred
{
    OLListIterator* first = [self beginImpl];
    OLListIterator* last = [self endImpl];
    OLListIterator* tmp;

    while (![first isEqual: last])
    {
        if ([pred performUnaryFunctionWithArg: [first dereference]])
        {
            tmp = [[OLListIterator alloc] initWithNode: [self eraseImpl: first]];
            OBJ_RELEASE(first);
            first = tmp;
        }
        else
        {
            [first advance];
        }
    }
    OBJ_RELEASE(first);
    OBJ_RELEASE(last);
}

- (OLReverseBidiIterator*) rend
{
    OLListIterator* first = [self beginImpl];
    OLReverseBidiIterator* re = OBJ_AUTORELEASE([[OLReverseBidiIterator alloc]
        initWithIterator: first]);

    OBJ_RELEASE(first);
    return re;
}

- (void) resize: (unsigned)count filledWith: (id)value
{
    OLListIterator* first = [self beginImpl];
    OLListIterator* last = [self endImpl];
    unsigned len = 0;

    for ( ; ![first isEqual: last] && len < count; [first advance], len++);

    if (len == count)
    {
        [self eraseImplFrom: first to: last needItor: NO];
    }
    else
    {
        [self insertAt: last count: count - len filledWith: value];
    }
    OBJ_RELEASE(first);
    OBJ_RELEASE(last);
}

- (void) reverse
{
    OLListNode* last = node;
    OLListNode* cur = last;
    OLListNode* tmp;

    do
    {
        tmp = cur->next;
        cur->next = cur->previous;
        cur->previous = tmp;
        cur = cur->previous;
    } while (cur != last);
}

- (unsigned) size
{
    OLListIterator* first = [self beginImpl];
    OLListIterator* last = [self endImpl];
    unsigned sz = [OLIterator distanceFrom: first to: last];

    OBJ_RELEASE(first);
    OBJ_RELEASE(last);
    return sz;
}

- (void) sort
{
    OLLess* less = [[OLLess alloc] init];

    [self sortWith: less];
    OBJ_RELEASE(less);
}

- (void) sortWith: (id<OLBoolBinaryFunction>)pred
{
    OLList*         counter[64];
    OLList*         carry;
    OLListIterator* carryBegin;
    OLListIterator* myBegin;
    int             fill = 0;
    int             i;

    if (node->next != node && node->next->next != node)
    {
        for (i = 0; i < 64; i++)
            counter[i] = [[OLList alloc] init];
        carry = [[OLList alloc] init];
        while (![self empty])
        {
            carryBegin = [carry beginImpl];
            myBegin = [self beginImpl];
            [carry spliceAt: carryBegin list: self from: myBegin];
            OBJ_RELEASE(carryBegin);
            OBJ_RELEASE(myBegin);
            i = 0;
            while (i < fill && ![counter[i] empty])
            {
                [counter[i] merge: carry withOrder: pred];
                [carry swap: counter[i++]];
            }
            [carry swap: counter[i]];
            if (i == fill)
                fill++;
        }
        for (i = 1; i < fill; i++)
            [counter[i] merge: counter[i - 1] withOrder: pred];
        [self swap: counter[fill - 1]];
        for (i = 0; i < 64; i++)
            OBJ_RELEASE(counter[i]);
        OBJ_RELEASE(carry);
    }
}

- (void) spliceAt: (OLListIterator*)where list: (OLList*)right
{
    OLListIterator* rightBegin = [right beginImpl];
    OLListIterator* rightEnd = [right endImpl];

    [self spliceAt: where list: right from: rightBegin to: rightEnd];
    OBJ_RELEASE(rightBegin);
    OBJ_RELEASE(rightEnd);
}

- (void) spliceAt: (OLListIterator*)where list: (OLList*)right from: (OLListIterator*)first
{
    OLListIterator* rightEnd = [[[OLListIterator alloc]
        initWithNode: [first node]] advance];

    [self spliceAt: where list: right from: first to: rightEnd];
    OBJ_RELEASE(rightEnd);
}

- (void) spliceAt: (OLListIterator*)where list: (OLList*)right from: (OLListIterator*)first to: (OLListIterator*)last
{
    if (![first isEqual: last])
        [self moveNodesTo: [where node] from: [first node] to: [last node]];
}

- (void) swap: (OLList*)right
{
    OL_PREPARE_SLOW_SWAP;

    if (self != right)
        OL_SLOW_SWAP(node, right->node);
}

- (void) unique
{
    OLEqualTo* equal = [[OLEqualTo alloc] init];

    [self uniqueWith: equal];
    OBJ_RELEASE(equal);
}

- (void) uniqueWith: (id<OLBoolBinaryFunction>)pred
{
    OLListIterator* first;
    OLListIterator* last;
    OLListIterator* next;

    if (![self empty])
    {
        first = [self beginImpl];
        last = [self endImpl];
        next = [self beginImpl];
        while (![[next advance] isEqual: last])
        {
            if ([pred performBinaryFunctionWithArg:
                    [first dereference] andArg: [next dereference]])
            {
                [self eraseImpl: next];
            }
            else
            {
                [first setNode: [next node]];
            }
            [next setNode: [first node]];
        }
        OBJ_RELEASE(first);
        OBJ_RELEASE(last);
        OBJ_RELEASE(next);
    }
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    writeContainer(self, @selector(beginImpl), @selector(endImpl),
        stream, @selector(writeObject:));
}

@end

@implementation OLListNode

@end

@implementation OLListIterator

- (id) advance
{
    node = node->next;
    return self;
}

- (id) assign: (id)object
{
    if (node->object != object)
    {
        OBJ_RELEASE(node->object);
        node->object = OBJ_RETAIN(object);
    }
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLListIterator alloc] initWithNode: node];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLListIterator allocWithZone: zone] initWithNode: node];
}
#endif

- (id) dereference
{
    return node->object;
    return self;
}

- (BOOL) isEqual: (id)object
{
    return [super isEqual: object] && node == ((OLListIterator*)object)->node;
}

- (id) reverse
{
    node = node->previous;
    return self;
}

@end

@implementation OLListIterator (PrivateMethods)

- (id) initWithNode: (OLListNode*)nde
{
    [super init];
    node = nde;
    return self;
}

- (OLListNode*) node
{
    return node;
}

- (void) setNode: (OLListNode*)n
{
    node = n;
}

@end

@implementation OLList (PrivateMethods)

- (OLListIterator*) beginImpl
{
    return [[OLListIterator alloc] initWithNode: node->next];
}

- (OLListIterator*) endImpl
{
    return [[OLListIterator alloc] initWithNode: node];
}

- (OLListNode*) eraseImpl: (OLListIterator*)where
{
    OLListNode* nde = [where node];
    OLListNode* next = nde->next;
    OLListNode* prev = nde->previous;

    prev->next = next;
    next->previous = prev;
    OBJ_RELEASE(nde->object);
    OBJ_RELEASE(nde);
    return next;
}

- (OLListIterator*) eraseImplFrom: (OLListIterator*)first to: (OLListIterator*)last needItor: (BOOL)needItor
{
    OLListIterator* firstCopy = [first copy];
    OLListIterator* nextCopy = nil;

    while (![firstCopy isEqual: last])
    {
        nextCopy = [[firstCopy copy] advance];
        [self eraseImpl: firstCopy];
        OBJ_RELEASE(firstCopy);
        firstCopy = nextCopy;
    }
    OBJ_RELEASE(firstCopy);
    return needItor ?
        [[OLListIterator alloc] initWithNode: [last node]] :
        nil;
}

- (OLListNode*) insertImpl: (OLListIterator*)where value: (id)object
{
    OLListNode* tmp = [[OLListNode alloc] init];
    OLListNode* next = [where node];
    OLListNode* prev = next->previous;

    tmp->object = OBJ_RETAIN(object);
    tmp->next = next;
    tmp->previous = prev;
    prev->next = tmp;
    next->previous = tmp;
    return tmp;
}

- (void) moveNodesTo: (OLListNode*)pos from: (OLListNode*)first to: (OLListNode*)last
{
    OLListNode* tmp;

    if (pos != last)
    {
        last->previous->next = pos;
        first->previous->next = last;
        pos->previous->next = first;
        tmp = pos->previous;
        pos->previous = last->previous;
        last->previous = first->previous;
        first->previous = tmp;
    }
}

@end
