//
// $Id: Iterator.m,v 1.34 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "Iterator.h"
#import "Macros.h"
#import "Tree.h"
#import "HashTable.h"
#import <stddef.h>

extern OLHashTableNode* __bumpHashTableNode(OLHashTableNode*, OLHashTable*);

@implementation OLIterator

+ (void) advanceIterator: (OLIterator*)itor distance: (int)count
{
    int i;

    if (IS_KIND_OF(itor, OLRandomAccessIterator))
    {
        [(OLRandomAccessIterator*)itor advanceBy: count];
    }
    else
    {
        if (count < 0)
        {
            if (IS_KIND_OF(itor, OLBidirectionalIterator))
            {
                for (i = 0; i > count; i--)
                    [(OLBidirectionalIterator*)itor reverse];
            }
        }
        else if (IS_KIND_OF(itor, OLForwardIterator))
        {
            for (i = 0; i < count; i++)
                [(OLForwardIterator*)itor advance];
        }
    }
}

+ (unsigned) distanceFrom: (OLIterator*)first to: (OLIterator*)last
{
    unsigned i = 0;
    OLForwardIterator* firstCopy;

    if (IS_KIND_OF(first, OLRandomAccessIterator) && IS_KIND_OF(last, OLRandomAccessIterator))
    {
        i = [(OLRandomAccessIterator*)last difference: (OLRandomAccessIterator*)first];
    }
    else if (IS_KIND_OF(first, OLForwardIterator) && IS_KIND_OF(last, OLForwardIterator))
    {
        firstCopy = [first copy];
        while (![firstCopy isEqual: last])
        {
            i++;
            [firstCopy advance];
        }
        OBJ_RELEASE(firstCopy);
    }
    return i;
}

- (id) assign: (id)object
{
    return self;
}

#if defined(OL_NO_OPENSTEP)

- (id) copy
{
    SUBCLASS_MUST_IMPLEMENT;
    return nil;
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    SUBCLASS_MUST_IMPLEMENT;
    return nil;
}

#endif

- (id) dereference
{
    return self;
}

- (BOOL) isEqual: (id)object
{
    return IS_KIND_OF(object, self);
}

@end

@implementation OLForwardIterator

- (id) advance
{
    return self;
}

@end

@implementation OLBidirectionalIterator

- (id) reverse
{
    return self;
}

@end

@implementation OLRandomAccessIterator

- (id) advanceBy: (int)count
{
    return self;
}

- (int) difference: (OLRandomAccessIterator*)other
{
    return 0;
}

@end

@implementation OLReverseBidiIterator

- (id) initWithIterator: (OLBidirectionalIterator*)itor
{
    [super init];
    current = OBJ_RETAIN(itor);
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(current);
    SUPER_FREE;
}

- (id) advance
{
    [current reverse];
    return self;
}

- (id) assign: (id)object
{
    [current assign: object];
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLReverseBidiIterator alloc] initWithIterator: current];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLReverseBidiIterator allocWithZone: zone] initWithIterator: current];
}
#endif

- (id) dereference
{
    id ref = [[current reverse] dereference];
    [current advance];
    return ref;
}

- (BOOL) isEqual: (id)object
{
    return [super isEqual: object] &&
           [current isEqual: ((OLReverseBidiIterator*)object)->current];
}

- (id) reverse
{
    [current advance];
    return self;
}

@end

@implementation OLReverseRandomIterator

- (id) advanceBy: (int)count
{
    if (IS_KIND_OF(current, OLRandomAccessIterator))
        [(OLRandomAccessIterator*)current advanceBy: -count];
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLReverseRandomIterator alloc] initWithIterator: current];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLReverseRandomIterator allocWithZone: zone] initWithIterator: current];
}
#endif

- (BOOL) isEqual: (id)object
{
    return [super isEqual: object] &&
           [current isEqual: ((OLReverseRandomIterator*)object)->current];
}

@end

@implementation OLBackInsertIterator

- (id) initWithBackInserter: (id<OLBackInserter>)cnr
{
    [super init];
    container = cnr;
    return self;
}

- (id) assign: (id)object
{
    [container pushBack: object];
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLBackInsertIterator alloc] initWithBackInserter: container];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLBackInsertIterator allocWithZone: zone] initWithBackInserter: container];
}
#endif

- (BOOL) isEqual: (id)object
{
    return [super isEqual: object] &&
           [container isEqual: ((OLBackInsertIterator*)object)->container];
}

@end

@implementation OLFrontInsertIterator

- (id) initWithFrontInserter: (id<OLFrontInserter>)cnr
{
    [super init];
    container = cnr;
    return self;
}

- (id) assign: (id)object
{
    [container pushFront: object];
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [((OLFrontInsertIterator*)[OLFrontInsertIterator alloc])
        initWithFrontInserter: container];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [((OLFrontInsertIterator*)[OLFrontInsertIterator allocWithZone: zone])
        initWithFrontInserter: container];
}
#endif

- (BOOL) isEqual: (id)object
{
    return [super isEqual: object] &&
           [container isEqual: ((OLFrontInsertIterator*)object)->container];
}


@end

@implementation OLInsertIterator

- (id) initWithInserter: (id<OLInserter>)cnr iterator: (OLForwardIterator*)itor
{
    [super init];
    container = cnr;
    iterator = OBJ_RETAIN(itor);
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(iterator);
    SUPER_FREE;
}

- (id) assign: (id)object
{
    OLForwardIterator* newItor = (OLForwardIterator*)[container insertAt: iterator value: object];

    OBJ_RELEASE(iterator);
    iterator = OBJ_RETAIN(newItor);
    [iterator advance];
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLInsertIterator alloc]
        initWithInserter: container iterator: iterator];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLInsertIterator allocWithZone: zone]
        initWithInserter: container iterator: iterator];
}
#endif

- (BOOL) isEqual: (id)object
{
    return [super isEqual: object] &&
           [container isEqual: ((OLInsertIterator*)object)->container] &&
           [iterator isEqual: ((OLInsertIterator*)object)->iterator];
}

@end

@implementation OLAssociativeIterator

- (id) advance
{
    node = [OLTreeNode increment: node];
    return self;
}

- (id) assign: (id)object
{
    if (node->object != object)
    {
        OBJ_RELEASE(node->object);
        node->object = OBJ_RETAIN(object);
    }
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    OLAssociativeIterator* copy = [[OLAssociativeIterator alloc] init];

    copy->node = node;
    return copy;
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    OLAssociativeIterator* copy = [[OLAssociativeIterator allocWithZone: zone] init];

    copy->node = node;
    return copy;
}
#endif

- (id) dereference
{
    return node->object;
}

- (BOOL) isEqual: (id)object
{
    return [super isEqual: object] &&
           node == ((OLAssociativeIterator*)object)->node;
}

- (id) reverse
{
    node = [OLTreeNode decrement: node];
    return self;
}

@end

@implementation OLArrayIterator

+ (OLPair*) pairWithPointer: (id*)p1 andPointer: (id*)p2
{
    OLArrayIterator* i1 = [[OLArrayIterator alloc] initWithPointer: p1];
    OLArrayIterator* i2 = [[OLArrayIterator alloc] initWithPointer: p2];
    OLPair* p = [[OLPair alloc] initWithFirst: i1 second: i2];

    OBJ_RELEASE(i1);
    OBJ_RELEASE(i2);
    return OBJ_AUTORELEASE(p);
}

+ (OLPair*) pairWithPointer: (id*)p distance: (unsigned)count
{
    return [OLArrayIterator pairWithPointer: p andPointer: p + count];
}

- (id) initWithPointer: (id*)ptr
{
    [super init];
    current = ptr;
    return self;
}

- (id) advance
{
    current++;
    return self;
}

- (id) advanceBy: (int)count
{
    current += count;
    return self;
}

- (id) assign: (id)object
{
    if (*current != object)
    {
        OBJ_RELEASE(*current);
        *current = OBJ_RETAIN(object);
    }
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLArrayIterator alloc] initWithPointer: current];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLArrayIterator allocWithZone: zone] initWithPointer: current];
}
#endif

- (id) dereference
{
    return *current;
}

- (int) difference: (OLRandomAccessIterator*)other
{
    int diff = 0;

    if (IS_KIND_OF(other, OLArrayIterator))
        diff = current - ((OLArrayIterator*)other)->current;
    return diff;
}

- (BOOL) isEqual: (id)object
{
    return [super isEqual: object] &&
           current == ((OLArrayIterator*)object)->current;
}

- (id) reverse
{
    current--;
    return self;
}

@end

@implementation OLHashIterator

- (id) advance
{
    current = __bumpHashTableNode(current, table);
    return self;
}

- (id) assign: (id)object
{
    if (current->value != object)
    {
        OBJ_RELEASE(current->value);
        current->value = OBJ_RETAIN(object);
    }
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    OLHashIterator* i = [[OLHashIterator alloc] init];

    i->table = table;
    i->current = current;
    return i;
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    OLHashIterator* i = [[OLHashIterator allocWithZone: zone] init];

    i->table = table;
    i->current = current;
    return i;
}
#endif

- (id) dereference
{
    return current->value;
}

- (BOOL) isEqual: (id)object
{
    return [super isEqual: object] &&
           table == ((OLHashIterator*)object)->table &&
           current == ((OLHashIterator*)object)->current;
}

@end

