//
// $Id: InStream.m,v 1.19 2007/03/20 04:10:25 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "InStreamPackage.h"
#import "RunTime.h"
#import "Macros.h"
#import "PointerData.h"
#import "ByteOrder.h"
#if defined(OL_NO_OPENSTEP)
#import "Text.h"
#import "Exception.h"
#else
#import <Foundation/NSString.h>
#import <Foundation/NSException.h>
#import <Foundation/NSData.h>
#endif
#include <stdlib.h>
#import <limits.h>

#if defined(OL_NO_OPENSTEP)
OLConstantString* const OLInputOutputException = @"OLInputOutputException";
OLConstantString* const OLEndOfStreamException = @"OLEndOfStreamException";
#else
NSString* const OLInputOutputException = @"OLInputOutputException";
NSString* const OLEndOfStreamException = @"OLEndOfStreamException";
#endif

extern uint64_t __swap64(uint64_t);

@interface OLInStream (PrivateMethods)

- (double) readDoubleImpl;
- (float) readFloatImpl;
- (unsigned) readIntImpl;
- (uint16_t) readInt16Impl;
- (SEL) readSelectorImpl;

@end

@implementation OLInStream

#if !defined(OL_NO_OPENSTEP)
- (BOOL) allowsKeyedCoding
{
    return NO;
}
#endif

- (void) close
{
}

#if !defined(OL_NO_OPENSTEP)
- (void*) decodeBytesWithReturnedLength: (unsigned*)numBytes
{
    NSMutableData* data;

    *numBytes = [self readIntImpl];
    data = [[NSMutableData alloc] initWithLength: *numBytes];
    [self completelyReadBytes: [data mutableBytes] count: *numBytes];
    OBJ_AUTORELEASE(data);
    return [data mutableBytes];
}

- (NSData*) decodeDataObject
{
    NSMutableData* data;
    unsigned numBytes;

    numBytes = [self readIntImpl];
    data = [[NSMutableData alloc] initWithLength: numBytes];
    [self completelyReadBytes: [data mutableBytes] count: numBytes];
    return OBJ_AUTORELEASE(data);
}

- (id) decodeObject
{
    RAISE_EXCEPTION(OLInputOutputException, @"Objects can only be read to OLObjectInStream");
    return nil;
}

- (void) decodeValueOfObjCType: (const char*)valueType at: (void*)address
{
    union MixedResults
    {
        uint8_t result8;
        uint16_t result16;
        uint32_t result32;
        uint64_t result64;
    };
    uint16_t strLen;
    uint16_t chunkCount;
    uint16_t i;
    union MixedResults results;
    long arrayCount;
    char* endOfNum;
    struct objc_struct_layout slay;
    unsigned offset;
    const char* fieldType;
    int typeSize;
    long cur;
    OLPointerData* pointerData;

    switch (*valueType)
    {
    case _C_CHR:
    case _C_UCHR:
        *(uint8_t*)address = [self readByte];
        break;
    case _C_SHT:
    case _C_USHT:
        [self completelyReadBytes: &results.result8 count: 2];
        *(short*)address = N16_TO_H(results.result16);
        break;
    case _C_INT:
    case _C_UINT:
        *(unsigned*)address = [self readIntImpl];
        break;
    case _C_LNG:
    case _C_ULNG:
        [self completelyReadBytes: &results.result8 count: 4];
        *(long*)address = N32_TO_H(results.result32);
        break;
    case _C_LNG_LNG:
    case _C_ULNG_LNG:
        [self completelyReadBytes: &results.result8 count: 8];
        *(long long*)address = N64_TO_H(results.result64);
        break;
    case _C_FLT:
        *(float*)address = [self readFloatImpl];
        break;
    case _C_DBL:
        *(double*)address = [self readDoubleImpl];
        break;
    case _C_SEL:
        *(SEL*)address = [self readSelectorImpl];
        break;
    case _C_CHARPTR:
        pointerData = [[OLPointerData alloc] init];
        strLen = [self readInt16Impl];
        if (strLen & BIG_STRING_TELL_BIT)
        {
            chunkCount = strLen & ~BIG_STRING_TELL_BIT;
            for (i = 0; i < chunkCount; i++)
                [pointerData readFromStream: self count: [self readInt16Impl]];
        }
        else
        {
            [pointerData readFromStream: self count: strLen];
        }
        [pointerData nullTerminate];
        *(uint8_t**)address = [pointerData bytes];
        OBJ_AUTORELEASE(pointerData);
        break;
    case _C_ARY_B:
        arrayCount = strtol(++valueType, &endOfNum, 10);
        typeSize = objc_sizeof_type(endOfNum);
        for (cur = 0; cur < arrayCount; cur++)
            [self decodeValueOfObjCType: endOfNum at: address + (typeSize * cur)];
        break;
    case _C_PTR:
        pointerData = [[OLPointerData alloc] initWithSize: objc_sizeof_type(++valueType)];
        *(void**)address = [pointerData bytes];
        [self decodeValueOfObjCType: valueType at: [pointerData bytes]];
        OBJ_AUTORELEASE(pointerData);
        break;
    case _C_STRUCT_B:
        objc_layout_structure(valueType, &slay);
        while (objc_layout_structure_next_member(&slay))
        {
            objc_layout_structure_get_info(&slay, &offset, NULL, &fieldType);
            [self decodeValueOfObjCType: fieldType at: (uint8_t*)address + offset];
        }
        break;
    case _C_ID:
        RAISE_EXCEPTION(INVALID_ARGUMENT,
            @"Objects can only be decoded by OLObjectInStream");
    case _C_CLASS:
        RAISE_EXCEPTION(INVALID_ARGUMENT,
            @"Classes can only be decoded by OLObjectInStream");
    default:
        RAISE_EXCEPTION(INVALID_ARGUMENT,
            @"Will not decode type \"%s\"", valueType);
    }
}
#endif

- (BOOL) readBool
{
    uint8_t value;

    value = [self readByte];
    return value ? YES : NO;
}

- (uint8_t) readByte
{
    uint8_t byte;

    [self completelyReadBytes: &byte count: 1];
    return byte;
}

- (unsigned) readBytes: (uint8_t*)buffer count: (unsigned)max
{
	SUBCLASS_MUST_IMPLEMENT;
    return UINT_MAX;
}

- (double) readDouble
{
    return [self readDoubleImpl];
}

- (float) readFloat
{
    return  [self readFloatImpl];
}

- (unsigned) readInt
{
    return [self readIntImpl];
}

- (uint16_t) readInt16
{
    return [self readInt16Impl];
}

- (uint32_t) readInt32
{
    uint32_t value;

    [self completelyReadBytes: (uint8_t*)&value count: 4];
    return N32_TO_H(value);
}

- (uint64_t) readInt64
{
    uint64_t value;

    [self completelyReadBytes: (uint8_t*)&value count: 8];
    return N64_TO_H(value);
}

- (SEL) readSelector
{
    return [self readSelectorImpl];
}

@end

@implementation OLInStream (PackageMethods)

- (void) completelyReadBytes: (uint8_t*)buffer count: (unsigned)count
{
    unsigned cur;
    unsigned total = 0;

    while (total < count)
    {
        cur = [self readBytes: buffer + total count: count - total];
        if (cur == UINT_MAX)
            RAISE_EXCEPTION(OLEndOfStreamException, @"");
        total += cur;
    }
}

@end

@implementation OLInStream (PrivateMethods)

- (double) readDoubleImpl
{
    union OLDoubleInt
    {
        uint64_t    integer;
        double      dbl;
        uint8_t     bytes[8];
    } result;

    [self completelyReadBytes: result.bytes count: 8];
    result.integer = N64_TO_H(result.integer);
    return result.dbl;
}

- (float) readFloatImpl
{
    union OLFloatInt
    {
        uint32_t    integer;
        float       flt;
        uint8_t     bytes[4];
    } result;

    [self completelyReadBytes: result.bytes count: 4];
    result.integer = N32_TO_H(result.integer);
    return result.flt;
}

- (unsigned) readIntImpl
{
    uint32_t value;

    [self completelyReadBytes: (uint8_t*)&value count: 4];
    return N32_TO_H(value);
}

- (uint16_t) readInt16Impl
{
    uint16_t value;

    [self completelyReadBytes: (uint8_t*)&value count: 2];
    return N16_TO_H(value);
}

- (SEL) readSelectorImpl
{
    uint16_t nameLen;
    uint16_t typesLen;
    uint8_t* nameBuf;
    uint8_t* typesBuf;
    SEL result;
#if defined(OL_NO_OPENSTEP)
    OLException* xcpt;
#else
    NSString* nameStr;
    NSString* typesStr;
#endif

    nameLen = [self readInt16Impl];
    if (nameLen != 0)
    {
        nameBuf = objc_malloc(nameLen + 1);
        [self completelyReadBytes: nameBuf count: nameLen];
        nameBuf[nameLen] = 0;
    }
    else
    {
        nameBuf = NULL;
    }
    typesLen = [self readInt16Impl];
    if (typesLen != 0)
    {
        typesBuf = objc_malloc(typesLen + 1);
        [self completelyReadBytes: typesBuf count: typesLen];
        typesBuf[typesLen] = 0;
    }
    else
    {
        typesBuf = NULL;
    }
    if (typesLen != 0)
        result = sel_get_typed_uid((const char*)nameBuf, (const char*)typesBuf);
    else
        result = sel_get_any_typed_uid((const char*)nameBuf);
    if (result == NULL)
    {
        if (typesLen != 0)
            result = sel_register_typed_name((const char*)nameBuf, (const char*)typesBuf);
        else
            result = sel_register_name((const char*)nameBuf);
        if (result == NULL)
        {
#if defined(OL_NO_OPENSTEP)
            xcpt = [[OLException alloc] initWithType: [OLInputOutputException cString]
                message: "Unable to register the selector \"%s\" with types \"%s\"",
                nameBuf, typesBuf];
            objc_free(nameBuf);
            objc_free(typesBuf);
            [xcpt raise];
#else
            nameStr = [NSString stringWithUTF8String: (const char*)nameBuf];
            typesStr = [NSString stringWithUTF8String: (const char*)typesBuf];
            objc_free(nameBuf);
            objc_free(typesBuf);
            RAISE_EXCEPTION(OLInputOutputException,
                @"Unable to register the selector \"%@\" with types \"%@\"",
                nameStr, typesStr);
#endif
        }
    }
    objc_free(nameBuf);
    objc_free(typesBuf);
    return result;
}

@end
