//
// $Id: HashSet.m,v 1.19 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "HashSet.h"
#import "HashTable.h"
#import "Macros.h"
#import "RunTime.h"
#import "ObjectInStream.h"
#import "ObjectOutStream.h"
#import "Utility.h"
#if !defined(OL_NO_OPENSTEP)
#import <Foundation/NSCoder.h>
#import <Foundation/NSAutoreleasePool.h>
#endif
#import <stdlib.h>

#if !defined(OL_NO_OPENSTEP)
extern NSString* const KEYEQUAL_KEY;
extern NSString* const TABLE_SIZE_KEY;
#endif

@implementation OLHashSet

+ (id) hashSet
{
    OL_BEGIN_AUTO_CTOR(OLHashSet)
        init
    OL_END_AUTO_CTOR;
}

+ (id) hashSetFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OL_BEGIN_AUTO_CTOR(OLHashSet)
        initFrom: first to: last
    OL_END_AUTO_CTOR;
}

+ (id) hashSetWithHashSet: (OLHashSet*)right
{
    OL_BEGIN_AUTO_CTOR(OLHashSet)
        initWithHashSet: right
    OL_END_AUTO_CTOR;
}

- (id) init
{
    return [self initWithTableSize: 100];
}

- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    [self init];
    [self insertFrom: first to: last];
    return self;
}

- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last tableSize: (unsigned)size
{
    [self initWithTableSize: size];
    [self insertFrom: first to: last];
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last tableSize: (unsigned)size keyEqual: (Object<OLBoolBinaryFunction>*)eq
#else
- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last tableSize: (unsigned)size keyEqual: (NSObject<OLBoolBinaryFunction>*)eq
#endif
{
    [self initWithTableSize: size keyEqual: eq];
    [self insertFrom: first to: last];
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    NSObject<OLBoolBinaryFunction>* eq;
    unsigned tableSize;

    [super init];
    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        eq = [decoder decodeObjectForKey: KEYEQUAL_KEY];
        tableSize = [decoder decodeIntForKey: TABLE_SIZE_KEY];
    }
    else
    {
        eq = [decoder decodeObject];
        [decoder decodeValueOfObjCType: @encode(unsigned) at: &tableSize];
    }
    table = [[OLHashTable alloc] initWithSize: tableSize equalFunc: eq];
    readContainerWithInsertRange(self, decoder, @selector(decodeObject));
    return self;
}
#endif

- (id) initWithHashSet: (OLHashSet*)right
{
    [super init];
    table = [[OLHashTable alloc] initWithHashTable: right->table];
    return self;
}

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
#if defined(OL_NO_OPENSTEP)
    Object<OLBoolBinaryFunction>* eq = [stream readObject];
#else
    NSObject<OLBoolBinaryFunction>* eq = [stream readObject];
#endif
    unsigned tableSize = [stream readInt];

    [super init];
    table = [[OLHashTable alloc] initWithSize: tableSize equalFunc: eq];
    readContainerWithInsertRange(self, stream, @selector(readObject));
    return self;
}

- (id) initWithTableSize: (unsigned)size
{
    OLEqualTo* eq = [[OLEqualTo alloc] init];

    [self initWithTableSize: size keyEqual: eq];
    OBJ_RELEASE(eq);
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) initWithTableSize: (unsigned)size keyEqual: (Object<OLBoolBinaryFunction>*)eq
#else
- (id) initWithTableSize: (unsigned)size keyEqual: (NSObject<OLBoolBinaryFunction>*)eq
#endif
{
    [super init];
    table = [[OLHashTable alloc] initWithSize: size equalFunc: eq];
    return self;
}


#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(table);
    SUPER_FREE;
}

- (OLHashIterator*) begin
{
    return [table begin];
}

- (void) clear
{
    [table clear];
}

- (int) compare: (id)other
{
    if (IS_KIND_OF(other, OLHashSet))
        return [table compare: ((OLHashSet*)other)->table];
    return -1;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLHashSet alloc] initWithHashSet: self];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLHashSet allocWithZone: zone] initWithHashSet: self];
}
#endif

- (unsigned) count: (id)value
{
    return [table count: value];
}

- (BOOL) empty
{
    return [table empty];
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    unsigned tableSize = [table tableSize];

    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeObject: [table keyEqual] forKey: KEYEQUAL_KEY];
        [encoder encodeInt: tableSize forKey: TABLE_SIZE_KEY];
    }
    else
    {
        [encoder encodeObject: [table keyEqual]];
        [encoder encodeValueOfObjCType: @encode(unsigned) at: &tableSize];
    }
    writeContainer(table, @selector(beginImpl), @selector(endImpl),
        encoder, @selector(encodeObject:));
}
#endif

- (OLHashIterator*) end
{
    return [table end];
}

- (OLPair*) equalRange: (id)value
{
    return [table equalRange: value];
}

- (void) erase: (OLHashIterator*)where
{
    [table erase: where];
}

- (void) eraseFrom: (OLHashIterator*)first to: (OLHashIterator*)last
{
    [table eraseFrom: first to: last];
}

- (unsigned) eraseValue: (id)value
{
    return [table eraseKey: value];
}

- (OLHashIterator*) find: (id)object
{
    return [table find: object];
}

- (id) insert: (id)object
{
    return [table insertUnique: object];
}

- (void) insertFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    [table insertUniqueFrom: first to: last];
}

- (BOOL) isEqual: (id)object
{
    return IS_KIND_OF(object, OLHashSet) &&
           [((OLHashSet*)object)->table isEqualUnique: table];
}

#if defined(OL_NO_OPENSTEP)
- (Object<OLBoolBinaryFunction>*) keyEqual
#else
- (NSObject<OLBoolBinaryFunction>*) keyEqual
#endif
{
    return [table keyEqual];
}

- (unsigned) maxSize
{
    return [table maxSize];
}

- (unsigned) size
{
    return [table size];
}

- (void) swap: (OLHashSet*)right
{
    [table swap: right->table];
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [stream writeObject: [table keyEqual]];
    [stream writeInt: [table tableSize]];
    writeContainer(table, @selector(beginImpl), @selector(endImpl),
        stream, @selector(writeObject:));
}

@end

@implementation OLHashMultiSet

+ (id) hashMultiSet
{
    OL_BEGIN_AUTO_CTOR(OLHashMultiSet)
        init
    OL_END_AUTO_CTOR;
}

+ (id) hashMultiSetFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OL_BEGIN_AUTO_CTOR(OLHashMultiSet)
        initFrom: first to: last
    OL_END_AUTO_CTOR;
}

+ (id) hashMultiSetWithHashSet: (OLHashSet*)right
{
    OL_BEGIN_AUTO_CTOR(OLHashMultiSet)
        initWithHashSet: right
    OL_END_AUTO_CTOR;
}

- (id) insert: (id)object
{
    return [table insertEqual: object];
}

- (void) insertFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    [table insertEqualFrom: first to: last];
}

- (BOOL) isEqual: (id)object
{
    return IS_KIND_OF(object, OLHashMultiSet) &&
           [((OLHashMultiSet*)object)->table isEqualNonUnique: table];
}

@end
