//
// $Id: FileInStream.m,v 1.17 2007/04/15 23:32:57 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "FileInStream.h"
#import "Macros.h"
#import "ConfigPrivate.h"
#import "Exception.h"
#if defined(OL_NO_OPENSTEP)
#import "Text.h"
#else
#import <Foundation/NSException.h>
#import <Foundation/NSString.h>
#endif
#import <unistd.h>
#import <errno.h>
#import <string.h>
#import <fcntl.h>
#import <limits.h>

#define FREE_MY_RESOURCES \
    [self close]

@implementation OLFileInStream

+ (id) streamWithPath: (const char*)path
{
    OL_BEGIN_AUTO_CTOR(OLFileInStream)
        initWithPath: path
    OL_END_AUTO_CTOR;
}

- (id) initWithFileDescriptor: (ol_fd)fileDesc
{
    [super init];
    fd = fileDesc;
    return self;
}

- (id) initWithPath: (const char*)path
{
#if defined(OL_WINDOWS)
    char msgBuf[8192];

    [super init];
    fd = CreateFile(path,
                    FILE_READ_DATA,
                    FILE_SHARE_READ,
                    NULL,
                    OPEN_EXISTING,
                    FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN,
                    NULL);
    if (fd == INVALID_HANDLE_VALUE)
    {
        if (FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,
                          NULL,
                          GetLastError(),
                          0,
                          msgBuf,
                          8191,
                          NULL) == 0)
        {
            msgBuf[0] = 0;
        }
        RAISE_EXCEPTION(OLInputOutputException, @"Error opening file - %s", msgBuf);
    }
#else
    [super init];
    fd = open(path, O_RDONLY);
    if (fd == -1)
        RAISE_EXCEPTION(OLInputOutputException, @"Error opening file - %s", strerror(errno));
#endif
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (void) dealloc
{
	FREE_MY_RESOURCES;
	SUPER_FREE;
}
#endif

- (void) close
{
#if defined(OL_WINDOWS)
    char msgBuf[8192];

    if (!(fd == INVALID_HANDLE_VALUE || fd == GetStdHandle(STD_INPUT_HANDLE)))
    {
        if (!CloseHandle(fd))
        {
            fd = INVALID_HANDLE_VALUE;
            if (FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,
                              NULL,
                              GetLastError(),
                              0,
                              msgBuf,
                              8191,
                              NULL) == 0)
            {
                msgBuf[0] = 0;
            }
            RAISE_EXCEPTION(OLInputOutputException, @"Error closing file - %s", msgBuf);
        }
    }
    fd = INVALID_HANDLE_VALUE;
#else
    if (!(fd == -1 || fd == STDIN_FILENO))
    {
        if (close(fd) != 0)
        {
            fd = -1;
            RAISE_EXCEPTION(OLInputOutputException, @"Error closing file - %s", strerror(errno));
        }
    }
    fd = -1;
#endif
}

- (ol_fd) fileDescriptor
{
    return fd;
}

#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources
{
	FREE_MY_RESOURCES;
	[super freeStreamResources];
}
#endif

- (unsigned) readBytes: (uint8_t*)buffer count: (unsigned)max
{
#if defined(OL_WINDOWS)
    DWORD readCount;
    char msgBuf[8192];

    if (!ReadFile(fd, buffer, max, &readCount, NULL))
    {
        if (FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,
                          NULL,
                          GetLastError(),
                          0,
                          msgBuf,
                          8191,
                          NULL) == 0)
        {
            msgBuf[0] = 0;
        }
        RAISE_EXCEPTION(OLInputOutputException, @"Error reading from file - %s", msgBuf);
    }
#else
    ssize_t readCount = read(fd, buffer, max);

    if (readCount == -1)
        RAISE_EXCEPTION(OLInputOutputException, @"Error reading from file - %s", strerror(errno));
#endif
    if (readCount == 0)
        readCount = UINT_MAX;
    return readCount;
}

@end
