//
// $Id: BzlibOutStream.m,v 1.9 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "BzlibOutStream.h"
#import "OutStreamPackage.h"
#import "Macros.h"
#import "RunTime.h"
#import "Exception.h"
#if defined(OL_NO_OPENSTEP)
#import "Text.h"
#else
#import <Foundation/NSString.h>
#import <Foundation/NSException.h>
#endif
#import <stddef.h>
#import <bzlib.h>
#import <stdlib.h>

#define FREE_MY_RESOURCES \
    [self close]; \
	objc_free(buffer)

void* __ol_bzalloc(void* alloc, int items, int size)
{
    return objc_malloc(items * size);
}

void __ol_bzfree(void* alloc, void* address)
{
    return objc_free(address);
}

@implementation OLBzlibOutStream

+ (id) streamWithOutStream: (OLOutStream*)underStream
{
    OL_BEGIN_AUTO_CTOR(OLBzlibOutStream)
        initWithOutStream: underStream
    OL_END_AUTO_CTOR;
}

+ (id) streamWithOutStream: (OLOutStream*)underStream blockSize: (int)block
{
    OL_BEGIN_AUTO_CTOR(OLBzlibOutStream)
        initWithOutStream: underStream blockSize: block
    OL_END_AUTO_CTOR;
}

- (id) initWithOutStream: (OLOutStream*)underStream
{
    return [self initWithOutStream: underStream blockSize: 9 bufferSize: 1024];
}

- (id) initWithOutStream: (OLOutStream*)underStream blockSize: (int)block
{
    return [self initWithOutStream: underStream blockSize: block bufferSize: 1024];
}

- (id) initWithOutStream: (OLOutStream*)underStream blockSize: (int)block bufferSize: (unsigned)size
{
    int rc;

    [super initWithOutStream: underStream];
    if (block < 1 || block > 9)
    {
        RAISE_EXCEPTION(INVALID_ARGUMENT, @"Block size must be in the range [1,9]");
    }
    bzstream = objc_malloc(sizeof(bz_stream));
    BZSTREAM->bzalloc = __ol_bzalloc;
    BZSTREAM->bzfree = __ol_bzfree;
    BZSTREAM->opaque = NULL;
    rc = BZ2_bzCompressInit(BZSTREAM, block, 0, 0);
    if (rc != BZ_OK)
    {
        buffer = NULL;
        bufferSize = 0;
        objc_free(bzstream);
        bzstream = NULL;
        RAISE_EXCEPTION(OLInputOutputException, @"Error initializing for decompression");
    }
    buffer = objc_malloc(size);
    bufferSize = size;
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (void) dealloc
{
	FREE_MY_RESOURCES;
	SUPER_FREE;
}
#endif

- (void) close
{
    int rc;

    if (bzstream != NULL)
    {
        BZSTREAM->next_in = NULL;
        BZSTREAM->avail_in = 0;
        do
        {
            BZSTREAM->next_out = (char*)buffer;
            BZSTREAM->avail_out = bufferSize;
            rc = BZ2_bzCompress(BZSTREAM, BZ_FINISH);
            [stream completelyWriteBytes: buffer count: bufferSize - BZSTREAM->avail_out];
        } while (rc == BZ_FINISH_OK);
        if (rc != BZ_STREAM_END)
        {
            BZ2_bzCompressEnd(BZSTREAM);
            RAISE_EXCEPTION(OLInputOutputException, @"Error closing OLBzlibOutStream");
        }
        BZ2_bzCompressEnd(BZSTREAM);
        objc_free(bzstream);
        bzstream = NULL;
    }
    [stream flush];
    [stream close];
}

#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources
{
	FREE_MY_RESOURCES;
	[super freeStreamResources];
}
#endif

- (unsigned) writeBytes: (const uint8_t*)bytes count: (unsigned)num
{
    BZSTREAM->next_in = (char*)bytes;
    BZSTREAM->avail_in = num;
    while (BZSTREAM->avail_in > 0)
    {
        BZSTREAM->next_out = (char*)buffer;
        BZSTREAM->avail_out = bufferSize;
        if (BZ2_bzCompress(BZSTREAM, BZ_RUN) != BZ_RUN_OK)
        {
            RAISE_EXCEPTION(OLInputOutputException, @"Error compressing");
        }
        [stream completelyWriteBytes: buffer count: bufferSize - BZSTREAM->avail_out];
    }
    return num;
}

@end
