/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "support.h"

START_TEST (check_san_new)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_USER, 0, "test", "/", SML_MIMETYPE_UNKNOWN, &error);
	fail_unless(san != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	smlNotificationFree(san);
}
END_TEST

START_TEST (check_san_empty)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_UNSPECIFIED, SML_SAN_INITIATOR_USER, 0, "tttt", "/", SML_MIMETYPE_UNKNOWN, &error);
	fail_unless(san != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x00, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0x00, 0x00, //Session id
		0x05, 0x74, 0x74, 0x74, 0x74, 0x00, //server identifier
		0x00}; // num syncs + future
	
	char *buffer = NULL;
	unsigned int size = 0;
	fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(size == sizeof(target), NULL);
	fail_unless(memcmp(target, buffer, size) == 0, NULL);
	
	g_free(buffer);
	smlNotificationFree(san);
}
END_TEST

START_TEST (check_san_empty2)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "ttttt", "/", SML_MIMETYPE_UNKNOWN, &error);
	fail_unless(san != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x06, 0x74, 0x74, 0x74, 0x74, 0x74, 0x00, //server identifier
		0x00}; // num syncs + future
	
	char *buffer = NULL;
	unsigned int size = 0;
	fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(size == sizeof(target), NULL);
	fail_unless(memcmp(target, buffer, size) == 0, NULL);
	
	g_free(buffer);
	smlNotificationFree(san);
}
END_TEST

START_TEST (check_san_alert)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "ttttt", "/", SML_MIMETYPE_UNKNOWN, &error);
	fail_unless(san != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlNotificationNewAlert(san, SML_ALERT_TWO_WAY_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "tttt", &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x06, 0x74, 0x74, 0x74, 0x74, 0x74, 0x00, //server identifier
		0x10,// num syncs + future
		0x60, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x05, 0x74, 0x74, 0x74, 0x74, 0x00 }; //Server URI;
	
	char *buffer = NULL;
	unsigned int size = 0;
	fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(size == sizeof(target), NULL);
	fail_unless(memcmp(target, buffer, size) == 0, NULL);
	
	g_free(buffer);
	smlNotificationFree(san);
}
END_TEST

START_TEST (check_san_alert2)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "ttttt", "/", SML_MIMETYPE_UNKNOWN, &error);
	fail_unless(san != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlNotificationNewAlert(san, SML_ALERT_TWO_WAY_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "tttt", &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlNotificationNewAlert(san, SML_ALERT_REFRESH_FROM_SERVER_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "tttt", &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x06, 0x74, 0x74, 0x74, 0x74, 0x74, 0x00, //server identifier
		0x20,// num syncs + future
		0x60, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x05, 0x74, 0x74, 0x74, 0x74, 0x00, //Server URI;
		0xA0, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x05, 0x74, 0x74, 0x74, 0x74, 0x00 }; //Server URI;
	
	char *buffer = NULL;
	unsigned int size = 0;
	fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(size == sizeof(target), NULL);
	fail_unless(memcmp(target, buffer, size) == 0, NULL);
	
	g_free(buffer);
	smlNotificationFree(san);
}
END_TEST

START_TEST (check_san_11_assemble_parse)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_11, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "ttttt", "/", SML_MIMETYPE_XML, &error);
	fail_unless(san != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlNotificationNewAlert(san, SML_ALERT_TWO_WAY_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "tttt", &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *buffer = NULL;
	unsigned int size = 0;
	fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlNotificationFree(san);
	
	
	san = smlNotificationParse(buffer, size, &error);
	fail_unless(san != NULL, NULL);
	fail_unless(error == NULL, NULL);
	g_free(buffer);
	
	fail_unless(smlNotificationGetVersion(san) == SML_SAN_VERSION_11, NULL);
	fail_unless(smlNotificationGetMode(san) == SML_SAN_UIMODE_USER, NULL);
	fail_unless(smlNotificationGetInitiator(san) == SML_SAN_INITIATOR_SERVER, NULL);
	fail_unless(smlNotificationGetSessionID(san) == 65535, NULL);
	fail_unless(!strcmp(smlNotificationGetIdentifier(san), "ttttt"), NULL);
	
	fail_unless(smlNotificationNumAlerts(san) == 1, NULL);
	SmlSanAlert *alert = smlNotificationNthAlert(san, 0);
	fail_unless(alert != NULL, NULL);
	fail_unless(smlSanAlertGetType(alert) == SML_ALERT_TWO_WAY_BY_SERVER, NULL);
	fail_unless(!strcmp(smlSanAlertGetContentType(alert), SML_ELEMENT_TEXT_VCARD), NULL);
	fail_unless(!strcmp(smlSanAlertGetServerURI(alert), "tttt"), NULL);
	
	smlNotificationFree(san);


}
END_TEST

START_TEST (check_san_parse)
{
	setup_testbed(NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x06, 0x74, 0x74, 0x74, 0x74, 0x74, 0x00, //server identifier
		0x20,// num syncs + future
		0x60, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x05, 0x74, 0x74, 0x74, 0x74, 0x00, //Server URI;
		0xA0, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x05, 0x74, 0x74, 0x74, 0x74, 0x00 }; //Server URI;
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationParse(target, sizeof(target), &error);
	fail_unless(san != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlNotificationGetVersion(san) == SML_SAN_VERSION_12, NULL);
	fail_unless(smlNotificationGetMode(san) == SML_SAN_UIMODE_USER, NULL);
	fail_unless(smlNotificationGetInitiator(san) == SML_SAN_INITIATOR_SERVER, NULL);
	fail_unless(smlNotificationGetSessionID(san) == 65535, NULL);
	fail_unless(!strcmp(smlNotificationGetIdentifier(san), "ttttt"), NULL);
	
	fail_unless(smlNotificationNumAlerts(san) == 2, NULL);
	SmlSanAlert *alert = smlNotificationNthAlert(san, 0);
	fail_unless(alert != NULL, NULL);
	fail_unless(smlSanAlertGetType(alert) == SML_ALERT_TWO_WAY_BY_SERVER, NULL);
	fail_unless(!strcmp(smlSanAlertGetContentType(alert), SML_ELEMENT_TEXT_VCARD), NULL);
	fail_unless(!strcmp(smlSanAlertGetServerURI(alert), "tttt"), NULL);
	
	alert = smlNotificationNthAlert(san, 1);
	fail_unless(alert != NULL, NULL);
	fail_unless(smlSanAlertGetType(alert) == SML_ALERT_REFRESH_FROM_SERVER_BY_SERVER, NULL);
	fail_unless(!strcmp(smlSanAlertGetContentType(alert), SML_ELEMENT_TEXT_VCARD), NULL);
	fail_unless(!strcmp(smlSanAlertGetServerURI(alert), "tttt"), NULL);
	
	smlNotificationFree(san);
}
END_TEST

START_TEST (check_san_parse2)
{
	setup_testbed(NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x00, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0x00, 0x00, //Session id
		0x00, //server identifier
		0x00}; //num syncs
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationParse(target, sizeof(target), &error);
	fail_unless(san != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlNotificationGetVersion(san) == SML_SAN_VERSION_12, NULL);
	fail_unless(smlNotificationGetMode(san) == SML_SAN_UIMODE_UNSPECIFIED, NULL);
	fail_unless(smlNotificationGetInitiator(san) == SML_SAN_INITIATOR_USER, NULL);
	fail_unless(smlNotificationGetSessionID(san) == 0, NULL);
	fail_unless(smlNotificationGetIdentifier(san) == NULL, NULL);
	
	fail_unless(smlNotificationNumAlerts(san) == 0, NULL);
	
	smlNotificationFree(san);
}
END_TEST

START_TEST (check_san_parse_error)
{
	setup_testbed(NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x00, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0x00, 0x00, //Session id
		0x00}; //num syncs
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationParse(target, sizeof(target), &error);
	fail_unless(san == NULL, NULL);
	fail_unless(error != NULL, NULL);
}
END_TEST

START_TEST (check_san_parse_error2)
{
	setup_testbed(NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x00, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0x00, 0x00, //Session id
		0x01, //server identifier
		0x00}; //num syncs
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationParse(target, sizeof(target), &error);
	fail_unless(san == NULL, NULL);
	fail_unless(error != NULL, NULL);
}
END_TEST

START_TEST (check_san_parse_error3)
{
	setup_testbed(NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x00, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0x00, 0x00, //Session id
		0x01, 0x00,//server identifier
		0x10}; //num syncs
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationParse(target, sizeof(target), &error);
	fail_unless(san == NULL, NULL);
	fail_unless(error != NULL, NULL);
}
END_TEST

START_TEST (check_san_parse_error4)
{
	setup_testbed(NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x02, 0x00, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0x00, 0x00, //Session id
		0x01, 0x00,//server identifier
		0x00}; //num syncs
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationParse(target, sizeof(target), &error);
	fail_unless(san == NULL, NULL);
	fail_unless(error != NULL, NULL);
}
END_TEST

START_TEST (check_san_parse_error5)
{
	setup_testbed(NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x06, 0x74, 0x74, 0x74, 0x74, 0x74, 0x00, //server identifier
		0x10,// num syncs + future
		0x50, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x05, 0x74, 0x74, 0x74, 0x74, 0x00}; //Server URI;
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationParse(target, sizeof(target), &error);
	fail_unless(san == NULL, NULL);
	fail_unless(error != NULL, NULL);
}
END_TEST

START_TEST (check_san_parse_error6)
{
	setup_testbed(NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x06, 0x74, 0x74, 0x74, 0x74, 0x74, 0x00, //server identifier
		0x10,// num syncs + future
		0x70, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x01}; //Server URI;
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationParse(target, sizeof(target), &error);
	fail_unless(san == NULL, NULL);
	fail_unless(error != NULL, NULL);
}
END_TEST

START_TEST (check_san_parse_error7)
{
	setup_testbed(NULL);
	
	char target[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //DIGEST
		0x03, 0x38, //version + ui + init + start of future
		0x00, 0x00, 0x00, //Future
		0xFF, 0xFF, //Session id
		0x06, 0x74, 0x74, 0x74, 0x74, 0x74, 0x00, //server identifier
		0x10,// num syncs + future
		0x70, //Sync type + future
		0x00, 0x00, 0x07, //Content type
		0x01, 0x00, 0x00}; //Server URI;
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationParse(target, sizeof(target), &error);
	fail_unless(san == NULL, NULL);
	fail_unless(error != NULL, NULL);
}
END_TEST

/*START_TEST (check_san_send)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "ttttt", "/", &error);
	fail_unless(san != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlNotificationNewAlert(san, SML_ALERT_TWO_WAY_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "tttt", &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlNotificationNewAlert(san, SML_ALERT_REFRESH_FROM_SERVER_BY_SERVER, SML_ELEMENT_TEXT_VCARD, "ttttaaaa", &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *buffer = NULL;
	unsigned int size = 0;
	fail_unless(smlNotificationAssemble(san, &buffer, &size, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool smlNotificationSend(SmlNotification *san, SmlTransport *tsp, SmlTransportSendCb callback, void *userdata, SmlError **error)
	
	
	fail_unless(size == sizeof(target), NULL);
	fail_unless(memcmp(target, buffer, size) == 0, NULL);
	
	g_free(buffer);
	smlNotificationFree(san);
}
END_TEST*/

Suite *san_suite(void)
{
	Suite *s = suite_create("San");
	//Suite *s2 = suite_create("San");
	
	create_case(s, "check_san_new", check_san_new);
	create_case(s, "check_san_empty", check_san_empty);
	create_case(s, "check_san_empty2", check_san_empty2);
	create_case(s, "check_san_alert", check_san_alert);
	create_case(s, "check_san_alert2", check_san_alert2);
	create_case(s, "check_san_parse", check_san_parse);
	create_case(s, "check_san_parse2", check_san_parse2);
	create_case(s, "check_san_11_assemble_parse", check_san_11_assemble_parse);
	create_case(s, "check_san_parse_error", check_san_parse_error);
	create_case(s, "check_san_parse_error2", check_san_parse_error2);
	create_case(s, "check_san_parse_error3", check_san_parse_error3);
	create_case(s, "check_san_parse_error4", check_san_parse_error4);
	create_case(s, "check_san_parse_error5", check_san_parse_error5);
	create_case(s, "check_san_parse_error6", check_san_parse_error6);
	create_case(s, "check_san_parse_error7", check_san_parse_error7);
	
	return s;
}

int main(void)
{
	int nf;

	Suite *s = san_suite();
	
	SRunner *sr;
	sr = srunner_create(s);
	srunner_run_all(sr, CK_NORMAL);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (nf == 0) ? 0 : 1;
}
