/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "syncml.h"

#include "syncml_internals.h"
#include "sml_parse_internals.h"
#include "sml_command_internals.h"
#include "sml_session_internals.h"
#include "sml_elements_internals.h"

/**
 * @defgroup GroupIDPrivate Group Description Internals
 * @ingroup ParentGroupID
 * @brief The private part
 * 
 */
/*@{*/

SmlCommandType smlCommandTypeFromString(const char *name, SmlError **error)
{
	if (!name)
		return SML_COMMAND_TYPE_UNKNOWN;
	
	if (!strcmp(name, SML_ELEMENT_ALERT)) {
		return SML_COMMAND_TYPE_ALERT;
	} else if (!strcmp(name, SML_ELEMENT_SYNC)) {
		return SML_COMMAND_TYPE_SYNC;
	} else if (!strcmp(name, SML_ELEMENT_PUT)) {
		return SML_COMMAND_TYPE_PUT;
	} else if (!strcmp(name, SML_ELEMENT_SYNCHDR)) {
		return SML_COMMAND_TYPE_HEADER;
	} else if (!strcmp(name, SML_ELEMENT_ADD)) {
		return SML_COMMAND_TYPE_ADD;
	} else if (!strcmp(name, SML_ELEMENT_REPLACE)) {
		return SML_COMMAND_TYPE_REPLACE;
	} else if (!strcmp(name, SML_ELEMENT_MAP)) {
		return SML_COMMAND_TYPE_MAP;
	} else if (!strcmp(name, SML_ELEMENT_DELETE)) {
		return SML_COMMAND_TYPE_DELETE;
	} else if (!strcmp(name, SML_ELEMENT_RESULTS)) {
		return SML_COMMAND_TYPE_RESULTS;
	} else if (!strcmp(name, SML_ELEMENT_GET)) {
		return SML_COMMAND_TYPE_GET;
	}
	
	smlErrorSet(error, SML_ERROR_GENERIC, "Unknown command name \"%s\"", name);
	return SML_COMMAND_TYPE_UNKNOWN;
}

const char *smlCommandTypeToString(SmlCommandType type, SmlError **error)
{
	switch (type) {
		case SML_COMMAND_TYPE_ALERT:
			return SML_ELEMENT_ALERT;
		case SML_COMMAND_TYPE_SYNC:
			return SML_ELEMENT_SYNC;
		case SML_COMMAND_TYPE_PUT:
			return SML_ELEMENT_PUT;
		case SML_COMMAND_TYPE_HEADER:
			return SML_ELEMENT_SYNCHDR;
		case SML_COMMAND_TYPE_ADD:
			return SML_ELEMENT_ADD;
		case SML_COMMAND_TYPE_REPLACE:
			return SML_ELEMENT_REPLACE;
		case SML_COMMAND_TYPE_DELETE:
			return SML_ELEMENT_DELETE;
		case SML_COMMAND_TYPE_MAP:
			return SML_ELEMENT_MAP;
		case SML_COMMAND_TYPE_GET:
			return SML_ELEMENT_GET;
		case SML_COMMAND_TYPE_RESULTS:
			return SML_ELEMENT_RESULTS;
		default:
			;
	}
		
	smlErrorSet(error, SML_ERROR_GENERIC, "Unknown command type \"%i\"", type);
	return NULL;
}

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/

SmlStatus *smlStatusNew(SmlErrorType data, unsigned int cmdref, unsigned int msgref, SmlLocation *sourceref, SmlLocation *targetref, SmlCommandType type, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%i, %i, %i, %p, %p, %i, %p)", __func__, data, cmdref, msgref, sourceref, targetref, type, error);
	SmlStatus *status = smlTryMalloc0(sizeof(SmlStatus), error);
	if (!status)
		goto error;
	
	status->refCount = 1;
	status->cmdRef = cmdref;
	status->msgRef = msgref;
	status->type = type;
	if (data != SML_ERROR_UNKNOWN)
		status->data = g_strdup_printf("%i", data);
	
	if (sourceref) {
		status->sourceRef = sourceref;
		smlLocationRef(sourceref);
	}
	
	if (targetref) {
		status->targetRef = targetref;
		smlLocationRef(targetref);
	}
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, status);
	return status;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlStatus *smlStatusRef(SmlStatus *status)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, status);
	smlAssert(status);
	
	g_atomic_int_inc(&(status->refCount));
	
	smlTrace(TRACE_EXIT, "%s: New refcount: %i", __func__, status->refCount);
	return status;
}

void smlStatusUnref(SmlStatus *status)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, status);
	smlAssert(status);
	
	if (g_atomic_int_dec_and_test(&(status->refCount))) {
		smlTrace(TRACE_INTERNAL, "Refcount == 0!");
		
		if (status->sourceRef)
			smlLocationUnref(status->sourceRef);
	
		if (status->targetRef)
			smlLocationUnref(status->targetRef);
		
		g_free(status->data);
		
		if (status->anchor)
			smlAnchorFree(status->anchor);
			
		g_free(status);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlErrorType smlStatusGetCode(SmlStatus *status)
{
	smlAssert(status);
	smlAssert(status->data);
	return atoi(status->data);
}

SmlErrorClass smlStatusGetClass(SmlStatus *status)
{
	smlAssert(status);
	smlAssert(status->data);
	return (atoi(status->data) / 100);
}

SmlCommand *smlStatusGetResult(SmlStatus *status)
{
	if (status->type == SML_COMMAND_TYPE_RESULTS)
		return status->result;
	return NULL;
}

SmlBool smlStatusIsResult(SmlStatus *status)
{
	return status->type == SML_COMMAND_TYPE_RESULTS ? TRUE : FALSE;
}

SmlCommand *smlCommandNew(SmlCommandType type, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%i, %p)", __func__, type, error);
	
	SmlCommand *cmd = smlTryMalloc0(sizeof(SmlCommand), error);
	if (!cmd)
		goto error;
	
	cmd->refCount = 1;
	cmd->type = type;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, cmd);
	return cmd;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlStatus *smlCommandNewReply(SmlCommand *cmd, SmlErrorType code, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p)", __func__, cmd, code, error);
	smlAssert(cmd);
	
	SmlStatus *reply = smlStatusNew(code, cmd->cmdID, cmd->msgID, cmd->source, cmd->target, cmd->type, error);
	if (!reply)
		goto error;
	
	switch (cmd->type) {
		case SML_COMMAND_TYPE_ALERT:
			if (cmd->private.alert.anchor) {
				reply->anchor = smlAnchorNew(NULL, cmd->private.alert.anchor->next, error);
				if (!reply->anchor)
					goto error;
			}
			break;
		default:
		;
	}
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, reply);
	return reply;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlCommand *smlCommandNewResult(SmlCommand *cmd, SmlLocation *source, char *data, unsigned int size, const char *contenttype, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %i, %s, %p)", __func__, cmd, source, data, size, contenttype, error);
	smlAssert(cmd);
	
	SmlCommand *result = smlCommandNew(SML_COMMAND_TYPE_RESULTS, error);
	if (!result)
		goto error;
	
	result->private.results.status = smlStatusNew(SML_NO_ERROR, cmd->cmdID, cmd->msgID, cmd->source, cmd->target, SML_COMMAND_TYPE_RESULTS, error);
	if (!result->private.results.status)
		goto error_free_cmd;
		
	result->private.results.status->item = smlItemNewForData(data, size, error);
	if (!result->private.results.status->item)
		goto error_free_cmd;
	
	result->private.results.status->item->contenttype = g_strdup(contenttype);
	result->private.results.status->item->source = smlLocationClone(source, error);
	
	if (!result->private.results.status->item->source)
		goto error_free_cmd;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, result);
	return result;
	
error_free_cmd:
	smlCommandUnref(result);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlCommand *smlCommandRef(SmlCommand *cmd)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, cmd);
	smlAssert(cmd);
	
	g_atomic_int_inc(&(cmd->refCount));
	
	smlTrace(TRACE_EXIT, "%s: New refcount: %i", __func__, cmd->refCount);
	return cmd;
}

void smlCommandUnref(SmlCommand *cmd)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, cmd);
	smlAssert(cmd);
	
	if (g_atomic_int_dec_and_test(&(cmd->refCount))) {
		smlTrace(TRACE_INTERNAL, "Refcount == 0!");
		
		if (cmd->parent) {
			cmd->parent->children = g_list_remove(cmd->parent->children, cmd);
		
			smlCommandUnref(cmd->parent);
			cmd->parent = NULL;
		}
		
		switch (cmd->type) {
			case SML_COMMAND_TYPE_UNKNOWN:
			case SML_COMMAND_TYPE_HEADER:
			case SML_COMMAND_TYPE_SYNC:
				break;
			case SML_COMMAND_TYPE_ALERT:
				if (cmd->private.alert.anchor)
					smlAnchorFree(cmd->private.alert.anchor);
				if (cmd->private.alert.contentType)
					g_free(cmd->private.alert.contentType);
				break;
			case SML_COMMAND_TYPE_PUT:
			case SML_COMMAND_TYPE_GET:
				if (cmd->private.access.type)
					g_free(cmd->private.access.type);
					
				if (cmd->private.access.item)
					smlItemUnref(cmd->private.access.item);
				break;
			case SML_COMMAND_TYPE_ADD:
			case SML_COMMAND_TYPE_REPLACE:
			case SML_COMMAND_TYPE_DELETE:
				if (cmd->private.change.item)
					smlItemUnref(cmd->private.change.item);
				break;
			case SML_COMMAND_TYPE_MAP:
				while (cmd->private.map.items) {
					SmlMapItem *item = cmd->private.map.items->data;
					smlMapItemUnref(item);
					cmd->private.map.items = g_list_delete_link(cmd->private.map.items, cmd->private.map.items);
				}
				break;
			case SML_COMMAND_TYPE_RESULTS:
				if (cmd->private.results.status)
					smlStatusUnref(cmd->private.results.status);
				break;
		}
		
		if (cmd->target)
			smlLocationUnref(cmd->target);
		
		if (cmd->source)
			smlLocationUnref(cmd->source);
		
		g_free(cmd);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlCommand *smlCommandNewChange(SmlChangeType type, const char *uid, const char *data, unsigned int size, const char *contenttype, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%i, %s, %p, %i, %s, %p)", __func__, type, uid, data, size, contenttype, error);
	SmlCommand *cmd = NULL;
	
	switch (type) {
		case SML_CHANGE_ADD:
			cmd = smlCommandNew(SML_COMMAND_TYPE_ADD, error);
			break;
		case SML_CHANGE_REPLACE:
			cmd = smlCommandNew(SML_COMMAND_TYPE_REPLACE, error);
			break;
		case SML_CHANGE_DELETE:
			cmd = smlCommandNew(SML_COMMAND_TYPE_DELETE, error);
			break;
		default:
			smlErrorSet(error, SML_ERROR_GENERIC, "Unknown changetype");
	}
	if (!cmd)
		goto error;
	
	cmd->private.change.item = smlItemNewForData(data, size, error);
	if (!cmd->private.change.item)
		goto error_free_cmd;
	
	SmlLocation *loc = smlLocationNew(uid, NULL, error);
	if (!loc)
		goto error_free_item;
	
	if (type != SML_CHANGE_ADD)
		cmd->private.change.item->target = loc;
	else
		cmd->private.change.item->source = loc;
	
	cmd->private.change.item->contenttype = g_strdup(contenttype);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, cmd);
	return cmd;

error_free_item:
	smlItemUnref(cmd->private.change.item);
error_free_cmd:
	smlCommandUnref(cmd);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

/** Send a fragmented change. You can use this command to fragment a very large change into several
 * objects.
 * @param complete_size The overall size of the object. must be the sum over all partial_sizes
 * @param partial_size The size of this part.
 */
SmlCommand *smlCommandNewPartialChange(SmlChangeType type, const char *uid, const char *data, unsigned int complete_size, unsigned int partial_size, const char *contenttype, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%i, %s, %p, %i, %i, %s, %p)", __func__, type, uid, data, complete_size, partial_size, contenttype, error);
	SmlCommand *cmd = NULL;
	
	switch (type) {
		case SML_CHANGE_ADD:
			cmd = smlCommandNew(SML_COMMAND_TYPE_ADD, error);
			break;
		case SML_CHANGE_REPLACE:
			cmd = smlCommandNew(SML_COMMAND_TYPE_REPLACE, error);
			break;
		case SML_CHANGE_DELETE:
			cmd = smlCommandNew(SML_COMMAND_TYPE_DELETE, error);
			break;
		default:
			smlErrorSet(error, SML_ERROR_GENERIC, "Unknown changetype");
	}
	if (!cmd)
		goto error;
	cmd->size = complete_size;
	
	cmd->private.change.item = smlItemNewForData(data, partial_size, error);
	if (!cmd->private.change.item)
		goto error_free_cmd;
	
	SmlLocation *loc = smlLocationNew(uid, NULL, error);
	if (!loc)
		goto error_free_item;
	
	if (type != SML_CHANGE_ADD)
		cmd->private.change.item->target = loc;
	else
		cmd->private.change.item->source = loc;
	
	cmd->private.change.item->moreData = TRUE;
	cmd->private.change.item->contenttype = g_strdup(contenttype);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, cmd);
	return cmd;

error_free_item:
	smlItemUnref(cmd->private.change.item);
error_free_cmd:
	smlCommandUnref(cmd);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlCommand *smlCommandNewAlert(SmlAlertType type, SmlLocation *target, SmlLocation *source, const char *next, const char *last, const char *contenttype, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%i, %p, %p, %s, %s, %s, %p)", __func__, type, target, source, next, last, contenttype, error);
	
	SmlCommand *cmd = smlCommandNew(SML_COMMAND_TYPE_ALERT, error);
	if (!cmd)
		goto error;
	
	if (target) {
		cmd->target = target;
		smlLocationRef(target);
	}
	
	if (source) {
		cmd->source = source;
		smlLocationRef(source);
	}
	
	if (type != SML_ALERT_NEXT_MESSAGE && type != SML_ALERT_TWO_WAY_BY_SERVER) {
		cmd->private.alert.anchor = smlAnchorNew(last, next, error);
		if (!cmd->private.alert.anchor)
			goto error;
	}
	cmd->private.alert.type = type;
	cmd->private.alert.contentType = g_strdup(contenttype);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, cmd);
	return cmd;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlCommand *smlCommandNewSync(SmlLocation *target, SmlLocation *source, unsigned int num_changes, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %i, %p)", __func__, target, source, num_changes, error);
	SmlCommand *cmd = smlCommandNew(SML_COMMAND_TYPE_SYNC, error);
	if (!cmd)
		goto error;
	
	cmd->target = target;
	smlLocationRef(target);
	
	cmd->source = source;
	smlLocationRef(source);
	
	cmd->private.sync.numChanged = num_changes;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, cmd);
	return cmd;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlCommand *smlCommandNewPut(SmlLocation *target, SmlLocation *source, const char *data, unsigned int size, const char *contenttype, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %i, %s, %p)", __func__, target, source, data, size, contenttype, error);
	
	SmlCommand *cmd = smlCommandNew(SML_COMMAND_TYPE_PUT, error);
	if (!cmd)
		goto error;
	
	cmd->private.access.item = smlItemNewForData(data, size, error);
	if (!cmd->private.access.item)
		goto error_free_cmd;
	
	if (target) {
		cmd->target = target;
		smlLocationRef(target);
		smlItemSetTarget(cmd->private.access.item, cmd->target);
	}
	
	if (source) {
		cmd->source = source;
		smlLocationRef(source);
		smlItemSetSource(cmd->private.access.item, cmd->source);
	}
	
	cmd->private.access.item->contenttype = g_strdup(contenttype);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, cmd);
	return cmd;

error_free_cmd:
	smlCommandUnref(cmd);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlCommand *smlCommandNewGet(SmlLocation *target, const char *contenttype, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %s, %p)", __func__, target, contenttype, error);
	smlAssert(target);
	
	SmlCommand *cmd = smlCommandNew(SML_COMMAND_TYPE_GET, error);
	if (!cmd)
		goto error;
	
	cmd->private.access.item = smlItemNew(0, error);
	if (!cmd->private.access.item)
		goto error_free_cmd;
	
	cmd->target = target;
	smlLocationRef(target);
	smlItemSetTarget(cmd->private.access.item, cmd->target);
	
	cmd->private.access.item->contenttype = g_strdup(contenttype);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, cmd);
	return cmd;

error_free_cmd:
	smlCommandUnref(cmd);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlCommand *smlCommandNewMap(SmlLocation *target, SmlLocation *source, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, target, source, error);
	
	SmlCommand *cmd = smlCommandNew(SML_COMMAND_TYPE_MAP, error);
	if (!cmd)
		goto error;
	
	cmd->target = target;
	smlLocationRef(target);
	
	cmd->source = source;
	smlLocationRef(source);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, cmd);
	return cmd;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlBool smlCommandAddMapItem(SmlCommand *map, SmlMapItem *item, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, map, item, error);
	smlAssert(map);
	smlAssert(map->type == SML_COMMAND_TYPE_MAP);
	smlAssert(item);
	
	smlMapItemRef(item);
	map->private.map.items = g_list_append(map->private.map.items, item);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
}

/*@}*/
