/**
 * @file header.c Zip header functions
 * 
 * $Id: header.c,v 1.9 2003/01/01 06:22:34 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/internal.h>
#include <string.h>
#include "zip.h"


ZipStatus
cxZipReadLocalHeader(ZipLocalHeader *header, CxFP *fp)
{
	char buffer[ZIP_LOCAL_HEADER_SIZE];
	int counter = 0;
	int signature;
	
	/* Read in the signature that is found at the file's current position */
	if (cxRead(buffer, ZIP_LOCAL_HEADER_SIZE, 1, fp) == 0)
	       return ZIP_ERROR;
	
	signature = cxZipGet32(buffer, &counter);
	if (signature != ZIP_LOCAL_SIG)
		return ZIP_ERROR; /* Must be at central directory or elsewhere */

	header->version          = cxZipGet16(buffer, &counter);
	header->flag		 = cxZipGet16(buffer, &counter);
	header->compression      = cxZipGet16(buffer, &counter);
	header->mtime		 = cxZipGet32(buffer, &counter);
	header->crc32		 = cxZipGet32(buffer, &counter);
	header->compressedSize   = cxZipGet32(buffer, &counter);
	header->uncompressedSize = cxZipGet32(buffer, &counter);
	header->filenameLength   = cxZipGet16(buffer, &counter);
	header->extraLength      = cxZipGet16(buffer, &counter);

	if (header->filenameLength == 0)
		return ZIP_ERROR;
	/* Grab the file name */
	
	MEM_CHECK(header->filename = (char *)malloc(header->filenameLength + 1));
	if (cxRead(header->filename, header->filenameLength, 1, fp) == 0)
		return ZIP_ERROR;
	header->filename[header->filenameLength] = '\0';
	
	/* Grab the extra field, if it's there */
	if (header->extraLength > 0)
	{	
		MEM_CHECK(header->extraField = (char *)malloc(header->extraLength + 1));
		if (cxRead(header->extraField, header->extraLength, 1, fp) == 0)
			return ZIP_ERROR;
	}
	else 
		header->extraField = NULL;

	/* Now, if bit 3 of the general flag is set, then there's a data descriptor
	 * with info about the crc, and (un)compressed file sizes.  Thus, the values 
	 * read above are all 0.  However, the data descriptor is _after_ the 
	 * data itself.  So, we have to seek to the correct location: currentPos + 
	 * compressedFileSize.  That's all I have to say about that.
	 */

	/* first, store the location of the data, which should should
	 * be at the current position
	 */
	header->fileOffset = cxTell(fp);

	/* This should be done anyway -- to position to the next header */
	cxSeek(fp, header->compressedSize, SEEK_CUR);		
	if(ZIP_HAS_DATADESCRIPTOR(header->flag))
	{
		counter = 0;
		if (cxRead(buffer, ZIP_DATADESCRIPTOR_SIZE, 1, fp) == 0)
			return ZIP_ERROR;
		header->crc32	          = cxZipGet32(buffer, &counter);
		header->compressedSize    = cxZipGet32(buffer, &counter);
		header->uncompressedSize  = cxZipGet32(buffer, &counter);
	}

	return ZIP_SUCCESS;
}



	
