/**
 * @file libcomprex/io.c Input/output functions
 * 
 * $Id: io.c,v 1.18 2003/01/26 19:54:52 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/internal.h>

CxFile *
cxMakeFile(const char *filename)
{
	CxFile *file;
	char   *temp;
	struct stat sb;

	if (filename == NULL)
		return NULL;

	if (stat(filename, &sb) != 0)
		return NULL;

	file = cxNewFile();

	temp = cxGetBaseName(filename);
	cxSetFileName(file, temp);
	free(temp);

	cxSetFilePhysicalPath(file, filename);
	cxSetFileSize(file, sb.st_size);
	cxSetFileMode(file, sb.st_mode);
	cxSetFileUid(file,  sb.st_uid);
	cxSetFileGid(file,  sb.st_gid);
	cxSetFileDate(file, sb.st_mtime);

	cxSetFileLocal(file, 1);

	return file;
}

CxFP *
cxOpenFile(const char *path, CxAccessMode mode)
{
	CxFP *fp;
	
	if (path == NULL)
		return NULL;

	cxOpenArchiveOrFile(path, mode, &fp, NULL);

	if (fp != NULL)
		cxSetFpAccessMode(fp, mode);

	return fp;
}

CxFP *
cxOpenFileHandle(CxFile *file, CxAccessMode mode)
{
	CxFP *fp;

	if (file == NULL)
		return NULL;

	fp = cxGetFileArchive(file)->module->ops.archive->openFile(file, mode);

	if (fp == NULL)
		return NULL;

	cxSetFpAccessMode(fp, mode);

	fp->file = file;
	fp->archive = cxGetFileArchive(file);
	
	CX_LINK(file);

	return fp;
}

CxFP *
cxOpenStream(FILE *stream)
{
	CxFP *fp;

	if (stream == NULL)
		return NULL;

	cxInternalOpenStream(stream, CX_MODE_READ_ONLY, &fp, NULL);

	if (fp != NULL)
		cxSetFpAccessMode(fp, CX_MODE_READ_ONLY);

	return fp;
}

CxFP *
cxOpenBuffer(char *buffer, size_t size)
{
	CxFP *fp;

	if (buffer == NULL || size == 0)
		return NULL;

	cxInternalOpenBuffer(buffer, size, CX_MODE_READ_ONLY, &fp, NULL);

	if (fp != NULL)
		cxSetFpAccessMode(fp, CX_MODE_READ_ONLY);

	return fp;
}

void
cxClose(CxFP *fp)
{
	CxFile *file;
	CxArchive *archive;
	
	if (fp == NULL || fp->refCount == 0)
		return;

	CX_UNLINK(fp);

	if (fp->refCount > 0)
		return;

	file = fp->file;
	archive = fp->archive;

	if (fp->ops.close != NULL)
		fp->ops.close(fp);

	if (fp->moduleData != NULL)
	{
		free(fp->moduleData);
		fp->moduleData = NULL;
	}

	free(fp);

	if (file != NULL)
	{
		cxDestroyFile(file);
	}
}

size_t
cxRead(void *ptr, size_t size, size_t nmemb, CxFP *fp)
{
	size_t result;

	if (fp == NULL || ptr == NULL || (size * nmemb) == 0 ||
		fp->ops.read == NULL)
	{
		return 0;
	}

	result = fp->ops.read(ptr, size, nmemb, fp);

	fp->pos += result * size;

#if 0
	if (cxTell(fp) == cxGetFileSize(fp->file))
	{
		fp->eof = 1;
	}
#endif

	return result;
}

size_t
cxWrite(const void *ptr, size_t size, size_t nmemb, CxFP *fp)
{
	size_t result;

	if (fp == NULL || ptr == NULL || fp->ops.write == NULL ||
		(size * nmemb) == 0)
	{
		return 0;
	}

	result = fp->ops.write(ptr, size, nmemb, fp);

	fp->pos += result * size;

	return result;
}

char *
cxGets(char *buffer, size_t size, CxFP *fp)
{
	size_t result;
	char *c;

	if (buffer == NULL || fp == NULL || size < 2)
		return NULL;

	result = cxRead(buffer, 1, size - 1, fp);

	if (result == 0)
		return NULL;

	buffer[size - 1] = '\0';

	/* Find either the newline or the NUL */
	if ((c = strchr(buffer, '\n')) != NULL)
	{
		size_t str_len = c - buffer;

		*(c + 1) = '\0';

		if (str_len < result)
		{
			/* Skip back to right after the newline. */
			cxSeek(fp, str_len - result + 1, SEEK_CUR);
		}
	}

	return buffer;
}

void
cxSeek(CxFP *fp, long offset, int whence)
{
	if (fp == NULL || fp->ops.seek == NULL ||
		(whence != SEEK_CUR && whence != SEEK_SET && whence != SEEK_END))
	{
		return;
	}

	fp->ops.seek(fp, offset, whence);

	switch (whence)
	{
		case SEEK_SET: fp->pos =  offset;                           break;
		case SEEK_CUR: fp->pos += offset;                           break;
		case SEEK_END: fp->pos =  cxGetFileSize(fp->file) - offset; break;
	}

	if (cxTell(fp) == cxGetFileSize(fp->file))
		fp->eof = 1;
	else
		fp->eof = 0;
}

void
cxRewind(CxFP *fp)
{
	if (fp == NULL)
		return;

	cxSeek(fp, 0, SEEK_SET);
}

long
cxTell(CxFP *fp)
{
	if (fp == NULL)
		return -1;

	return fp->pos;
}

void
cxClearErr(CxFP *fp)
{
	if (fp == NULL)
		return;

	if (fp->errStr != NULL)
		free(fp->errStr);

	fp->errStr = NULL;
	fp->error  = 0;
}

int
cxEof(CxFP *fp)
{
	if (fp == NULL)
		return 1;

	return (cxTell(fp) >= cxGetFileSize(fp->file));
}

int
cxError(CxFP *fp)
{
	if (fp == NULL)
		return 0;

	return fp->error;
}

const char *
cxStrError(CxFP *fp, int errnum)
{
	if (fp == NULL)
		return NULL;

	return fp->errStr;
}

CxStatus
cxExtractFile(CxFile *file, const char *destPath)
{
	CxArchive *archive;

	archive = cxGetFileArchive(file);

	if (archive == NULL)
		return CX_ERROR;

	/* Let them know we're about to extract this file */
	if (archive->ex_callback != NULL)
		archive->ex_callback(archive, file, 0, 0);

	return cxInternalExtractFile(file, destPath);
}

