/***************************************************************************
	sprites.c - sprite routines

    begin                : 29 Mar 2003
    copyright            : (C) 2003 by Paul Rahme
****************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
****************************************************************************/

#include <string.h>
#include "SDL/SDL_image.h"

#include "main.h"
#include "tuning.h"
#include "sprites.h"
#include "spritesData.h"
#include "helpers.h"
#include "timer.h"

// -------------------------------------------------------------------------------------------------
// Functions
// -------------------------------------------------------------------------------------------------

// Resets all the sprite structures
void SPR_Init()
{
	Uint32			sprID;
	SPR_Sprite		*sprite;
	spriteStruct	*sprData;

	#ifdef DEBUG_MODE1_SPR
	printf("SPR_Init\n");
	#endif

	for (sprID=0; sprID<noOfSprites; sprID++)
	{
		sprite = &(sprites[sprID]);
		sprData = (spriteStruct*)(&(spriteData[sprID]));

		// Set the sprite's constant data which does not depend on the surface
		sprite->name = sprID;
		sprite->xOffset = sprData->xOffset;
		sprite->yOffset = sprData->yOffset;
		sprite->frameRate = sprData->frameRate;
		sprite->noOfDirs = sprData->noOfDirs;

		// Set the surface to NULL values, it will be created when this sprite is used for the first time
		sprite->surface = NULL;
		sprite->width = 0;
		sprite->height = 0;
		sprite->totalFrames = 0;
	}
}

// Loads a bitmap from the speciified filename and converts it to SDL-optimized format.
SDL_Surface* SPR_LoadConvert(const char *_fileName, Uint8 _alpha)
{
	SDL_Surface *tempSurface, *finalSurface;
	Uint32 colorKey;

	#ifdef DEBUG_MODE1_SPR
	printf("SPR_LoadConvert : loading %s\n", _fileName);
	#endif

	// Load the original image into the temp surface, and get colorkey value
	tempSurface = IMG_Load(_fileName);
	colorKey = tempSurface->format->colorkey;

	// Convert it into SDL-optimized format on the final surface, and free up the temp one
	finalSurface = SDL_DisplayFormat(tempSurface);
	SDL_FreeSurface(tempSurface);

	// Set up the correct alpha value on the final surface
	SDL_SetColorKey(finalSurface, SDL_SRCCOLORKEY, colorKey);

	// Prepare the per-surface alpha value on the final surface
	if (SDL_SetAlpha(finalSurface, SDL_SRCALPHA, _alpha) == -1)
	{
		Lockup("Failed to set alpha.");
	}

	return finalSurface;
}

// Loads a sprite's surface from disk into the sprite structure & corresponding SDL_Surface
void SPR_LoadSprite(SPR_Sprite *_sprite)
{
	char currSpriteFileName[256];
	spriteStruct	*sprData;

	#ifdef DEBUG_MODE1_SPR
	printf("SPR_LoadSprite called for sprite no. %i\n", _sprite->name);
	#endif

	#ifdef DEBUG_CHECKS
	if (_sprite->surface != NULL)
	{
		Lockup("SPR_LoadSprite: surface is not NULL, sprite may already have been loaded.");
	}
	#endif

	sprData = (spriteStruct*)(&(spriteData[_sprite->name]));

	// Load the file
	strcpy(currSpriteFileName, BASE_DIR);		// copy it into a long string so that the filename can be appended
	strcat(currSpriteFileName, sprData->fileName);

	// Allocate, load and set up the sprite's surface values
	_sprite->surface = SPR_LoadConvert(currSpriteFileName, 255); // TODO: modify this if sprite transparency is required
	_sprite->width = _sprite->surface->w / (sprData->noOfDirs);
	_sprite->height = _sprite->surface->h / (sprData->noOfFrames);
	_sprite->totalFrames = (int)(_sprite->surface->h / _sprite->height);
}

// Attempts to reinvent what SDL_BlitSurface does and has wild dreams of actually being faster
// Note : _x and _y are treated as absolute screen coordinates
void SPR_BlitSprite(SPR_Sprite *_sprite, int _x, int _y, Uint32 _dir, Uint32 _frame)
{
	int srcCount, destCount, srcBase, destBase, yOffset, xOffset;
	int srcSkipPixels, destSkipPixels; // how many pixels to skip at the end of each row
	int	lineCount; // counter for how many pixels it's drawn in ths current row
	int clippedWidth, clippedHeight; // clipped to never draw off the surface
	Uint32 *srcPos;
	Uint32 *destPos;
	Uint32 pixel;
	SDL_Surface *src;
	SDL_Surface *dest;
	int	dontDraw;

	#ifdef DEBUG_POINTS_SPR
	Uint32 midpoint; // used for drawing a dot at the orginal position after the sprite's been drawn
	#endif

	#ifdef DEBUG_MODE3_SPR
	printf("SPR_BlitSprite() called for x=%i, y=%i\n", _x, _y);
	#endif

	// If it's the empty sprite, don't bother drawing it
	if (_sprite->name == SPR_None)
	{
		return;
	}

	#ifdef DEBUG_CHECKS
	if (_sprite->surface == NULL)
	{
		Lockup("SPR_BlitSprite: Sprite has not been loaded.");
	}
	#endif

	src = _sprite->surface;
	dest = vScreen;

	// Initialise some stuff
	dontDraw = false;

	// Lock surfaces if necessary
	if (SDL_MUSTLOCK(src))
	{
		SDL_LockSurface(src);
	}
	if (SDL_MUSTLOCK(dest))
	{
		SDL_LockSurface(dest);
	}

	#ifdef DEBUG_POINTS_SPR
	// Save the object's position for after it's been drawn
	if ((_x < SCREEN_MAX_X) && (_y < SCREEN_MAX_Y))
	{
		midpoint = HLP_GetPos(dest, _x, _y);
	} else {
		printf("sprite offscreen\n");
		midpoint = -1;
	}
	#endif

	// Add/subtract the sprite's offsets to get its top left corner
	_x += _sprite->xOffset;
	_y += _sprite->yOffset;

	// If it's off the screen, don't draw
	if ((_x > SCREEN_MAX_X) || (_y > SCREEN_MAX_Y)
	 || (_x + _sprite->width < 0) || (_y + _sprite->height < 0))
	{
		dontDraw = true;
	}

	// Calculate clipped part of sprite to be drawn
	clippedWidth = _sprite->width;
	if (_x < 0)
	{
		clippedWidth += _x;					// subtract the no of columns to the left of the screen...
		srcCount = HLP_BPP(-_x);			// ...and start further into the sprite to compensate
		_x = 0;
	} else {
		srcCount = 0; // start from the very left side of the sprite
	}
	if (_x + clippedWidth > SCREEN_MAX_X)
	{
		clippedWidth = SCREEN_MAX_X - _x;
	}
	clippedHeight = _sprite->height;
	if (_y < 0)
	{
		clippedHeight += _y; 				// subtract the no of rows above the top of the screen...
		srcCount -= HLP_BPP(src->w*_y);		// ...and start further down the sprite to compensate
		_y = 0;
	}
	destCount = 0; // start from the top of the sprite's screen position
	if (_y + clippedHeight > SCREEN_MAX_Y)
	{
		clippedHeight = SCREEN_MAX_Y - _y;
	}

	#ifdef DEBUG_MODE3_SPR
	printf("Offsetted x, y = %i, %i; clippedWidth = %i, clippedHeight = %i\n", _x, _y, clippedWidth, clippedHeight);
	#endif

	// If it's clipped to 0 on either width or height, don't draw it
	if ((clippedWidth == 0) || (clippedHeight == 0))
	{
		dontDraw = true;
	}

	// Calculate how many bytes to skip at the end of each row
	srcSkipPixels = HLP_BPP(src->w - clippedWidth);
	destSkipPixels = HLP_BPP(SCREEN_WIDTH - clippedWidth);

	// set initial counter values
	lineCount = clippedWidth;

	// Calculate starting positions
	yOffset = _frame * _sprite->height;				// frames go down the sprite
	xOffset = _dir * _sprite->width;				// directions go across the sprite
	srcBase = HLP_GetPos(src, xOffset, yOffset);
	destBase = HLP_GetPos(dest, _x, _y);

	#ifdef DEBUG_POINTS_SPR
	// Draw a dot at the top left corner
	if ((_x < SCREEN_MAX_X) && (_y < SCREEN_MAX_Y))
	{
		*(Uint32*)(destBase) = 0xFFFF00FF;
	}
	#endif

	// Only blit the sprite if nothing's set dontDraw to true up to now
	if (dontDraw == false)
	{
		// Actual blit loop
		while (destCount < (HLP_BPP(SCREEN_WIDTH*clippedHeight))) // clippedHeight * bytes per row
		{
			srcPos = (Uint32*)(srcBase + srcCount);
			destPos = (Uint32*)(destBase + destCount);

			pixel = *(Uint32*)(srcPos);
			if (pixel != 0) // TODO: change this to colorkey - might not always be 0
			{
//				pixel |= 0xFF000000;			// alpha
				*(Uint32*)(destPos) = pixel;
			}

			srcCount += BPP;
			destCount += BPP;
			lineCount--;
			if (lineCount == 0)
			{
				srcCount += srcSkipPixels;
				destCount += destSkipPixels;
				lineCount = clippedWidth;
			}
		}

		#ifdef DEBUG_POINTS_SPR
		// Draw a dot at the object's original position, over the top of the drawn sprite
		if (midpoint != -1)
		{
			*(Uint32*)(midpoint) = 0xFFFFFF00;
			*(Uint32*)(midpoint + 2*BPP) = 0xFFFFFF00;
			*(Uint32*)(midpoint - 2*BPP) = 0xFFFFFF00;
			*(Uint32*)(midpoint + 2*BPP*SCREEN_WIDTH) = 0xFFFFFF00;
			*(Uint32*)(midpoint - 2*BPP*SCREEN_WIDTH) = 0xFFFFFF00;
		}
		#endif
	}

	// Unlock surfaces if necessary
	if (SDL_MUSTLOCK(src))
	{
		SDL_UnlockSurface(src);
	}
	if (SDL_MUSTLOCK(dest))
	{
		SDL_UnlockSurface(dest);
	}

	SDL_UpdateRect(dest, 0, 0, dest->w, dest->h);
}
