/***************************************************************************
	main.c  -  main program core

	begin                : 22 Feb 2003
    copyright            : (C) 2003 by Paul Rahme
****************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
****************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include "errno.h"

#include "main.h"
#include "backgrounds.h"
#include "timer.h"
#include "input.h"
#include "tuning.h"
#include "objects.h"
#include "sprites.h"
#include "helpers.h"
#include "audio.h"

// --------------------------------------------------------------------------
// Constants
// --------------------------------------------------------------------------

// --------------------------------------------------------------------------
// Global Variables
// --------------------------------------------------------------------------

// Config
char	*saveFileName;					// filename of save game

// Gameplay variables
bool	MN_Quit;						// True means the game is over
int		MN_CurrentLevel;				// Level no currently in play
int		MN_HighestLevel;				// The highest level no that the player has reached
int		MN_Score;						// score for current level

// Graphics variables
SDL_Surface	*screen;					// The actual screen
SDL_Surface	*vScreen;					// Virtual screen which gets copied over the main screen once a frame

// --------------------------------------------------------------------------
// Functions
// --------------------------------------------------------------------------

// Saves the game
// if newGame is TRUE, it creates a clean savegame, else it saves the current game.
static void MN_SaveGame(bool newGame)
{
	FILE *saveFile;

	saveFile = fopen(saveFileName, "w");
	if (saveFile == NULL)
	{
		const char err[256];
		perror(err);
		Lockup("MN_LoadGame : Could not create a new save file.");
	}
	if (newGame)
	{
		fprintf(saveFile, "1 ");
	} else {
		fprintf(saveFile, "%i ", MN_HighestLevel);
	}
	fclose(saveFile);
}

// Loads the save game, if not found then creates a new one
static void MN_LoadGame()
{
	FILE *saveFile;
	char tempStr[5];

	saveFile = fopen(saveFileName, "r");

	#ifdef DEBUG_MODE1_FILEIO
	if (saveFile == NULL)
	{
		printf("File does not exist, attempting to create a new one.\n");
	} else {
		printf("Previous save file found and opened, reading save game values.\n");
	}
	#endif

	if (saveFile == NULL)
	{
		MN_SaveGame(true);

		// Open it as read-only
		saveFile = fopen(saveFileName, "r");
		if (saveFile == NULL)
		{
			const char err[256];
			perror(err);
			Lockup("MN_LoadGame : Could not reopen newly created save file.");
		#ifdef DEBUG_MODE1_FILEIO
		} else {
			printf("Reopened file for reading.\n");
		#endif
		}
	}

	// Read the values from the save file into the score structure
	fscanf(saveFile, "%s ", tempStr);
	MN_HighestLevel = atoi(tempStr);
	MN_CurrentLevel = MN_HighestLevel;

	fclose(saveFile);
}

// Initialise the global variables
void MN_InitVars()
{
	MN_Quit = false;
	IN_Event.type = -1;

	saveFileName = strcat(getenv("HOME"), SAVE_FILENAME);

	#ifdef DEBUG_MODE1_FILEIO
	printf("Save filename = %s\n", saveFileName);
	#endif

	MN_LoadGame();
}

// Initialise the game
void MN_Init()
{
	#ifdef DEBUG_MODE1
	printf("MN_Init\n");
	#endif

	// Create the main screen

	screen = SDL_SetVideoMode(SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_DEPTH, SDL_HWSURFACE|SDL_DOUBLEBUF
	#ifdef FULLSCREEN_FLAG
	|SDL_FULLSCREEN
	#endif
	);

	if (screen == NULL)
	{
		Lockup("Could not change to video mode.");
	}

	// Open audio
	AUD_Init();

	// Create the virtual screen, same size & depth as the actual screen
	#if SDL_BYTEORDER == SDL_BIGENDIAN
	vScreen = SDL_CreateRGBSurface(SDL_HWSURFACE, SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_DEPTH,
									0xff000000, 0x00ff0000, 0x0000ff00, 0x000000ff);
	#else
	vScreen = SDL_CreateRGBSurface(SDL_HWSURFACE, SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_DEPTH,
									0x000000ff, 0x0000ff00, 0x00ff0000, 0xff000000);
	#endif

}

// Initialises versatile things that may be reset at any time, eg. when player dies
void MN_Reset()
{
	#ifdef DEBUG_MODE1
	printf("\n\n---------------\nResetting level\n---------------\n\n");
	#endif

	TM_Init();
	BG_Reset();
	OBJ_Init();
	IN_Reset();
	MN_SaveGame(false);

	MN_Score = 0;
	switch (MN_CurrentLevel)
	{
		default:
		{
			AUD_StartSong(SONG_DUMMY, true);
/*
			#ifdef DEBUG_CHECKS
			Lockup("MN_Reset: Unahndled/invalid level for song");
			#endif
*/
			break;
		}
	}
}

// Call this once when MN_CurrentLevel contains the correct level, to load that level
void MN_InitLevel()
{
	// Init functions that need to be called each time MN_CurrentLevel changes
	BG_FreeSurfaces();
	BG_Init();

	// Reset functions that can be recalled at any time (eg after player dies)
	MN_Reset();
}

// Instantly jumps to the specified level
void MN_JumpToLevel(int _level)
{
	MN_CurrentLevel = _level;
	MN_InitLevel();
}

// Main function
int main(int argc, char *argv[])
{
	if (SDL_Init(SDL_INIT_VIDEO|SDL_INIT_AUDIO|SDL_INIT_JOYSTICK) == -1)
	{
		Lockup("SDL did not initialise.");
	}

	// Init functions that need to only happen once on startup
	MN_InitVars();
	MN_Init();
	SPR_Init();
	HLP_Init();
	TM_Init();
	IN_Init();

	MN_InitLevel();

	while (!MN_Quit)
	{
		// Update the game timer and frame rate counter
  		TM_UpdateTimer();

		// Process input
		IN_ProcessInput();

		if (!gamePaused)
		{
			// Update & draw all the layers, and draw all the objects
			BG_UpdateLayers();

			// Update & draw all the objects
			OBJ_UpdateObjects();
		}

		// Equivalent to calling SDL_BlitSurface then SDL_UpdateRect
		BG_BlitScreen();

		// Update the audio
		AUD_Update();
	}

	MN_SaveGame(false);

	AUD_Kill();

	SDL_FreeSurface(screen);
	SDL_FreeSurface(vScreen);

	if (joystick)
	{
		SDL_JoystickClose(joystick);
	}

	SDL_Quit();
	exit(0);

}
