/*-
 * Copyright (c) 2005 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *   @(#)$Id: ipastat_rules.h,v 1.3.2.1 2011/11/15 18:12:29 simon Exp $
 */

#ifndef IPASTAT_RULES_H
#define IPASTAT_RULES_H

#include "ipastat_limits.h"
#include "ipastat_thresholds.h"

#ifndef RULEPAT_NSIZE
# define RULEPAT_NSIZE	10
#endif
#ifndef RULEPAT_NALLOC
# define RULEPAT_NALLOC	10
#endif

/*
 * rulepat{} section.
 */
struct rulepat {
	STAILQ_ENTRY(rulepat) link;	/* For list building. */

	char		*pat;		/* Regular expression string. */
	regex_t		re;		/* Compiled regular expression. */

	unsigned int	no;		/* Ordinal number of this section. */

	signed char	check_next;	/* { check_next_rulepat } */

	const struct st_list *st_list;	/* { st_list } */

#ifdef WITH_LIMITS
	struct limits_list limits;	/* { limit } */
#endif

#ifdef WITH_THRESHOLDS
	struct thresholds_list thresholds; /* rulepat { threshold } */
#endif
};

/*
 * List of all rulepat{} sections.
 */
STAILQ_HEAD(rulepats_list, rulepat);

extern struct rulepats_list rulepats_list;

extern ipa_mzone *rulepat_mzone;

#ifndef RULE_NSIZE
# define RULE_NSIZE	30
#endif
#ifndef RULE_NALLOC
# define RULE_NALLOC	20
#endif

#define RULE_FREE_NAME	0x01		/* Free 'name' in struct rule{}. */

/*
 * rule{} section.
 */
struct rule {
	STAILQ_ENTRY(rule) list;	/* All rules list. */
	SLIST_ENTRY(rule) hlink;	/* All rules list in one hash bucket. */

	char		*name;		/* Name of this rule. */
	char		*info;		/* Info for this rule. */
	unsigned int	no;		/* Rule ordinal number. */

	const struct st_list *st_list;	/* { st_list } */

#ifdef WITH_LIMITS
	struct limits_list limits;	/* All { limit } */
#endif

#ifdef WITH_THRESHOLDS
	struct thresholds_list thresholds; /* All { thresholds } */
#endif

	unsigned int	free_mask;	/* RULE_FREE_xxx bits. */

	int		inited;		/* Set if rule was initialized. */

	unsigned int	name_hash;	/* Hashed value of rule's name. */
};

/*
 * List of all rules.
 */
STAILQ_HEAD(rules_list, rule);

enum {
#ifdef WITH_LIMITS
	OPT_RULE_LIMIT,			/* -r ... -l ... */
#endif
#ifdef WITH_THRESHOLDS
	OPT_RULE_THRESHOLD,		/* -r ... -t ... */
#endif
	OPT_RULE_RULE			/* -r ... */
};

/*
 * Rule in query: -q -r name
 */
struct opt_rule {
	STAILQ_ENTRY(opt_rule) link;	/* For list building. */

	const char	*name;		/* -r name */
	struct rule	*rule;		/* Corresponding rule structure. */

	const struct opt_st *opt_st;	/* Corresponding -s ... */

	unsigned int	type;		/* OPT_RULE_xxx. */

	void		*data;		/* Opaque data. */

#ifdef WITH_LIMITS
	struct opt_limits opt_limits;	/* Its limits. */
#endif
#ifdef WITH_THRESHOLDS
	struct opt_thresholds opt_thresholds; /* Its thresholds. */
#endif
};

/*
 * All rules in query: -q -r name1 -r name2 ...
 */
STAILQ_HEAD(opt_rules_list, opt_rule);

extern signed char dynamic_rules;

extern struct opt_rule *cur_opt_rule;

extern struct rules_list rules_list;
extern ipa_mzone *rule_mzone;

extern unsigned int nrules;

extern struct opt_rules_list opt_rules_list;

extern struct rule *alloc_rule(char *);

extern void	rules_hash_init(void);
extern void	rules_hash_add(struct rule *rule);

extern int	init_rules(void);
extern int	deinit_rules(void);
extern void	free_rules(void);
extern int	rules_inherit(void);

extern void	rulepats_inherit(void);
extern void	free_rulepats(void);

extern int	opt_rule_add(const char *);
extern int	opt_rules_parse(void);
extern void	opt_rules_free(void);

extern int	mod_rule_inherit(const struct rulepat *, const struct rule *);

extern struct rule *rule_by_name(const char *);

#endif /* IPASTAT_RULES_H */
