/*-
 * Copyright (c) 1999-2004 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa_rules.c,v 1.3.2.1 2011/11/15 18:12:29 simon Exp $";
#endif /* !lint */

#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"
#include "pathnames.h"

#include "ipa_ac.h"
#include "ipa_db.h"
#include "ipa_ctl.h"
#include "ipa_cmd.h"
#include "ipa_time.h"

#include "ipa_conf.h"
#include "ipa_log.h"
#include "ipa_main.h"
#include "ipa_rules.h"
#include "ipa_autorules.h"

#ifndef RULES_HASH_BUCKETS
# define RULES_HASH_BUCKETS 64		/* Must be power of 2. */
#endif

unsigned int	nstatrules;		/* Number of static rules. */

signed char	keep_rules_order;	/* keep_rules_order parameter. */

ipa_mzone	*rule_mzone;		/* Mzone for all struct rule{}. */

#ifdef WITH_RULES
signed char	has_ac_gather;		/* Set if has "ac_gather_*". */
ipa_mzone	*acg_mzone;		/* Mzone for all struct acg{}. */
struct acg_list	acg_list;		/* All rules with "ac_gather_*". */
#endif

ipa_mzone	*rulepat_mzone;		/* Mzone for all struct rulepat{}. */

/* Time when to check inactive rules queue. */
unsigned int	rules_inactive_check_sec;

/* List of all rules. */
struct rules_list rules_list = TAILQ_HEAD_INITIALIZER(rules_list);

/* List of all rulepats. */
struct rulepats_list rulepats_list = STAILQ_HEAD_INITIALIZER(rulepats_list);

/* Active rules queue. */
struct rules_queue rules_active = TAILQ_HEAD_INITIALIZER(rules_active);

/* Inactive rules queue. */
struct rules_queue rules_inactive = TAILQ_HEAD_INITIALIZER(rules_inactive);

LIST_HEAD(rules_hash, rule);

static struct rules_hash rules_hash[RULES_HASH_BUCKETS];

#ifdef WITH_RULES
/*
 * Add chunk to all rules that are listed in rule's acgs list.
 */
static int
rules_add_chunk_acg(const struct rule *rule, const uint64_t *chunk_ptr)
{
	const struct acg *acg;
	struct rule *rule2;

	SLIST_FOREACH(acg, &rule->acgs, link) {
		rule2 = acg->rule;
		if (RULE_IS_INACTIVE(rule2))
			continue;
		if (acg->addition) {
			if (rule_add_chunk(rule2, chunk_ptr) < 0)
				goto failed;
		} else {
			if (rule_sub_chunk(rule2, chunk_ptr) < 0)
				goto failed;
		}
	}
	return (0);

failed:
	logbt("rule_add_chunk_acg");
	return (-1);
}
#endif

/*
 * Add chunk to the rule's counter and to counters of its limits and
 * thresholds.  Check if there is statistics in rule's negative
 * counter, check for counter overflowing.  Add chunk to all rules
 * that are listed in this rule's acgs list.
 */
int
rule_add_chunk(struct rule *rule, const uint64_t *chunk_ptr)
{
	uint64_t chunk;

	chunk = *chunk_ptr;

	if (rule->cnt_neg >= chunk)
		rule->cnt_neg -= chunk;
	else {
		chunk -= rule->cnt_neg;
		rule->cnt_neg = 0;
		if (rule->cnt <= UINT64_MAX - chunk)
			rule->cnt += chunk;
		else {
			if (db_update_rule(rule, &uint64_max) < 0)
				goto failed;
			rule->cnt = UINT64_MAX - rule->cnt;
			rule->cnt = chunk - rule->cnt;
			if (db_append_rule(rule, &rule->cnt, 0) < 0)
				goto failed;
		}
	}

#ifdef WITH_LIMITS
	if (!STAILQ_EMPTY(&rule->limits))
		if (limits_add_chunk(rule, chunk_ptr) < 0)
			goto failed;
#endif

#ifdef WITH_THRESHOLDS
	if (!STAILQ_EMPTY(&rule->thresholds))
		if (thresholds_add_chunk(rule, chunk_ptr) < 0)
			goto failed;
#endif

#ifdef WITH_RULES
	if (!SLIST_EMPTY(&rule->acgs))
		if (rules_add_chunk_acg(rule, chunk_ptr) < 0)
			goto failed;
#endif

	return (0);

failed:
	logbt("rule_add_chunk");
	return (-1);
}

#ifdef WITH_RULES
/*
 * Subtract chunk from all rules that are listed im rule's acgs list.
 */
static int
rules_sub_chunk_acg(const struct rule *rule, const uint64_t *chunk_ptr)
{
	const struct acg *acg;
	struct rule *rule2;

	SLIST_FOREACH(acg, &rule->acgs, link) {
		rule2 = acg->rule;
		if (RULE_IS_INACTIVE(rule2))
			continue;
		if (acg->addition) {
			if (rule_sub_chunk(rule2, chunk_ptr) < 0)
				goto failed;
		} else {
			if (rule_add_chunk(rule2, chunk_ptr) < 0)
				goto failed;
		}
	}
	return (0);

failed:
	logbt("rules_sub_chunk_acg");
	return (-1);
}
#endif

/*
 * Subtract chunk from the rule's counter and from counters of its
 * limits and thresholds.  Check if there is statistics in negative
 * rule's counter, check for counter overflowing.  Subtract chunk from
 * all rules that are listed in this rule's acgs list.
 */
int
rule_sub_chunk(struct rule *rule, const uint64_t *chunk_ptr)
{
	uint64_t chunk;

	chunk = *chunk_ptr;

	if (rule->cnt >= chunk)
		rule->cnt -= chunk;
	else {
		chunk -= rule->cnt;
		if (rule->cnt_neg > UINT64_MAX - chunk) {
			logmsgx(IPA_LOG_ERR, "rule %s: rule_sub_chunk: "
			    "negative statistics counter overflowed",
			    rule->name);
			logmsgx(IPA_LOG_ERR, "this means that something is "
			    "wrong in configuration");
			return (-1);
		}
		rule->cnt_neg += chunk;
		rule->cnt = 0;
	}

#ifdef WITH_LIMITS
	if (!STAILQ_EMPTY(&rule->limits))
		if (limits_sub_chunk(rule, chunk_ptr) < 0)
			goto failed;
#endif
#ifdef WITH_THRESHOLDS
	if (!STAILQ_EMPTY(&rule->thresholds))
		if (thresholds_sub_chunk(rule, chunk_ptr) < 0)
			goto failed;
#endif
#ifdef WITH_RULES
	if (!SLIST_EMPTY(&rule->acgs))
		if (rules_sub_chunk_acg(rule, chunk_ptr) < 0)
			goto failed;
#endif

	return (0);

#if defined(WITH_RULES) || defined(WITH_ANY_LIMITS)
failed:
	logbt("rule_sub_chunk");
	return (-1);
#endif
}

/*
 * Add a rule to rules_active queue.
 */
void
queue_active_rule(struct rule *rule1)
{
	struct rule *rule2;

#ifdef WITH_AUTORULES
	/* Dynamic rules are always linked to the head. */
	if (RULE_IS_DYNAMIC(rule1)) {
		TAILQ_INSERT_HEAD(&rules_active, rule1, queue);
		return;
	}
#endif

	if (keep_rules_order)
		/* Keep rules order or have ac_gather_*. */
		TAILQ_FOREACH(rule2, &rules_active, queue)
			if (rule1->orderno > rule2->orderno) {
				TAILQ_INSERT_BEFORE(rule2, rule1, queue);
				return;
			}

	/* Do not keep rules order or no dependency was found. */
	TAILQ_INSERT_TAIL(&rules_active, rule1, queue);
}

#ifdef WITH_RULES
/*
 * Add all static rules to active rules queue, this function is called
 * after configuration/reconfiguration.
 */
void
init_active_rules(void)
{
	const struct acg *acg;
	struct rule *rule1, *rule2;
	unsigned int orderno;

	TAILQ_FOREACH(rule1, &rules_list, list) {
		RULE_SET_QUEUED(rule1);
		if (!keep_rules_order && has_ac_gather) {
			TAILQ_FOREACH(rule2, &rules_active, queue) {
				/*
				 * The given rule must be below than rule,
				 * which has this rule in its acgs.
				 */
				SLIST_FOREACH(acg, &rule2->acgs, link)
					if (acg->rule == rule1)
						goto next_rule2;
				/*
				 * Rules listed in acgs of the given rule
				 * must below than this rule.
				 */
				SLIST_FOREACH(acg, &rule1->acgs, link)
					if (acg->rule == rule2) {
						TAILQ_INSERT_BEFORE(rule2,
						    rule1, queue);
						goto next_rule1;
					}
next_rule2:			;
			}
		}
		/*
		 * Keep rules order or no dependency was found,
		 * add it to the end of the active rules queue.
		 * Side effect: by default static rules are initially
		 * placed in rules_active queue in the original order.
		 */
		TAILQ_INSERT_TAIL(&rules_active, rule1, queue);
next_rule1:	;
	}

	/* Let's combine these two flags for queue_active_rule(). */
	keep_rules_order |= has_ac_gather;

	if (keep_rules_order) {
		orderno = 0;
		TAILQ_FOREACH_REVERSE(rule1, &rules_active, rules_queue, queue)
			rule1->orderno = orderno++;
	}
}
#endif /* WITH_RULES */

/*
 * Set rule active or inactive in modules it uses.
 */
int
mod_set_rule_active(struct rule *rule, int active)
{
	if ((rule->rule_flags & RULE_FLAG_ACTIVE) ==
	    (active ? RULE_FLAG_ACTIVE : 0)) {
		logmsgx(IPA_LOG_ERR, "internal error: mod_set_rule_active"
		    "(%s, %d): rule is already %s", rule->name, active,
		    active_msg[active]);
		return (-1);
	}

	if (debug_worktime)
		logdbg("rule %s: set rule %s", rule->name, active_msg[active]);

	if (active)
		RULE_SET_ACTIVE(rule);
	else
		RULE_SET_INACTIVE(rule);

	if (ac_set_rule_active(rule, active) < 0)
		goto failed;
	if (db_set_rule_active(rule, active) < 0)
		goto failed;

	return (0);

failed:
	logbt("mod_set_rule_active");
	return (-1);
}

static void
show_inactive_rules(void)
{
	const struct rule *rule;

	logdbg("inactive rules after sorting:");
	TAILQ_FOREACH(rule, &rules_inactive, queue)
		logdbg("    active %s, rule %s",
		    sec_str(rule->worktime->active_sec), rule->name);
}

/*
 * Move a rule from the active rules queue to the inactive rules queue.
 */
int
set_rule_inactive(struct rule *rule1)
{
	struct rule *rule2;

	/* Inform modules that rule becomes inactive. */
	if (mod_set_rule_active(rule1, 0) < 0) {
		logbt("set_rule_inactive");
		return (-1);
	}

	/* Remove the rule from the active rules queue. */
	TAILQ_REMOVE(&rules_active, rule1, queue);

	/* Add the rule to the inactive rules queue, keep that queue sorted. */
	if (rule1->worktime->active_sec != EVENT_NOT_SCHEDULED) {
		/* Will be active in current day. */
		TAILQ_FOREACH(rule2, &rules_inactive, queue)
			if (rule1->worktime->active_sec <
			    rule2->worktime->active_sec) {
				TAILQ_INSERT_BEFORE(rule2, rule1, queue);
				goto done;
			}
	}
	TAILQ_INSERT_TAIL(&rules_inactive, rule1, queue);
done:
	if (debug_worktime)
		show_inactive_rules();

	rules_inactive_check_sec =
	    TAILQ_FIRST(&rules_inactive)->worktime->active_sec;

	return (0);
}

/*
 * Move a rule from the inactive rules queue to the active rules queue.
 */
int
set_rule_active(struct rule *rule)
{
	/* Inform modules that rule becomes active. */
	if (mod_set_rule_active(rule, 1) < 0)
		goto failed;

	/* Get new stat for the rule. */
	rule->newstat = 1;

	/* Remove the rule from the inactive rules queue. */
	TAILQ_REMOVE(&rules_inactive, rule, queue);

	/* Add the rule to the active rules queue. */
	queue_active_rule(rule);

	/* Get new value of rules_inactive_check_sec. */
	rules_inactive_check_sec = TAILQ_EMPTY(&rules_inactive) ?
	    EVENT_NOT_SCHEDULED :
	    TAILQ_FIRST(&rules_inactive)->worktime->active_sec;

	/* Append new record for the rule. */
	if (db_append_rule(rule, &uint64_zero, 1) < 0)
		goto failed;

	rule->check_sec = cursec;

	return (0);

failed:
	logbt("set_rule_active");
	return (-1);
}

/*
 * Sort inactive rules queue;
 */
void
sort_inactive_rules(void)
{
	struct rule *r1, *r1_next, *r2;

	if (TAILQ_EMPTY(&rules_inactive))
		rules_inactive_check_sec = EVENT_NOT_SCHEDULED;
	else {
		r1 = TAILQ_FIRST(&rules_inactive);
		TAILQ_INIT(&rules_inactive);
		for (; r1 != NULL; r1 = r1_next) {
			r1_next = TAILQ_NEXT(r1, queue);
			if (r1->worktime->active_sec != EVENT_NOT_SCHEDULED) {
				/* Will be active in current day. */
				TAILQ_FOREACH(r2, &rules_inactive, queue)
					if (r1->worktime->active_sec <=
					    r2->worktime->active_sec) {
						TAILQ_INSERT_BEFORE(r2, r1,
						    queue);
						goto next;
					}
			}
			TAILQ_INSERT_TAIL(&rules_inactive, r1, queue);
next:			;
		}
		rules_inactive_check_sec =
		    TAILQ_FIRST(&rules_inactive)->worktime->active_sec;
	}

	if (debug_worktime)
		show_inactive_rules();
}

/*
 * Check if time to make some rule active come.  This function
 * is called when rules_inactive_check_sec <= cursec.
 */
int
check_inactive_rules(void)
{
	struct rule *rule, *rule_next;

	for (rule = TAILQ_FIRST(&rules_inactive); rule != NULL;
	    rule = rule_next)
		if (rule->worktime->active_sec <= cursec) {
			/* It's time to make a rule active. */
			rule_next = TAILQ_NEXT(rule, queue);
			if (set_rule_active(rule) < 0) {
				logbt("check_inactive_rules");
				return (-1);
			}
		} else {
			/* Set next time of this function invocation. */
			rules_inactive_check_sec = rule->worktime->active_sec;
			return (0);
		}

	/* No more inactive rules. */
	rules_inactive_check_sec = EVENT_NOT_SCHEDULED;
	return (0);
}

/*
 * Log messages about lost negative statistics.
 */
static void
check_cnt_neg(const struct rule *rule, int only_for_limits)
{
#ifdef WITH_LIMITS
	const struct limit *limit;
#endif
#ifdef WITH_THRESHOLDS
	const struct threshold *threshold;
#endif

	if (!only_for_limits && rule->cnt_neg > 0)
		logmsgx(IPA_LOG_WARNING, "rule %s: counter is -%"PRIu64", "
		    "this statistics is lost", rule->name, rule->cnt_neg);

#ifdef WITH_LIMITS
	STAILQ_FOREACH(limit, &rule->limits, link)
		if (limit->cnt_neg > 0)
			logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: "
			    "counter is -%"PRIu64", this statistics is lost",
			    rule->name, limit->name, limit->cnt_neg);
#endif

#ifdef WITH_THRESHOLDS
	STAILQ_FOREACH(threshold, &rule->thresholds, link)
		if (threshold->cnt_neg > 0)
			logmsgx(IPA_LOG_WARNING, "rule %s, threshold %s: "
			    "counter is -%"PRIu64", this statistics is lost",
			    rule->name, threshold->name, threshold->cnt_neg);
#endif
}

#ifdef WITH_RULES
/*
 * Initialize one static rule.
 */
static int
init_stat_rule(struct rule *rule)
{
	rule->newstat = 1;
	if (rule->append_tevent == NULL)
		rule->append_sec = EVENT_NOT_SCHEDULED;
	if (init_acg(rule) < 0)
		goto failed;
	if (ac_init_statrule(rule) < 0)
		goto failed;
	if (db_init_statrule(rule) < 0)
		goto failed;
	return (0);

failed:
	logbt("init_stat_rule");
	return (-1);
}
#endif /* WITH_RULES */

#ifdef WITH_RULES
/*
 * Build reverse list for "ac_gather_*" parameters: if rule1 matches rule2's
 * "ac_gather_*" parameters, then rule2 is added to rule1's acg list.
 */
int
init_acg(struct rule *rule1)
{
	struct rule *rule2;
	struct acg *acg;

	SLIST_FOREACH(rule2, &acg_list, acg_link) {
		if (rule1 == rule2)
			continue;
		if (rule2->acg_add_pat != NULL)
			if (regexec_simple(&rule2->acg_add_re, rule1->name) ==
			    0) {
				acg = mzone_alloc(acg_mzone);
				if (acg == NULL)
					goto failed;
				SLIST_INSERT_HEAD(&rule1->acgs, acg, link);
				acg->rule = rule2;
				acg->addition = 1;
			}
		if (rule2->acg_sub_pat != NULL)
			if (regexec_simple(&rule2->acg_sub_re, rule1->name) ==
			    0) {
				acg = mzone_alloc(acg_mzone);
				if (acg == NULL)
					goto failed;
				SLIST_INSERT_HEAD(&rule1->acgs, acg, link);
				acg->rule = rule2;
				acg->addition = 0;
			}
	}
	return (0);

failed:
	logmsgx(IPA_LOG_ERR, "rule %s: init_acg: mzone_alloc failed",
	    rule1->name);
	return (-1);
}

/*
 * Release memory used by acg list for one rule.
 */
static void
deinit_acg(struct rule *rule)
{
	struct acg *acg, *acg_next;

	SLIST_FOREACH_SAFE(acg, &rule->acgs, link, acg_next)
		mzone_free(acg_mzone, acg);
}
#endif /* WITH_RULES */

/*
 * Initialize all rules and their limits and thresholds.  This function is
 * called for initial initialization of static rules (first_call == 0),
 * or when there are problems with time.
 */
int
init_rules(int first_call)
{
	struct rule *rule;

	TAILQ_FOREACH(rule, &rules_list, list) {
#ifdef WITH_RULES
		if (first_call) {
			if (init_stat_rule(rule) < 0)
				goto failed;
		} else
#endif
			check_cnt_neg(rule, 1);
#ifdef WITH_LIMITS
		if (init_limits(rule) < 0)
			goto failed;
#endif
#ifdef WITH_THRESHOLDS
		if (init_thresholds(rule) < 0)
			goto failed;
#endif
	}

	return (0);

#if defined(WITH_RULES) || defined(WITH_ANY_LIMITS)
failed:
	logbt("init_rules");
	return (-1);
#endif
}

/*
 * Unlink rule from rules lists and release memory used by one rule.
 */
void
free_rule(struct rule *rule)
{
	int dyn_flag;

	dyn_flag = RULE_IS_DYNAMIC(rule);
	mem_free(rule->name, m_anon);
	mem_free(rule->info, dyn_flag ? m_anon : m_parser);

	cmds_rule_free(&rule->rc[RC_STARTUP]);
	cmds_rule_free(&rule->rc[RC_SHUTDOWN]);

#ifdef WITH_LIMITS
	free_limits(rule->rule_flags, &rule->limits, dyn_flag);
#endif
#ifdef WITH_THRESHOLDS
	free_thresholds(rule->rule_flags, &rule->thresholds, dyn_flag);
#endif

#ifdef WITH_RULES
	if (rule->acg_add_pat != NULL) {
		mem_free(rule->acg_add_pat, m_parser);
		regfree(&rule->acg_add_re);
	}
	if (rule->acg_sub_pat != NULL) {
		mem_free(rule->acg_sub_pat, m_parser);
		regfree(&rule->acg_sub_re);
	}
	deinit_acg(rule);
#endif

	/* Remove rule from rules list. */
	TAILQ_REMOVE(&rules_list, rule, list);

	/* Remove rule from rules hash. */
	rules_hash_rem(rule);

	mzone_free(rule_mzone, rule);
}

/*
 * Call free_rule() for each rule.
 */
void
free_rules(void)
{
	struct rule *rule;

	while ((rule = TAILQ_FIRST(&rules_list)) != NULL)
		free_rule(rule);
}

/*
 * Deinitialize one rule.
 */
int
deinit_rule(struct rule *rule)
{
#ifdef WITH_LIMITS
	struct limit *limit;
#endif
#ifdef WITH_SUBLIMITS
	struct sublimit *sublimit;
#endif
#ifdef WITH_THRESHOLDS
	struct threshold *threshold;
#endif
	int rv;

	check_cnt_neg(rule, 0);

	if (RULE_IS_QUEUED(rule)) {
		/* Remove rule from active or inactive queue. */
		if (RULE_IS_ACTIVE(rule))
			TAILQ_REMOVE(&rules_active, rule, queue);
		else {
			TAILQ_REMOVE(&rules_inactive, rule, queue);
			rules_inactive_check_sec =
			    TAILQ_EMPTY(&rules_inactive) ?
			    EVENT_NOT_SCHEDULED :
			    TAILQ_FIRST(&rules_inactive)->worktime->active_sec;
		}
	}

	rv = 0;

	/* Deinitialize in modules. */
#ifdef WITH_LIMITS
	STAILQ_FOREACH(limit, &rule->limits, link) {
		wpid_hash_rem(&limit->wpid);
# ifdef WITH_SUBLIMITS
		STAILQ_FOREACH(sublimit, &limit->sublimits, link)
			wpid_hash_rem(&sublimit->wpid);
# endif
		if (db_deinit_limit(rule, limit) < 0)
			rv = -1;
	}
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
	STAILQ_FOREACH(threshold, &rule->thresholds, link) {
		wpid_hash_rem(&threshold->wpid);
		if (db_deinit_threshold(rule, threshold) < 0)
			rv = -1;
	}
#endif

	if (ac_deinit_rule(rule) < 0)
		rv = -1;
	if (db_deinit_rule(rule) < 0)
		rv = -1;

	/* Decrease ref_count of accounting systems if rule is active. */
	if (RULE_IS_ACTIVE(rule))
		if (ac_dec_ref_count(rule->ac_list) < 0) {
			logmsgx(IPA_LOG_ERR, "rule %s: deinit_rule: "
			    "ac_dec_ref_count failed", rule->name);
			rv = -1;
		}

	if (rv != 0)
		logbt("deinit_rule");
	return (rv);
}

static int
deinit_stat_rule(struct rule *rule)
{
#ifdef WITH_AUTORULES
	if (rules_ptr_marray != NULL)
		/* Mark rule as unused. */
		marray_free(rules_ptr_marray, rule->no);
#endif

	if (deinit_rule(rule) < 0) {
		logbt("deinit_stat_rule");
		return (-1);
	}

	return (0);
}

/*
 * Deinitialize all static and dynamic rules.
 */
int
deinit_rules(void)
{
	struct rule *rule;
	int rv;

	rv = 0;
	TAILQ_FOREACH(rule, &rules_list, list) {
#ifdef WITH_AUTORULES
		if (RULE_IS_DYNAMIC(rule)) {
			if (deinit_dyn_rule(rule) < 0)
				rv = -1;
		} else
#endif
		{
			if (deinit_stat_rule(rule) < 0)
				rv = -1;
		}
	}

	if (rv != 0)
		logbt("deinit_rules");
	return (rv);
}

/*
 * Return name hash value for the given name;
 */
static unsigned int
rule_name_hash(const char *name)
{
	unsigned int value;

	for (value = 0; *name != '\0'; ++name)
		value += (unsigned char)*name;
	return (value);
}

#define rule_hash_bucket(x) ((x) & (RULES_HASH_BUCKETS - 1))

/*
 * Add rule to rules_hash.
 */
void
rules_hash_add(struct rule *rule)
{
	rule->name_hash = rule_name_hash(rule->name);
	LIST_INSERT_HEAD(&rules_hash[rule_hash_bucket(rule->name_hash)],
	    rule, hlink);
}

/*
 * Remove rule from rules_hash.
 */
void
rules_hash_rem(struct rule *rule)
{
	LIST_REMOVE(rule, hlink);
}

/*
 * Initialize rules_hash.
 */
void
rules_hash_init(void)
{
	struct rules_hash *hash;

	for (hash = rules_hash; hash < rules_hash + RULES_HASH_BUCKETS; ++hash)
		LIST_INIT(hash);
}

/*
 * Return non-zero value if there are no entries in rules_hash table.
 */
int
rules_hash_is_empty(void)
{
	const struct rules_hash *hash;

	for (hash = rules_hash; hash < rules_hash + RULES_HASH_BUCKETS; ++hash)
		if (!LIST_EMPTY(hash))
			return (0);
	return (1);
}

/*
 * Return pointer to rule with the given name.
 */
struct rule *
rule_by_name(const char *name)
{
	const struct rules_hash *hash;
	struct rule *rule;
	unsigned int name_hash;

	name_hash = rule_name_hash(name);
	hash = &rules_hash[rule_hash_bucket(name_hash)];

	LIST_FOREACH(rule, hash, hlink)
		if (rule->name_hash == name_hash &&
		    strcmp(rule->name, name) == 0)
			break;

	return (rule);
}

/*
 * Set check_sec for all active rules to 0, so they
 * will be checked immediately.
 */
void
set_rules_for_check(void)
{
	struct rule *rule;

	TAILQ_FOREACH(rule, &rules_active, queue)
		rule->check_sec = 0;
}

/*
 * Release memory used by one rulepat.
 */
static void
free_rulepat(struct rulepat *rulepat)
{
	mem_free(rulepat->pat, m_parser);
	regfree(&rulepat->re);
	cmds_rule_free(&rulepat->rc[RC_STARTUP]);
	cmds_rule_free(&rulepat->rc[RC_SHUTDOWN]);
#ifdef WITH_LIMITS
	free_limits(RULE_FLAG_FREE_LIMITS, &rulepat->limits, 0);
#endif
#ifdef WITH_THRESHOLDS
	free_thresholds(RULE_FLAG_FREE_THRESHOLDS, &rulepat->thresholds, 0);
#endif
}

/*
 * Release memory used by all rulepats.
 */
void
free_rulepats(void)
{
	struct rulepat *rulepat;

	STAILQ_FOREACH(rulepat, &rulepats_list, link)
		free_rulepat(rulepat);
	mzone_deinit(rulepat_mzone);
}

void
rule_init_cmds(struct rule *rule)
{
	cmds_rule_init(&rule->rc[RC_STARTUP]);
	cmds_rule_init(&rule->rc[RC_SHUTDOWN]);
}

/*
 * Inherit settings from rulepat{} from modules for rule{}.
 */
int
mod_rule_inherit(const struct rulepat *rulepat, const struct rule *rule)
{
	const struct ac_mod *ac_mod;
	const struct db_mod *db_mod;

	SLIST_FOREACH(ac_mod, &ac_mod_list, link)
		if (ac_mod->ipa_ac_mod->conf_inherit != NULL &&
		    ac_mod->ipa_ac_mod->conf_inherit(rulepat->no, rule->no,
		    rule->name) < 0) {
			xlogmsgx(IPA_LOG_ERR, "module %s: conf_inherit"
			    "(%s, %s) failed", ac_mod->mod_file,
			    parser_stringify(rulepat->pat), rule->name);
			return (-1);
		}

	SLIST_FOREACH(db_mod, &db_mod_list, link)
		if (db_mod->ipa_db_mod->conf_inherit != NULL &&
		    db_mod->ipa_db_mod->conf_inherit(rulepat->no, rule->no,
		    rule->name) < 0) {
			xlogmsgx(IPA_LOG_ERR, "module %s: conf_inherit"
			    "(%s, %s) failed", db_mod->mod_file,
			    parser_stringify(rulepat->pat), rule->name);
			return (-1);
		}

	return (0);
}

/*
 * Inherit settings from rulepat{} in rule{}.
 */
static int
rule_inherit_rulepat(const struct rulepat *rulepat, struct rule *rule)
{
	unsigned int x;

	if (rule->update_tevent == NULL)
		rule->update_tevent = rulepat->update_tevent;
	if (rule->append_tevent == NULL)
		rule->append_tevent = rulepat->append_tevent;
	if (rule->worktime == NULL)
		rule->worktime = rulepat->worktime;
	if (rule->ac_list == NULL) {
		rule->ac_list = rulepat->ac_list;
		if (rule->ac_list != NULL)
			ac_inc_ref_count(rule->ac_list);
	}
	if (rule->db_list == NULL)
		rule->db_list = rulepat->db_list;
	if (rule->debug_exec < 0)
		rule->debug_exec = rulepat->debug_exec;
#ifdef CTL_CHECK_CREDS
	if (rule->ctl_rule_acl == NULL)
		rule->ctl_rule_acl = rulepat->ctl_rule_acl;
# endif
	for (x = 0; x < 2; ++x) {
		if (rule->rc[x].cmds.sect_set || !rulepat->rc[x].cmds.sect_set)
			continue;
		if (cmds_rule_copy(rule, &rule->rc[x], &rulepat->rc[x]) < 0) {
			xlogmsgx(IPA_LOG_ERR, "rule %s: cannot copy all "
			    "commands from rulepat %s { %s {}}", rule->name,
			    parser_stringify(rulepat->pat), rc_sect_name[x]);
			return (-1);
		}
	}
#ifdef WITH_LIMITS
	if (rule->debug_limit < 0)
		rule->debug_limit = rulepat->debug_limit;
	if (rule->debug_limit_init < 0)
		rule->debug_limit_init = rulepat->debug_limit_init;
	if (STAILQ_EMPTY(&rule->limits) &&
	    !STAILQ_EMPTY(&rulepat->limits)) {
		if (copy_limits(rule, &rulepat->limits) < 0) {
			xlogmsgx(IPA_LOG_ERR, "rule %s: cannot copy all "
			    "limits from rulepat %s {}", rule->name,
			    parser_stringify(rulepat->pat));
			return (-1);
		}
		rule->rule_flags &= ~RULE_FLAG_FREE_LIMITS;
	}
#endif
#ifdef WITH_THRESHOLDS
	if (rule->debug_threshold < 0)
		rule->debug_threshold = rulepat->debug_threshold;
	if (rule->debug_threshold_init < 0)
		rule->debug_threshold_init = rulepat->debug_threshold_init;
	if (STAILQ_EMPTY(&rule->thresholds) &&
	    !STAILQ_EMPTY(&rulepat->thresholds)) {
		if (copy_thresholds(rule, &rulepat->thresholds) < 0) {
			xlogmsgx(IPA_LOG_ERR, "rule %s: cannot copy all "
			    "thresholds from rulepat %s {}", rule->name,
			    parser_stringify(rulepat->pat));
			return (-1);
		}
		rule->rule_flags &= ~RULE_FLAG_FREE_THRESHOLDS;
	}
#endif
	if (mod_rule_inherit(rulepat, rule) < 0) {
		xlogmsgx(IPA_LOG_ERR, "rule %s: cannot inherit settings "
		    "from rulepat %s {} from some module", rule->name,
		    parser_stringify(rulepat->pat));
		return (-1);
	}
	return (0);
}

/*
 * Inherit settings from all rulepats in rule{}.
 * Return:
 *   0 -- settings were copied, but matched rulepat had check_next == 0;
 *   1 -- settings were copied and matched rulepat had check_next == 1;
 *  -1 -- some error occurred.
 */
static int
rule_inherit_rulepats(struct rule *rule)
{
	const struct rulepat *rulepat;

	STAILQ_FOREACH(rulepat, &rulepats_list, link) {
		if (regexec_simple(&rulepat->re, rule->name) != 0)
			continue;
		if (rule_inherit_rulepat(rulepat, rule) < 0)
			return (-1);
		if (rulepat->check_next == 0)
			return (1);
	}
	return (0);
}

/*
 * Inherit settings from global{} in rule{}.
 */
static void
rule_inherit_global(struct rule *rule)
{
	if (rule->update_tevent == NULL)
		rule->update_tevent = global_update_tevent;
	if (rule->append_tevent == NULL)
		rule->append_tevent = global_append_tevent;
	if (rule->worktime == NULL)
		rule->worktime = global_worktime;
	if (rule->ac_list == NULL) {
		rule->ac_list = global_ac_list;
		ac_inc_ref_count(rule->ac_list);
	}
	if (rule->db_list == NULL)
		rule->db_list = global_db_list;
	if (rule->debug_exec < 0)
		rule->debug_exec = global_debug_exec;
#ifdef WITH_LIMITS
	if (rule->debug_limit < 0)
		rule->debug_limit = global_debug_limit;
	if (rule->debug_limit_init < 0)
		rule->debug_limit_init = global_debug_limit_init;
#endif
#ifdef WITH_THRESHOLDS
	if (rule->debug_threshold < 0)
		rule->debug_threshold = global_debug_threshold;
	if (rule->debug_threshold_init < 0)
		rule->debug_threshold_init = global_debug_threshold_init;
#endif
#ifdef CTL_CHECK_CREDS
	if (rule->ctl_rule_acl == NULL)
		rule->ctl_rule_acl = global_ctl_rule_acl;
#endif
}

/*
 * Inherit settings for a rule from a matched rule pattern,
 * then from global{}.  Make final tests of rule's settings.
 */
int
rule_inherit(struct rule *rule)
{
#ifdef WITH_LIMITS
	struct limit *limit;
#endif
#ifdef WITH_THRESHOLDS
	struct threshold *threshold;
#endif

	switch (rule_inherit_rulepats(rule)) {
	case 0:
		rule_inherit_global(rule);
		break;
	case -1:
		xlogmsgx(IPA_LOG_ERR, "rule %s: cannot inherit settings "
		    "from rule patterns", rule->name);
		return (-1);
	}

	cmds_rule_set_sync(&rule->rc[RC_STARTUP]);
	cmds_rule_set_sync(&rule->rc[RC_SHUTDOWN]);

#ifdef WITH_LIMITS
	STAILQ_FOREACH(limit, &rule->limits, link) {
		if (check_worktime_subset(rule->worktime,
		    limit->worktime) < 0) {
			xlogmsgx(IPA_LOG_ERR, "rule %s, limit %s: limit's "
			    "worktime must be a subset of rule's worktime",
			    rule->name, limit->name);
			return (-1);
		}
		if (rule->rule_flags & RULE_FLAG_FREE_LIMITS)
			limit_inherit(limit);
		if (limit->db_list == NULL)
			limit->db_list = rule->db_list;
		if (limit->worktime == NULL)
			limit->worktime = rule->worktime;
	}
#endif
#ifdef WITH_THRESHOLDS
	STAILQ_FOREACH(threshold, &rule->thresholds, link) {
		if (check_worktime_subset(rule->worktime,
		    threshold->worktime) < 0) {
			xlogmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
			    "threshold's worktime must be a subset of rule's "
			    "worktime", rule->name, threshold->name);
			return (-1);
		}
		if (rule->rule_flags & RULE_FLAG_FREE_THRESHOLDS)
			threshold_inherit(threshold);
		if (threshold->db_list == NULL)
			threshold->db_list = rule->db_list;
		if (threshold->worktime == NULL)
			threshold->worktime = rule->worktime;
	}
#endif
	return (0);
}

#ifdef WITH_RULES
int
rules_inherit(void)
{
	struct rule *rule;

	TAILQ_FOREACH(rule, &rules_list, list)
		if (rule_inherit(rule) < 0)
			return (-1);
	return (0);
}
#endif

/*
 * A rule pattern inherits settings from global{}, only if its
 * "check_next_rulepat" parameter is ``no'', since it is the
 * last rule pattern in the search for the matched rule.
 */
static void
rulepat_inherit_global(struct rulepat *rulepat)
{
	rulepat->check_next = 0;
	if (rulepat->update_tevent == NULL)
		rulepat->update_tevent = global_update_tevent;
	if (rulepat->append_tevent == NULL)
		rulepat->append_tevent = global_append_tevent;
	if (rulepat->worktime == NULL)
		rulepat->worktime = global_worktime;
	if (rulepat->ac_list == NULL)
		rulepat->ac_list = global_ac_list;
	if (rulepat->db_list == NULL)
		rulepat->db_list = global_db_list;
	if (rulepat->debug_exec < 0)
		rulepat->debug_exec = global_debug_exec;
#ifdef WITH_LIMITS
	if (rulepat->debug_limit < 0)
		rulepat->debug_limit = global_debug_limit;
	if (rulepat->debug_limit_init < 0)
		rulepat->debug_limit_init = global_debug_limit_init;
#endif
#ifdef WITH_THRESHOLDS
	if (rulepat->debug_threshold < 0)
		rulepat->debug_threshold = global_debug_threshold;
	if (rulepat->debug_threshold_init < 0)
		rulepat->debug_threshold_init = global_debug_threshold_init;
#endif
#ifdef CTL_CHECK_CREDS
	if (rulepat->ctl_rule_acl == NULL)
		rulepat->ctl_rule_acl = global_ctl_rule_acl;
#endif
}

/*
 * Set default settings and inherit settings from global{} in rulepat{}.
 */
void
rulepats_inherit(void)
{
	struct rulepat *rulepat;
#ifdef WITH_LIMITS
	struct limit *limit;
#endif
#ifdef WITH_THRESHOLDS
	struct threshold *threshold;
#endif

	STAILQ_FOREACH(rulepat, &rulepats_list, link) {
		if (rulepat->check_next <= 0)
			rulepat_inherit_global(rulepat);
		cmds_rule_set_sync(&rulepat->rc[RC_STARTUP]);
		cmds_rule_set_sync(&rulepat->rc[RC_SHUTDOWN]);
#ifdef WITH_LIMITS
		STAILQ_FOREACH(limit, &rulepat->limits, link)
			limit_inherit(limit);
#endif
#ifdef WITH_THRESHOLDS
		STAILQ_FOREACH(threshold, &rulepat->thresholds, link)
			threshold_inherit(threshold);
#endif
	}
}
