package netfilter

import (
	"testing"

	"github.com/stretchr/testify/assert"

	"github.com/mdlayher/netlink"
)

func TestHeaderMarshalTwoWay(t *testing.T) {
	refHdr := Header{
		SubsystemID: SubsystemID(NFSubsysCTNetlinkTimeout),
		MessageType: MessageType(123),

		Family:     255,
		Version:    1,
		ResourceID: 2,
	}

	refMsg := netlink.Message{Header: netlink.Header{Type: 0x087B}, Data: []byte{255, 1, 0, 2}}

	var gotHdr Header
	assert.Nil(t, gotHdr.unmarshal(refMsg))
	assert.Equal(t, refHdr, gotHdr, "unexpected unmarshal")

	gotMsg := netlink.Message{Data: make([]byte, 4)}
	assert.Nil(t, gotHdr.marshal(&gotMsg))
	assert.Equal(t, refMsg, gotMsg, "unexpected marshal")

	// unmarshal error
	assert.Equal(t, errMessageLen, gotHdr.unmarshal(netlink.Message{}))

	// marshal error
	assert.Equal(t, errMessageLen, gotHdr.marshal(&netlink.Message{}))
}

func TestHeaderString(t *testing.T) {
	ht := Header{
		SubsystemID: NFSubsysIPSet,
		MessageType: 123,
	}

	want := "<Subsystem: NFSubsysIPSet, Message Type: 123, Family: ProtoUnspec, Version: 0, ResourceID: 0>"

	assert.Equal(t, want, ht.String())
}

func TestProtoFamilyString(t *testing.T) {
	if ProtoFamily(255).String() == "" {
		t.Fatal("ProtoFamily string representation empty - did you run `go generate`?")
	}

	pf := map[ProtoFamily]string{
		ProtoUnspec: "ProtoUnspec",
		ProtoInet:   "ProtoInet",
		ProtoIPv4:   "ProtoIPv4",
		ProtoARP:    "ProtoARP",
		ProtoNetDev: "ProtoNetDev",
		ProtoBridge: "ProtoBridge",
		ProtoIPv6:   "ProtoIPv6",
		ProtoDECNet: "ProtoDECNet",
	}

	for k, v := range pf {
		assert.Equal(t, k.String(), v)
	}
}

func TestSubsystemIDString(t *testing.T) {

	if SubsystemID(255).String() == "" {
		t.Fatal("SubsystemID string representation empty - did you run `go generate`?")
	}
}
