// Package util contains various utilities utilized for rq
package util

import (
	"encoding/json"
	"errors"
	"fmt"
	"reflect"
	"regexp"
	"sort"
	"strconv"
	"strings"
	"text/template"
)

// jsonNumberRegex matches valid JSON numbers, taken from here:
// https://stackoverflow.com/a/13340826
var jsonNumberRegex *regexp.Regexp = regexp.MustCompile(`^-?(?:0|[1-9]\d*)(?:\.\d+)?(?:[eE][+-]?\d+)?$`)

func IsValidJSONNumber(s string) bool {
	// avoid dealing with multiline regex matching shenanigans
	if strings.ContainsAny(s, "\n") {
		return false
	}

	return jsonNumberRegex.Match([]byte(s))
}

// TemplateFakeOptions stores the fake options used with GetTemplateFuncs().
var TemplateFakeOptions FakeOptions = GetDefaultFakeOptions()

// Unescape unescapes escapes escape codes. I spent at least 15 minutes googling how to
// do this and couldn't find anything that does, or even anyone else who wanted
// to do this. I find this pretty shocking and I'm sure there's a better
// version of this out there somewhere. I also tried using Sprintf(), and that
// didn't work either.
//
// This unescapes the following sequences: \a, \b, \e, \f, \n, \r, \t, \v, \\,
// \', and \"
func Unescape(s string) (string, error) {
	builder := strings.Builder{}

	var i int = 0
	for {
		if i >= len(s) {
			return builder.String(), nil
		}

		var cursor rune

		cursor = []rune(s)[i]
		i++

		if cursor != '\\' {
			builder.WriteRune(cursor)
			continue
		}

		if i >= len(s) {
			return builder.String(), nil
		}

		cursor = []rune(s)[i]

		switch {
		case cursor == 'a':
			builder.WriteRune(0x7)
		case cursor == 'b':
			builder.WriteRune(0x8)
		case cursor == 'e':
			builder.WriteRune(0x1b)
		case cursor == 'f':
			builder.WriteRune(0x0c)
		case cursor == 'n':
			builder.WriteRune(0x0a)
		case cursor == 'r':
			builder.WriteRune(0x0d)
		case cursor == 't':
			builder.WriteRune(0x09)
		case cursor == 'v':
			builder.WriteRune(0x0b)
		case cursor == '\\':
			builder.WriteRune(0x5c)
		case cursor == '\'':
			builder.WriteRune(0x27)
		case cursor == '"':
			builder.WriteRune(0x22)
		case cursor == '?':
			builder.WriteRune(0x3f)
		default:
			return "", fmt.Errorf("unknown escape code \\%c", cursor)
		}

		i++

	}
}

// StringToValue attempts to convert the given string into a value. This
// supports booleans, integers, and floats. The following values are considered
// booleans, case-insensitive: true, yes, false, no.
//
// Integer radix is inferred by prefix, supporting 0b, 0o, or 0x (per
// strconv.ParseInt()).
func StringToValue(s string) interface{} {
	l := strings.ToLower(s)

	// First check if the value is a boolean
	switch {
	case l == "true":
		return true
	case l == "yes":
		return true
	case l == "false":
		return false
	case l == "no":
		return false
	}

	// Next, check if the value is a valid integer
	i, err := strconv.ParseInt(s, 0, 0)
	if err == nil {
		return int(i)
	}

	// Finally, check if it's a float
	f, err := strconv.ParseFloat(s, 64)
	if err == nil {
		return f
	}

	// We couldn't infer a type, so we just keep it as a string.
	return s
}

// ValueToString can be used to convert primitive types to strings.
//
// NOTE: because int32 aliases with rune, I have chosen to support rune out of
// the two. Thus any int32 values will treated as runes. A workaround for this
// behavior is to case them to int64 first.
func ValueToString(v interface{}) string {
	// TODO: Sprintf() for float types is probably not ideal, but we want
	// to avoid inserting E in where it is not needed just for human
	// readability.

	switch v.(type) {
	case int:
		return strconv.FormatInt(int64(v.(int)), 10)
	case int8:
		return strconv.FormatInt(int64(v.(int8)), 10)
	case int16:
		return strconv.FormatInt(int64(v.(int16)), 10)
	case int64:
		return strconv.FormatInt(int64(v.(int64)), 10)
	case uint:
		return strconv.FormatInt(int64(v.(uint)), 10)
	case uint8:
		return strconv.FormatInt(int64(v.(uint8)), 10)
	case uint16:
		return strconv.FormatInt(int64(v.(uint16)), 10)
	case uint32:
		return strconv.FormatInt(int64(v.(uint32)), 10)
	case uint64:
		return strconv.FormatInt(int64(v.(uint64)), 10)
	case float32:
		// We don't want to introduce extra trailing zeros either.
		s := fmt.Sprintf("%f", v.(float32))
		s = strings.TrimRight(s, "0")
		s = strings.TrimRight(s, ".")
		return s
	case float64:
		s := fmt.Sprintf("%f", v.(float64))
		s = strings.TrimRight(s, "0")
		s = strings.TrimRight(s, ".")
		return s
	case string:
		return v.(string)
	case bool:
		if v.(bool) {
			return "true"
		}
		return "false"
	case rune:
		return string(v.(rune))
	default:
		result, err := json.Marshal(v)
		if err != nil {
			return fmt.Sprintf("%v", v)
		}
		return string(result)
	}
}

// IsPrimitive returns true if and only if the type of `v` is primitive.
func IsPrimitive(v interface{}) bool {
	switch v.(type) {
	case int:
		return true
	case int8:
		return true
	case int16:
		return true
	case int64:
		return true
	case uint:
		return true
	case uint8:
		return true
	case uint16:
		return true
	case uint32:
		return true
	case uint64:
		return true
	case float32:
		return true
	case float64:
		return true
	case string:
		return true
	case bool:
		return true
	case rune:
		return true
	}

	return false
}

// Truthy is the canonical way of determining if a string value is "truthy" in
// rq.
//
// Truthy values are: 1, true, t, yes, y
//
// Whitespace and case are ignored. All other values are considered falsey.
//
// This method should NOT be used for deserializing string values into
// booleans, StringToValue() should be used instead. This method is appropriate
// for user-supplied values that need to be interpreted as booleans.
func Truthy(s string) bool {
	s = strings.ToLower(s)
	s = strings.Trim(s, " \t\r\n")

	switch s {
	case "1":
		return true
	case "true":
		return true
	case "t":
		return true
	case "yes":
		return true
	case "y":
		return true
	default:
		return false
	}
}

// tabularizeTable uses reflection to convert the given interface to a table.
// It assumes that the interface is already a slice of slices.
func tabularizeTable(v interface{}) ([][]string, error) {
	rv := reflect.ValueOf(v)
	if rv.Kind() != reflect.Slice {
		return nil, fmt.Errorf("don't know how to tabularize value of type '%T'", v)
	}

	result := make([][]string, rv.Len())

	for i := 0; i < rv.Len(); i++ {
		row := reflect.ValueOf(rv.Index(i).Interface())

		if row.Kind() == reflect.Slice {
			result[i] = make([]string, row.Len())
			for j := 0; j < row.Len(); j++ {
				result[i][j] = ValueToString(row.Index(j).Interface())
			}
		} else {
			result[i] = []string{ValueToString(row)}
		}
	}

	return result, nil
}

// MapPath represents a path within a nested collection of maps.
type MapPath struct {
	path []string
}

// Child creates a new map path with a single new path element appended.
func (p MapPath) Child(elem string) MapPath {
	n := MapPath{path: make([]string, len(p.path)+1)}
	for i, v := range p.path {
		n.path[i] = v
	}
	n.path[len(n.path)-1] = elem
	return n
}

// Suffix creates a new map path containing all but the rightmost element of p.
func (p MapPath) Suffix() MapPath {
	if len(p.path) == 0 {
		return MapPath{}
	}

	n := MapPath{path: make([]string, len(p.path)-1)}
	for i := range n.path {
		n.path[i] = p.path[i+1]
	}
	return n
}

// String generates a canonical string representation of p, suitable for use
// with ParseMapPath().
func (p MapPath) String() string {
	// Escape instances of the . delimiter in any path elements.
	temp := make([]string, len(p.path))
	for i, v := range p.path {
		temp[i] = strings.ReplaceAll(v, ".", "\\.")
	}

	return strings.Join(temp, ".")
}

// Slice returns a slice representation of the MapPath elements.
func (p MapPath) Slice() []string {
	return p.path
}

// Access attempts to retrieve the value from v at the path specified.
func (p MapPath) Access(v interface{}) (interface{}, bool) {
	if len(p.path) == 0 {
		return v, true
	}

	rv := reflect.ValueOf(v)
	kind := rv.Kind()
	if kind == reflect.Map {
		// If v is a map, then we try to access the key corresponding
		// to the leftmost path element of p.

		rc := rv.MapIndex(reflect.ValueOf(p.path[0]))
		if !reflect.Value.IsValid(rc) || rc.IsZero() {
			return nil, false
		}
		if len(p.path) == 1 {
			return rc.Interface(), true
		}

		// There are still remaining elements in p that we haven't
		// resolved.
		return p.Suffix().Access(rc.Interface())

	} else if (kind == reflect.Array) || (kind == reflect.Slice) {
		// If v is an array, then we try to cast the leftmost element
		// of p to an integer and use that as an index.
		i, err := strconv.Atoi(p.path[0])
		if err != nil {
			return nil, false
		}

		if (i < 0) || (i >= rv.Len()) {
			return nil, false
		}

		rc := rv.Index(i)
		if len(p.path) == 1 {
			return rc.Interface(), true
		}
		return p.Suffix().Access(rc.Interface())

	}

	return nil, false
}

// SortMapPaths sorts a slice of MapPath objects in-place. If one path is
// shorter and the shared elements are identical, then the longer path collates
// last.
func SortMapPaths(paths []MapPath) {
	sort.Slice(paths, func(i, j int) bool {
		min := len(paths[i].path)
		if len(paths[j].path) < min {
			min = len(paths[j].path)
		}

		for k := 0; k < min; k++ {
			if paths[i].path[k] != paths[j].path[k] {
				return paths[i].path[k] < paths[j].path[k]
			}
		}

		return len(paths[i].path) < len(paths[j].path)
	})
}

// ParseMapPath parses a string to a new MapPath object. It is suitable for
// use on the output of MapPath.String().
func ParseMapPath(s string) MapPath {
	p := MapPath{path: make([]string, 0)}

	elem := ""
	cursor := 0
	N := 0
	escape := false
	r := []rune(s)
	for {
		if cursor >= len(r) {
			break
		}

		c := r[cursor]
		if escape {
			elem += string(c)
			escape = false
		} else if c == '\\' {
			escape = true
		} else if c == '.' {
			p.path = append(p.path, elem)
			elem = ""
			N++
		} else {
			elem += string(c)
		}

		cursor++
	}

	if elem != "" {
		p.path = append(p.path, elem)
	}

	return p
}

// Keyspace returns one MapPath object for every value recursively nested under
// the map v. All generate MapPath objects are created by calling .Child() on
// prefix.
func Keyspace(prefix MapPath, v interface{}) []MapPath {
	rv := reflect.ValueOf(v)
	if rv.Kind() != reflect.Map {
		return nil
	}

	if prefix.path == nil {
		prefix = MapPath{path: make([]string, 0)}
	}

	paths := make([]MapPath, 0)

	for _, k := range rv.MapKeys() {
		mv := reflect.ValueOf(rv.MapIndex(k).Interface())
		ks := ValueToString(k.Interface())
		kind := mv.Kind()

		if kind == reflect.Map {
			// If the value stored under the current key is another map, we
			// recurse. We don't need an explicit termination condition
			// since if there are no child paths, the returned slice will
			// be empty and the append will be a NO-OP.
			paths = append(paths, Keyspace(prefix.Child(ks), mv.Interface())...)

		} else if (kind == reflect.Array) || (kind == reflect.Slice) {
			// If the value is an array, create a path for every index in
			// it.
			lp := prefix.Child(ks)
			for i := 0; i < mv.Len(); i++ {
				paths = append(paths, lp.Child(strconv.Itoa(i)))
			}
		} else {
			// If the value is a scalar, we can just add the key
			// and be done.
			paths = append(paths, prefix.Child(ValueToString(ks)))
		}
	}

	return paths
}

// tabularizeMaps uses reflection to convert a slice of maps to a table,
// with the first row being the keyspace of the maps.
func tabularizeMaps(v interface{}, headers bool) ([][]string, error) {

	rv := reflect.ValueOf(v)
	if (rv.Kind() != reflect.Array) && (rv.Kind() != reflect.Slice) {
		return nil, fmt.Errorf("don't know how to tabularize value of type '%T' which is not an array or slice", v)
	}

	// First we have to determine the keyspace so we know what columns to
	// create.
	keyspaceSet := make(map[string]interface{})
	for i := 0; i < rv.Len(); i++ {
		wv := rv.Index(i)
		for _, p := range Keyspace(MapPath{}, wv.Interface()) {
			keyspaceSet[p.String()] = true
		}
	}

	keyspace := make([]MapPath, 0)
	for k := range keyspaceSet {
		keyspace = append(keyspace, ParseMapPath(k))
	}
	SortMapPaths(keyspace)

	// Now that we know the keyspace, we simply iterate over every row, and
	// use MapPath.Access() to attempt to retrieve the pertinent value for
	// each column.
	result := make([][]string, rv.Len()+1)
	result[0] = make([]string, len(keyspace))
	for i, k := range keyspace {
		result[0][i] = k.String()
	}

	for i := 0; i < rv.Len(); i++ {
		wv := rv.Index(i)
		result[i+1] = make([]string, len(keyspace))
		for j := 0; j < len(keyspace); j++ {
			s, ok := keyspace[j].Access(wv.Interface())
			if ok {
				result[i+1][j] = ValueToString(s)
			}
		}
	}

	if !headers {
		return result[1:], nil
	}
	return result, nil

}

// Tabularize will attempt to convert the given value to a table of strings.
//
// If the value is a list of maps, then each map key will become a column,
// and each map will become one row. If any of the map elements is a slice,
// then each index in the slice will be a separate column. If any map element
// contains another map, then each key in that map will be a column as well.
//
// If the value is a list of lists, then the list elements will simply be
// converted to strings.
//
// If the value is a primitive type, then the resulting table will have just a
// single cell being the stringification of that value.
//
// Any other shape or type will result in an error.
//
// If headers is false, then the header row will not be generated for the
// slice-of-maps case.
func Tabularize(v interface{}, headers bool) ([][]string, error) {
	// Case where v is primitive.
	if IsPrimitive(v) {
		return [][]string{[]string{ValueToString(v)}}, nil
	}

	// Case where rv is a slice, could be a slice of slices or a slice of
	// maps. We need to detect which one. We assume that map valued are
	// intended to be scalars UNLESS every row is a map.
	rv := reflect.ValueOf(v)
	rowsAreMaps := true
	if rv.Kind() == reflect.Slice {
		for i := 0; i < rv.Len(); i++ {
			row := reflect.ValueOf(rv.Index(i).Interface())
			if row.Kind() != reflect.Map {
				rowsAreMaps = false
				break
			}
		}
	}

	if rowsAreMaps {
		return tabularizeMaps(v, headers)
	}
	return tabularizeTable(v)
}

const templTypeHint string = "(HINT: consider using 'str' or 'sprintf', e.g. '{{ lower ( str .somekey ) }}')"

func eWrap1[T func(string) string | func(string) (string, error)](name string, f T) func(string) (string, error) {
	if g, ok := any(f).(func(string) string); ok {
		return func(s string) (string, error) {
			return g(s), nil
		}
	}

	if g, ok := any(f).(func(string) (string, error)); ok {
		return func(s string) (string, error) {
			res, err := g(s)
			if err != nil {
				return "", fmt.Errorf("template function '%s' failed: %w", name, err)
			}
			return res, nil
		}
	}

	panic("eWrap1 called with invalid type signature (this should be unreachable")
}

func eWrap3[T func(string, string, string) string | func(string, string, string) (string, error)](name string, f T) func(string, string, string) (string, error) {
	if g, ok := any(f).(func(string, string, string) string); ok {
		return func(s, t, u string) (string, error) {
			return g(s, t, u), nil
		}
	}

	if g, ok := any(f).(func(string, string, string) (string, error)); ok {
		return func(s, t, u string) (string, error) {
			res, err := g(s, t, u)
			if err != nil {
				return "", fmt.Errorf("template function '%s' failed: %w", name, err)
			}
			return res, nil
		}
	}

	panic("eWrap3 called with invalid type signature (this should be unreachable")
}

// wrapper for template functions that take 1 string argument, provides more
// helpful error messages than the default template library
func mustString1[T func(string) string | func(string) (string, error)](name string, f T) func(any) (string, error) {
	g := eWrap1(name, f)
	return func(v any) (string, error) {
		if s, ok := v.(string); ok {
			return g(s)
		}
		return "", fmt.Errorf("template function '%s' requires a string argument, not %T %s", name, v, templTypeHint)
	}
}

func mustString2(name string, f func(string, string) string) func(any, any) (string, error) {
	g := func(v, w, x string) string {
		return f(v, w)
	}

	h := mustString3(name, g)

	return func(v, w any) (string, error) {
		return h(v, w, "")
	}
}

func mustString3[T func(string, string, string) string | func(string, string, string) (string, error)](name string, f T) func(any, any, any) (string, error) {
	g := eWrap3(name, f)

	return func(v, w, x any) (string, error) {
		vs, vok := v.(string)
		ws, wok := w.(string)
		xs, xok := x.(string)

		errs := []error{}

		if !vok {
			errs = append(errs, fmt.Errorf("argument 0 is '%T', not string", v))
		}

		if !wok {
			errs = append(errs, fmt.Errorf("argument 1 is '%T', not string", w))
		}

		if !xok {
			errs = append(errs, fmt.Errorf("argument 2 is '%T', not string", x))
		}

		if len(errs) > 0 {
			return "", errors.Join(
				append([]error{
					fmt.Errorf("template function '%s' has 1 or more arguments of incorrect types %s", name, templTypeHint),
				},
					errs...)..., // ... really, go fmt?
			)
		}

		return g(vs, ws, xs)
	}
}

// GetTemplateFuncs returns the standard template utility functions used
// wherever string templating happens in rq.
func GetTemplateFuncs() template.FuncMap {
	return map[string]any{
		"lower":   mustString1("lower", strings.ToLower),
		"upper":   mustString1("upper", strings.ToUpper),
		"replace": mustString3("replace", strings.ReplaceAll),

		"reverse": mustString1("reverse", func(s string) string {
			c := []rune(s)
			for i, j := 0, len(c)-1; i < j; i, j = i+1, j-1 {
				c[i], c[j] = c[j], c[i]
			}
			return string(c)
		}),

		"trim":        mustString2("trim", strings.Trim),
		"trim_left":   mustString2("trim_left", strings.TrimLeft),
		"trim_right":  mustString2("trim_right", strings.TrimRight),
		"trim_space":  mustString1("trim_space", strings.TrimSpace),
		"trim_suffix": mustString2("trim_suffix", strings.TrimSuffix),
		"trim_prefix": mustString2("trim_prefix", strings.TrimPrefix),

		"repeat": func(v, countI any) (string, error) {
			errs := []error{}
			s, sok := v.(string)
			count, cok := countI.(int)

			if !sok {
				errs = append(errs, fmt.Errorf("template function 'repeat' value must be string, not %T %s", v, templTypeHint))
			}

			if !cok {
				errs = append(errs, fmt.Errorf("template function 'repeat' count must be int, not %T", countI))
			}

			if len(errs) > 0 {
				return "", errors.Join(errs...)
			}

			return strings.Repeat(s, count), nil
		},

		"quote":   mustString1("quote", strconv.Quote),
		"unquote": mustString1("unquote", strconv.Unquote),

		"sprintf": func(formatI any, arg ...interface{}) (string, error) {
			format, ok := formatI.(string)
			if !ok {
				return "", fmt.Errorf("template function 'sprintf' format string must be string, not %T %s", formatI, templTypeHint)
			}
			return fmt.Sprintf(format, arg...), nil
		},

		"fake": mustString1("fake", func(kind string) string {
			v, err := TemplateFakeOptions.Fake(kind)
			if err != nil {
				return fmt.Sprintf("error creating fake value of kind '%s': %s\n", kind, err.Error())
			}
			return ValueToString(v)
		}),

		"resub": mustString3("resub", func(s, pattern, replacement string) (string, error) {
			re, err := regexp.Compile(pattern)
			if err != nil {
				return "", fmt.Errorf("failed to compile regex: %w", err)
			}

			return re.ReplaceAllString(s, replacement), nil
		}),

		"str": func(v any) string {
			return ValueToString(v)
		},
	}
}
