#!/bin/sh

# This script installs an appropriate rq binary release automatically.
#
# The following environment variables are influential:
#
# RQ_INSTALL_DIR . . . . . Directory where the rq binary should be installed.
#                          (default: /usr/bin)
#
# RQ_INSTALL_VERSION . . . What version of rq should be installed. 'latest' may
#                          be specified to automatically detect the latest
#                          version. (default: latest)
#
# RQ_INSTALL_OS  . . . . . Change the OS the binary should be retrieved for.
#                          'auto' may be specified to automatically detect the
#                          OS. (default: auto).
#
# RQ_INSTALL_ARCH  . . . . Change the CPU architecture the binary should be
#                          retrieved for. 'auto' may be specified to
#                          automatically detect the CPU architecture.
#                          (default: auto)

if [ -z "$RQ_INSTALL_DIR"     ] ; then export RQ_INSTALL_DIR=/usr/local/bin; fi
if [ -z "$RQ_INSTALL_VERSION" ] ; then export RQ_INSTALL_VERSION=latest; fi
if [ -z "$RQ_INSTALL_OS"      ] ; then export RQ_INSTALL_OS=auto; fi
if [ -z "$RQ_INSTALL_ARCH"    ] ; then export RQ_INSTALL_ARCH=auto; fi

RQ_REPO='https://git.sr.ht/~charles/rq'

for dep in git curl grep awk cut chmod gunzip ; do
	if [ ! -x "$(which "$dep")" ] ; then
		echo "FATAL: required tool '$dep' not found in PATH" 1>&2
		exit 1
	fi
done

set -e
set -u

if [ ! -w "$RQ_INSTALL_DIR" ] ; then
	echo "'$RQ_INSTALL_DIR' is not writable, bailing out as installation will fail (did you forget sudo?)" 1>&2
	exit 1
fi

if [ "$RQ_INSTALL_VERSION" = "latest" ] ; then
	export RQ_INSTALL_VERSION="$(git ls-remote --tags "$RQ_REPO" | awk '$2 ~ /[}]$/ {print($2)}' | cut -d/ -f3 | cut -d^ -f1 | grep -v '[-]RC' | sort | tail -n1)"
fi

if [ "$RQ_INSTALL_OS" = "auto" ] ; then
	UNAME="$(uname)"
	if [ "$UNAME" = "Linux" ] ; then
		export RQ_INSTALL_OS=linux
	elif [ "$UNAME" = "Darwin" ] ; then
		export RQ_INSTALL_OS=darwin
	else
		# Currently, we only build for Windows, Linux, and MacOS.
		echo "FATAL: unsupported OS '$UNAME'" 1>&2
		exit 1
	fi
fi

if [ "$RQ_INSTALL_ARCH" = "auto" ] ; then
	if [ "$RQ_INSTALL_OS" = linux ] ; then
		DETECT="$(lscpu | awk '$1 == "Architecture:" {print($2)}')"
		if [ "$DETECT" = i686 ] ; then
			export RQ_INSTALL_ARCH=386
		elif [ "$DETECT" = x86_64 ] ; then
			export RQ_INSTALL_ARCH=amd64
		elif [ "$DETECT" = armv7l ] ; then
			export RQ_INSTALL_ARCH=arm
		elif [ "$DETECT" = aarch64 ] ; then
			export RQ_INSTALL_ARCH=arm64
		else
			echo "FATAL: unsupported CPU architecture '$DETECT' for OS '$RQ_INSTALL_OS'" 1>&2
			exit 1
		fi

	elif [ "$RQ_INSTALL_OS" = darwin ] ; then
		if [ "$(uname -m)" = arm64 ] ; then
			export RQ_INSTALL_ARCH=arm64
		elif [ "$(uname -m)" = x86_64 ] ; then
			export RQ_INSTALL_ARCH=amd64
		else
			echo "FATAL: unsupported CPU architecture '$DETECT' for OS '$RQ_INSTALL_OS'" 1>&2
			exit 1
		fi
	else
		echo "FATAL: don't know how to auto-detect CPU architecture for OS '$RQ_INSTALL_OS'" 1>&2
		exit 1
	fi
fi

cd "$RQ_INSTALL_DIR"

if [ -e ./rq ] ; then
	echo "INFO: detected an existing copy of rq in '$RQ_INSTALL_DIR', renaming it to rq.bak" 1>&2
	mv ./rq ./rq.bak
fi

BARE_VERSION="$(echo "$RQ_INSTALL_VERSION" | sed 's/^v//g')"
BINARY_URL="$RQ_REPO/refs/download/$RQ_INSTALL_VERSION/rq-$RQ_INSTALL_OS-$RQ_INSTALL_ARCH-$BARE_VERSION.gz"
echo "INFO: beginning download from computed URL '$BINARY_URL'" 1>&2

curl -LSsf "$BINARY_URL" > rq.gz
gunzip ./rq.gz
chmod +x rq

echo "INFO: finished installing rq $RQ_INSTALL_VERSION into '$RQ_INSTALL_DIR'" 1>&2
