package io

import (
	"io"

	"git.sr.ht/~charles/rq/util"
	"github.com/alecthomas/chroma/v2/quick"
	"github.com/pelletier/go-toml"
)

func init() {
	registerOutputHandler("toml", func() OutputHandler { return &TOMLOutputHandler{} })
}

// Declare conformance with OutputHandler interface.
var _ OutputHandler = &TOMLOutputHandler{}

// TOMLOutputHandler handles serializing TOML data.
type TOMLOutputHandler struct {
	style       string
	colorize    bool
	initialized bool
}

func (t *TOMLOutputHandler) init() {
	if t.initialized {
		return
	}

	t.colorize = false
	t.style = "native"
	t.initialized = true
}

// Name implements OutputHandler.Name().
func (t *TOMLOutputHandler) Name() string {
	return "toml"
}

// SetOption implements OutputHandler.SetOption().
func (t *TOMLOutputHandler) SetOption(name string, value string) error {
	t.init()

	switch name {
	case "output.colorize":
		t.colorize = util.StringToValue(value).(bool)
	case "output.style":
		t.style = value
	}

	return nil
}

// Format implements OutputHandler.Format()
func (t *TOMLOutputHandler) Format(writer io.Writer, data interface{}) error {
	var err error
	var tomlBytes []byte

	t.init()

	tomlBytes, err = toml.Marshal(data)
	if err != nil {
		return err
	}

	if t.colorize {
		err := quick.Highlight(writer, string(tomlBytes), "toml", "terminal", t.style)
		if err != nil {
			return err
		}
	} else {
		_, err := writer.Write(tomlBytes)
		if err != nil {
			return err
		}
	}

	return nil
}
