package io

import (
	"encoding/json"
	"io"

	"git.sr.ht/~charles/rq/util"
	"github.com/alecthomas/chroma/v2/quick"
	"github.com/gowebpki/jcs"
)

func init() {
	registerOutputHandler("json", func() OutputHandler { return &JSONOutputHandler{} })

	// aliases
	registerOutputHandler("jcs", func() OutputHandler {
		return &JSONOutputHandler{
			canonical:   true,
			initialized: true,
		}
	})
}

// Declare conformance with OutputHandler interface.
var _ OutputHandler = &JSONOutputHandler{}

// JSONOutputHandler handles serializing JSON data.
type JSONOutputHandler struct {
	indent      string
	style       string
	colorize    bool
	pretty      bool
	initialized bool
	canonical   bool
}

func (j *JSONOutputHandler) init() {
	if j.initialized {
		return
	}

	j.colorize = false
	j.pretty = false
	j.indent = "\t"
	j.style = "native"
	j.initialized = true
	j.canonical = false
}

// Name implements OutputHandler.Name().
func (j *JSONOutputHandler) Name() string {
	return "json"
}

// SetOption implements OutputHandler.SetOption().
func (j *JSONOutputHandler) SetOption(name string, value string) error {
	j.init()

	switch name {
	case "output.colorize":
		j.colorize = util.StringToValue(value).(bool)
	case "output.pretty":
		j.pretty = util.StringToValue(value).(bool)
	case "output.indent":
		j.indent = value
	case "output.style":
		j.style = value
	case "json.canonical":
		j.canonical = util.StringToValue(value).(bool)
	}

	return nil
}

func (j *JSONOutputHandler) formatCanonical(writer io.Writer, data any) error {
	jsonBytes, err := json.Marshal(data)
	if err != nil {
		return err
	}

	jcsBytes, err := jcs.Transform(jsonBytes)
	if err != nil {
		return err
	}

	_, err = writer.Write(jcsBytes)
	if err != nil {
		return err
	}

	return nil
}

// Format implements OutputHandler.Format()
func (j *JSONOutputHandler) Format(writer io.Writer, data interface{}) error {
	var err error
	var jsonBytes []byte

	j.init()

	if j.canonical {
		return j.formatCanonical(writer, data)
	}

	if j.pretty {
		jsonBytes, err = json.MarshalIndent(data, "", j.indent)
		if err != nil {
			return err
		}
	} else {
		jsonBytes, err = json.Marshal(data)
		if err != nil {
			return err
		}
	}

	if j.colorize {
		err := quick.Highlight(writer, string(jsonBytes), "json", "terminal", j.style)
		if err != nil {
			return err
		}
	} else {
		_, err := writer.Write(jsonBytes)
		if err != nil {
			return err
		}
	}

	return nil
}
